# $Id: redirect.rb,v 1.3 2004/11/25 05:43:36 toki Exp $

require 'rucy/error'
require 'rucy/document'

include Rucy

class RedirectDocument < Document
  def initialize(location, status=302)
    @location = location
    @status = status
  end

  def publish(script_name, request, response, logger)
    logger.debug("enter document: #{self.class}")
    case (request.method)
    when 'GET', 'HEAD'
      response.status = @status
      response.set_header('Location', @location)
      response.set_header('Content-Type', 'text/plain')
      response.absolute_location(request)
      response.start_body
      response << "Jump to #{@location}.\n" if (request.method != 'HEAD')
    else
      ex = HTTPError.new(405)	# Method Not Allowed
      ex.set_header('Allow', 'GET, HEAD')
      raise ex
    end
    nil
  end
end

class RedirectDocumentBuilder < DocumentBuilder
  def doc_name
    'Redirect'
  end

  def doc_args
    [ [ 'location', :string, nil ],
      [ 'status', :select, [
	  '302 Found',
	  '303 See Other',
	  '301 Moved Permanently',
	  '307 Temporary Redirect'
	]
      ]
    ]
  end

  def new(location, status)
    if (! location || location.empty?) then
      raise 'required location path.'
    end
    case (status)
    when '302 Found'
      code = 302
    when '303 See Other'
      code = 303
    when '301 Moved Permanently'
      code = 301
    when '307 Temporary Redirect'
      code = 307
    else
      raise "unknown redirect status: #{status.inspect}"
    end
    RedirectDocument.new(location, code)
  end
end

class RegexpRedirectDocument < Document
  def initialize(req_path_pattern, replace_location, status=302)
    @req_path_pattern = req_path_pattern
    @replace_location = replace_location
    @status = status
  end

  def publish(script_name, request, response, logger)
    logger.debug("enter document: #{self.class}")
    case (request.method)
    when 'GET', 'HEAD'
      loc_path = request.path.dup
      if (request.query) then
	loc_path << '?' << request.query
      end
      if (loc_path.gsub!(@req_path_pattern, @replace_location)) then
	response.status = @status
	response.set_header('Location', loc_path)
	response.set_header('Content-Type', 'text/plain')
	response.absolute_location(request)
	response.start_body
	response << "Jump to #{loc_path}.\n" if (request.method != 'HEAD')
      else
	raise HTTPError.new(404) # Not Found
      end
    else
      ex = HTTPError.new(405)	# Method Not Allowed
      ex.set_header('Allow', 'GET, HEAD')
      raise ex
    end
    nil
  end
end

class RegexpRedirectDocumentBuilder < DocumentBuilder
  def doc_name
    'RegexpRedirect'
  end

  def doc_args
    [ [ 'request path pattern (regexp)', :regexp, nil ],
      [ %q"replace location (usable \&, \0, \1 ... \9, \`, \', \+)", :string, nil ],
      [ 'status', :select, [
	  '302 Found',
	  '303 See Other',
	  '301 Moved Permanently',
	  '307 Temporary Redirect'
	]
      ]
    ]
  end

  def new(req_path_pattern, replace_location, status)
    if (! req_path_pattern) then
      raise 'required request path pattern.'
    end
    if (! replace_location || replace_location.empty?) then
      raise 'required replace location.'
    end
    case (status)
    when '302 Found'
      code = 302
    when '303 See Other'
      code = 303
    when '301 Moved Permanently'
      code = 301
    when '307 Temporary Redirect'
      code = 307
    else
      raise "unknown redirect status: #{status.inspect}"
    end
    RegexpRedirectDocument.new(req_path_pattern, replace_location, status)
  end
end
