# $Id: wpm.rb,v 1.52 2004/12/03 02:37:30 toki Exp $
# -
# Web Page Maker - web application framework like CGIKit

require 'thread'
require 'weakref'
require 'delegate'
require 'forwardable'

module WPM
  XMLNS_DOC_URI = 'http://www.freedom.ne.jp/toki/ruby/PageMaker'
  XMLNS_MAP_URI = 'http://www.freedom.ne.jp/toki/ruby/PageMaker/Map'
  XMLNS_HTML_URI = 'http://www.w3.org/1999/xhtml'

  # system components
  COMPO_PATH = File.join(File.dirname(__FILE__), 'wpm', 'compo')

  class XMLNamespaceMap
    def initialize
      @stack = Array.new
    end

    def start_element
      @stack.push(Hash.new)
      nil
    end

    def end_element
      @stack.pop or raise 'internal error'
      nil
    end

    def add_ns(prefix, ns_uri)
      ns_map = @stack.last or raise 'internal error'
      ns_map[prefix] = ns_uri
      nil
    end

    def [](prefix)
      @stack.reverse_each do |ns_map|
	if (ns_map.include? prefix) then
	  return ns_map[prefix]
	end
      end

      nil
    end
  end

  class XMLAttributeMap
    include Enumerable

    def initialize
      @map = Hash.new
    end

    def add_attr(ns_uri, prefix, name, value)
      @map[ns_uri] = Hash.new unless (@map.include? ns_uri)
      @map[ns_uri][name] = { :prefix => prefix, :value => value }
      nil
    end

    def [](*args)
      case (args.length)
      when 1
	ns_uri, name = '', *args
      when 2
	ns_uri, name = args
      else
	raise 'invalid arguments: ' + args.map{|a| a.inspect }.join(', ')
      end

      if (@map.include? ns_uri) then
	if (@map[ns_uri].include? name) then
	  return @map[ns_uri][name][:value]
	end
      end

      nil
    end

    def prefix(ns_uri, name)
      if (@map.include? ns_uri) then
	if (@map[ns_uri].include? name) then
	  return @map[ns_uri][name][:prefix]
	end
      end

      nil
    end

    def each
      @map.each do |ns_uri, name_map|
	name_map.each do |name, value_map|
	  prefix = value_map[:prefix]
	  value = value_map[:value]
	  yield(ns_uri, prefix, name, value)
	end
      end

      nil
    end
  end

  class XMLReader
    def start_element(ns_uri, prefix, name, attr_map)
    end

    def end_element(ns_uri, prefix, name)
    end

    def processing_instruction(target, data)
    end

    def character(data)
    end

    def comment(data)
    end
  end

  module CharacterEntity
    CHARACTER_ENTITY = [
      # symbol      code    utf-8
      [ "nbsp",     160,  "\302\240"     ], # ISO 8859-1
      [ "iexcl",    161,  "\302\241" 	 ],
      [ "cent",     162,  "\302\242" 	 ],
      [ "pound",    163,  "\302\243" 	 ],
      [ "curren",   164,  "\302\244" 	 ],
      [ "yen",      165,  "\302\245" 	 ],
      [ "brvbar",   166,  "\302\246" 	 ],
      [ "sect",     167,  "\302\247" 	 ],
      [ "uml",      168,  "\302\250" 	 ],
      [ "copy",     169,  "\302\251" 	 ],
      [ "ordf",     170,  "\302\252" 	 ],
      [ "laquo",    171,  "\302\253" 	 ],
      [ "not",      172,  "\302\254" 	 ],
      [ "shy",      173,  "\302\255" 	 ],
      [ "reg",      174,  "\302\256" 	 ],
      [ "macr",     175,  "\302\257" 	 ],
      [ "deg",      176,  "\302\260" 	 ],
      [ "plusmn",   177,  "\302\261" 	 ],
      [ "sup2",     178,  "\302\262" 	 ],
      [ "sup3",     179,  "\302\263" 	 ],
      [ "acute",    180,  "\302\264" 	 ],
      [ "micro",    181,  "\302\265" 	 ],
      [ "para",     182,  "\302\266" 	 ],
      [ "middot",   183,  "\302\267" 	 ],
      [ "cedil",    184,  "\302\270" 	 ],
      [ "sup1",     185,  "\302\271" 	 ],
      [ "ordm",     186,  "\302\272" 	 ],
      [ "raquo",    187,  "\302\273" 	 ],
      [ "frac14",   188,  "\302\274" 	 ],
      [ "frac12",   189,  "\302\275" 	 ],
      [ "frac34",   190,  "\302\276" 	 ],
      [ "iquest",   191,  "\302\277" 	 ],
      [ "Agrave",   192,  "\303\200" 	 ],
      [ "Aacute",   193,  "\303\201" 	 ],
      [ "Acirc",    194,  "\303\202" 	 ],
      [ "Atilde",   195,  "\303\203" 	 ],
      [ "Auml",     196,  "\303\204" 	 ],
      [ "Aring",    197,  "\303\205" 	 ],
      [ "AElig",    198,  "\303\206" 	 ],
      [ "Ccedil",   199,  "\303\207" 	 ],
      [ "Egrave",   200,  "\303\210" 	 ],
      [ "Eacute",   201,  "\303\211" 	 ],
      [ "Ecirc",    202,  "\303\212" 	 ],
      [ "Euml",     203,  "\303\213" 	 ],
      [ "Igrave",   204,  "\303\214" 	 ],
      [ "Iacute",   205,  "\303\215" 	 ],
      [ "Icirc",    206,  "\303\216" 	 ],
      [ "Iuml",     207,  "\303\217" 	 ],
      [ "ETH",      208,  "\303\220" 	 ],
      [ "Ntilde",   209,  "\303\221" 	 ],
      [ "Ograve",   210,  "\303\222" 	 ],
      [ "Oacute",   211,  "\303\223" 	 ],
      [ "Ocirc",    212,  "\303\224" 	 ],
      [ "Otilde",   213,  "\303\225" 	 ],
      [ "Ouml",     214,  "\303\226" 	 ],
      [ "times",    215,  "\303\227" 	 ],
      [ "Oslash",   216,  "\303\230" 	 ],
      [ "Ugrave",   217,  "\303\231" 	 ],
      [ "Uacute",   218,  "\303\232" 	 ],
      [ "Ucirc",    219,  "\303\233" 	 ],
      [ "Uuml",     220,  "\303\234" 	 ],
      [ "Yacute",   221,  "\303\235" 	 ],
      [ "THORN",    222,  "\303\236" 	 ],
      [ "szlig",    223,  "\303\237" 	 ],
      [ "agrave",   224,  "\303\240" 	 ],
      [ "aacute",   225,  "\303\241" 	 ],
      [ "acirc",    226,  "\303\242" 	 ],
      [ "atilde",   227,  "\303\243" 	 ],
      [ "auml",     228,  "\303\244" 	 ],
      [ "aring",    229,  "\303\245" 	 ],
      [ "aelig",    230,  "\303\246" 	 ],
      [ "ccedil",   231,  "\303\247" 	 ],
      [ "egrave",   232,  "\303\250" 	 ],
      [ "eacute",   233,  "\303\251" 	 ],
      [ "ecirc",    234,  "\303\252" 	 ],
      [ "euml",     235,  "\303\253" 	 ],
      [ "igrave",   236,  "\303\254" 	 ],
      [ "iacute",   237,  "\303\255" 	 ],
      [ "icirc",    238,  "\303\256" 	 ],
      [ "iuml",     239,  "\303\257" 	 ],
      [ "eth",      240,  "\303\260" 	 ],
      [ "ntilde",   241,  "\303\261" 	 ],
      [ "ograve",   242,  "\303\262" 	 ],
      [ "oacute",   243,  "\303\263" 	 ],
      [ "ocirc",    244,  "\303\264" 	 ],
      [ "otilde",   245,  "\303\265" 	 ],
      [ "ouml",     246,  "\303\266" 	 ],
      [ "divide",   247,  "\303\267" 	 ],
      [ "oslash",   248,  "\303\270" 	 ],
      [ "ugrave",   249,  "\303\271" 	 ],
      [ "uacute",   250,  "\303\272" 	 ],
      [ "ucirc",    251,  "\303\273" 	 ],
      [ "uuml",     252,  "\303\274" 	 ],
      [ "yacute",   253,  "\303\275" 	 ],
      [ "thorn",    254,  "\303\276" 	 ],
      [ "yuml",     255,  "\303\277" 	 ],
      [ "fnof",     402,  "\306\222" 	 ],
      [ "Alpha",    913,  "\316\221" 	 ], # Mathematical, Greek and Symbolic characters for HTML
      [ "Beta",     914,  "\316\222" 	 ],
      [ "Gamma",    915,  "\316\223" 	 ],
      [ "Delta",    916,  "\316\224" 	 ],
      [ "Epsilon",  917,  "\316\225" 	 ],
      [ "Zeta",     918,  "\316\226" 	 ],
      [ "Eta",      919,  "\316\227" 	 ],
      [ "Theta",    920,  "\316\230" 	 ],
      [ "Iota",     921,  "\316\231" 	 ],
      [ "Kappa",    922,  "\316\232" 	 ],
      [ "Lambda",   923,  "\316\233" 	 ],
      [ "Mu",       924,  "\316\234" 	 ],
      [ "Nu",       925,  "\316\235" 	 ],
      [ "Xi",       926,  "\316\236" 	 ],
      [ "Omicron",  927,  "\316\237" 	 ],
      [ "Pi",       928,  "\316\240" 	 ],
      [ "Rho",      929,  "\316\241" 	 ],
      [ "Sigma",    931,  "\316\243" 	 ],
      [ "Tau",      932,  "\316\244" 	 ],
      [ "Upsilon",  933,  "\316\245" 	 ],
      [ "Phi",      934,  "\316\246" 	 ],
      [ "Chi",      935,  "\316\247" 	 ],
      [ "Psi",      936,  "\316\250" 	 ],
      [ "Omega",    937,  "\316\251" 	 ],
      [ "alpha",    945,  "\316\261" 	 ],
      [ "beta",     946,  "\316\262" 	 ],
      [ "gamma",    947,  "\316\263" 	 ],
      [ "delta",    948,  "\316\264" 	 ],
      [ "epsilon",  949,  "\316\265" 	 ],
      [ "zeta",     950,  "\316\266" 	 ],
      [ "eta",      951,  "\316\267" 	 ],
      [ "theta",    952,  "\316\270" 	 ],
      [ "iota",     953,  "\316\271" 	 ],
      [ "kappa",    954,  "\316\272" 	 ],
      [ "lambda",   955,  "\316\273" 	 ],
      [ "mu",       956,  "\316\274" 	 ],
      [ "nu",       957,  "\316\275" 	 ],
      [ "xi",       958,  "\316\276" 	 ],
      [ "omicron",  959,  "\316\277" 	 ],
      [ "pi",       960,  "\317\200" 	 ],
      [ "rho",      961,  "\317\201" 	 ],
      [ "sigmaf",   962,  "\317\202" 	 ],
      [ "sigma",    963,  "\317\203" 	 ],
      [ "tau",      964,  "\317\204" 	 ],
      [ "upsilon",  965,  "\317\205" 	 ],
      [ "phi",      966,  "\317\206" 	 ],
      [ "chi",      967,  "\317\207" 	 ],
      [ "psi",      968,  "\317\210" 	 ],
      [ "omega",    969,  "\317\211" 	 ],
      [ "thetasym", 977,  "\317\221" 	 ],
      [ "upsih",    978,  "\317\222" 	 ],
      [ "piv",      982,  "\317\226"     ],
      [ "bull",     8226, "\342\200\242" ],
      [ "hellip",   8230, "\342\200\246" ],
      [ "prime",    8242, "\342\200\262" ],
      [ "Prime",    8243, "\342\200\263" ],
      [ "oline",    8254, "\342\200\276" ],
      [ "frasl",    8260, "\342\201\204" ],
      [ "weierp",   8472, "\342\204\230" ],
      [ "image",    8465, "\342\204\221" ],
      [ "real",     8476, "\342\204\234" ],
      [ "trade",    8482, "\342\204\242" ],
      [ "alefsym",  8501, "\342\204\265" ],
      [ "larr",     8592, "\342\206\220" ],
      [ "uarr",     8593, "\342\206\221" ],
      [ "rarr",     8594, "\342\206\222" ],
      [ "darr",     8595, "\342\206\223" ],
      [ "harr",     8596, "\342\206\224" ],
      [ "crarr",    8629, "\342\206\265" ],
      [ "lArr",     8656, "\342\207\220" ],
      [ "uArr",     8657, "\342\207\221" ],
      [ "rArr",     8658, "\342\207\222" ],
      [ "dArr",     8659, "\342\207\223" ],
      [ "hArr",     8660, "\342\207\224" ],
      [ "forall",   8704, "\342\210\200" ],
      [ "part",     8706, "\342\210\202" ],
      [ "exist",    8707, "\342\210\203" ],
      [ "empty",    8709, "\342\210\205" ],
      [ "nabla",    8711, "\342\210\207" ],
      [ "isin",     8712, "\342\210\210" ],
      [ "notin",    8713, "\342\210\211" ],
      [ "ni",       8715, "\342\210\213" ],
      [ "prod",     8719, "\342\210\217" ],
      [ "sum",      8721, "\342\210\221" ],
      [ "minus",    8722, "\342\210\222" ],
      [ "lowast",   8727, "\342\210\227" ],
      [ "radic",    8730, "\342\210\232" ],
      [ "prop",     8733, "\342\210\235" ],
      [ "infin",    8734, "\342\210\236" ],
      [ "ang",      8736, "\342\210\240" ],
      [ "and",      8743, "\342\210\247" ],
      [ "or",       8744, "\342\210\250" ],
      [ "cap",      8745, "\342\210\251" ],
      [ "cup",      8746, "\342\210\252" ],
      [ "int",      8747, "\342\210\253" ],
      [ "there4",   8756, "\342\210\264" ],
      [ "sim",      8764, "\342\210\274" ],
      [ "cong",     8773, "\342\211\205" ],
      [ "asymp",    8776, "\342\211\210" ],
      [ "ne",       8800, "\342\211\240" ],
      [ "equiv",    8801, "\342\211\241" ],
      [ "le",       8804, "\342\211\244" ],
      [ "ge",       8805, "\342\211\245" ],
      [ "sub",      8834, "\342\212\202" ],
      [ "sup",      8835, "\342\212\203" ],
      [ "nsub",     8836, "\342\212\204" ],
      [ "sube",     8838, "\342\212\206" ],
      [ "supe",     8839, "\342\212\207" ],
      [ "oplus",    8853, "\342\212\225" ],
      [ "otimes",   8855, "\342\212\227" ],
      [ "perp",     8869, "\342\212\245" ],
      [ "sdot",     8901, "\342\213\205" ],
      [ "lceil",    8968, "\342\214\210" ],
      [ "rceil",    8969, "\342\214\211" ],
      [ "lfloor",   8970, "\342\214\212" ],
      [ "rfloor",   8971, "\342\214\213" ],
      [ "lang",     9001, "\342\214\251" ],
      [ "rang",     9002, "\342\214\252" ],
      [ "loz",      9674, "\342\227\212" ],
      [ "spades",   9824, "\342\231\240" ],
      [ "clubs",    9827, "\342\231\243" ],
      [ "hearts",   9829, "\342\231\245" ],
      [ "diams",    9830, "\342\231\246" ],
#     [ "quot",     34,   "\""           ], # Special characters for HTML
#     [ "amp",      38,   "&"            ],
#     [ "lt",       60,   "<" 		 ],
#     [ "gt",       62,   ">" 		 ],
      [ "OElig",    338,  "\305\222"     ],
      [ "oelig",    339,  "\305\223" 	 ],
      [ "Scaron",   352,  "\305\240" 	 ],
      [ "scaron",   353,  "\305\241" 	 ],
      [ "Yuml",     376,  "\305\270" 	 ],
      [ "circ",     710,  "\313\206" 	 ],
      [ "tilde",    732,  "\313\234" 	 ],
      [ "ensp",     8194, "\342\200\202" ],
      [ "emsp",     8195, "\342\200\203" ],
      [ "thinsp",   8201, "\342\200\211" ],
      [ "zwnj",     8204, "\342\200\214" ],
      [ "zwj",      8205, "\342\200\215" ],
      [ "lrm",      8206, "\342\200\216" ],
      [ "rlm",      8207, "\342\200\217" ],
      [ "ndash",    8211, "\342\200\223" ],
      [ "mdash",    8212, "\342\200\224" ],
      [ "lsquo",    8216, "\342\200\230" ],
      [ "rsquo",    8217, "\342\200\231" ],
      [ "sbquo",    8218, "\342\200\232" ],
      [ "ldquo",    8220, "\342\200\234" ],
      [ "rdquo",    8221, "\342\200\235" ],
      [ "bdquo",    8222, "\342\200\236" ],
      [ "dagger",   8224, "\342\200\240" ],
      [ "Dagger",   8225, "\342\200\241" ],
      [ "permil",   8240, "\342\200\260" ],
      [ "lsaquo",   8249, "\342\200\271" ],
      [ "rsaquo",   8250, "\342\200\272" ],
      [ "euro",     8364, "\342\202\254" ]
    ]

    CHAR_CODE = Hash.new
    CHAR_SYMBOL = Hash.new

    for sym, code, utf8 in CHARACTER_ENTITY
      CHAR_CODE[utf8] = code
      CHAR_SYMBOL[utf8] = sym
    end

    CHAR_PATTERN = Regexp.compile(CHARACTER_ENTITY.map{|sym, code, utf8|
				    Regexp.quote(utf8, 'u')
				  }.join('|'),
				  false, 'u')

    def escape_specials(raw_string, mode=:number)
      raw_string.gsub(CHAR_PATTERN) {|utf8|
	case (mode)
	when :number
	  '&#' + CHAR_CODE[utf8].to_s + ';'
	when :symbol
	  '&' + CHAR_SYMBOL[utf8] + ';'
	else
	  raise "internal error: #{mode.inspect}"
	end
      }
    end
    module_function :escape_specials
  end

  module Escape
    def escapeURL(raw_string)
      url_string = raw_string.gsub(/[^ _0-9A-Za-z.-]/) {|special|
	special.unpack('C*').map{|c|
	  format('%%%02X', c)
	}.join
      }
      url_string.gsub!(/ /, '+')
      url_string
    end
    module_function :escapeURL

    def unescapeURL(url_string)
      raw_string = url_string.gsub(/\+/, ' ')
      raw_string.gsub!(/%([0-9A-Fa-f][0-9A-Fa-f])/) {|special| $1.hex.chr }
      raw_string
    end
    module_function :unescapeURL

    def escapeHTML(raw_string, mode=:number)
      html_string = raw_string.gsub(/&/, '&amp;')
      html_string.gsub!(/</, '&lt;')
      html_string.gsub!(/>/, '&gt;')
      html_string.gsub!(/"/, '&quot;')
      CharacterEntity.escape_specials(html_string, mode)
    end
    module_function :escapeHTML
  end

  class Driver
    def handler
      # return a native message handler object of dirver.
      raise NotImplementedError, 'abstract method'
    end

    def env
      # return a Hash of CGI meta variables.
      raise NotImplementedError, 'abstract method'
    end

    def params
      # return a Hash of form parameters.
      raise NotImplementedError, 'abstract method'
    end

    def header(name)
      # return a value of header field.
      raise NotImplementedError, 'abstract method'
    end

    def set_header(name, value)
      # set value to header field.
      raise NotImplementedError, 'abstract method'
    end

    def write(messg_body)
      # wirte to body of a message.
      raise NotImplementedError, 'abstract method'
    end

    def close
      # finish of write
      raise NotImplementedError, 'abstract method'
    end
  end

  class MessageManipulator < Driver
    extend Forwardable
    include Escape

    NO_FILTER = proc{|messg_body|
      messg_body
    }

    def initialize(driver)
      @driver = driver
      @input_filter = NO_FILTER
      @output_filter = NO_FILTER
    end

    def set_input_filter(&block)
      @input_filter = block
      nil
    end

    def set_output_filter(&block)
      @output_filter = block
      nil
    end

    # definitions as a driver.

    def_delegator :@driver, :handler
    def_delegator :@driver, :env
    def_delegator :@driver, :header
    def_delegator :@driver, :set_header
    def_delegator :@driver, :close

    def params
      utf8_params = Hash.new
      for name, value in @driver.params
	utf8_name = @input_filter.call(name)
	utf8_value = @input_filter.call(value)
	utf8_params[utf8_name] = utf8_value
      end

      utf8_params
    end

    def write(utf8_messg_body)
      messg_body = @output_filter.call(utf8_messg_body)
      @driver.write(messg_body)
    end

    # definitions as a message manipulator.

    MSG = {
      100 => 'Continue',
      101 => 'Switching Protocols',
      200 => 'OK',
      201 => 'Created',
      202 => 'Accepted',
      203 => 'Non-Authoritative Information',
      204 => 'No Content',
      205 => 'Reset Content',
      206 => 'Partial Content',
      300 => 'Multiple Choices',
      301 => 'Moved Permanently',
      302 => 'Found',
      303 => 'See Other',
      304 => 'Not Modified',
      305 => 'Use Proxy',
      307 => 'Temporary Redirect',
      400 => 'Bad Request',
      401 => 'Unauthorized',
      402 => 'Payment Required',
      403 => 'Forbidden',
      404 => 'Not Found',
      405 => 'Method Not Allowed',
      406 => 'Not Acceptable',
      407 => 'Proxy Authentication Required',
      408 => 'Request Timeout',
      409 => 'Conflict',
      410 => 'Gone',
      411 => 'Length Required',
      412 => 'Precondition Failed',
      413 => 'Request Entity Too Large',
      414 => 'Request-URI Too Long',
      415 => 'Unsupported Media Type',
      416 => 'Requested Range Not Satisfiable',
      417 => 'Expectation Failed',
      500 => 'Internal Server Error',
      501 => 'Not Implemented',
      502 => 'Bad Gateway',
      503 => 'Service Unavailable',
      504 => 'Gateway Timeout',
      505 => 'HTTP Version Not Supported'
    }

    def status
      if (field_value = header('Status')) then
	code, reason = field_value.split(/\s+/, 2)
	return code.to_i
      else
	return 200
      end
    end

    def set_status(code)
      set_header('Status', "#{code} #{MSG[code] || 'Unknown'}")
      nil
    end

    def location
      header('Location')
    end

    def set_location(uri, code=302)
      set_status(code)
      set_header('Location', uri)
      nil
    end

    def curr_page
      path_info = env['PATH_INFO']
      if (path_info) then
	if (path_info =~ %r"^/?([^/]+)") then
	  return $1
	end
      end

      nil
    end

    def path_info
      path_info = env['PATH_INFO']
      if (path_info) then
	if (path_info =~ %r"^/?[^/]+") then
	  return $'
	end
      end

      ''
    end

    def page_path(page_name=curr_page, path_info=path_info)
      env['SCRIPT_NAME'] + '/' + page_name + path_info
    end

    def redirect(page_name, query_params=nil)
      env = self.env
      uri = 'http://'
      uri << env['SERVER_NAME'] << ':' << env['SERVER_PORT'] << page_path(page_name)
      if (query_params) then
	sep = '?'
	for name, value in query_params
	  uri << sep << escapeURL(name)
	  uri << '=' << escapeURL(value) if value
	  sep = '&'
	end
      end
      set_location(uri, 303)
      nil
    end
  end

  class Logger
    def write_access(status, environ, params)
      raise NotImplementedError, 'not implemented.'
    end
  end

  class DummyLogger < Logger
    def write_access(status, environ, params)
      nil
    end
  end

  class StandardErrorOutputLogger < Logger
    def write_access(status, environ, params)
      timestamp = Time.now.gmtime.to_s
      rhost = environ['REMOTE_ADDR']
      if (environ.include? 'REMOTE_HOST') then
	rhost += ' '
	rhost += environ['REMOTE_HOST']
      end
      method = environ['REQUEST_METHOD']
      page = environ['PATH_INFO'] || ''
      if (method != 'POST') then
	query = params.map{|name, value| "#{name}=#{value}" }.join('&')
	unless (query.empty?) then
	  page += '?'
	  page += query
	end
      end

      STDERR.print "[WPM debug] #{timestamp}: [#{rhost}] #{method} #{page} - #{status}\n"
      STDERR.flush

      nil
    end
  end

  class AppendLogger < Logger
    NO_FILTER = MessageManipulator::NO_FILTER

    def initialize(filename, input_filter=NO_FILTER, output_filter=NO_FILTER)
      @filename = filename
      @input_filter = input_filter
      @output_filter = output_filter
    end

    def write_access(status, environ, params)
      timestamp = Time.now.gmtime.to_s
      rhost = environ['REMOTE_ADDR']
      if (environ.include? 'REMOTE_HOST') then
	rhost += ' '
	rhost += environ['REMOTE_HOST']
      end
      method = environ['REQUEST_METHOD']
      page = environ['PATH_INFO'] || ''
      if (method != 'POST') then
	query = params.map{|name, value| "#{name}=#{value}" }.join('&')
	unless (query.empty?) then
	  page += '?'
	  page += query
	end
      end
      user_agent = environ['HTTP_USER_AGENT']
      referer = environ['HTTP_REFERER']
      if (referer) then
	referer = Escape.unescapeURL(referer)
	referer = @input_filter.call(referer)
      end

      log = "#{timestamp}: [#{rhost}] #{method} #{page} - #{status}: #{user_agent} < #{referer}\n"
      File.open(@filename, 'a') {|output|
	output.print @output_filter.call(log)
      }

      nil
    end
  end

  module CheckUtil
    def check_integer(string)
      case (string.strip)
      when /^[+-]?\d+$/
	return
      else
	raise "not a integer: #{string.inspect}"
      end
    end
    module_function :check_integer

    def check_float(string)
      case (string.strip)
      when /^[+-]?\d+(\.\d+)?$/, /^[+-]?\.\d+$/
	return
      when /^[+-]?\d+(\.\d+)?[Ee][+-]?\d+$/, /^[+-]?\.\d+[Ee][+-]?\d+$/
	return
      else
	raise "not a float: #{string.inspect}"
      end
    end
    module_function :check_float

    def check_number(string)
      if (string !~ /^(\+|\-)?\d+$/) then
	raise "not a number: #{string.inspect}"
      end
      if (iterator?) then
	unless (yield(string.to_i)) then
	  raise "out of range: #{string.inspect}"
	end
      end
      nil
    end
    module_function :check_number

    def check_not_empty(string, strip=false)
      if (string == nil || (strip ? string.strip.empty? : string.empty?)) then
	raise "empty string: #{string.inspect}"
      end
      nil
    end
    module_function :check_not_empty

    def empty_string?(string, strip=false)
      string == nil || (strip ? string.strip.empty? : string.empty?)
    end
    module_function :empty_string?
  end

  class PageContext
    extend Forwardable
    include CheckUtil

    def initialize(page, option={})
      @page = page
      @option = option
    end

    def_delegator :@page, :driver
    def_delegator :@page, :xml_assist
    def_delegator :@page, :ns_map
    def_delegator :@page, :base_dir
    def_delegator :@page, :last_error_page
    def_delegator :@page, :last_error_widget
    def_delegator :@page, :last_error_info

    def page_option(name)
      @option[name]
    end
    private :page_option

    def init_context
    end
  end

  class ComponentError < StandardError
  end

  class WidgetError < ComponentError
  end

  class PropertyError < WidgetError
  end

  class WidgetMappingError < WidgetError
  end

  class WidgetBindingError < WidgetError
  end

  class PageError < ComponentError
  end

  # memo: event flow.
  # (1) call WPM::Component#compile -> building a tree of components.
  # (2) call WPM::Component#setup -> setup each component.
  # (3) call WPM::Component#run -> executing actions at each component.
  # (4) call WPM::Component#make_page -> building a page from a tree of components.
  #
  class Component
    extend Forwardable
    include Escape

    def initialize
      @parent_component = nil
      @page = nil		# optimized
      @page_name = nil		# optimized
    end

    attr_accessor :parent_component
    def_delegator :@parent_component, :driver
    def_delegator :@parent_component, :xml_assist
    def_delegator :@parent_component, :base_dir
    def_delegator :@page, :ns_map        # optimized
    def_delegator :@page, :expand_string # optimized

    # optimized
    def page_name
      # [cacheable reason]
      # if @page_name is changed, page cache entry in WPM::PageCache
      # will be expired.
      @page_name = @parent_component.page_name unless @page_name
      @page_name
    end

    def component_name
      name = self.class.to_s
      name.gsub!(/^WPM::/, '')
      page_name + '.#' + name
    end

    def find(component_name)
      if (self.component_name == component_name) then
	return self
      else
	return nil
      end
    end

    def parent_page
      component = self
      loop do
	case (component)
	when Page
	  return component
	when NilClass
	  raise WidgetError, "not found a page widget upper #{component_name}"
	end
	component = component.parent_component
      end

      raise 'internal error'
    end
    private :parent_page

    def compile
      # [cacheable reason]
      # if @page is changed, page cache entry in WPM::PageCache
      # will be expired.
      @page = parent_page	# optimized
      nil
    end

    def recompile
    end

    def call_reset
    end

    def setup
    end

    def run
    end

    def make_page
      yield(escapeHTML(self.class.to_s))
      nil
    end
  end

  module ComponentParent
    include Enumerable

    def initialize(*args, &block)
      super
      @child_components = Array.new
    end

    def add(component)
      component.parent_component = self
      @child_components.push(component)
      nil
    end

    def has_children?
      ! @child_components.empty?
    end

    def find(component_name)
      if (self.component_name == component_name) then
	return self
      else
	for component in @child_components
	  if (found_component = component.find(component_name)) then
	    return found_component
	  end
	end

	return nil
      end
    end

    def compile
      super
      for component in @child_components
	component.compile
      end

      nil
    end

    def recompile
      super
      for component in @child_components
	component.recompile
      end

      nil
    end

    def call_reset
      super
      for component in @child_components
	component.call_reset
      end

      nil
    end

    def setup
      super
      for component in @child_components
	component.setup
      end

      nil
    end

    def run
      super
      for component in @child_components
	component.run
      end

      nil
    end

    def each
      for component in @child_components
	yield(component)
      end

      nil
    end
  end

  class Element < Component
    include ComponentParent

    def initialize(ns_uri, prefix, name, attr_map)
      super()

      @xmlns_alist = Array.new
      @xmlns_alist.push([ ns_uri, prefix ])

      @prefix = prefix
      @name = name
      @attr_map = attr_map

      for attr_ns_uri, attr_prefix, attr_name, attr_value in @attr_map
	unless (attr_prefix.empty?) then
	  @xmlns_alist.push([ attr_ns_uri, attr_prefix ])
	end
      end
    end

    def make_xmlns_desc
      xmlns_desc = ''
      for ns_uri, prefix in @xmlns_alist
	if (ns_map[prefix] != ns_uri) then
	  ns_map.add_ns(prefix, ns_uri)
	  xmlns_desc += ' xmlns'
	  xmlns_desc += ':' + prefix unless prefix.empty?
	  xmlns_desc += '="' + escapeHTML(ns_uri) + '"'
	end
      end

      xmlns_desc
    end
    private :make_xmlns_desc

    def expand_string_key(value)
      src_list = value.scan(/\$|[^\$]+/)
      dst = ''
      while (fragment = src_list.shift)
	if (fragment == '$') then
	  if (src_list.empty?) then
	    dst.concat(fragment)
	  else
	    next_fragment = src_list.shift
	    if (next_fragment == '$') then
	      dst.concat('$')
	    elsif (next_fragment =~ /^\{(.*?)\}/) then
	      key = $1
	      following = $'
	      dst.concat(expand_string(key))
	      dst.concat(following)
	    else
	      dst.concat(fragment)
	      dst.concat(next_fragment)
	    end
	  end
	else
	  dst.concat(fragment)
	end
      end

      dst
    end
    private :expand_string_key

    def make_page
      start_elem = '<'
      start_elem += @prefix + ':' unless @prefix.empty?
      start_elem += @name

      attr_desc = ''
      for attr_ns_uri, attr_prefix, attr_name, attr_value in @attr_map
	attr_desc += ' '
	attr_desc += attr_prefix + ':' unless attr_prefix.empty?
	attr_desc += attr_name
	attr_desc += '="'
	attr_desc += escapeHTML(expand_string_key(attr_value))
	attr_desc += '"'
      end

      end_elem = '</'
      end_elem += @prefix + ':' unless @prefix.empty?
      end_elem += @name + '>'

      ns_map.start_element
      begin
	if (has_children?) then
	  yield(start_elem + make_xmlns_desc + attr_desc + '>')
	  for component in @child_components
	    component.make_page do |page_text|
	      yield(page_text)
	    end
	  end
	  yield(end_elem)
	else
	  yield(start_elem + make_xmlns_desc + attr_desc + ' />')
	end
      ensure
	ns_map.end_element
      end

      nil
    end
  end

  class StringComponent < Component
    def initialize(value)
      super()
      @value = value
    end

    def make_page
      yield(@value)
      nil
    end
  end

  class WidgetProperties
    extend Forwardable

    def initialize(widget_id, page_context, running_context)
      @widget_id = widget_id
      @page_context = page_context
      @running_context = running_context
      @call_index_setter = nil
      @instance_number = 0
    end

    def instance_number
      n = @instance_number
      @instance_number += 1
      n
    end

    def create_widget
      raise NotImplementedError, 'abstract method'
    end

    attr_reader :widget_id
    def_delegator :@running_context, :call_reset
    def_delegator :@running_context, :call_index

    def call_count(name)
      @running_context.call_count(name)
      @call_index_setter.call(@running_context.call_index(name)) if @call_index_setter
      nil
    end

    def mapped(attr_map)
    end

    def check_mapping
    end

    def set_property(name, attr_map, value)
      raise "unknown property #{name.inspect} at #{@widget_id.inspect}"
    end

    def set_common_property(name, attr_map, value)
      case (name)
      when 'call-index'
	# common property
	@call_index_setter = get_property(attr_map, value, 'accessor')
      else
	# local property
	set_property(name, attr_map, value)
      end

      nil
    end

    # utility method for subclasses
    def check_type(name, value, klass)
      unless (value.kind_of? klass) then
	raise PropertyError, "not an object of #{klass} at the #{name.inspect} of #{@widget_id.inspect}: #{value.inspect}"
      end

      nil
    end
    private :check_type

    # utility method for subclasses
    def fetch_value(prop_name, prop_value, check_type=nil, not_allow_nil=false)
      case (prop_value)
      when Proc, Method
	ret_val = prop_value.call
      else
	ret_val = prop_value
      end

      unless (not_allow_nil) then
	if (ret_val.nil?) then
	  return nil
	end
      end

      if (check_type) then
	check_type(prop_name, ret_val, check_type)
      end

      ret_val
    end
    private :fetch_value

    # utility method for subclasses
    def get_property(attr_map, value, *types)
      prop_type = attr_map['type']
      unless (types.include? prop_type) then
	raise PropertyError, "not allowd property type: #{prop_type.inspect}: at #{@widget_id.inspect}"
      end
      prop_value = value.strip
      case (prop_type)
      when 'string'
	return prop_value
      when 'bool'
	return get_bool_property(prop_value)
      when 'number'
	return get_number_property(prop_value)
      when 'method'
	return get_method_property(prop_value)
      when 'accessor'
	return get_accessor_property(prop_value)
      when 'eval'
	return get_eval_property(prop_value)
      else
	raise PropertyError, "unknown property type: #{prop_type.inspect}: at #{@widget_id.inspect}"
      end
    end
    private :get_property

    def get_bool_property(prop_value)
      case (prop_value.downcase)
      when 'true'
	return true
      when 'false'
	return false
      else
	raise PropertyError, "unknown boolean format: #{prop_value.inspect}: at #{@widget_id.inspect}"
      end
    end
    private :get_bool_property

    def get_number_property(prop_value)
      if (prop_value !~ /^[\+\-]?\d+$/) then
	raise PropertyError, "invalid number format: #{prop_value.inspect}: at #{@widget_id.inspect}"
      end
      prop_value.to_i
    end
    private :get_number_property

    def get_method_property(prop_value, context=@page_context)
      if (prop_value !~ /^[_A-Za-z][_A-Za-z0-9]*$/) then
	raise PropertyError, "invalid accessor format: #{prop_value.inspect}: at #{@widget_id.inspect}"
      end

      method_name = prop_value.intern

      ## NoMethodError is too late. why?
      #
      # proc{
      #   context.__send__(method_name)
      # nil
      # }

      context.method(method_name)
    end
    private :get_method_property

    def get_accessor_property(prop_value, context=@page_context)
      if (prop_value !~ /^[_A-Za-z][_A-Za-z0-9]*\??$/) then
	raise PropertyError, "invalid accessor format: #{prop_value.inspect}: at #{@widget_id.inspect}"
      end

      ## NoMethodError is too late. why?
      #
      # attr_reader = prop_value.intern
      # if (prop_value =~ /\?$/) then
      #   attr_writer = nil
      # else
      #   attr_writer = "#{prop_value}=".intern
      # end
      #
      # proc{|*args|
      #   case (args.length)
      #   when 0
      #     context.__send__(attr_reader)
      #   when 1
      #     if (attr_writer) then
      #       context.__send__(attr_writer, args.first)
      #     else
      #       raise PropertyError, "read only accessor: #{attr_reader}"
      #     end
      #   else
      #     raise PropertyError, 'invalid accessor arguments: ' + args.map{|a| a.inspect }.joni(', ') + ": at #{@widget_id.inspect}"
      #   end
      # }

      if (context.public_methods(true).include? prop_value) then
	attr_reader = context.method(prop_value)
      else
	attr_reader = nil
      end

      if (prop_value =~ /\?$/) then
	attr_writer = nil
      elsif (context.public_methods(true).include? "#{prop_value}=") then
	attr_writer = context.method("#{prop_value}=")
      else
	attr_writer = nil
      end

      proc{|*args|
	case (args.length)
	when 0
	  if (attr_reader) then
	    attr_reader.call
	  else
	    raise PropertyError, "not readable accessor: #{prop_value.inspect}"
	  end
	when 1
	  if (attr_writer) then
	    attr_writer.call(args[0])
	  else
	    raise PropertyError, "not writable accessor: #{prop_value.inspect}"
	  end
	else
	  raise PropertyError, 'invalid accessor arguments: ' + args.map{|a| a.inspect }.joni(', ') + ": at #{@widget_id.inspect}"
	end
      }
    end
    private :get_accessor_property

    def make_eval_bind(__context__)
      __module__ = Module.new
      __module__.const_set(:CONTEXT, __context__)

      class << __module__
	def include(mod)
	  extend(mod)
	  super(mod)
	end
	private :include
      end

      __module__.module_eval %q{
        include Escape
        binding
      }
    end
    private :make_eval_bind

    def get_eval_property(prop_value, context=@page_context)
      eval_expr = prop_value.dup
      eval_expr.untaint
      eval_context = make_eval_bind(context)
      proc{
	RunningContext.safe_run(3) {
	  eval(eval_expr, eval_context)
	}
      }
    end
    private :get_eval_property
  end

  # memo: event flow.
  # (1) mapping phase.
  #   (1.1) call WPM::WidgetProperties#mapped -> start mapping.
  #   (1.2) call WPM::WidgetProperties#set_property -> setting widget properties.
  #   (1.3) call WPM::WidgetProperties#check_mapping -> end of mapping.
  # (2) binding phase.
  #   (2.1) call WPM::Component#compile -> building a tree of components.
  #     (2.1.1) call WPM::WidgetProperties#create_widget -> creating a widget having properties.
  #   (2.2) call WPM::Widget#init -> initialize each widget.
  #   (2.3) call WPM::Widget#invoke -> executing an action at a widget.
  #   (2.4) call WPM::Widget#view -> building widget view.
  #
  class Widget < Component
    extend Forwardable
    include ComponentParent

    def initialize(properties)
      super()
      @instance_number = properties.instance_number
      @properties = properties
      @component_name = nil	# optimized
    end

    attr_accessor :instance_number
    def_delegator :@properties, :widget_id

    def call_reset
      @properties.call_reset(component_name)
      super			# traversing child components.
      nil
    end

    def call_index
      @properties.call_index(component_name)
    end

    # optimized
    def component_name
      # [cacheable reason]
      # if @component_name is changed, page cache entry in
      # WPM::PageCache will be expired.
      @component_name = page_name + '.' + widget_id unless @component_name
      @component_name
    end

    def action_id
      component_name + '.' + call_index.to_s
    end

    def init
    end

    def invoke
    end

    def view
      yield(escapeHTML(self.class.to_s))
      nil
    end

    def widget_scope
      old_widget = @page.curr_widget
      @page.curr_widget = action_id
      yield
      @page.curr_widget = old_widget
      nil
    end
    private :widget_scope

    def setup
      @properties.call_count(component_name)
      widget_scope{ init }

      nil
    end

    def run
      @properties.call_count(component_name)
      widget_scope{ invoke }

      nil
    end

    def make_page
      @properties.call_count(component_name)
      widget_scope{
	view do |page_text|
	  yield(page_text)
	end
      }

      nil
    end

    # utility method fo subclasses
    def make_attr_desc
      ns_map.start_element
      begin
	xmlns_desc = ''
	attr_desc = ''
	if (ns_map[''] != XMLNS_HTML_URI) then
	  xmlns_desc.concat(' xmlns="' + escapeHTML(XMLNS_HTML_URI) + '"')
	  ns_map.add_ns('', XMLNS_HTML_URI)
	end
	yield(xmlns_desc + attr_desc)
      ensure
	ns_map.end_element
      end

      nil
    end
    private :make_attr_desc
  end

  class WidgetLocator < XMLReader
    include Enumerable

    FACTORY_MAP = Hash.new

    def self.add(name, properties_factory)
      FACTORY_MAP[name] = properties_factory
      nil
    end

    def initialize(page_context, running_context)
      @depth = 0
      @page_context = page_context
      @running_context = running_context
      @properties_map = Hash.new
      @string_map = Hash.new
      @curr_properties = nil
      @curr_attr_map = nil
      @curr_value = nil
    end

    def create_widget(widget_id)
      unless (@properties_map.include? widget_id) then
	raise "not found a widget with id: #{widget_id.inspect}"
      end
      @properties_map[widget_id].create_widget
    end

    def expand_string(key)
      if (@string_map.include? key) then
	properties = @string_map[key]
	widget = properties.create_widget
	return widget.to_s
      end

      raise "not found a string widget of #{key.inspect}"
    end

    def start_element(ns_uri, prefix, name, attr_map)
      if (ns_uri == XMLNS_MAP_URI) then
	@depth += 1
	case (@depth)
	when 1
	  if (name != 'map') then
	    raise WidgetMappingError, "invalid top level element of map: #{name.inspect}"
	  end
	when 2
	  if (FACTORY_MAP.include? name) then
	    unless (attr_map['name']) then
	      raise WidgetMappingError, "undefined name attribute: #{name.inspect}"
	    end
	    widget_id = attr_map['name']
	    @curr_properties = FACTORY_MAP[name].new(widget_id, @page_context, @running_context)
	    if (@properties_map.include? widget_id) then
	      raise WidgetMappingError, "duplicated name: #{widget_id.inspect}"
	    end
	    @curr_properties.mapped(attr_map)
	    @properties_map[widget_id] = @curr_properties
	    @string_map[widget_id] = @curr_properties if (name == 'string')
	  else
	    raise WidgetMappingError, "unknown widget: #{name.inspect}"
	  end
	when 3
	  @curr_attr_map = attr_map
	  @curr_value = ''
	else
	  raise WidgetMappingError, "too deep element: #{name.inspect}"
	end
      else
	if (@depth >= 3) then
	  raise WidgetMappingError, "not allowed element: #{name.inspect}"
	end
      end

      nil
    end

    def character(data)
      case (@depth)
      when 3
	@curr_value.concat(data)
      end

      nil
    end

    def end_element(ns_uri, prefix, name)
      if (ns_uri == XMLNS_MAP_URI) then
	case (@depth)
	when 1
	  if (name != 'map') then
	    raise WidgetMappingError, "invalid top level element: #{name.inspect}"
	  end
	when 2
	  if (FACTORY_MAP.include? name) then
	    @curr_properties.check_mapping
	    @curr_properties = nil
	  else
	    raise WidgetMappingError, "unknown widget: #{name.inspect}"
	  end
	when 3
	  @curr_properties.set_common_property(name, @curr_attr_map, @curr_value)
	  @curr_attr_map = nil
	  @curr_value = nil
	else
	  raise WidgetMappingError, "too deep element: #{name.inspect}"
	end
	@depth -= 1
      else
	if (@depth >= 3) then
	  raise WidgetMappingError, "not allowed element: #{name.inspect}"
	end
      end
    end

    nil
  end

  class IfWidgetProperies < WidgetProperties
    WidgetLocator.add('if', self)

    def mapped(attr_map)
      @condition = nil
      @negate = false
      nil
    end

    def check_mapping
      if (@condition == nil) then
	raise PropertyError, "required `condition' property at #{widget_id.inspect}"
      end
      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'condition'
	@condition = get_property(attr_map, value, 'bool', 'accessor', 'eval')
      when 'negate'
	@negate = get_property(attr_map, value, 'bool', 'accessor', 'eval')
      else
	raise PropertyError, "unknown property: #{name.inspect}: at #{widget_id.inspect}"
      end

      nil
    end

    def condition
      fetch_value('condition', @condition)
    end

    def negate
      fetch_value('negate', @negate)
    end

    def create_widget
      IfWidget.new(self)
    end
  end

  class IfWidget < Widget
    def execute
      if (@properties.negate) then
	if (! @properties.condition) then
	  for component in @child_components
	    yield(component)
	  end
	end
      else
	if (@properties.condition) then
	  for component in @child_components
	    yield(component)
	  end
	end
      end

      nil
    end
    private :execute

    def init
      execute do |component|
	component.setup
      end

      nil
    end

    def invoke
      execute do |component|
	component.run
      end

      nil
    end

    def view
      execute do |component|
	component.make_page do |page_text|
	  yield(page_text)
	end
      end

      nil
    end
  end

  class RepeatWidgetProperties < WidgetProperties
    WidgetLocator.add('repeat', self)

    def mapped(attr_map)
      @times = nil
      @index = nil
      nil
    end

    def check_mapping
      unless (@times) then
	raise PropertyError, "required `condition' property at #{widget_id.inspect}"
      end
      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'times'
	@times = get_property(attr_map, value, 'number', 'accessor', 'eval')
      when 'index'
	@index = get_property(attr_map, value, 'accessor')
      else
	raise PropertyError, "unknown property: #{name.inspect}: at #{widget_id.inspect}"
      end
    end

    def times
      fetch_value('times', @times, Numeric, true)
    end

    def has_index?
      @index != nil
    end

    def index=(new_index)
      @index.call(new_index)
    end

    def create_widget
      RepeatWidget.new(self)
    end
  end

  class RepeatWidget < Widget
    def execute
      for i in 1..(@properties.times)
	@properties.index = i if @properties.has_index?
	for component in @child_components
	  yield(component)
	end
      end
    end
    private :execute

    def init
      execute do |component|
	component.setup
      end

      nil
    end

    def invoke
      execute do |component|
	component.run
      end

      nil
    end

    def view
      execute do |component|
	component.make_page do |page_text|
	  yield(page_text)
	end
      end

      nil
    end
  end

  class ForeachWidgetProperties < WidgetProperties
    WidgetLocator.add('foreach', self)

    def mapped(attr_map)
      @list = nil
      @item = nil
      @index = nil
      nil
    end

    def check_mapping
      [ [ 'list',  @list,  true  ],
	[ 'item',  @item,  true  ],
	[ 'index', @index, false ]
      ].each do |prop_name, prop_value, required|
	case (prop_value)
	when NilClass
	  if (required) then
	    raise PropertyError, "required `#{prop_name}' property at #{widget_id.inspect}"
	  else
	    # nothing to do.
	  end
	when Proc, Method
	  # nothing to do.
	else
	  raise PropertyError, "invlid #{prop_name} property type at #{widget_id.inspect}"
	end
      end

      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'list'
	@list = get_property(attr_map, value, 'accessor', 'eval')
      when 'item'
	@item = get_property(attr_map, value, 'accessor')
      when 'index'
	@index = get_property(attr_map, value, 'accessor')
      else
	raise PropertyError, "unknown property: #{name.inspect}: at #{widget_id.inspect}"
      end

      nil
    end

    def list
      fetch_value('list', @list, Enumerable, true)
    end

    def item=(new_item)
      @item.call(new_item)
    end

    def has_index?
      @index != nil
    end

    def index=(new_index)
      @index.call(new_index)
    end

    def create_widget
      ForeachWidget.new(self)
    end
  end

  class ForeachWidget < Widget
    def execute
      count = 0
      for item in @properties.list
	count += 1
	@properties.item = item
	@properties.index = count if @properties.has_index?
	for component in @child_components
	  yield(component)
	end
      end

      nil
    end
    private :execute

    def init
      execute do |component|
	component.setup
      end

      nil
    end

    def invoke
      execute do |component|
	component.run
      end

      nil
    end

    def view
      execute do |component|
	component.make_page do |page_text|
	  yield(page_text)
	end
      end

      nil
    end
  end

  class ImportWidgetProperties < WidgetProperties
    WidgetLocator.add('import', self)

    def mapped(attr_map)
      @page = attr_map['page'] or raise PropertyError, "required `page' attribute at #{widget_id.inspect}"
      @params = Array.new
      nil
    end

    def set_property(name, attr_map, value)
      value = get_property(attr_map, value, 'string', 'bool', 'number', 'accessor', 'eval')
      @params.push([ name, value ])
      nil
    end

    attr_reader :page

    def setup_import_params(context)
      for name, value in @params
	name_setter = get_accessor_property(name.gsub(/-/, '_'), context)
	name_setter.call(fetch_value(name, value))
      end

      nil
    end

    def create_widget
      ImportWidget.new(self)
    end
  end

  class ImportWidget < Widget
    def find(component_name)
      if (self.component_name == component_name) then
	return self
      elsif (@another_page) then
	if (found_component = @another_page.find(component_name)) then
	  return found_component
	end
      end

      nil
    end

    def init_import_page
      @another_page = parent_page.new_page(@properties.page, self) {|another_page|
	another_page.context_hook{|context|
	  @properties.setup_import_params(context)
	}
      }

      nil
    end
    private :init_import_page

    def compile
      init_import_page
      super			# traversing child components.
      nil
    end

    def recompile
      init_import_page
      super			# traversing child components.
      nil
    end

    def call_reset
      @another_page.call_reset
      super			# traversing child components.
      nil
    end

    def init
      @another_page.setup
      # child components traversed in WPM::ContentWidget.
      nil
    end

    def invoke
      @another_page.run
      # child components traversed in WPM::ContentWidget.
      nil
    end

    def view
      @another_page.make_page do |page_text|
	# child components traversed in WPM::ContentWidget.
	yield(page_text)
      end

      nil
    end
  end

  class ContentWidgetProperties < WidgetProperties
    WidgetLocator.add('content', self)

    def set_property(name, attr_map, value)
      raise PropertyError, "unknown property: #{name.inspect}: at #{widget_id.inspect}"
    end

    def create_widget
      ContentWidget.new(self)
    end
  end

  class ContentWidget < Widget
    def execute
      component = @parent_component
      loop do
	case (component)
	when ImportWidget
	  break
	when NilClass
	  raise WidgetError, "not found an import widget upper #{@widget_id.inspect}."
	end
	component = component.parent_component
      end
      import_widget = component

      for component in import_widget
	yield(component)
      end

      nil
    end
    private :execute

    def init
      execute do |component|
	component.setup
      end

      nil
    end

    def invoke
      execute do |component|
	component.run
      end

      nil
    end

    def view
      execute do |component|
	component.make_page do |page_text|
	  yield(page_text)
	end
      end

      nil
    end
  end

  class StringWidgetProperties < WidgetProperties
    WidgetLocator.add('string', self)

    def mapped(attr_map)
      @escape = true
      @default = ''
      @value = nil
      nil
    end

    def check_mapping
      unless (@value) then
	raise PropertyError, "required `value' property at #{widget_id.inspect}"
      end
      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'value'
	@value = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'default'
	@default = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'escape'
	@escape = get_property(attr_map, value, 'bool', 'accessor', 'eval')
      else
	raise PropertyError, "unknown property: #{name.inspect}: at #{widget_id.inspect}"
      end

      nil
    end

    def escape?
      fetch_value('escape', @escape) ? true : false
    end

    def has_default?
      @default != nil
    end

    def default
      fetch_value('default', @default, String)
    end

    def value
      fetch_value('value', @value)
    end

    def create_widget
      StringWidget.new(self)
    end
  end

  class StringWidget < Widget
    # hook for expand_string
    def to_s
      string = @properties.value.to_s
      if (@properties.has_default? && string.empty?) then
	string = @properties.default.to_s
      end

      if (@properties.escape?) then
	string = escapeHTML(string)
      end

      string
    end

    def view
      yield(self.to_s)
      nil
    end
  end

  class HyperlinkWidgetProperties < WidgetProperties
    WidgetLocator.add('hyperlink', self)

    def mapped(attr_map)
      @action = nil
      @page = nil
      @href = nil
      @target = nil
      @string = nil
      @query = nil
      nil
    end

    def check_mapping
      if (! @action && ! @page && ! @href) then
	raise PropertyError, "required `action', `page' or href property at #{widget_id.inspect}"
      end
      if (@action && @target) then
	raise PropertyError, "conflicted action and target property at #{widget_id.inspect}"
      end

      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'action'
	@page and raise PropertyError, "conflicted `page' and `action' property at #{widget_id.inspect}"
	@href and raise PropertyError, "conflicted `href' and `action' property at #{widget_id.inspect}"
	@action = get_property(attr_map, value, 'method', 'eval')
      when 'page'
	@action and raise PropertyError, "conflicted `action' and `page' property at #{widget_id.inspect}"
	@href and raise PropertyError, "conflicted `href' and `page' property at #{widget_id.inspect}"
	@page = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'href'
	@action and raise PropertyError, "conflicted `action' and `href' property at #{widget_id.inspect}"
	@page and raise PropertyError, "conflicted `page' and `href' property at #{widget_id.inspect}"
	@href = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'target'
	@target = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'string'
	@string = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'query'
	@query = get_property(attr_map, value, 'accessor', 'eval')
      else
	raise PropertyError, "unknown property: #{name.inspect}: at #{widget_id.inspect}"
      end

      nil
    end

    def has_action?
      @action != nil
    end

    def action
      @action.call
    end

    def has_page?
      @page != nil
    end

    def page
      fetch_value('page', @page, String, true)
    end

    def has_href?
      @href != nil
    end

    def href
      fetch_value('href', @href, String, true)
    end

    def has_target?
      @target != nil
    end

    def target
      fetch_value('target', @target, String, true)
    end

    def has_string?
      @string != nil
    end

    def string
      fetch_value('string', @string, String)
    end

    def has_query?
      @query != nil
    end

    def each_query_pair
      query = fetch_value('query', @query, Hash, true)
      for name, value in query
	check_type('query', name, String)
	check_type('query', value, String)
	yield(name, value)
      end

      nil
    end

    def create_widget
      HyperlinkWidget.new(self)
    end
  end

  class HyperlinkWidget < Widget
    def init
      for component in @child_components
	component.setup
      end

      nil
    end

    def invoke
      if (@properties.has_action?) then
	if (driver.params['action'] == action_id) then
	  @properties.action
	end
      end

      for component in @child_components
	component.run
      end

      nil
    end

    def view
      if (@properties.has_action?) then
	href = driver.page_path + "?action=#{escapeURL(action_id)}"
      elsif (@properties.has_page?) then
	page_name = @properties.page
	href = driver.page_path(page_name)
      elsif (@properties.has_href?) then
	href = @properties.href
      else
	raise 'internal error'
      end

      if (@properties.has_query?) then
	if (@properties.has_action?) then
	  sep = '&'
	elsif (@properties.has_page?) then
	  sep = '?'
	end
	if (@properties.has_action? || @properties.has_page?) then
	  @properties.each_query_pair do |name, value|
	    href += sep; sep = '&'
	    href += escapeURL(name)
	    href += '='
	    href += escapeURL(value)
	  end
	end
      end

      make_attr_desc{|attr_desc|
	attr_desc += ' href="' + escapeHTML(href) + '"'
	attr_desc += ' target="' + escapeHTML(@properties.target) + '"' if @properties.has_target?
	yield('<a' + attr_desc + '>')
	if (has_children?) then
	  for component in @child_components
	    component.make_page do |page_text|
	      yield(page_text)
	    end
	  end
	elsif (@properties.has_string?) then
	  yield(escapeHTML(@properties.string || ''))
	end
	yield('</a>')
      }

      nil
    end
  end

  class ImageWidgetProperties < WidgetProperties
    WidgetLocator.add('image', self)

    def mapped(attr_map)
      @src = nil
      @alt = nil
      nil
    end

    def check_mapping
      unless (@src) then
	raise PropertyError, "required `src' property at #{widget_id.inspect}"
      end

      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'src'
	@src = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'alt'
	@alt = get_property(attr_map, value, 'string', 'accessor', 'eval')
      else
	raise PropertyError, "unknown property: #{name.inspect}: at #{widget_id.inspect}"
      end

      nil
    end

    def src
      fetch_value('src', @src, String, true)
    end

    def has_alt?
      @alt != nil
    end

    def alt
      fetch_value('alt', @alt, String, true)
    end

    def create_widget
      ImageWidget.new(self)
    end
  end

  class ImageWidget < Widget
    def view
      make_attr_desc{|attr_desc|
	attr_desc += ' src="' + escapeHTML(@properties.src) + '"'
	attr_desc += ' alt="' + escapeHTML(@properties.alt) + '"' if @properties.has_alt?
	yield('<img' + attr_desc + ' />')
      }

      nil
    end
  end

  class FrameWidgetProperties < WidgetProperties
    WidgetLocator.add('frame', self)

    def mapped(attr_map)
      @name = nil
      @page = nil
      @src = nil
      nil
    end

    def check_mapping
      unless (@name) then
	raise PropertyError, "required `name' property at #{widget_id.inspect}"
      end
      unless (@page || @src) then
	raise PropertyError, "required `page' or `src' property at #{widget_id.inspect}"
      end

      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'name'
	@name = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'page'
	@src and raise PropertyError, "conflicted `src' and `page' property at #{widget_id.inspect}"
	@page = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'src'
	@page and raise PropertyError, "conflicted `page' and `src' property at #{widget_id.inspect}"
	@src = get_property(attr_map, value, 'string', 'accessor', 'eval')
      else
	raise PropertyError, "unknown property: #{name.inspect}: at #{widget_id.inspect}"
      end

      nil
    end

    def name
      fetch_value('name', @name, String, true)
    end

    def has_page?
      @page != nil
    end

    def page
      fetch_value('page', @page, String, true)
    end

    def has_src?
      @src != nil
    end

    def src
      fetch_value('src', @src, String, true)
    end

    def create_widget
      FrameWidget.new(self)
    end
  end

  class FrameWidget < Widget
    def view
      make_attr_desc{|attr_desc|
	attr_desc += ' name="' + escapeHTML(@properties.name) + '"'
	if (@properties.has_page?) then
	  attr_desc += ' src="' + escapeHTML(driver.page_path(@properties.page)) + '"'
	elsif (@properties.has_src?) then
	  attr_desc += ' src="' + escapeHTML(@properties.src) + '"'
	else
	  raise 'internal error'
	end
	yield('<frame' + attr_desc + ' />')
      }

      nil
    end
  end

  class FormWidgetProperties < WidgetProperties
    WidgetLocator.add('form', self)

    def mapped(attr_map)
      @method = 'post'
      @enctype = 'application/x-www-form-urlencoded'
      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'method'
	@method = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'enctype'
	@enctype = get_property(attr_map, value, 'string', 'accessor', 'eval')
      else
	raise PropertyError, "unknown property: #{name.inspect}: at #{widget_id.inspect}"
      end

      nil
    end

    def http_method
      fetch_value('method', @method, String, true)
    end

    def enctype
      fetch_value('enctype', @enctype, String, true)
    end

    def create_widget
      FormWidget.new(self)
    end
  end

  class FormWidget < Widget
    def compile
      @params = driver.params
      @control = Hash.new
      super			# traversing child components.
      nil
    end

    def recompile
      @params = driver.params
      super			# traversing child components.
      nil
    end

    attr_reader :control
    attr_reader :params

    def submit?
      @params['_wpm_submit_'] == action_id
    end

    def init
      for component in @child_components
	component.setup
      end

      nil
    end

    def invoke
      for component in @child_components
	component.run
      end

      nil
    end

    def view
      make_attr_desc{|attr_desc|
	http_method = @properties.http_method
	attr_desc += ' id="' + escapeHTML(action_id) + '"'
	attr_desc += ' method="' + escapeHTML(http_method) + '"'
	attr_desc += ' enctype="' + escapeHTML(@properties.enctype) + '"' if (http_method == 'post')
	yield('<form' + attr_desc + '>')
	yield('<div style="display: none">')
	yield('<input type="hidden" name="_wpm_submit_" value="' + escapeHTML(action_id) + '" />')
	yield('</div>')
	for component in @child_components
	  component.make_page do |page_text|
	    yield(page_text)
	  end
	end
	yield('</form>')
      }


      nil
    end
  end

  class FormElementWidgetProperties < WidgetProperties
    def initialize(*args)
      super
      @created = false
    end

    def create_widget
      if (@created) then
	raise WidgetBindingError, "duplicated form element widget call: #{widget_id.inspect}"
      end
      @created = true
      create_form_element
    end

    def create_form_element
      raise NotImplementedError, 'abstract method'
    end
  end

  class FormElementWidget < Widget
    def form
      component = @parent_component
      loop do
	case (component)
	when FormWidget
	  return component
	when Page, NilClass
	  raise WidgetError, "not found a form widget upper #{widget_id.inspect}"
	end
	component = component.parent_component
      end

      raise 'internal error'
    end
    private :form
  end

  class LabelWidgetProperties < FormElementWidgetProperties
    WidgetLocator.add('label', self)

    def mapped(attr_map)
      @for_id = nil
      @string = nil
      nil
    end

    def check_mapping
      unless (@for_id) then
	 raise PropertyError, "required `for' property at #{widget_id.inspect}"
      end

      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'for'
	@for_id = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'string'
	@string = get_property(attr_map, value, 'string', 'accessor', 'eval')
      else
	raise PropertyError, "unknown property: #{name.inspect}: at #{widget_id.inspect}"
      end

      nil
    end

    def for_id
      fetch_value('for', @for_id, String, true)
    end

    def has_string?
      @string != nil
    end

    def string
      fetch_value('string', @string, String, true)
    end

    def create_form_element
      LabelWidget.new(self)
    end
  end

  class LabelWidget < FormElementWidget
    def init
      for component in @child_components
	component.setup
      end

      nil
    end

    def invoke
      control_widget = form.control[@properties.for_id]
      unless (control_widget) then
	raise WidgetError, "not found a widget of #{@properties.for_id}"
      end

      for component in @child_components
	component.run
      end

      nil
    end

    def view
      control_widget = form.control[@properties.for_id]
      unless (control_widget) then
	raise WidgetError, "not found a widget of #{for_id}"
      end
      ctl_id = control_widget.component_name + '.' + @properties.call_index(component_name).to_s

      make_attr_desc{|attr_desc|
	attr_desc += ' for="' + escapeHTML(ctl_id) + '"'
	yield('<label' + attr_desc + '>')
	if (has_children?) then
	  for component in @child_components
	    component.make_page do |page_text|
	      yield(page_text)
	    end
	  end
	elsif (@properties.has_string?) then
	  yield(escapeHTML(@properties.string))
	end
	yield('</label>')
      }

      nil
    end
  end

  class FormControlWidget < FormElementWidget
    def compile
      form.control[widget_id] = self

      # traversing child components.
      super

      nil
    end

    def make_attr_desc
      super{|attr_desc|
	attr_desc += ' id="' + escapeHTML(action_id) + '"'
	yield(attr_desc)
      }

      nil
    end
    private :make_attr_desc
  end

  module DisabledWidgetProperty
    def mapped(attr_map)
      super
      @disabled = false
      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'disabled'
	@disabled = get_property(attr_map, value, 'bool', 'accessor', 'eval')
      else
	super
      end

      nil
    end

    def disabled
      fetch_value('disabled', @disabled)
    end
  end

  module DisabledWidgetAttribute
    def make_attr_desc
      super{|attr_desc|
	if (@properties.disabled) then
	  attr_desc += ' disabled="disabled"'
	end
	yield(attr_desc)
      }

      nil
    end
    private :make_attr_desc
  end

  module ReadOnlyWidgetProperty
    def mapped(attr_map)
      super
      @readonly = false
      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'readonly'
	@readonly = get_property(attr_map, value, 'bool', 'accessor', 'eval')
      else
	super
      end

      nil
    end

    def readonly
      fetch_value('readonly', @readonly)
    end
  end

  module ReadOnlyWidgetAttribute
    def make_attr_desc
      super{|attr_desc|
	if (@properties.readonly) then
	  attr_desc += ' readonly="readonly"'
	end
	yield(attr_desc)
      }

      nil
    end
    private :make_attr_desc
  end

  class TextFieldWidgetProperties < FormElementWidgetProperties
    WidgetLocator.add('textfield', self)

    include DisabledWidgetProperty
    include ReadOnlyWidgetProperty

    def mapped(attr_map)
      super
      @value = nil
      @size = nil
      @data_type = 'string'
      @nonnil = false
      nil
    end

    def check_mapping
      unless (@value) then
	raise PropertyError, "required `value' property at #{widget_id.inspect}"
      end

      case (@data_type)
      when 'string', 'integer', 'float'
	# nothing to do.
      else
	raise PropertyError, "unknown property value #{@data_type.inspect} of data-type"
      end

      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'value'
	@value = get_property(attr_map, value, 'accessor')
      when 'size'
	@size = get_property(attr_map, value, 'number', 'accessor', 'eval')
      when 'data-type'
	@data_type = get_property(attr_map, value, 'string')
      when 'nonnil'
	@nonnil = get_property(attr_map, value, 'bool')
      else
	super
      end

      nil
    end

    def value
      case (@data_type)
      when 'string'
	fetch_value('value', @value, String)
      when 'integer'
	fetch_value('value', @value, ::Integer).to_s
      when 'float'
	fetch_value('value', @value, ::Float).to_s
      else
	raise "internal error: #{@data_type.inspect}"
      end
    end

    def value=(new_value)
      case (@data_type)
      when 'string'
	if (new_value && new_value.empty? && ! @nonnil) then
	  @value.call(nil)
	else
	  @value.call(new_value)
	end
      when 'integer'
	CheckUtil.check_integer(new_value)
	@value.call(new_value.to_i)
      when 'float'
	CheckUtil.check_float(new_value)
	@value.call(new_value.to_f)
      else
	raise "internal error: #{@data_type.inspect}"
      end
    end

    def has_size?
      @size != nil
    end

    def size
      fetch_value('size', @size, Numeric, true)
    end

    def create_form_element
      TextFieldWidget.new(self)
    end
  end

  class TextFieldWidget < FormControlWidget
    include DisabledWidgetAttribute
    include ReadOnlyWidgetAttribute

    def init
      form_params = form.params
      action_id = self.action_id
      if (form_params.include? action_id) then
	@properties.value = form_params[action_id]
      end

      nil
    end

    def view
      make_attr_desc{|attr_desc|
	attr_desc += ' name="' + escapeHTML(action_id) + '"'
	attr_desc += ' type="text"'
	attr_desc += ' value="' + escapeHTML(@properties.value || '') + '"'
	attr_desc += ' size="' + escapeHTML(@properties.size.to_s) + '"' if @properties.has_size?
	yield('<input' + attr_desc + ' />')
      }

      nil
    end
  end

  class PasswordWidgetProperties < FormElementWidgetProperties
    WidgetLocator.add('password', self)

    include DisabledWidgetProperty
    include ReadOnlyWidgetProperty

    def mapped(attr_map)
      super
      @value = nil
      @size = nil
      nil
    end

    def check_mapping
      unless (@value) then
	raise PropertyError, "required `value' property at #{widget_id.inspect}"
      end

      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'value'
	@value = get_property(attr_map, value, 'accessor')
      when 'size'
	@size = get_property(attr_map, value, 'number', 'accessor', 'eval')
      else
	super
      end

      nil
    end

    def value=(new_value)
      @value.call(new_value)
    end

    def has_size?
      @size != nil
    end

    def size
      fetch_value('size', @size, Numeric, true)
    end

    def create_form_element
      PasswordWidget.new(self)
    end
  end

  class PasswordWidget < FormControlWidget
    include DisabledWidgetAttribute
    include ReadOnlyWidgetAttribute

    def init
      form_params = form.params
      action_id = self.action_id
      if (form_params.include? action_id) then
	@properties.value = form_params[action_id]
      end

      nil
    end

    def view
      make_attr_desc{|attr_desc|
	attr_desc += ' name="' + escapeHTML(action_id) + '"'
	attr_desc += ' type="password"'
	attr_desc += ' size="' + escapeHTML(@properties.size.to_s) + '"' if @properties.has_size?
	yield('<input' + attr_desc + ' />')
      }

      nil
    end
  end

  class CheckboxWidgetProperties < FormElementWidgetProperties
    WidgetLocator.add('checkbox', self)

    include ReadOnlyWidgetProperty
    include DisabledWidgetProperty

    def mapped(attr_map)
      super
      @checked = nil
      @value = nil
      @selected = nil
      nil
    end

    def check_mapping
      unless (@checked || (@value && @selected)) then
	raise PropertyError, "required `checked' or `value' and `selected' property at #{@widget_id.inspect}"
      end

      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'checked'
	@checked = get_property(attr_map, value, 'accessor')
      when 'value'
	@value = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'selected'
	@selected = get_property(attr_map, value, 'accessor')
      else
	super
      end

      nil
    end

    def has_checked?
      @checked != nil
    end

    def checked
      @checked.call
    end

    def checked=(new_checked)
      @checked.call(new_checked)
    end

    def has_value?
      @value != nil
    end

    def value
      fetch_value('value', @value, String)
    end

    def has_selected?
      @selected != nil
    end

    def selected
      fetch_value('selected', @selected, String)
    end

    def selected=(new_selected)
      @selected.call(new_selected)
    end

    def create_form_element
      CheckboxWidget.new(self)
    end
  end

  class CheckboxWidget < FormControlWidget
    include ReadOnlyWidgetAttribute
    include DisabledWidgetAttribute

    def init
      if (form.submit?) then
	form_params = form.params
	action_id = self.action_id

	if (@properties.has_checked?) then
	  if (form_params.include? action_id) then
	    @properties.checked = true
	  else
	    @properties.checked = false
	  end
	end

	if (@properties.has_selected?) then
	  if (form_params.include? action_id) then
	    @properties.selected = form_params[action_id]
	  end
	end
      end

      nil
    end

    def view
      make_attr_desc{|attr_desc|
	attr_desc += ' name="' + escapeHTML(action_id) + '"'
	attr_desc += ' type="checkbox"'
	attr_desc += ' value="' + escapeHTML((@properties.has_value?) ? @properties.value : '') + '"'
	if (@properties.has_checked?) then
	  if (@properties.checked) then
	    attr_desc += ' checked="checked"'
	  end
	elsif (@properties.has_value? && @properties.has_selected?) then
	  if (@properties.value == @properties.selected) then
	    attr_desc += ' checked="checked"'
	  end
	end
	yield('<input' + attr_desc + ' />')
      }

      nil
    end
  end

  class RadioButtonWidgetProperties < FormElementWidgetProperties
    WidgetLocator.add('radio', self)

    include DisabledWidgetProperty
    include ReadOnlyWidgetProperty

    def mapped(attr_map)
      super
      @checked = nil
      @name = nil
      @value = nil
      @selected = nil
      nil
    end

    def check_mapping
      unless (@checked || (@value && @selected)) then
	raise PropertyError, "required `checked' property at #{@widget_id.inspect}"
      end

      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'checked'
	@checked = get_property(attr_map, value, 'accessor')
      when 'name'
	@name = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'value'
	@value = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'selected'
	@selected = get_property(attr_map, value, 'accessor')
      else
	super
      end

      nil
    end

    def has_checked?
      @checked != nil
    end

    def checked
      fetch_value('checked', @checked)
    end

    def checked=(new_checked)
      @checked.call(new_checked)
    end

    def has_name?
      @name != nil
    end

    def name
      fetch_value('name', @name, String, true)
    end

    def has_value?
      @value != nil
    end

    def value
      fetch_value('value', @value, String, true)
    end

    def has_selected?
      @selected != nil
    end

    def selected
      fetch_value('selected', @selected, String)
    end

    def selected=(new_selected)
      @selected.call(new_selected)
    end

    def create_form_element
      RadioButtonWidget.new(self)
    end
  end

  class RadioButtonWidget < FormControlWidget
    include DisabledWidgetAttribute
    include ReadOnlyWidgetAttribute

    def init
      if (form.submit?) then
	form_params = form.params
	name = (@properties.has_name?) ? @properties.name : component_name
	value = (@properties.has_value?) ? @properties.value : action_id
	if (@properties.has_checked?) then
	  @properties.checked = form_params[name] == value
	end
	if (@properties.has_selected?) then
	  if (form_params.include? name) then
	    @properties.selected = form_params[name]
	  end
	end
      end
    end

    def view
      make_attr_desc{|attr_desc|
	attr_desc += ' name="' + escapeHTML((@properties.has_name?) ? @properties.name : component_name) + '"'
	attr_desc += ' type="radio"'
	attr_desc += ' value="' + escapeHTML((@properties.has_value?) ? @properties.value : action_id) + '"'
	if (@properties.has_checked?) then
	  if (@properties.checked) then
	    attr_desc += ' checked="checked"'
	  end
	elsif (@properties.has_value? && @properties.has_selected?) then
	  if (@properties.value == @properties.selected) then
	    attr_desc += ' checked="checked"'
	  end
	end
	yield('<input' + attr_desc + ' />')
      }

      nil
    end
  end

  class SubmitButtonWidgetProperties < FormElementWidgetProperties
    WidgetLocator.add('submit', self)

    include DisabledWidgetProperty
    include ReadOnlyWidgetProperty

    def mapped(attr_map)
      super
      @action = nil
      @value = nil
      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'action'
	@action = get_property(attr_map, value, 'method', 'eval')
      when 'value'
	@value = get_property(attr_map, value, 'string', 'accessor', 'eval')
      else
	super
      end

      nil
    end

    def has_action?
      @action != nil
    end

    def action
      @action.call
    end

    def has_value?
      @value != nil
    end

    def value
      fetch_value('value', @value, String, true)
    end

    def create_form_element
      SubmitButtonWidget.new(self)
    end
  end

  class SubmitButtonWidget < FormControlWidget
    include DisabledWidgetAttribute
    include ReadOnlyWidgetAttribute

    def invoke
      if (@properties.has_action?) then
	form_params = form.params
	action_id = self.action_id
	if (form_params.include? action_id) then
	  @properties.action
	end
      end

      nil
    end

    def view
      make_attr_desc{|attr_desc|
	attr_desc += ' name="' + escapeHTML(action_id) + '"'
	attr_desc += ' type="submit"'
	attr_desc += ' value="' + escapeHTML(@properties.value) + '"' if @properties.has_value?
	yield('<input' + attr_desc + ' />')
      }

      nil
    end
  end

  class ResetButtonWidgetProperties < FormElementWidgetProperties
    WidgetLocator.add('reset', self)

    include DisabledWidgetProperty
    include ReadOnlyWidgetProperty

    def mapped(attr_map)
      super
      @value = nil
      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'value'
	@value = get_property(attr_map, value, 'string', 'accessor', 'eval')
      else
	super
      end

      nil
    end

    def has_value?
      @value != nil
    end

    def value
      fetch_value('value', @value, String, true)
    end

    def create_form_element
      ResetButtonWidget.new(self)
    end
  end

  class ResetButtonWidget < FormControlWidget
    include DisabledWidgetAttribute
    include ReadOnlyWidgetAttribute

    def view
      make_attr_desc{|attr_desc|
	attr_desc += ' type="reset"'
	attr_desc += ' value="' + escapeHTML(@properties.value) + '"' if @properties.has_value?
	yield('<input' + attr_desc + ' />')
      }

      nil
    end
  end

  class HiddenAttributeWidgetProperties < FormElementWidgetProperties
    WidgetLocator.add('hidden', self)

    include DisabledWidgetProperty
    include ReadOnlyWidgetProperty

    def mapped(attr_map)
      super
      @value = nil
      @name = nil
      @data_type = 'string'
      @nonnil = false
      @putonly = false
      nil
    end

    def check_mapping
      unless (@value) then
	raise PropertyError, "required `value' property at #{@widget_id.inspect}"
      end

      case (@data_type)
      when 'string', 'integer', 'float', 'bool'
	# nothing to do.
      else
	raise PropertyError, "unknown property value #{@data_type.inspect} of data-type"
      end

      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'value'
	@value = get_property(attr_map, value, 'accessor')
      when 'name'
	@name = get_property(attr_map, value, 'string', 'accessor', 'eval')
      when 'data-type'
	@data_type = get_property(attr_map, value, 'string')
      when 'nonnil'
	@nonnil = get_property(attr_map, value, 'bool')
      when 'putonly'
	@putonly = get_property(attr_map, value, 'bool', 'accessor', 'eval')
      else
	super
      end

      nil
    end

    def value
      case (@data_type)
      when 'string'
	fetch_value('value', @value, String)
      when 'integer'
	fetch_value('value', @value, ::Integer).to_s
      when 'float'
	fetch_value('value', @value, ::Float).to_s
      when 'bool'
	fetch_value('value', @value) ? 'true' : 'false'
      else
	raise "internal error: #{@data_type.inspect}"
      end
    end

    def value=(new_value)
      case (@data_type)
      when 'string'
	if (new_value && new_value.empty? && ! @nonnil) then
	  @value.call(nil)
	else
	  @value.call(new_value)
	end
      when 'integer'
	CheckUtil.check_integer(new_value)
	@value.call(new_value.to_i)
      when 'float'
	CheckUtil.check_float(new_value)
	@value.call(new_value.to_f)
      when 'bool'
	case (new_value)
	when 'true'
	  @value.call(true)
	when 'false'
	  @value.call(false)
	else
	  raise "not a boolean: #{new_value.inspect}"
	end
      else
	raise "internal error: #{@data_type.inspect}"
      end
    end

    def has_name?
      @name != nil
    end

    def name
      fetch_value('name', @name, String, true)
    end

    def putonly
      fetch_value('putonly', @putonly)
    end

    def create_form_element
      HiddenAttributeWidget.new(self)
    end
  end

  class HiddenAttributeWidget < FormControlWidget
    include DisabledWidgetAttribute
    include ReadOnlyWidgetAttribute

    def init
      unless (@properties.putonly) then
	form_params = form.params
	if (@properties.has_name?) then
	  name = @properties.name
	else
	  name = action_id
	end
	if (form_params.include? name) then
	  @properties.value = form_params[name]
	end
      end

      nil
    end

    def view
      make_attr_desc{|attr_desc|
	attr_desc += ' name="' + escapeHTML((@properties.has_name?) ? @properties.name : action_id) + '"'
	attr_desc += ' type="hidden"'
	attr_desc += ' value="' + escapeHTML(@properties.value || '') + '"'
	yield('<input' + attr_desc + ' />')
      }

      nil
    end
  end

  class TextAreaWidgetProperties < FormElementWidgetProperties
     WidgetLocator.add('textarea', self)

    include DisabledWidgetProperty
    include ReadOnlyWidgetProperty

    def mapped(attr_map)
      super
      @value = nil
      @rows = nil
      @cols = nil
      nil
    end

    def check_mapping
      unless (@value) then
	raise "required `value' property at #{@widget_id.inspect}"
      end

      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'value'
	@value = get_property(attr_map, value, 'accessor')
      when 'rows'
	@rows = get_property(attr_map, value, 'number', 'accessor', 'eval')
      when 'cols'
	@cols = get_property(attr_map, value, 'number', 'accessor', 'eval')
      else
	super
      end

      nil
    end

    def value
      fetch_value('value', @value, String)
    end

    def value=(new_value)
      @value.call(new_value)
    end

    def has_rows?
      @rows != nil
    end

    def rows
      fetch_value('rows', @rows, Numeric, true)
    end

    def has_cols?
      @cols != nil
    end

    def cols
      fetch_value('cols', @cols, Numeric, true)
    end

    def create_form_element
      TextAreaWidget.new(self)
    end
  end

  class TextAreaWidget < FormControlWidget
    include DisabledWidgetAttribute
    include ReadOnlyWidgetAttribute

    def init
      form_params = form.params
      action_id = self.action_id
      if (form_params.include? action_id) then
	@properties.value = form_params[action_id]
      end

      nil
    end

    def view
      make_attr_desc{|attr_desc|
	attr_desc += ' name="' + escapeHTML(action_id) + '"'
	attr_desc += ' rows="' + escapeHTML(@properties.rows.to_s) + '"' if @properties.has_rows?
	attr_desc += ' cols="' + escapeHTML(@properties.cols.to_s) + '"' if @properties.has_cols?
	yield('<textarea' + attr_desc + '>')
	yield(escapeHTML(@properties.value || ''))
	yield('</textarea>')
      }

      nil
    end
  end

  class SelectWidgetProperties < FormElementWidgetProperties
    WidgetLocator.add('select', self)

    include DisabledWidgetProperty

    def mapped(attr_map)
      super
      @list = nil
      @size = nil
      @selected = nil
      nil
    end

    def check_mapping
      unless (@list) then
	raise PropertyError, "required `list' property at #{@widget_id.inspect}"
      end

      nil
    end

    def set_property(name, attr_map, value)
      case (name)
      when 'list'
	@list = get_property(attr_map, value, 'accessor', 'eval')
      when 'size'
	@size = get_property(attr_map, value, 'number', 'accessor', 'eval')
      when 'selected'
	@selected = get_property(attr_map, value, 'accessor')
      else
	super
      end

      nil
    end

    def list
      fetch_value('list', @list, Enumerable, true)
    end

    def has_size?
      @size != nil
    end

    def size
      fetch_value('size', @size, Numeric, true)
    end

    def has_selected?
      @selected != nil
    end

    def selected
      fetch_value('selected', @selected, String)
    end

    def selected=(new_selected)
      @selected.call(new_selected)
    end

    def create_form_element
      SelectWidget.new(self)
    end
  end

  class SelectWidget < FormControlWidget
    include DisabledWidgetAttribute

    def init
      if (@properties.has_selected?) then
	form_params = form.params
	action_id = self.action_id
	if (form_params.include? action_id) then
	  @properties.selected = form_params[action_id]
	end
      end

      nil
    end

    def view
      make_attr_desc{|attr_desc|
	attr_desc += ' name="' + escapeHTML(action_id) + '"'
	attr_desc += ' size="' + escapeHTML(@properties.size.to_s) + '"' if @properties.has_size?
	yield('<select' + attr_desc + '>')
	for opt_value, opt_label in @properties.list
	  opt_label = opt_value unless opt_label
	  opt_attr_desc = ' value="' + escapeHTML(opt_value) + '"'
	  if (@properties.has_selected? && opt_value == @properties.selected)
	    opt_attr_desc += ' selected="selected"'
	  end
	  yield('<option' + opt_attr_desc + '>')
	  yield(escapeHTML(opt_label))
	  yield('</option>')
	end
	yield('</select>')
      }

      nil
    end
  end

  class ComponentAssembler < XMLReader
    include Escape

    def initialize(widget_locator, root_component)
      @widget_locator = widget_locator
      @root_component = root_component
      @depth = 0
      @stack = [ @root_component ]
      @ns_map = XMLNamespaceMap.new
    end

    attr_accessor :root_component

    def start_element(ns_uri, prefix, name, attr_map)
      @depth += 1
      case (ns_uri)
      when XMLNS_DOC_URI
	case (name)
	when 'widget'
	  widget_id = attr_map['name']
	  unless (widget_id) then
	    raise WidgetBindingError, 'not found a name attribute'
	  end
	  widget =  @widget_locator.create_widget(widget_id)
	  unless (widget) then
	    raise WidgetBindingError, "not found a widget: #{widget_id.inspect}"
	  end
	  @stack.last.add(widget)
	  @stack.push(widget)
	when 'import'
	  if (@depth != 1) then
	    raise WidgetBindingError, "not a root element: #{name}@#{ns_uri}"
	  end
	else
	  raise WidgetBindingError, "unknown element: #{name}@#{ns_uri}"
	end
      else
	element = Element.new(ns_uri, prefix, name, attr_map)
	@stack.last.add(element)
	@stack.push(element)
      end

      nil
    end

    def end_element(ns_uri, prefix, name)
      case (ns_uri)
      when XMLNS_DOC_URI
	case (name)
	when 'widget'
	  @stack.pop
	else
	  # nothing to do.
	end
      else
	@stack.pop
      end
      @depth -= 1

      nil
    end

    def processing_instruction(target, data)
    end

    def character(data)
      string = StringComponent.new(escapeHTML(data))
      @stack.last.add(string)
      nil
    end

    def comment(data)
      string = StringComponent.new('<!-- ' + data + ' -->')
      @stack.last.add(string)
      nil
    end
  end

  class Loader
    extend Forwardable

    def initialize(filename)
      @mod = Module.new
      File.open(filename) {|input|
	@mod.module_eval(input.read, filename)
      }
    end

    def_delegator :@mod, :const_get
  end

  class LoadPath
    def initialize(path)
      @path = path
      @list = Array.new
    end

    def first_path
      @path
    end

    def scan
      yield(@path)
      for path in @list
	yield(path)
      end

      nil
    end
    private :scan

    def look_up(name)
      scan do |path|
	compo_path = File.join(path, name)
	if (File.directory? compo_path) then
	  return compo_path
	end
      end

      nil
    end

    def add(path)
      @list.push(path)
      nil
    end

    def <<(path)
      add(path)
      self
    end
  end

  class PageCache
    def initialize
      @map = Hash.new
      @lock = Mutex.new
    end

    def store(path, page_id, cache_entry)
      @lock.synchronize{
	@map[[path, page_id]] = WeakRef.new(cache_entry)
      }

      nil
    end

    def fetch(path, page_id)
      @lock.synchronize{
	if (cache_ref = @map[[path, page_id]]) then
	  begin
	    if (cache_ref.weakref_alive?) then
	      return cache_ref.__getobj__
	    end
	  rescue WeakRef::RefError
	    # nothing to do.
	  end
	  @map.delete([path, page_id])
	end
      }

      nil
    end
  end

  class RunningContext
    extend Forwardable

    def initialize(load_path, reserve_last_cache=false)
      @load_path = load_path
      @reserve_last_cache = reserve_last_cache
      @page_cache = PageCache.new
    end

    def_delegator :@load_path, :first_path, :base_dir
    def_delegator :@load_path, :look_up

    def new_page(name, xml_assist, option={}, parent_component=nil)
      path = @load_path.look_up(name)
      page_id = Page.page_id(name, parent_component)
      if (page = @page_cache.fetch(path, page_id)) then
	if (! page.modified?) then
	  if (page.parent_component == parent_component) then
	    yield(page) if iterator?
	    page.recompile
	    return page
	  end
	end
      end

      page = Page.new(name, self, xml_assist, option)
      page.parent_component = parent_component
      yield(page) if iterator?
      page.compile
      @page_cache.store(path, page_id, page)

      page
    end

    def self.eval_page(page)
      page.call_reset
      page.setup
      page.call_reset
      page.run
      page.call_reset
      messg_body = ''
      page.make_page{|s| messg_body << s }
      messg_body
    end

    def transaction(driver)
      begin
	Thread.current[:_wpm_context_] = {
	  :driver => driver,
	  :ns_map => nil,
	  :page_context => Hash.new,
	  :call_index => Hash.new,
	  :curr_widget => nil,
	  :last_error_page => nil,
	  :last_error_widget => nil,
	  :last_error_info => nil
	}
	return yield
      ensure
        Thread.current[:_wpm_context_] = nil unless @reserve_last_cache
      end
    end

    def self.safe_run(safe_level=$SAFE)
      curr_context = Thread.current[:_wpm_context_]
      Thread.new{
	Thread.current[:_wpm_context_] = curr_context
	$SAFE = safe_level
	yield
      }.value
    end

    def curr_context
      Thread.current[:_wpm_context_]
    end
    private :curr_context

    def driver
      curr_context[:driver]
    end

    def ns_map
      curr_context[:ns_map]
    end

    def ns_map=(new_map)
      curr_context[:ns_map] = new_map
    end

    def page_context(name)
      curr_context[:page_context][name]
    end

    def set_page_context(name, page_context)
      curr_context[:page_context][name] = page_context
      nil
    end

    def call_reset(name)
      curr_context[:call_index][name] = -1
      nil
    end

    def call_count(name)
      curr_context[:call_index][name] += 1
      nil
    end

    def call_index(name)
      curr_context[:call_index][name]
    end

    def curr_widget
      curr_context[:curr_widget]
    end

    def curr_widget=(new_widget)
      curr_context[:curr_widget] = new_widget
    end

    def last_error_page
      curr_context[:last_error_page]
    end

    def last_error_page=(new_page)
      curr_context[:last_error_page] = new_page
    end

    def last_error_widget
      curr_context[:last_error_widget]
    end

    def last_error_widget=(new_widget)
      curr_context[:last_error_widget] = new_widget
    end

    def last_error_info
      curr_context[:last_error_info]
    end

    def last_error_info=(new_info)
      curr_context[:last_error_info] = new_info
    end
  end

  class PageContextProxy < Delegator
    def initialize(page_id, running_context)
      @page_id = page_id
      @running_context = running_context
      super(__getobj__)
    end

    def __getobj__
      @running_context.page_context(@page_id)
    end

    def method(*args)
      begin
	super
      rescue Exception
	$!.message.gsub!(/WPM::PageContextProxy/, __getobj__.class.to_s)
	raise
      end
    end
  end

  class Page < Component
    extend Forwardable
    include ComponentParent

    NULL_PROC = proc{|context|
      # nothing to do.
    }

    def initialize(name, running_context, xml_assist, option={})
      super()
      @name = name
      @running_context = running_context
      @xml_assist = xml_assist
      @option = option
      @widget_locator = nil
      @src_path = nil
      @map_path = nil
      @xml_path = nil
      @src_stat = nil
      @map_stat = nil
      @xml_stat = nil
      @context_class = nil
      @context_hook = NULL_PROC
      @context = nil
    end

    def_delegator :@running_context, :driver
    def_delegator :@running_context, :ns_map
    def_delegator :@running_context, :base_dir
    def_delegator :@running_context, :curr_widget
    def_delegator :@running_context, :curr_widget=
    def_delegator :@running_context, :last_error_page
    def_delegator :@running_context, :last_error_widget
    def_delegator :@running_context, :last_error_info
    def_delegator :@widget_locator, :expand_string

    attr_reader :xml_assist
    attr_reader :context

    def new_page(name, parent_component=nil, &block)
      @running_context.new_page(name, @xml_assist, @option, parent_component, &block)
    end

    def self.page_name(name, parent_component=nil)
      if (parent_component) then
	return parent_component.component_name + '.' + name
      else
	return name
      end
    end

    def self.page_id(name, parent_component=nil)
      name = page_name(name, parent_component)
      if (parent_component) then
	return name + ':' + parent_component.instance_number.to_s
      else
	return name + ':0'
      end
    end

    def page_name
      Page.page_name(@name, @parent_component)
    end

    def page_id
      Page.page_id(@name, @parent_component)
    end

    def component_name
      page_name
    end

    def context_hook(&block)
      @context_hook = block
      nil
    end

    # for debug
    def clear
      @child_components.clear
      @context = nil
      @context_hook = NULL_PROC
    end

    def modified?
      stat = File.stat(@src_path)
      if (stat.mtime != @src_stat.mtime || stat.ino != @src_stat.ino) then
	return true
      end

      stat = File.stat(@map_path)
      if (stat.mtime != @map_stat.mtime || stat.ino != @map_stat.ino) then
	return true
      end

      stat = File.stat(@xml_path)
      if (stat.mtime != @xml_stat.mtime || stat.ino != @xml_stat.ino) then
	return true
      end

      false
    end

    def compile
      unless (@running_context.ns_map) then
	@running_context.ns_map = XMLNamespaceMap.new
      end

      if (@name !~ /^[A-Z][_0-9A-Za-z]*$/) then
	raise PageError, "invalid page name: #{@name.inspect}"
      end

      compo_path = @running_context.look_up(@name)
      if (! compo_path) then
	raise "not found a component: #{@name.inspect}"
      end
      @src_path = File.join(compo_path, @name + '.rb')
      @map_path = File.join(compo_path, @name + '.map')
      @xml_path = File.join(compo_path, @name + '.xml')
      @src_stat = File.stat(@src_path)
      @map_stat = File.stat(@map_path)
      @xml_stat = File.stat(@xml_path)

      loader = Loader.new(@src_path)
      @context_class = loader.const_get(@name)
      unless (@context_class < PageContext) then
	raise PageError, "not inherited from #{PageContext.inspect}: #{@context_class}"
      end

      page_id = self.page_id
      @running_context.set_page_context(page_id, @context_class.new(self, @option))
      @context = PageContextProxy.new(page_id, @running_context)
      @widget_locator = WidgetLocator.new(@context, @running_context)
      assembler = ComponentAssembler.new(@widget_locator, self)

      @widget_locator.extend(@xml_assist)
      assembler.extend(@xml_assist)

      File.open(@map_path) {|input|
	@widget_locator.read(input)
      }
      File.open(@xml_path) {|input|
	assembler.read(input)
      }

      # traversing child components.
      super

      nil
    end

    def recompile
      unless (@running_context.ns_map) then
	@running_context.ns_map = XMLNamespaceMap.new
      end

      page_id = self.page_id
      @running_context.set_page_context(page_id, @context_class.new(self, @option))

      # traversing child components.
      super

      nil
    end

    def setup
      @context.init_context
      @context_hook.call(@context)

      # traversing child components.
      super

      nil
    end

    def run
      # traversing child components.
      super

      nil
    end

    def make_page
      for component in @child_components
	component.make_page do |page_text|
	  yield(page_text)
	end
      end

      nil
    end
  end

  class Writer
    extend Forwardable
    include Escape

    NO_FILTER = MessageManipulator::NO_FILTER

    def initialize(xml_assist)
      # Page
      @xml_assist = xml_assist
      @main_page = 'MainPage'
      @page_option = Hash.new
      @error_display = 'WPMErrorPage'
      self.base_dir = Dir.getwd

      # HTTP
      @status = 200
      @content_type = 'text/html'
      @charset = nil

      # Logging
      @logger = DummyLogger.new
      @filename = nil
      @input_filter = NO_FILTER
      @output_filter = NO_FILTER
    end

    def_delegator :@running_context, :base_dir

    def base_dir=(new_dir)
      @load_path = LoadPath.new(new_dir)
      @load_path << COMPO_PATH
      @running_context = RunningContext.new(@load_path)
      new_dir
    end

    attr_accessor :main_page
    attr_accessor :status
    attr_accessor :content_type
    attr_accessor :charset
    attr_accessor :logger

    def setup_logfile
      if (@filename) then
	@logger = AppendLogger.new(@filename, @input_filter, @output_filter)
      end
      nil
    end
    private :setup_logfile

    def set_logfile(filename)
      @filename = filename
      setup_logfile
      nil
    end

    def set_input_filter(&block)
      @input_filter = block
      setup_logfile
      nil
    end

    def set_output_filter(&block)
      @output_filter = block
      setup_logfile
      nil
    end

    def set_page_option(name, value)
      @page_option[name] = value
      nil
    end

    def eval_page(driver, page, http_method)
      messg_body = RunningContext.eval_page(page)
      driver.set_header('Content-Length', messg_body.length.to_s)
      driver.write(messg_body) if (http_method != 'HEAD')
      nil
    end
    private :eval_page

    def debug_print(driver, exception, http_method)
      driver.set_header('Content-Type', 'text/html')
      if (http_method != 'HEAD') then
	driver.write("<html>\n")
	driver.write("<head><title>500 Internal Server Error</title></head>\n")
	driver.write("<body>\n")
	driver.write("<h1>500 Internal Server Error</h1>\n")
	driver.write("<h2>Error message</h2>\n")
	driver.write("<p>#{escapeHTML(exception.message)}</p>\n")
	driver.write("<h2>Error type</h2>\n")
	driver.write("<p>#{escapeHTML(exception.class.to_s)}</p>\n")
	driver.write("<h2>Backtrace</h2>\n")
	if (exception.backtrace) then
	  driver.write("<ol>\n")
	  for frame in exception.backtrace
	    driver.write("<li>#{escapeHTML(frame)}</li>\n")
	  end
	  driver.write("</ol>\n")
	  driver.write("</body>\n")
	  driver.write("</html>\n")
	else
	  driver.write("<p><em>None.</em></p>\n")
	end
      end

      nil
    end
    private :debug_print

    def run(driver)
      # setup driver
      driver = MessageManipulator.new(driver)
      driver.set_input_filter(&@input_filter)
      driver.set_output_filter(&@output_filter)

      # setup response
      driver.set_status(@status)
      if (@charset && @content_type =~ %r"text/\S+"i) then
	driver.set_header('Content-Type', "#{@content_type}; charset=#{@charset}")
      else
	driver.set_header('Content-Type', @content_type)
      end

      # run page
      begin
	@running_context.transaction(driver) {
	  http_method = driver.env['REQUEST_METHOD']
	  begin
	    case (http_method)
	    when 'GET', 'HEAD', 'POST'
	      if (page_name = driver.curr_page) then
		page = @running_context.new_page(page_name, @xml_assist, @page_option)
		eval_page(driver, page, http_method)
	      else
		driver.set_location(driver.page_path(@main_page))
		driver.set_header('Content-Type', 'text/plain')
		driver.write("Jump to #{driver.page_path(@main_page)}.") if (http_method != 'HEAD')
	      end
	    else
	      driver.set_status(405)
	      driver.set_header('Allow', 'GET, HEAD, POST')
	      driver.set_header('Content-Type', 'text/plain')
	      driver.write("Method Not Alllowed.\n") if (http_method != 'HEAD')
	    end
	  rescue StandardError, ScriptError
	    exception = $!
	    driver.set_status(500)
	    if (@load_path.look_up(@error_display)) then
	      begin
		@running_context.last_error_page = page_name
		@running_context.last_error_widget = @running_context.curr_widget
		@running_context.last_error_info = exception
		error_page = @running_context.new_page(@error_display, @xml_assist, @page_option)
		eval_page(driver, error_page, http_method)
	      rescue StandardError, ScriptError
		# debug for error display
		debug_print(driver, $!, http_method)
	      end
	    else
	      driver.set_header('Content-Type', 'text/plain')
	      driver.write($!.to_s) if (http_method != 'HEAD')
	    end
	  ensure
	    @logger.write_access(driver.header('Status'), driver.env, driver.params)
	  end
	}
      ensure
	driver.close
      end

      nil
    end
  end
end
