# $Id: monitor.rb,v 1.8 2004/10/12 14:36:38 toki Exp $

require 'forwardable'

module Rucy
  class RuntimeErrorSample
    extend Forwardable

    def initialize(exception, timestamp=Time.now)
      @exception = exception
      @timestamp = timestamp
    end

    def_delegator :@exception, :message, :error_message
    def_delegator :@exception, :class, :error_type
    def_delegator :@exception, :backtrace
    attr_reader :timestamp
  end

  class LoadErrorSample < RuntimeErrorSample
    def initialize(exception, mod_path, timestamp=Time.now)
      super(exception, timestamp)
      @mod_path = mod_path
    end

    attr_reader :mod_path
  end

  class LoggingErrorSample < RuntimeErrorSample
    def initialize(exception, log_path, timestamp=Time.now)
      super(exception, timestamp)
      @log_path = log_path
    end

    attr_reader :log_path
  end

  class AliasErrorSample < RuntimeErrorSample
    def initialize(exception, alias_path, orig_path, virtual_host, timestamp=Time.now)
      super(exception, timestamp)
      @alias_path = alias_path
      @orig_path = orig_path
      @virtual_host = virtual_host
    end

    attr_reader :alias_path
    attr_reader :orig_path
    attr_reader :virtual_host
  end

  class DocumentErrorSample < RuntimeErrorSample
    def initialize(exception, doc_name, doc_args, mount_path, mount_mask, virtual_host, timestamp=Time.now)
      super(exception, timestamp)
      @document = doc_name
      @arguments = doc_args
      @mount_path = mount_path
      @mount_mask = mount_mask
      @virtual_host = virtual_host
    end

    attr_reader :document
    attr_reader :mount_path
    attr_reader :mount_mask
    attr_reader :virtual_host

    def arguments
      @arguments.map{|name, value| [ name, value ] } # deep copy
    end
  end

  class FilterErrorSample < RuntimeErrorSample
    def initialize(exception, filter_name, filter_args, attach_path, attach_mask, virtual_host, timestamp=Time.now)
      super(exception, timestamp)
      @filter = filter_name
      @arguments = filter_args
      @attach_path = attach_path
      @attach_mask = attach_mask
      @virtual_host = virtual_host
    end

    attr_reader :filter
    attr_reader :attach_path
    attr_reader :attach_mask
    attr_reader :virtual_host

    def arguments
      @arguments.map{|name, value| [ name, value ] } # deep copy
    end
  end

  class SetupErrorMonitor
    def initialize
      @load_errors = Array.new
      @logging_errors = Array.new
      @alias_errors = Array.new
      @document_errors = Array.new
      @filter_errors = Array.new
    end

    def load_errors
      @load_errors.dup
    end

    def logging_errors
      @logging_errors.dup
    end

    def alias_errors
      @alias_errors.dup
    end

    def document_errors
      @document_errors.dup
    end

    def filter_errors
      @filter_errors.dup
    end

    def capture_load_error(exception, mod_path)
      @load_errors.push(LoadErrorSample.new(exception, mod_path))
      nil
    end

    def capture_logging_error(exception, log_path)
      @logging_errors.push(LoggingErrorSample.new(exception, log_path))
      nil
    end

    def capture_alias_error(exception, alias_path, orig_path, virtual_host)
      @alias_errors.push(AliasErrorSample.new(exception, alias_path, orig_path, virtual_host))
      nil
    end

    def capture_document_error(exception, doc_name, doc_args, mount_path, mount_mask, virtual_host)
      @document_errors.push(DocumentErrorSample.new(exception, doc_name, doc_args, mount_path, mount_mask, virtual_host))
      nil
    end

    def capture_filter_error(exception, filter_name, filter_args, attach_path, attach_mask, virtual_host)
      @filter_errors.push(FilterErrorSample.new(exception, filter_name, filter_args, attach_path, attach_mask, virtual_host))
      nil
    end
  end
end
