# $Id: messenger.rb,v 1.13 2004/12/11 14:32:39 toki Exp $

require 'time'
require 'thread'
require 'rucy/wait'
require 'rucy/priv'
require 'rucy/writer'
require 'rucy/error'
require 'rucy/request'
require 'rucy/response'

module Rucy
  class SocketQueue
    CMD_CLOSE = :close

    def initialize(size=16)
      @size = size
      @sock_queue = Array.new
      @cmd_queue = Array.new
      @lock = Mutex.new
      @push_cond = ConditionVariable.new
      @pop_cond = ConditionVariable.new
    end

    def empty?
      @lock.synchronize{
	@sock_queue.empty? && @cmd_queue.empty?
      }
    end

    def resize(new_size)
      @lock.synchronize{
	@size = new_size
      }
      nil
    end

    def push(socket)
      @lock.synchronize{
	while (@sock_queue.size >= @size)
	  @push_cond.wait(@lock)
	end
	@sock_queue.push(socket)
	@pop_cond.signal
      }
      nil
    end

    def push_close
      @lock.synchronize{
	@cmd_queue.push(CMD_CLOSE)
	@pop_cond.signal
      }
      nil
    end

    def pop
      @lock.synchronize{
	while (@sock_queue.empty? && @cmd_queue.empty?)
	  @pop_cond.wait(@lock)
	end

	if (! @cmd_queue.empty?) then
	  return @cmd_queue.shift
	elsif (! @sock_queue.empty?) then
	  socket = @sock_queue.shift
	  @push_cond.signal
	  return socket
	end
      }
    end
  end

  class Messenger
    TIMEOUT = 60 * 5		# seconds
    KEEP_ALIVE = 8		# seconds
    MAX_REQUESTS = 32
    QUEUE_LENGTH = 16
    MESSENGERS = 8
    MESSENGER_THREADS = 4
    MESSENGER_QUEUE_LENGTH = 4

    def initialize(document, logger, access_log)
      @document = document
      @logger = logger
      @access_log = access_log
      @closed = false
      @timeout = TIMEOUT
      @keep_alive = KEEP_ALIVE
      @max_requests = MAX_REQUESTS
    end

    attr_accessor :timeout
    attr_accessor :keep_alive
    attr_accessor :max_requests

    def open(server)
      # hook for subclasses.
    end

    def close
      # hook for subclasses.
    end

    def closed?
      @closed
    end

    def accept(queue)
      @closed and raise 'closed'
      begin
	loop do
	  case (socket = queue.pop)
	  when SocketQueue::CMD_CLOSE
	    break
	  else
	    begin
	      receive(socket)
	    rescue StandardError, ScriptError
	      @logger.warn("failed to dispatch: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}")
	    ensure
	      begin
		socket.close unless socket.closed?
	      rescue
		@logger.warn("failed to close socket: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}")
	      end
	    end
	  end
	end
      ensure
	begin
	  close
	rescue
	  @logger.warn("failed to close messenger: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}")
	ensure
	  @closed = true
	end
      end

      nil
    end
  end

  class MultiThreadMessenger < Messenger
    def build_response
      response = Response.new
      response.set_header('Server', SERVER_TOKEN_LIST)
      response.set_header('Date', Time.now.httpdate)

      response
    end
    private :build_response

    def receive(socket)
      req_count = 0
      loop do
	begin
	  req_count += 1
	  request = Request.new
	  srv_family, srv_port, srv_host, srv_addr = socket.addr
	  request.set_server(srv_host, srv_addr, srv_port)
	  cli_family, cli_port, cli_host, cli_addr = socket.peeraddr
	  request.set_client(cli_host, cli_addr, cli_port)
	  unless (Wait.wait(socket, req_count > 1 ? @keep_alive : @timeout)) then
	    @logger.debug("close connection.")
	    break
	  end
	  request.parse(socket)
	  request.set_reader(socket)
	  if (request.version == 'HTTP/1.1') then
	    unless (request.has_header? 'Host') then
	      raise ParseError, 'required Host header.'
	    end
	  end

	  response = build_response
	  if (req_count >= @max_requests || request.conn_closed?) then
	    response.conn_close
	    writer = HTTPThroughWriter.new(socket, request)
	  else
	    writer = HTTPSpoolWriter.new(socket, request)
	  end
	  response.set_writer(writer)

	  @document.publish('', request, response, @logger)
	  writer.close
	  @access_log.write_log(request, response, Time.now)
	rescue ParseError
	  if (! writer || ! writer.writing?) then
	    write_parse_error(socket, request, $!)
	  end
	  break			# abort connection
	rescue HTTPError
	  if (! writer || ! writer.writing?) then
	    if (write_http_error(socket, request, $!, req_count)) then
	      break		# abort connection
	    else
	      next		# keep-alive connection
	    end
	  else
	    break		# abort connection
	  end
	rescue StandardError, ScriptError
	  @logger.warn("failed to receive request: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}")
	  if (! writer || ! writer.writing?) then
	    write_server_error(socket, request, $!)
	  end
	  break			# abort connection
	end

	if (request.conn_closed? || response.conn_closed?) then
	  break
	end
      end

      nil
    end

    def write_error_page(socket, request)
      response = build_response
      response.set_header('Content-Type', 'text/plain')
      response.conn_close

      writer = HTTPSpoolWriter.new(socket, request)
      response.set_writer(writer)
      yield(response)
      writer.close
      @access_log.write_log(request, response, Time.now)

      nil
    end
    private :write_error_page

    def write_parse_error(socket, request, exception)
      write_error_page(socket, request) { |response|
	response.status = 400	# Bad Request
	response.start_body
	if (request.method != 'HEAD') then
	  response << exception.message << "\n"
	end
      }

      nil
    end
    private :write_parse_error

    def write_http_error(socket, request, exception, req_count)
      err_messg = exception.message + "\n"
      response = build_response
      response.status = exception.status
      response.set_header('Content-Type', 'text/plain')
      response.set_header('Content-Length', err_messg.length.to_s)
      exception.each_header do |name, value|
	response.set_header(name, value)
      end
      if (req_count >= @max_requests || request.conn_closed?) then
	response.conn_close
	writer = HTTPThroughWriter.new(socket, request)
      else
	case (response.status)
	when 200...400, 401, 404, 412, 416
	  writer = HTTPSpoolWriter.new(socket, request)
	else
	  response.conn_close
	  writer = HTTPThroughWriter.new(socket, request)
	end
      end
      response.set_writer(writer)
      response.start_body
      if (request.method != 'HEAD') then
	response << err_messg
      end
      writer.close
      @access_log.write_log(request, response, Time.now)

      # Keep-Alive check
      request.conn_closed? || response.conn_closed?
    end
    private :write_http_error

    def write_timeout_error(socket, request, exception)
      write_error_page(socket, request) { |response|
	response.status = 408	# Request Timeout
	response.start_body
	if (request.method != 'HEAD') then
	  response << exception.message << "\n"
	end
      }

      nil
    end
    private :write_timeout_error

    def write_server_error(socket, request, exception)
      write_error_page(socket, request) { |response|
	response.status = 500
	response.start_body
	if (request.method != 'HEAD') then
	  response << exception.message << "\n"
	end
      }

      nil
    end
    private :write_server_error
  end

  module MessengerParent
    def start_messengers(num_of_messengers, socket_queue, document, logger, access_log, factory, restart_signal=nil)
      messenger_list = Array.new
      num_of_messengers.times do
	messenger = factory.new(document, logger, access_log)
	messenger.timeout = self.timeout
	messenger.keep_alive = self.keep_alive
	messenger.max_requests = self.max_requests
	messenger.open(self)
	messenger_list.push(messenger)
      end

      messenger_thgrp = ThreadGroup.new
      if (restart_signal) then
	thread = Thread.new{
	  restart_signal.wait
	}
	messenger_thgrp.add(thread)
      end

      messenger_list.each do |messenger|
	thread = Thread.new{
	  logger.info("start #{factory} thread.")
	  begin
	    messenger.accept(socket_queue)
	  rescue StandardError, ScriptError
	    logger.warning("aborted #{factory} thread: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}")
	  ensure
	    logger.info("stop #{factory} thread.")
	  end
	}
	messenger_thgrp.add(thread)
      end

      messenger_thgrp
    end
    module_function :start_messengers

    def stop_messengers(num_of_messengers, socket_queue, messenger_thgrp, restart_signal=nil)
      num_of_messengers.times do
	socket_queue.push_close
      end
      if (restart_signal) then
	restart_signal.cancel
      end
      for thread in messenger_thgrp.list
	thread.join
      end

      nil
    end
    module_function :stop_messengers
  end

  class MultiProcessMessenger < Messenger
    include MessengerParent

    CMD_LEN = 1
    SEND_IO = 'S'
    RECV_IO = 'R'
    CLOSE   = 'C'

    attr_reader :messenger_threads
    attr_reader :messenger_queue_length

    def open(server)
      @messenger_threads = server.messenger_threads
      @messenger_queue_length = server.messenger_queue_length
      @privilege = server.privilege

      @child_socket, @parent_socket = UNIXSocket.socketpair
      if (@pid = fork) then
	@child_socket.close
      else
	@parent_socket.close
	begin
	  child_process
	rescue StandardError, ScriptError
	  @logger.err("error: aborted child process: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}")
	ensure
	  exit!
	end
      end

      nil
    end

    def receive(socket)
      @parent_socket.write(SEND_IO)
      @parent_socket.send_io(socket); socket.close
      @parent_socket.read(CMD_LEN)
      nil
    end

    def close
      @parent_socket.write(CLOSE)
      @parent_socket.read(CMD_LEN)
      @parent_socket.close
      Process.waitpid(@pid)
      nil
    end

    def child_process
      if (@privilege.privileged_user?) then
	begin
	  @privilege.cancel_privilege
	rescue
	  @logger.err("error: failed to change subprocess privilege: #{$!.message} (#{$!.class}): #{$!.backtrace[0]}")
	end
      end

      socket_queue = SocketQueue.new(@messenger_queue_length)
      messenger_thgrp = start_messengers(@messenger_threads, socket_queue, @document, @logger, @access_log, MultiThreadMessenger)
      while (cmd = @child_socket.read(CMD_LEN))
	case (cmd)
	when SEND_IO
	  socket = @child_socket.recv_io(TCPSocket)
	  socket_queue.push(socket)
	  @child_socket.write(RECV_IO)
	when CLOSE
	  until (socket_queue.empty?)
	    sleep(0.1)
	  end
	  stop_messengers(@messenger_threads, socket_queue, messenger_thgrp)
	  @child_socket.write(CLOSE)
	  @child_socket.close
	  break
	else
	  raise 'internal error.'
	end
      end

      nil
    end
    private :child_process
  end
end
