// -*- C++ -*-
/*!
 * @file Time_posix.h
 * @brief Time functions
 * @date $Date$
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2008
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#ifndef COIL_TIME_H
#define COIL_TIME_H

#ifdef __T_KERNEL__
#include <nes_posix_unistd.h>
#include <nes_posix_time.h>
#include <nes_posix_proc.h>
#include <btron/bsocket.h>
#else
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#endif
#include <time.h>
#include <iostream>
#include <coil/config_coil.h>
#include <coil/TimeValue.h>

namespace coil
{
#ifdef __T_KERNEL__
#define USEC_IN_NSECS 1000.0 // 1 [usec] = 1000 [nsec]
#endif
  inline unsigned int sleep(unsigned int seconds)
  {
    return ::sleep(seconds);
  }

  inline int sleep(TimeValue interval)
  {
    timeval tv;
    tv.tv_sec = interval.sec();
    tv.tv_usec = interval.usec();
#ifdef __T_KERNEL__
    return so_select(0, 0, 0, 0, &tv);
#else
    return ::select(0, 0, 0, 0, &tv);
#endif
  }

  inline int usleep(useconds_t usec)
  {
    return ::usleep(usec);
  }

  inline int gettimeofday(struct timeval *tv, struct timezone *tz)
  {
    return ::gettimeofday(tv, tz);
  }

  inline TimeValue gettimeofday()
  {
    timeval tv;
#ifdef __T_KERNEL__
    ::gettimeofday(&tv, NULL);
#else
    ::gettimeofday(&tv, 0);
#endif
    return TimeValue(tv.tv_sec, tv.tv_usec);
  }
  
#ifdef __T_KERNEL__
  inline int clock_gettime(clockid_t clock_id, struct timespec *tp)
  {
    return ::clock_gettime(clock_id, tp);
  }

  inline TimeValue clock_gettime()
  {
    timespec tp;
    ::clock_gettime(CLOCK_REALTIME, &tp);
    return TimeValue(tp.tv_sec, (int)(tp.tv_nsec / USEC_IN_NSECS));
  }
#endif

  inline int settimeofday(const struct timeval *tv , const struct timezone *tz)
  {
#ifdef __T_KERNEL__
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    return 0;
#else
    return ::settimeofday(tv, tz);
#endif
  }


};

#endif // COIL_TIME_H
