// -*- C++ -*-
/*!
 * @file Singleton.h
 * @brief Singleton template class
 * @date $Date$
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2009
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#ifndef COIL_SINGLETON_H
#define COIL_SINGLETON_H

#include <coil/Mutex.h>
#include <coil/Guard.h>

namespace coil
{
  /*!
   * @if jp
   * @class Singleton NXev[g
   *
   * ̃ev[ǵACӂ̃NX Singleton ɂev[głB
   * ȉ̂悤ɂĎgpB
   *
   * class A { // };
   * typedef coil::Singleton<A> A_;
   *
   * Cӂ̏ꏊ
   *
   * A& a(A_:instance()); // a  A ̗B̃CX^X
   *
   * AẪRXgN^͎gpł̂ŁÃ\[XŁA
   *
   * A* a = new A();
   *
   * ̂悤ɂ邱Ƃł邽߁AӂKvłB
   * ΏۂƂNX new 邱Ƃ֎~邽߂ɂ́Aȉ̂悤ɁA
   * ΏۃNX Singelton p (CRTP)  friend 錾KvB
   *
   * class A
   *  : public coil::Singleton<A>
   * {
   * public:
   * private:
   *   A(){}
   * 
   *   friend class coil::Singelton<A>;
   * };
   *
   * 邱ƂŁA
   *
   * A* a = new A(); // ͋֎~
   * A& a(A::instance()); // B̃CX^X𓾂B̕@
   *
   * @else
   * @class Singleton class template
   *
   * This class template makes any classed into Singleton classes.
   * Usage is as follows.
   *
   * class A { // };
   * typedef coil::Singleton<A> A_;
   *
   * In the any places,
   * A& a(A_:instance()); // a has singular instance of A
   *
   * Since the constructor of A is still public, however, user can
   * create other instance of A as follows.
   *
   * A* a = new A();
   *
   * If you want to prohibit user from creating new instance, please
   * inherit Singleton class (CRTP) and declare it as a friend class
   * in the target class.
   *
   * class A
   *  : public coil::Singleton<A>
   * {
   * public:
   * private:
   *   A(){}
   * 
   *   friend class coil::Singelton<A>;
   * };
   *
   * A* a = new A(); // compile error
   * A& a(A::instance()); // This is the only method to get unique instance
   *
   * @endif
   */
  template <class SingletonClass>
  class Singleton
  {
  public:
    typedef SingletonClass* SingletonClassPtr;
    typedef ::coil::Mutex Mutex;
    static SingletonClass& instance()
    {

      // DLC pattern
      if (!m_instance)
      {
        coil::Guard<coil::Mutex> guard(m_mutex);
	if (!m_instance)
	  {
	    m_instance = new SingletonClass();
	  }
      }
      return *m_instance;
    }

  protected:
    Singleton(){};
    ~Singleton(){};

  private:
    Singleton(const Singleton& x);
    Singleton& operator=(const Singleton& x);

  protected:
    static coil::Mutex m_mutex;
    static SingletonClass* m_instance;
  };

  template <class SingletonClass>
  typename Singleton<SingletonClass>::SingletonClassPtr
  Singleton<SingletonClass>::m_instance;
  
  template <class SingletonClass>
  typename Singleton<SingletonClass>::Mutex
  Singleton<SingletonClass>::m_mutex;
}; // namepsace coil

#endif // COIL_SINGLETON_H
