/**
 * @file nes_posix_access.c
 * @brief This file is part of the posix library
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <tstring.h>
#include <btron/file.h>
#include <btron/errcode.h>
#include <errno.h>
#include <tcode.h>

#include "common.h"
#include "nes_posix_access.h"


/*****************************************************************************
 *
 * Function     : conv_pathname
 * Purpose      : convert cli's path name to TC to used in b_get_lnk
 *                
 * Returns      : void
 *
 ****************************************************************************/
static void conv_pathname( const char *src, TC *dest )
{
	char c;
	const char *p;
	int n, k;
	TC buf;

	p = src;
	n = 0;

#ifndef PROCESS_BASE
	if (*p != '/') {
		dest[n++] = TC_FDLM;
		dest[n++] = TK_S;
		dest[n++] = TK_Y;
		dest[n++] = TK_S;
		dest[n++] = TC_FDLM;
	}
#endif
	for ( ; (c = *p) != '\0' && n < FILENAME_MAX - 1; p++) {
		if (c == '/') {
			dest[n++] = TC_FDLM;
		} else if (c == ':') {
			dest[n++] = TC_FSEP;
		} else if ((k = euctotc(&buf, p)) > 0) {
			dest[n++] = buf;
			p += (k - 1);
		}
	}
	dest[n] = 0x0000;
}



/**
 * access - check user's permissions for a file
 * @param 	pathname: file name
 * @param   mode: attribute
 *          R_OK: read 
 *          W_OK: write 
 *          X_OK: exexute
 *          F_OK: read  
 * @retval int
 *      		0: succeed
 *          -1: do not suit to the attibute
 *              EACCES : The requested access would be denied
 *                       to the file or search permission is denied
 *                       for one of the directories in the path 
 *                       prefix of pathname. (See also path_resolution(2).) 
 *              EROFS :  Write permission was requested for a file on a read-only filesystem.
 *              EFAULT:  pathname points outside your accessible address space. 
 *              EINVAL:  mode was incorrectly specified. 
 *              EIO   :  An I/O error occurred. 
 *              ENOMEM : Insufficient kernel memory was available. 
 * 
 */
int nes_posix_access(const char *pathname, int mode)
{
	WERR wRet = 0;
	TC *tcpath;
	LINK lnk;

	if(!pathname || (strlen(pathname) == 0))
	{
		wRet = -1;
		errno = EINVAL;
		return wRet;
	}		

	//tranfer the path to TC type
	tcpath = (TC*)malloc(strlen(pathname)*2 + 10);

	memset((char*)tcpath, 0, strlen((char*)tcpath));
	DP(("access pathname: [%s]\n", pathname));
	conv_pathname(pathname, tcpath);	


	/*
	 *  get link
	 */
	lnk.atr1 = 0;
	lnk.atr2 = 0;
	lnk.atr3 = 0; 
	lnk.atr4 = 0;
	lnk.atr5 = 0;
	wRet = b_get_lnk(tcpath, &lnk, F_DIRECT);

	//	DP(("b_get_lnk : [%d]\n", wRet));

	switch(wRet)
	{
		case   0:
		case   1:
		case   2:
			//get link successfully
			wRet = 0;
			break;
		case   ER_ACCES:			
		case   ER_ADR:  
			wRet = -1;
			errno = EACCES;
			goto exit;

		case   ER_NOFS:			
		case   ER_NOEXS:
			//operation not permitted
			wRet = -1;
			errno = EFAULT;
			DP(("b_get_lnk : [%d]\n", errno));
			goto exit;

		case   ER_IO:
			//I/O error 
			wRet = -1;
			errno = EIO;
			goto exit;

		case   ER_FNAME:
		case   ER_PAR:
			//Path or mode is invalid
			wRet = -1;
			errno = EINVAL;
			goto exit;

		case   ER_NOSPC:
			//out of memory 
			wRet = -1;
			errno = ENOMEM;
			goto exit;

		default :
			wRet = -1;
			errno = EINVAL;
			goto exit;


	}
	//checke the attribute 
	wRet = b_chk_fil(&lnk, mode, NULL);
	if(wRet < 0){
		DP(("b_chk_fil failed:[%d]\n", wRet));	
		switch(wRet){
			case ER_ACCES:
				wRet = -1;
				errno = EACCES;				
				goto exit;

			case ER_ADR:
				wRet = -1;
				errno = EFAULT;			
				goto exit;

			case ER_IO:
				wRet = -1;
				errno = EIO;				
				goto exit;

			case ER_NOEXS:
				wRet = -1;
				errno = EFAULT;				
				goto exit;					

			case ER_NOFS:
				wRet = -1;
				errno = EFAULT;			
				goto exit;	

			case ER_NOSPC:
				wRet = -1;
				errno = ENOMEM;				
				goto exit;									

			case ER_PAR:
				wRet = -1;
				errno = EINVAL;				
				goto exit;		

			case ER_RONLY:
				wRet = -1;
				errno = EROFS;				
				goto exit;	
			default:
				wRet = -1;
				errno = EFAULT;			
				goto exit;
		}
	}
	else{
		wRet = 0;
		goto exit;
	}
exit:
	if(tcpath)
	{
		free(tcpath);
		tcpath = NULL;
	}
	return wRet;
}

