/**
 * @file internal_lock.c
 * @define implement of internal_lock fucntion
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <tkse/errno.h>
#include <tkse/extension/tkcall.h>
#include <tkse/extension/typedef.h>

// #include "error.h"
#include "common.h"
#include "internal_lock.h"

/**********************************************************************/
/* Function name: __internal_lock_init                                */
/* Description:  Init internal lock                                   */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - __internal_lock_t* lock:                                */
/**********************************************************************/
int __internal_lock_init(__internal_lock_t* lock)
{
	INT err = 0;		// T-Engine call result
	int ret = 0;		// Linux type return value
	T_CSEM sem = {0};
	
	if( !lock ) 
	{
		return EINVAL;
	}

	sem.exinf = NULL;
	sem.sematr = TA_TFIFO | TA_FIRST  | TA_DELEXIT;
	sem.isemcnt = 1;
	sem.maxsem = 1;

	err = tkse_cre_sem(&sem);

	/* Create successfully */
	if ( err > 0 )
	{
		*lock = err;		 	
		return 0;
	} 

	/* Error code mapping */
	switch( err )
	{
	case E_OK:
		ret = 0;
		break;
	case E_NOMEM:
		ret = ENOMEM;
		break;
	case E_LIMIT:
		ret = ERANGE;
		break;
	case E_ID:
	case E_PAR:
	case E_RSATR:
	default:
		ret = EINVAL;
		break;
	}

	return ret;
}

/**********************************************************************/
/* Function name: __internal_lock_destroy                             */
/* Description:  Destroy internal MUTEX.                              */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - __internal_lock_t* lock:                                */
/**********************************************************************/
int __internal_lock_destroy(__internal_lock_t* lock)
{
	INT wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value

	if( !lock ) 
	{
		return EINVAL;
	}

	wCall = tkse_del_sem(*lock);

	/* Error code mapping */
	switch(wCall)
	{
	case E_OK:
		nRet = 0;
		break;
	case E_ID:			
	case E_NOEXS:
	default:
		nRet = EINVAL;
		break;
	}

	return nRet;
}

/**********************************************************************/
/* Function name: __internal_lock                                     */
/* Description:  Wait until lock for internal MUTEX becomes available */
/*               and lock it.                                         */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - __internal_lock_t* lock:                                */
/**********************************************************************/
int __internal_lock(__internal_lock_t* lock)
{
	INT wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value

	if( !lock ) 
	{
		return EINVAL;
	}

	wCall = tkse_wai_sem(*lock, 1, T_FOREVER);

	switch(wCall)
	{
	case E_OK:
		nRet = 0;
		break;
	case E_ID:			/* Invalid sem_t */
	case E_NOEXS:
	case E_PAR:
	case E_CTX:
		nRet = EINVAL;
		break;
	case E_DLT:
	case E_RLWAI:
	case E_DISWAI:
		nRet = EINTR;
		break;
	case E_TMOUT: 			/* Never reach here */
		nRet = EBUSY;
		break;
	default:				/* Unknow error */
		nRet = EINVAL;
		break;
	}

	return nRet;
}

/**********************************************************************/
/* Function name: __internal_unlock                                   */
/* Description: Unlock internal MUTEX                                 */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - __internal_lock_t* lock:                                */
/**********************************************************************/
int __internal_unlock(__internal_lock_t* lock)
{
	INT wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value
	
	if( !lock ) 
	{
		return EINVAL;
	}
	wCall = tkse_sig_sem(*lock, 1);

	switch ( wCall )
	{
	case E_OK:
		nRet = 0;
		break;
	case E_ID:
	case E_NOEXS:
		nRet = EINVAL;
		break;
	case E_QOVR:
		/* nRet = ERANGE; */
		nRet = 0; /* Compatible with pmc */
		break;
	case E_ILUSE:
		nRet = EPERM;
		break;
	default:
		nRet = EINVAL;
		break;
	}

	return nRet;
}

/**********************************************************************/
/* Function name: __internal_try_lock                                 */
/* Description:  try lock                                             */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - __internal_lock_t* lock:                                */
/**********************************************************************/
int __internal_try_lock(__internal_lock_t* lock)
{
	INT wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value

	if( !lock ) 
	{
		return EINVAL;
	}

	wCall = tkse_wai_sem(*lock, 1, T_NOWAIT);

	switch(wCall)
	{
	case E_OK:
		nRet = 0;
		break;
	case E_ID:			/* Invalid sem_t */
	case E_NOEXS:
	case E_PAR:
	case E_CTX:
		nRet = EINVAL;
		break;
	case E_DLT:
	case E_RLWAI:
	case E_DISWAI:
		nRet = EINTR;
		break;
	case E_TMOUT:
		nRet = EBUSY;
		break;
	default:				/* Unknow error */
		nRet = EINVAL;
		break;
	}

	return nRet;
}
