/*
 * Copyright (c) 2008, AIST.
 * All rights reserved. This program is made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * National Institute of Advanced Industrial Science and Technology (AIST)
 */
//
// Client.cc - This is the source code of StructTest-client.
//
// Usage: Client -ORBInitRef NameService=<corbaloc>
//
//  e.g.: Client -ORBInitRef NameService=corbaloc::localhost/NameService
//
//  On startup, Client looks up the object reference from the COS naming service.
//
//  The name which the object is bound to is as follows:
//        id:root   kind:context
//         |
//        id:StructTester kind:object
//

#include <sys/time.h>

#include <iostream>
using namespace std;

#ifdef ORB_IS_RTORB
#include "BorderTest.h"
#else
#include "BorderTest.hh"
#endif

//////////////////////////////////////////////////////////////////////

class Client
{
public:
	Client(CORBA::ORB_var& orb);
	CORBA::Object_ptr getObject(const CosNaming::Name& name);
protected:
	CosNaming::NamingContext_ptr getRootContext();
private:
	CORBA::ORB_var& orb;
	CosNaming::NamingContext_var root_context;
};

Client::Client(CORBA::ORB_var& _orb): orb(_orb)
{
	this->root_context = getRootContext();
}

CosNaming::NamingContext_ptr
Client::getRootContext()
{
	CORBA::Object_var obj = orb->resolve_initial_references("NameService");
	return CosNaming::NamingContext::_narrow(obj);
}

CORBA::Object_ptr
Client::getObject(const CosNaming::Name& name)
{
	return root_context->resolve(name);
}

//////////////////////////////////////////////////////////////////////

class TesterClient: public Client
{
public:
	TesterClient(CORBA::ORB_var& orb);
	void run();
	void test_ushort_border();

protected:
	BorderTest::Tester_ptr getTester();
private:
	BorderTest::Tester_var tester;
};

TesterClient::TesterClient(CORBA::ORB_var& orb): Client(orb)
{
	this->tester = getTester();
}

BorderTest::Tester_ptr
TesterClient::getTester()
{
	CosNaming::Name name;
	name.length(1);
	name[0].id   = (const char*)"BorderTester";
	name[0].kind = (const char*)"object";

	CORBA::Object_var obj = getObject(name);
	return BorderTest::Tester::_narrow(obj);
}

void 
TesterClient::test_ushort_border()
{
	printf("test_ushort_border:\n");

	BorderTest::UShortData p_out;
	BorderTest::UShortData p_in;

#define GET() \
	tester->get_ushort(p_out); \
	printf("  tester->get_ushort() ==> %u\n", p_out.us)

#define SET(data) \
	p_in.us = data; \
	tester->set_ushort(p_in); \
	printf("  tester->set_ushort(%u)\n", p_in.us)

#define SET_GET(data) \
	SET(data); \
	GET()

	GET();

	SET_GET(1);

	SET_GET(0);

	SET_GET(65535);

	printf("\n");

}

void 
TesterClient::run()
{
	test_ushort_border();
}

//////////////////////////////////////////////////////////////////////

int
main (int argc, char **argv) 
{
	try {
		CORBA::ORB_var orb = CORBA::ORB_init(argc, argv);
		TesterClient tester(orb);
		tester.run();
		orb->destroy();
	}
	catch(CosNaming::NamingContext::NotFound& ex) {
		// This exception is thrown if any of the components of the
		// path [contexts or the object] aren't found:
		cerr << "Context not found." << endl;
	}
	catch(CORBA::COMM_FAILURE& ex) {
		cerr << "Caught system exception COMM_FAILURE -- unable to contact the "
				<< "object." << endl;
	}
	catch(CORBA::SystemException&) {
		cerr << "Caught CORBA::SystemException." << endl;
	}
	catch(CORBA::Exception&) {
		cerr << "Caught CORBA::Exception." << endl;
	}
#ifdef ORB_IS_OMNIORB
	catch(omniORB::fatalException& fe) {
		cerr << "Caught omniORB::fatalException:" << endl;
		cerr << "  file: " << fe.file() << endl;
		cerr << "  line: " << fe.line() << endl;
		cerr << "  mesg: " << fe.errmsg() << endl;
	}
#endif
	catch(...) {
		cerr << "Caught unknown exception." << endl;
	}
	return 0;
}
