// -*- C++ -*-
/*!
 * @file NamingManager.h
 * @brief naming Service helper class
 * @date $Date: 2007-12-31 03:08:04 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: NamingManager.h 1766 2010-01-23 17:19:09Z n-ando $
 *
 */

#ifndef RTC_NAMINGMANAGER_H
#define RTC_NAMINGMANAGER_H

#include <rtm/RTC.h>

#include <coil/Task.h>
#include <coil/Mutex.h>
#include <coil/Guard.h>
#include <rtm/CorbaNaming.h>
#include <rtm/RTObject.h>
#include <rtm/SystemLogger.h>
#include <rtm/ManagerServant.h>

namespace RTC
{
  class Manager;
  /*!
   * @if jp
   *
   * @class NamingBase
   * @brief NamingService ݥ饹
   *
   * NamingServer ݥ󥿡ե饹
   * ݴ饹ϡʲν貾۴ؿμ󶡤ʤФʤʤ
   * - bindObject() : ꤷ֥ȤNamingServiceؤΥХ
   * - unbindObject() : ꤷ֥ȤNamingServiceΥХ
   *
   * @since 0.4.0
   *
   * @else
   *
   * @class NamingBase
   * @brief NamingService management abstract class
   *
   * This is the abstract interface class for NamingServer management.
   * Concrete management classes must implement the following pure virtual 
   * functions.
   * - bindObject() : Bind the specified object to NamingService
   * - unbindObject() : Unbind the specified object from NamingService
   *
   * @since 0.4.0
   *
   * @endif
   */
  class NamingBase
  {
    typedef coil::Mutex Mutex;
    typedef coil::Guard<Mutex> Guard;
  public:
    /*!
     * @if jp
     *
     * @brief 󥹥ȥ饯
     *
     * @else
     *
     * @brief Constructor
     *
     * @endif
     */
    NamingBase() {};
    
    /*!
     * @if jp
     *
     * @brief ǥȥ饯
     *
     * @else
     *
     * @brief Destructor
     *
     * @endif
     */
    virtual ~NamingBase(void) {};
    
    /*!
     * @if jp
     *
     * @brief ꤷ֥ȤNamingServiceإХɤ貾۴ؿ
     *
     * @param name Хɻ̾
     * @param rtobj Хоݥ֥
     *
     * @else
     *
     * @brief Pure virtual function to bind the specified objects 
     *        to NamingService
     *
     * @param name Names at the binding
     * @param rtobj The target objects for the binding
     *
     * @endif
     */
    virtual void bindObject(const char* name, const RTObject_impl* rtobj) = 0;

    /*!
     * @if jp
     *
     * @brief ꤷManagerServantNamingServiceإХɤ貾۴ؿ
     *
     * @param name Хɻ̾
     * @param rtobj ХоManagerServant
     *
     * @else
     *
     * @brief Pure virtual function to bind the specified ManagerServants 
     *        to NamingService
     *
     * @param name Names at the binding
     * @param mgr The target ManagerServants for the binding
     *
     * @endif
     */
    virtual void bindObject(const char* name, const RTM::ManagerServant* mgr) = 0;
    
    /*!
     * @if jp
     *
     * @brief ꤷ֥ȤNamingService饢Хɤ뤿
     *        貾۴ؿ
     *
     * @param name Хоݥ֥
     *
     * @else
     *
     * @brief Pure virtual function to unbind the specified objects from 
     *        NamingService
     *
     * @param name The target objects for the unbinding
     *
     * @endif
     */
    virtual void unbindObject(const char* name) = 0;

    /*!
     * @if jp
     *
     * @brief ͡ॵФΥ롼ȥƥȤ¸ߤǧ롣
     * 
     * @return true:¸ߤ,false:¸ߤʤ
     *
     * @else
     *
     * @brief Existence of route context of Nameserver 
     * 
     * @return true:existent,false:non existent
     *
     * @endif
     */
    virtual bool isAlive() = 0;
  };
  
  /*!
   * @if jp
   *
   * @class NamingOnCorba
   * @brief CORBA  NamingServer 饹
   *
   * CORBA  NamingServer ѥ饹
   * CORBA ݡͥȤNamingServiceؤϿʤɤ롣
   *
   * @since 0.4.0
   *
   * @else
   *
   * @class NamingOnCorba
   * @brief NamingServer management class for CORBA
   *
   * NamingServer management class for CORBA.
   * Manage to register and unregister CORBA components to NamingService.
   *
   * @since 0.4.0
   *
   * @endif
   */
  class NamingOnCorba
    : public virtual NamingBase
  {
  public:
    /*!
     * @if jp
     *
     * @brief 󥹥ȥ饯
     *
     * 󥹥ȥ饯
     *
     * @param orb ORB
     * @param names NamingServer ̾
     *
     * @else
     *
     * @brief Constructor
     *
     * Constructor
     *
     * @param orb ORB
     * @param names Name of NamingServer
     *
     * @endif
     */
    NamingOnCorba(CORBA::ORB_ptr orb, const char* names);
    
    /*!
     * @if jp
     *
     * @brief ǥȥ饯
     *
     * @else
     *
     * @brief Destructor
     *
     * @endif
     */
    virtual ~NamingOnCorba(void){};
    
    /*!
     * @if jp
     *
     * @brief ꤷ CORBA ֥ȤNamingServiceإХ
     * 
     * ꤷ CORBA ֥Ȥꤷ̾Τ CORBA NamingService 
     * Хɤ롣
     * 
     * @param name Хɻ̾
     * @param rtobj Хоݥ֥
     *
     * @else
     *
     * @brief Bind the specified CORBA objects to NamingService
     * 
     * Bind the specified CORBA objects to CORBA NamingService
     * by specified names.
     * 
     * @param name Names at the binding
     * @param rtobj The target objects for the binding
     *
     * @endif
     */
    virtual void bindObject(const char* name, const RTObject_impl* rtobj);
    /*!
     * @if jp
     *
     * @brief ꤷManagerServantNamingServiceإХ
     *
     * @param name Хɻ̾
     * @param rtobj ХоManagerServant
     *
     * @else
     *
     * @brief Bind the specified ManagerServants to NamingService
     *
     * @param name Names at the binding
     * @param mgr The target ManagerServants for the binding
     *
     * @endif
     */
    virtual void bindObject(const char* name, const RTM::ManagerServant* mgr);
    
    /*!
     * @if jp
     *
     * @brief ꤷ CORBA ֥ȤNamingService饢Х
     * 
     * ꤷ CORBA ֥Ȥ CORBA NamingService 饢Хɤ롣
     * 
     * @param name Хоݥ֥
     *
     * @else
     *
     * @brief Unbind the specified CORBA objects from NamingService
     * 
     * Unbind the specified CORBA objects from CORBA NamingService.
     * 
     * @param name The target objects for the unbinding
     *
     * @endif
     */
    virtual void unbindObject(const char* name);
    
    /*!
     * @if jp
     *
     * @brief ͡ॵФΥ롼ȥƥȤ¸ߤǧ롣
     * 
     * @return true:¸ߤ,false:¸ߤʤ
     *
     * @else
     *
     * @brief Existence of route context of Nameserver 
     * 
     * @return true:existent,false:non existent
     *
     * @endif
     */
    virtual bool isAlive();

  private:
    Logger rtclog;
    CorbaNaming m_cosnaming;
    std::string m_endpoint;
    bool m_replaceEndpoint;
    std::map<std::string, RTObject_impl*> m_names;
  };
  
  /*!
   * @if jp
   *
   * @class NamingManager
   * @brief NamingServer 饹
   *
   * NamingServer ѥ饹
   * ݡͥȤNamingServiceؤϿʤɤ롣
   *
   * @since 0.4.0
   *
   * @else
   *
   * @class NamingManager
   * @brief NamingServer management class
   *
   * NamingServer management class.
   * Manage to register and unregister components to NamingService.
   *
   * @since 0.4.0
   *
   * @endif
   */
  class NamingManager
  {
    typedef coil::Mutex Mutex;
    typedef coil::Guard<Mutex> Guard;
  public:
    /*!
     * @if jp
     *
     * @brief 󥹥ȥ饯
     *
     * 󥹥ȥ饯
     *
     * @param manager ޥ͡㥪֥
     *
     * @else
     *
     * @brief Constructor
     *
     * Constructor
     *
     * @param manager Manager object
     *
     * @endif
     */
    NamingManager(Manager* manager);
    
    /*!
     * @if jp
     *
     * @brief ǥȥ饯
     *
     * @else
     *
     * @brief Destructor
     *
     * @endif
     */
    virtual ~NamingManager(void);
    
    /*!
     * @if jp
     *
     * @brief NameServer Ͽ
     *
     * ꤷ NameServer Ͽ롣
     * ߻ǽʷ CORBA Τߡ
     *
     * @param method NamingService η
     * @param name_server Ͽ NameServer ̾
     *
     * @else
     *
     * @brief Regster the NameServer
     *
     * Register NameServer by specified format.
     * Currently. only CORBA can be specified.
     *
     * @param method Format of NamingService
     * @param name_server Name of NameServer for registration
     *
     * @endif
     */
    void registerNameServer(const char* method, const char* name_server);
    
    /*!
     * @if jp
     *
     * @brief ꤷ֥ȤNamingServiceإХ
     * 
     * ꤷ֥Ȥꤷ̾Τ CORBA NamingService إХɤ롣
     * 
     * @param name Хɻ̾
     * @param rtobj Хоݥ֥
     *
     * @else
     *
     * @brief Bind the specified objects to NamingService
     * 
     * Bind the specified objects to CORBA NamingService by specified names.
     * 
     * @param name Names at the binding
     * @param rtobj The target objects for the binding
     *
     * @endif
     */
    void bindObject(const char* name, const RTObject_impl* rtobj);
    /*!
     * @if jp
     *
     * @brief ꤷManagerServantNamingServiceإХ
     * 
     * ꤷManagerServantꤷ̾Τ 
     * CORBA NamingService إХɤ롣
     * 
     * @param name Хɻ̾
     * @param mgr ХоManagerServant
     *
     * @else
     *
     * @brief Bind the specified ManagerServants to NamingService
     * 
     * Bind the specified ManagerServants to CORBA NamingService 
     * by specified names.
     * 
     * @param name Names at the binding
     * @param mgr The target ManagerServants for the binding
     *
     * @endif
     */
    void bindObject(const char* name, const RTM::ManagerServant* mgr);
    
    /*!
     * @if jp
     *
     * @brief NamingServer ξι
     * 
     * ꤵƤ NameServer ϿƤ륪֥Ȥξ
     * 롣
     * 
     * @else
     *
     * @brief Update information of NamingServer
     * 
     * Update the object information registered in the specified NameServer.
     * 
     * @endif
     */
    void update();
    
    /*!
     * @if jp
     *
     * @brief ꤷ֥ȤNamingService饢Х
     * 
     * ꤷ֥Ȥ NamingService 饢Хɤ롣
     * 
     * @param name Хоݥ֥
     *
     * @else
     *
     * @brief Unbind the specified objects from NamingService
     * 
     * Unbind the specified objects from NamingService.
     * 
     * @param name The target objects for the unbinding
     *
     * @endif
     */
    void unbindObject(const char* name);
    
    /*!
     * @if jp
     *
     * @brief ƤΥ֥ȤNamingService饢Х
     * 
     * ƤΥ֥Ȥ CORBA NamingService 饢Хɤ롣
     * 
     * @else
     *
     * @brief Unbind all objects from NamingService
     * 
     * Unbind all objects from CORBA NamingService.
     * 
     * @endif
     */
    void unbindAll();
    
    /*!
     * @if jp
     *
     * @brief ХɤƤƤΥ֥Ȥ
     * 
     * ХɤƤƤΥ֥Ȥ 롣
     *
     * @return ХɺѤߥ֥ ꥹ
     * 
     * @else
     *
     * @brief Get all bound objects
     * 
     * Get all bound objects.
     *
     * @return Bound object list
     * 
     * @endif
     */
    std::vector<RTObject_impl*> getObjects();
    
  protected:
    /*!
     * @if jp
     *
     * @brief NameServer ѥ֥Ȥ
     * 
     * ꤷNameServer ѥ֥Ȥ롣
     *
     * @param method NamingService 
     * @param name_server NameServer ̾
     * 
     * @return  NameServer ֥
     * 
     * @else
     *
     * @brief Create objects for NameServer management
     * 
     * Create objects of specified type for NameServer management.
     *
     * @param method NamingService format
     * @param name_server NameServer name
     * 
     * @return Created NameServer objects
     * 
     * @endif
     */
    NamingBase* createNamingObj(const char* method, const char* name_server);
    
    /*!
     * @if jp
     *
     * @brief ѤߥݡͥȤ NameServer Ͽ
     * 
     * ѤߥݡͥȤꤷ NameServer Ͽ롣
     *
     * @param ns Ͽо NameServer
     * 
     * @else
     *
     * @brief Register the configured component to NameServer
     * 
     * Register the already configured components to NameServer.
     *
     * @param ns The target NameServer for the registration
     * 
     * @endif
     */
    void bindCompsTo(NamingBase* ns);
    
    /*!
     * @if jp
     *
     * @brief NameServer Ͽ륳ݡͥȤ
     * 
     * NameServer Ͽ륳ݡͥȤꤹ롣
     *
     * @param name ݡͥȤϿ̾
     * @param rtobj Ͽоݥ֥
     * 
     * @else
     *
     * @brief Configure the components that will be registered to NameServer
     * 
     * Configure the components that will be registered to NameServer.
     *
     * @param name Names of components at the registration
     * @param rtobj The target objects for registration
     * 
     * @endif
     */
    void registerCompName(const char* name, const RTObject_impl* rtobj);
    /*!
     * @if jp
     *
     * @brief NameServer ϿManagerServant
     * 
     * NameServer ϿManagerServantꤹ롣
     *
     * @param name ManagerServantϿ̾
     * @param mgr ϿоManagerServant
     * 
     * @else
     *
     * @brief Configure the ManagerServants that will be registered 
     * to NameServer
     * 
     * Configure the ManagerServants that will be registered to NameServer.
     *
     * @param name Names of ManagerServants at the registration
     * @param mgr The target ManagerServants for registration
     * 
     * @endif
     */
    void registerMgrName(const char* name, const RTM::ManagerServant* mgr);
    
    /*!
     * @if jp
     *
     * @brief NameServer Ͽ륳ݡͥȤ
     * 
     * NameServer Ͽ륳ݡͥȤ롣
     *
     * @param name оݥݡͥȤ̾
     * 
     * @else
     *
     * @brief Unregister the components that will be registered to NameServer
     * 
     * Unregister the components that will be registered to NameServer.
     *
     * @param name Names of the target components for unregistration
     * 
     * @endif
     */
    void unregisterCompName(const char* name);
    /*!
     * @if jp
     *
     * @brief NameServer ϿManagerServant
     * 
     * NameServer ϿManagerServant롣
     *
     * @param name оManagerServant̾
     * 
     * @else
     *
     * @brief Unregister the ManagerServants that will be registered 
     * to NameServer
     * 
     * Unregister the ManagerServants that will be registered to NameServer.
     *
     * @param name Names of the target ManagerServants for unregistration
     * 
     * @endif
     */
    void unregisterMgrName(const char* name);

    /*!
     * @if jp
     *
     * @brief ݥͥȤХɤ
     * 
     * ͡ॵФ³ƥݥͥȤХɤ롣
     *
     * @param ns NameServer
     * 
     * @else
     *
     * @brief Rebind the component to NameServer
     * 
     * Connect with the NameServer and rebind the component. 
     *
     * @param ns NameServer
     * 
     * @endif
     */
    class Names;
    void retryConnection(Names* ns);
    
  protected:
    // Name Servers' method/name and object
    /*!
     * @if jp
     * @brief NameServer ѹ¤
     * @else
     * @brief Structure for NameServer management
     * @endif
     */
    class Names
    {
    public:
      Names(const char* meth, const char* name, NamingBase* naming)
	: method(meth), nsname(name), ns(naming)
      {
      }
      
      ~Names()
      {
        delete ns;
      }
      
      std::string method;
      std::string nsname;
      NamingBase* ns;
    };
    /*!
     * @if jp
     * @brief NameServer ꥹ
     * @else
     * @brief NameServer list
     * @endif
     */
    std::vector<Names*> m_names;
    /*!
     * @if jp
     * @brief NameServer ꥹȤmutex
     * @else
     * @brief Mutex of NameServer list
     * @endif
     */
    Mutex m_namesMutex;
    
    // Components' name and object
    /*!
     * @if jp
     * @brief ݡͥȴѹ¤
     * @else
     * @brief Structure for component management
     * @endif
     */
    struct Comps
    {
      Comps(const char* n, const RTObject_impl* obj)
	: name(n), rtobj(obj)
      {}
      std::string name;
      const RTObject_impl* rtobj;
    };
    /*!
     * @if jp
     * @brief ManagerServantѹ¤
     * @else
     * @brief Structure for ManagerServant management
     * @endif
     */
    struct Mgr
    {
      Mgr(const char* n, const RTM::ManagerServant* obj)
	: name(n), mgr(obj)
      {}
      std::string name;
      const RTM::ManagerServant* mgr;
    };
    /*!
     * @if jp
     * @brief ݡͥȥꥹ
     * @else
     * @brief Component list
     * @endif
     */
    std::vector<Comps*> m_compNames;
    /*!
     * @if jp
     * @brief ݡͥȥꥹȤmutex
     * @else
     * @brief Mutex of Component list
     * @endif
     */
    Mutex m_compNamesMutex;
    /*!
     * @if jp
     * @brief ManagerServantꥹ
     * @else
     * @brief ManagerServant list
     * @endif
     */
    std::vector<Mgr*> m_mgrNames;
    /*!
     * @if jp
     * @brief ManagerServantꥹȤmutex
     * @else
     * @brief Mutex of ManagerServant list
     * @endif
     */
    Mutex m_mgrNamesMutex;
    
    /*!
     * @if jp
     * @brief ޥ͡㥪֥
     * @else
     * @brief Manager object
     * @endif
     */
    Manager* m_manager;
    
    /*!
     * @if jp
     * @brief ȥ꡼
     * @else
     * @brief Logger stream
     * @endif
     */
    Logger rtclog;
  }; // class NamingManager
}; // namespace RTC

#endif // RTC_NAMINGMANAGER_H
