// -*- C++ -*-
/*!
 * @file  MutexPosix.h
 * @brief RT-Middleware Service interface
 * @date  $Date$
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2008
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#include <coil/UUID.h>
#include <iostream>
#include <string.h>

#ifdef COIL_OS_FREEBSD
void error_code(uint32_t status)
{
  if (status == uuid_s_ok)
    std::cout << "uuid_s_ok" << std::endl;
  else if (status == uuid_s_bad_version)
    std::cout << "uuid_s_bad_version" << std::endl;
  else if (status == uuid_s_invalid_string_uuid)
    std::cout << "uuid_s_invalid_string_uuid" << std::endl;
  else if (status == uuid_s_no_memory)
    std::cout << "uuid_s_no_memory" << std::endl;
  else
    std::cout << "other error" << std::endl;
}

void uuid_clear(uuid_t& uu)
{
  uint32_t status;
  uuid_create_nil(&uu, &status);
}
void uuid_unparse(uuid_t& uu, char*& uuidstr)
{
  uint32_t status;
  uuid_to_string(&uu, &uuidstr, &status);
}
void uuid_generate(uuid_t& out)
{
  uint32_t status;
  uuid_create(&out, &status);
}
#endif

namespace coil
{

#ifdef COIL_OS_FREEBSD
  UUID::UUID()
    : m_uuidstr(0)
  {
    ::uuid_clear(m_uuid);
  }
  UUID::UUID(const uuid_t& uuid)
    : m_uuid(uuid), m_uuidstr(0)
  {
  }

  UUID::~UUID()
  {
    free(m_uuidstr);
  }

  const char* UUID::to_string()
  {
    uuid_unparse(m_uuid, m_uuidstr);
    return m_uuidstr;
  }
    

  UUID_Generator::UUID_Generator()
  {
  }

  UUID_Generator::~UUID_Generator()
  {
  }

  void UUID_Generator::init()
  {
  }

  UUID* UUID_Generator::generateUUID(int n, int h)
  {
    uuid_t uuid;
    uuid_generate(uuid);
    return new UUID(uuid);
  }
#endif

#ifdef COIL_OS_LINUX

  UUID_Generator::UUID_Generator(){}
  
  void UUID_Generator::init(){}
  UUID* UUID_Generator::generateUUID(int varsion, int variant){
#ifdef __T_KERNEL__
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    return new UUID();
#else    
    uuid_t uuid;
    uuid_generate(uuid);
    return new UUID(&uuid);
#endif
  }
  
  UUID::UUID(){
#ifdef __T_KERNEL__
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    memset(&_uuid,0,sizeof(uuid_t));
    uuid_generate_time(&_uuid);
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
#else
    uuid_clear(this->_uuid);
#endif
  }
  
  UUID::UUID(uuid_t *uuid){
#ifdef __T_KERNEL__
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    if(uuid)
    {
        memcpy(&_uuid,uuid,sizeof(uuid_t));
    }
    else
    {
        memset(&_uuid,0,sizeof(uuid_t));
        uuid_generate_time(&_uuid);
    }
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
#else
    strcpy((char *)this->_uuid, (char *)(*uuid));
#endif
  }
  
  const char* UUID::to_string(){
#ifdef __T_KERNEL__
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    sprintf (buf,
                     "%08x-%04x-%04x-%04x-%02x%02x%02x%02x%02x%02x",
                     _uuid.time_low,
                     _uuid.time_mid,
                     _uuid.time_hi_and_version,
                     _uuid.clock_seq,
                     _uuid.node[0],
                     _uuid.node[1],
                     _uuid.node[2],
                     _uuid.node[3],
                     _uuid.node[4],
                     _uuid.node[5]
                     );
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
#else
    uuid_unparse(this->_uuid, buf);
#endif
    return buf;
  }

#endif
};
