/**
 * @file nes_posix_time.c
 * @brief This file is part of the posix library
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <time.h>
#include <errno.h>
#include <btron/clk.h>
#include <btron/errcode.h>
#include "nes_posix_time.h"

/**
 * nes_posix_gettimeofday - get the time as well as a timezone.
 * @param *tv	time info struct pointer
 * @param *tz	timezone pointer
 * @retval int	0 success -1 failed
 * 
 * @note
 */
int nes_posix_gettimeofday(struct timeval *tv, struct timezone *tz)
{
	int ret = 0;
	STIME t = 0;
	TIMEZONE zone;	

	if( !tv || !tz ) {
		errno = EFAULT;
		return -1;
	}

	ret = b_get_tim(&t, &zone);
	if( ret != 0 ) {
		errno = EINVAL;
		return -1;
	}

	tv->tv_sec = t;
	tv->tv_usec = 0;  /* Not support */

	tz->tz_minuteswest = zone.adjust / 60;
	tz->tz_dsttime = 0;

	return 0;
}

/*
   int settimeofday(const struct timeval *tv , const struct timezone *tz)
   {
   return 0;
   }
 */

/**********************************************************************/ 
/* Function name: nes_posix_clock_gettime                                       */ 
/* Description:   this syscall return the current value tp for the    */
/*                specified clock(only support CLOCK_REALTIME)        */
/* Return value:  On sucessfully:  clock's current time               */
/*                erro: -1                                            */
/*                      EPERM : operation is forbidded                */
/*                      EINVAL: parameter invalid                     */
/* Argument -     clock_id:     clock type                            */
/*                tp            the buffer which save clock's time    */
/**********************************************************************/ 
int nes_posix_clock_gettime(clockid_t clock_id, struct timespec *tp)
{
	ERR eRet = 0;
	STIME time;

	//only support the clock of CLOCK_REALTIME
	if( (clock_id != CLOCK_REALTIME) || !tp)
	{
		eRet = -1;
		errno = EINVAL;
		return eRet;
	}

	//get the systime
	eRet = b_get_tim(&time, NULL);
	if(eRet != 0)
	{
		eRet = -1;
		errno = EPERM;
		return eRet;
	}
	tp->tv_sec = time;		
	//at T-Engine ,only can get the second
	tp->tv_nsec = 0;		
	return eRet;	
}


