/**
 * @file nes_posix_cond.c
 * @define implement of condition operation
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <tkse/extension/typedef.h>
#include <tkse/seio/sys/time.h>
#include <tkse/extension/clk.h>
#include <tkse/extension/tkcall.h>
#include <tkse/extension/proctask.h>

#include <string.h>

#include "nes_posix_cond.h"
#include "common.h"
#include "utility/queue.h"

/**********************************************************************/
/* Function name:   nes_posix_pthread_condattr_init                             */
/* Description:                                                       */
/*             Initializes a condition variable attributes object with*/
/*             default attributes.in fact, the default value of attr  */
/*             is PTHREAD_ PROCESS_PRIVATE                            */
/* Return type: 0           always return 0;                          */
/* Argument - pthread_condattr_t * attr:                              */
/**********************************************************************/

int nes_posix_pthread_condattr_init(pthread_condattr_t * attr)
{
	/*set the flg of condition can be deleted */
	attr->busy_check = 0;
	return 0;
}

/**********************************************************************/
/* Function name:  nes_posix_pthread_condattr_destroy                           */
/* Description:    destroy the attribute of pthread                   */
/*                 Initializes a condition variable attributes object */
/*                 with default attributes.in fact, the value of attr */
/*                 is NULL,so the function do nothing.                */
/* Return type: 0           always return 0;                          */
/* Argument - pthread_condattr_t * attr:                              */
/**********************************************************************/

int nes_posix_pthread_condattr_destroy(pthread_condattr_t *attr)
{
	return 0;
}

/**********************************************************************/
/* Function name:  nes_posix_pthread_cond_init                                  */
/* Description:    This function initializes a condition variable     */
/*                 Initializes a condition variable attributes object */
/*                 with default attributes.in fact, the value of attr */
/*                 is NULL,so the function do nothing.                */
/* Return type:  =0  successfully created condition variable          */
/*              EINVAL          'attr' is invalid or unknow error     */
/*              EAGAIN          insufficient resources (other than    */
/*                              memory,                               */
/*              ENOMEM          insufficient memory,                  */
/* Argument - cond : pointer to an instance of pthread_cond_t         */
/*            attr : specifies optional creation attributes           */
/**********************************************************************/

int nes_posix_pthread_cond_init(pthread_cond_t *cond, pthread_condattr_t *cond_attr)
{
	INT wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value	
	T_CFLG flg = {0};

	if( !cond )
	{
		nRet = EINVAL;
		goto exit;
	}
	
	cond->busy_check = cond_attr ? cond_attr->busy_check : 0;

	flg.exinf     = NULL;
	flg.flgatr    = TA_TFIFO | TA_WMUL | TA_DELEXIT;
	flg.iflgptn   = 0;
	flg.dsname[0] = 0;
	wCall = tkse_cre_flg(&flg);

	// Create successfully
	if ( wCall > 0 )
	{
		// save cond,  Init data queue
		cond ->iCond_id = wCall;
		(struct queue*)cond ->myqueue = (struct queue*)malloc(sizeof(struct queue));
		queue_init((struct queue*)cond ->myqueue);		

		//initial internal lock
		nRet = nes_posix_pthread_mutex_init(&cond->lock, NULL);

		/* Added by wang-yg @ NES(Jinan) K2G2. 2007/07/03 */
		if( nRet != 0 ) {
			tkse_del_flg(wCall);
		}
		/* End added */
		goto exit;

	}
		
	// Error code mapping
	switch(wCall)
	{
	case E_NOMEM:			//insufficient memory
		nRet = ENOMEM;
		break;
	case E_LIMIT:			// Reach  eventflag max count limit
		nRet = EAGAIN;		
		break;
	case E_RSATR:			// Invalid parameter
	case E_PAR:			// Invalid parameter address
	default:			// Unknow error
		nRet = EINVAL;
		break;
	}
exit:
	return nRet;  
}

/**********************************************************************/
/* Function name: nes_posix_pthread_cond_destroy                                */
/* Description:   This function destroys a condition variable         */
/*                if no thread is using the cond Variable             */
/* Return type:   0  successfully destroy condition variable          */
/*              EINVAL          parameters is invalid                 */
/*              EBUSY          insufficient memory,                   */
/* Argument - cond: pointer to an instance of pthread_cond_t          */
/**********************************************************************/   
 
int nes_posix_pthread_cond_destroy(pthread_cond_t *cond)
{	
	INT wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value	

	if(!cond)
	{
		nRet = EINVAL;
		goto exit;
	}
		
	if(!cond ->iCond_id)
	{
		nRet = EINVAL;
		goto exit;
	}
	
	//delete the source of cond queuefree(cond ->myqueue)
	nes_posix_pthread_mutex_lock(&cond->lock);
	
	/*set the flg of condition can be deleted or not
	 *when value=1;if condition being used by thread,it cann't be deleted
	 *when value=0;if condition being used by thread,it can be deleted
	 */
	if( cond->busy_check ) 
	{
		//if condition is used by other thread,it cann't be deleted
		struct node* head = cond->myqueue->head;
		if( head )
		{
			nes_posix_pthread_mutex_unlock(&cond->lock);
			return EBUSY;
		}
	}	
 
	//condition can be deleted or the queue of cond is NULL
	nRet = queue_destroy((struct queue*)cond ->myqueue);
	//the return value of queue_destroy is unnecessary to judge
	
	/* 
	 * added by liu on 20060313
	 * if condtion is not initialed,don't destroy it
	 */
	if(cond ->myqueue)
	{
		free(cond ->myqueue);
	}
	
	/* Reset wait queue */
	cond ->myqueue = NULL;
	
	nes_posix_pthread_mutex_unlock(&cond->lock);
	
	//delete internal lock	
	nRet = nes_posix_pthread_mutex_destroy(&cond->lock);
	if (nRet != 0)
	{
		wCall = tkse_del_flg(cond->iCond_id);
		goto exit;
	}
	wCall = tkse_del_flg(cond->iCond_id);

	// code mapping
	switch(wCall)
	{
	case E_OK:
		nRet = 0;		//delete flg successfully
		break;			
	case E_ID:			// Invalid parameter
	case E_NOEXS:			// Invalid parameter 
	default:			// flg is using by other thread
		nRet = EINVAL;		
		break;
	}
exit:
	return nRet;  
	
}

/**********************************************************************/
/* Function name: nes_posix_pthread_cond_wait                                   */
/* Description:   This function waits on a condition variable until   */
/*                awakened by a signal or broadcast                   */
/* Return type:   >0    :  wait condition successfully                */
/*                EINVAL:  prameter is wrong                          */
/*                ETIMEDOUT :other error                              */
/* Argument - cond: pointer to an instance of pthread_cond_t          */
/*            mutex:pointer to an instance of pthread_mutex_t         */
/**********************************************************************/
int nes_posix_pthread_cond_wait(pthread_cond_t *cond, pthread_mutex_t *mutex)
{
	INT wCall;
	int nRet = 0;
	INT flgptn = 0;
	static int i = 1;
	int ifindptn = 0;
	node *pnode;
	
	/* because pthread_cond_wait always is used with
	 * pthread_mutex_lock and pthread_mutex_unlock 
	 * in Linux
	 */
	nes_posix_pthread_mutex_unlock(mutex);
	//Check the cond and mutex is invalid
	if(!cond)
	{
		nRet = EINVAL;
		goto exit;
	}
	if(!mutex)
	{
		nRet = EINVAL;
		goto exit;
	}	

	// Check whether the mutex is locked and owned by this thread.
	if (mutex->__m_kind != PTHREAD_MUTEX_TIMED_NP && mutex->__m_kind != PTHREAD_MUTEX_ADAPTIVE_NP)	      
	{
		nRet = EINVAL;
		goto exit;
	}
	
	//check the condID is existing or not in the queue
	nes_posix_pthread_mutex_lock(&cond->lock);
	ifindptn = queue_find((struct queue*)cond->myqueue, pthread_self());
	nes_posix_pthread_mutex_unlock(&cond->lock);
		
	/* in the condition queue,if cond don't exist,
	 * push the cond into the condition queue.
	 * then execute the tkse_wai_flg()
	 */
	if (ifindptn == -1 )
	{
		/* transfer iSelf (thread id),i to *cond
		 * save the them to the cond
		 */
		pnode = (node*)malloc(sizeof(node));
		if( !pnode )
		{
			nRet= ENOMEM;
			goto exit;
		}
	
		pnode->itid = pthread_self();
		
		/* give the pattern value to the cond ,so when signal
		 * is executed, the phread (has the pattern value)
		 * is executed.
		 */
		pnode->iptn = i++;
		ifindptn = pnode->iptn;	
		
		
		/* if the condID is not existing in the queue,push
		 * it into the queue.
		 */
		nes_posix_pthread_mutex_lock(&cond->lock);
		queue_push((struct queue*)cond->myqueue, pnode);
		pnode = NULL;		
		nes_posix_pthread_mutex_unlock(&cond->lock);
	}
	
	wCall = tkse_wai_flg(cond->iCond_id, ifindptn, TWF_ANDW, &flgptn, T_FOREVER);
	
	//at Linux, pthread_cond_wait always return 0
	if (wCall > 0)
	{
		nRet = 0;
		goto exit;		
	}

	// Error code mapping
	switch(wCall)
	{
	case E_ID:
	case E_PAR:
	case E_NOEXS:
	case E_OBJ:
		nRet = EINVAL;		
		break;
	case E_DLT:
	case E_RLWAI:
	case E_DISWAI:
		nRet = EINTR;
		break;
	case E_TMOUT: 			/* Never reach here */
		nRet = ETIMEDOUT;
		break;
	default:
		nRet = EINVAL;
		break;
	}

	nes_posix_pthread_mutex_lock(&cond->lock);	
	queue_search_del((struct queue*)cond->myqueue, pthread_self());
	nes_posix_pthread_mutex_unlock(&cond->lock);	
exit:
	nes_posix_pthread_mutex_lock(mutex);
	return nRet;	
}

/**********************************************************************/
/* Function name: nes_posix_pthread_cond_timedwait                              */
/* Description:   This function waits on a condition variable either  */
/*                until awakened by a signal or broadcast; or until   */
/*                the time specified by abstime passes                */
/* Return type:   0    :  wait condition successfully                 */
/*                EINVAL: 'cond', 'mutex', or abstime is invalid      */
/*                EINVAL:  different mutexes for concurrent waits     */
/*                EINVAL:  mutex is not held by the calling thread    */
/*                EINTR;@ pthread_cond_timedwait was interrupted by  */
/*                         a signal                                   */
/*                ETIMEDOUT:abstime ellapsed before cond was signaled */
/* Argument - cond: pointer to an instance of pthread_cond_t          */
/*            mutex:pointer to an instance of pthread_mutex_t         */
/*            abstime: pointer to an instance of                      */
/*                     (const struct timespec)                        */
/**********************************************************************/

int nes_posix_pthread_cond_timedwait(pthread_cond_t *cond, pthread_mutex_t *mutex, const struct timespec *abstime)
{
	static int i = 1;

	INT wCall = 0;
	int nRet = 0;
	int ifindptn = 0;
	UINT flgptn = 0;
	node *pnode = NULL;
	STIME currenttime;
	TMOUT tmo = {0};
	
	/* because pthread_cond_wait always is used with
	 * pthread_mutex_lock and pthread_mutex_unlock 
	 * in Linux
	 */
	nes_posix_pthread_mutex_unlock(mutex);
	//Check the cond and mutex is invalid or not
	if(!cond)
	{
		nRet = EINVAL;
		goto exit;
	}
	if(!mutex)
	{
		nRet = EINVAL;
		goto exit;
	}
	if(!abstime)
	{
		nRet = EINVAL;
		goto exit;
	}
	
	/* check whether the mutex is locked and owned by this thread.
	 * check the cond and time varible are existing or not
	 */ 
	
	if (mutex->__m_kind != PTHREAD_MUTEX_TIMED_NP && mutex->__m_kind != PTHREAD_MUTEX_ADAPTIVE_NP) 	      
	{
		nRet = EINVAL;
		goto exit;
	}
	
	//get now time
	tkse_get_tim2(&currenttime, NULL);
	
	
	tmo = ( (abstime->tv_sec) * 1000 + (abstime->tv_nsec)/1000000) - currenttime * 1000;
	if(tmo < 0)
	{
		nRet = EINVAL;
		goto exit;	
	}
	//check the condID is existing or not in the queue
	nes_posix_pthread_mutex_lock(&cond->lock);
	ifindptn = queue_find((struct queue*)cond->myqueue, pthread_self());
	nes_posix_pthread_mutex_unlock(&cond->lock);

	/* in the condition queue,if cond don't exist,
	 * push the cond into the condition queue.
	 * then execute the tkse_wai_flg()
	 */
	if (ifindptn == -1 )
	{
		/* transfer iSelf (thread id),i to *cond
		 * save the them to the cond
		 */
		// Create new data and put to queue
		pnode = (node*)malloc(sizeof(node));
		pnode->itid = pthread_self();
		
		/* give the pattern value to the cond ,so when signal
		 * is executed, the phread (has the pattern value)
		 * is executed.
		 */
		pnode->iptn = i++;
		ifindptn = pnode->iptn;	
			
		/* if the condID is not existing in the queue,push
		 * it into the queue.
		 */
		nes_posix_pthread_mutex_lock(&cond->lock);	
		queue_push((struct queue*)cond->myqueue, pnode);
		nes_posix_pthread_mutex_unlock(&cond->lock);
	}
	wCall = tkse_wai_flg(cond->iCond_id, ifindptn, TWF_ANDW, &flgptn, tmo);
	
	//the thread waiting for the conditon is waken successfully by other thread
	if (wCall >= E_OK)
	{
		nRet = 0;	
		goto exit;		
	}

	// Error code mapping
	switch(wCall)
	{
	case E_ID:
	case E_PAR:
	case E_NOEXS:
	case E_OBJ:
		nRet = EINVAL;		
		break;
	case E_DLT:
	case E_RLWAI:
	case E_DISWAI:
		nRet = EINTR;
		break;
	case E_TMOUT: 			/* Time out! */
		nRet = ETIMEDOUT;
		// queue_search_del((struct queue*)cond->myqueue, pthread_self());
		break;
	default:
		nRet = EINVAL;
		break;
	}

	nes_posix_pthread_mutex_lock(&cond->lock);	
	queue_search_del((struct queue*)cond->myqueue, pthread_self());
	nes_posix_pthread_mutex_unlock(&cond->lock);	
exit:
	nes_posix_pthread_mutex_lock(mutex);
	return nRet;  
}

/**********************************************************************/
/* Function name: nes_posix_pthread_cond_signal                                 */
/* Description:   This function signals a condition variable, waking  */
/*                one waiting thread.(FIFO)                           */
/* Return type:   0     : successfully signaled condition             */
/*                EINVAL: parameter is invalid                        */
/* Argument - cond: pointer to an instance of pthread_cond_t          */
/**********************************************************************/
int nes_posix_pthread_cond_signal(pthread_cond_t *cond)
{	
	INT wCall;
	int nRet = 0;
	int iptn = 0;
	node* pnode = NULL;

	//conditon queue is not existing	
	if (!cond->myqueue) 
	{
		nRet = EINVAL;
		goto exit;
	}
	else
	{
		//get the first inputed node(waiting thread)from the queue of condition 
		nes_posix_pthread_mutex_lock(&cond->lock);
		pnode = queue_pop((struct queue*)cond->myqueue);
		nes_posix_pthread_mutex_unlock(&cond->lock);
		//there is not waiting pthread in the condition queue
		if( !pnode )
		{
			goto exit;
		}
		else
		{
			//save the iptn
			iptn = pnode->iptn;
			free(pnode);
			pnode = NULL;				
		}
	}
	
	/* tkse_set_flg() the default value of flg is suit to multi-task to wake.
	 * to void it,set the thread's pattern (iptn) into the second pramater
	 * of tkse_set_flg.
	 */
	wCall = tkse_set_flg(cond->iCond_id, iptn);
	if (wCall == E_OK )
	{
		nRet = 0;
		goto exit;		
	}
	else
	{
		// Error code mapping
		switch(wCall)
		{
		case E_ID:			// Invalid parameter
		case E_PAR:			// Invalid parameter
		case E_NOEXS:			// Invalid parameter 
		default:			
			nRet = EINVAL;		
			break;
		}
	}			
exit:
	return nRet;
	
}

/**********************************************************************/
/* Function name: nes_posix_pthread_cond_broadcast                              */
/* Description:   This function signals a condition variable, waking  */
/*                all waiting threads                                 */
/* Return type:   0     : successfully signaled condition             */
/*                EINVAL: parameter is invalid                        */
/* Argument - cond: pointer to an instance of pthread_cond_t          */
/**********************************************************************/
int nes_posix_pthread_cond_broadcast(pthread_cond_t *cond)
{
	INT wCall;
	int nRet = 0;
	node* pnode = NULL;
	
	//conditon queue is not existing	
	if (!cond || !cond->myqueue) 
	{
		nRet = EINVAL;
		goto exit;
	}
	//first get the first inputed node(the infor waiting pthread)from the queue of condition 			
	nes_posix_pthread_mutex_lock(&cond->lock);
	pnode = queue_pop((struct queue*)cond->myqueue);
	//pop all the node(the infor waiting pthread) from the condtion queue
	while(pnode != NULL)
	{
		
		/* set the thread's pattern (iptn) into the second pramater
		 * of tkse_set_flg,so awake the pthread to execute
		 */
		wCall = tkse_set_flg(cond->iCond_id, pnode->iptn);
		//free the memory
		free(pnode);
		pnode = NULL;
		pnode = queue_pop((struct queue*)cond->myqueue);
	}
	nes_posix_pthread_mutex_unlock(&cond->lock);
exit:
	return nRet;
}

