/**
 * @file manager.c
 * @define implement of internal core operation
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <tkse/extension/proctask.h>

#include "nes_posix_error.h"
#include "nes_posix_siginfo.h"

#include "manager.h"
#include "./utility/hashtable.h"

extern sigset_t __g_block_sigmask;			/* sigmanager.c */

/* Global data define */
struct hashtable* __g_threadhash = 0;	/* Global hashtable to store all thread's control block */
pthread_mutex_t __g_threadlock;		/* Global lock to protect __g_threadhash */	

pthread_t __g_main_thread = 0;		/* Main thread id */

int __pthread_lock(void);
int __pthread_unlock(void);

/**********************************************************************/
/* Function name: int __pthread_hashfromkey                           */
/* Description:hash key create algorithm. here just return source     */
/*             value. complete depend hashtable's hash algorithm      */
/* Return: the key that created by hash algorithm                     */
/* Argument - void* key:  source index/key                            */
/**********************************************************************/
static unsigned int __pthread_hashfromkey(void* key)
{
	return (int)key;
}

/**********************************************************************/
/* Function name: __pthread_equalkeys                                 */
/* Description:  Use for hashtable to compare keys                    */
/* Return type: static int  Compare result. return 0 if not equal     */
/* Argument - void* key1:  first key                                  */
/* Argument - void* key2:  second key                                 */
/**********************************************************************/
static int __pthread_equalkeys(void* key1, void* key2)
{
	return (key1 == key2);
}

/**********************************************************************/
/* Function name: __phread_init                                       */
/* Description: Init global data. Here is __g_threadhash and          */
/*              __g_threadlock This function must be called at first. */
/*              The Macro PTHREAD_INIT() can hide the call detail     */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - void:                                                   */
/**********************************************************************/
int __phread_init(void)
{
	/* 
	 * !!! ATTENTION !!!
	 * Because here, we haven't any valid lock to protect self, so
	 * This function NOT SUPPORT muti thread, and user must make sure 
	 * that only one thread call it at one time.
	 */
	int ret = 0;
	pthread_mutexattr_t attr;

	/* For main thread */
	__pthread_desr* desr = NULL;
	pthread_attr_t mainattr;

	if( __g_threadhash ) 	/* Init already */
	{
		return 0;
	}

	/*
	 *  Create global thread lock with PTHREAD_MUTEX_RECURSIVE_NP
	 *  It protect the hashtable that store all thread's control block 
	 */
	attr.__mutexkind = PTHREAD_MUTEX_RECURSIVE_NP;
	if( nes_posix_pthread_mutex_init(&__g_threadlock, &attr) ) 	/* Return !0 is failed */
	{
		ret = EPERM;
		goto exit;
	}

	/* 
	 * Init descriptor strorage manager (hash table), supportedmax thread count is 1k.
	 */
	__g_threadhash = create_hashtable(1024, __pthread_hashfromkey, __pthread_equalkeys);
	if ( !__g_threadhash ) 
	{
		ret = EPERM;
		goto exit;
	}

	/*
	 * Must create and record 'Main' thread' 
	 * And Must be the 'Main' thread do this init works.
	 */

	/* Create main thread descriptor struct */
	desr = (__pthread_desr*)malloc(sizeof(__pthread_desr));
	if( !desr ) 
	{
		ret = ENOMEM;
		goto exit;
	}

	/* For no one can join main thread, so make it run under PTHREAD_CREATE_DETACHED mode */
	mainattr.__detachstate = PTHREAD_CREATE_DETACHED;
	if(  __pthread_desr_init(desr, &mainattr, 1)  ) 
	{
		free(desr);
		ret = EAGAIN;
		goto exit;
	}

	/* Welcome the first member of us ! */
	GLOBAL_THREADMGR_LOCK();
	__pthread_add_desr(pthread_self(), desr);	/* The caller thread must be main thread */
	GLOBAL_THREADMGR_UNLOCK();

exit:
	return ret;
}

/**********************************************************************/
/* Function name: __phread_finish                                     */
/* Description:  Destroy global data. Here is __g_threadhash and      */
/* __g_threadlock. This function must be called at first. The Macro   */
/* PTHREAD_FINISH() can hide the detail                               */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - void:                                                   */
/**********************************************************************/
int __phread_finish(void)
{
	if( __g_threadhash ) 
	{
		/*
		 * Must clean 'Main' thread resource
		 */
		pthread_exit(NULL);		/* The caller thread must be main thread */

		hashtable_destroy(__g_threadhash, 1);
		__g_threadhash = 0;
	}

	nes_posix_pthread_mutex_destroy(&__g_threadlock);
	return 0;
}

/**********************************************************************/
/* Function name: __pthread_lock                                      */
/* Description: We don't recommend you use __g_threadlock directly.   */
/*              So use this wrapper                                   */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - void:                                                   */
/**********************************************************************/
int __pthread_lock(void)
{
	return nes_posix_pthread_mutex_lock(&__g_threadlock);
}

/**********************************************************************/
/* Function name: __pthread_unlock                                    */
/* Description: We don't recommend you use __g_threadlock directly.   */
/*              So use this wrapper                                   */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - void:                                                   */
/**********************************************************************/
int __pthread_unlock(void)
{
	return nes_posix_pthread_mutex_unlock(&__g_threadlock);
}

/**********************************************************************/
/* Function name: __pthread_desr_init                                 */
/* Description: Every active thread has a 'Descriptor', also call it  */
/*              'Thread Control Block'.  This function init this      */
/*               Descriptor. We give it a default pthread_attr_t. if  */
/*               need under PTHREAD_CREATE_JOINABLE, we must init the */ 
/*               lock to make join action is available.               */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - __pthread_desr* desr:                                   */
/* Argument -unsigned char is_mainthread:  */
/**********************************************************************/
int __pthread_desr_init(__pthread_desr* desr, const pthread_attr_t* attr, unsigned char is_mainthread)
{
	int nRet = 0;
	P_STATE PS;		/* Use for get 'main' thread priority */

	if( !desr ) 
	{
		return EINVAL;
	}

	/* Init pthread attr */
	if( !attr ) 
	{
		pthread_attr_init(&(desr->__attr));
	} 
	else 
	{
		memcpy(&(desr->__attr), attr, sizeof(pthread_attr_t));
	}
	
	desr->__status = STATUS_THREAD_DEFAULT;
	desr->__lock = 0;
	desr->__exitcode = 0;
	memset(desr->__tsd_buf, 0, sizeof(desr->__tsd_buf));

	//------------------------------------------------->by liu-cm
	desr->cancelState = PTHREAD_CANCEL_ENABLE;			/* cancel state */
	desr->cancelType = PTHREAD_CANCEL_DEFERRED;			/* cancel type */
	desr->p_cleanup = NULL;
	//<------------------------------------------------- by liu-cm

	/* 
	 * Modified by wang-yg @ NECSoft(JiNan) May 25 2006 
	 * Thread's block mask inherit from process
	 * JUST READ, not use lock
	 */
	desr->__block_sigmask = __g_block_sigmask;


	/* Main thread specific setting */
	if( is_mainthread )  	
	{
		/* Set main thread flag bit */
		desr->__status |= STATUS_THREAD_MAIN_THREAD;

		/* Get 'main' thread priority and save to TCB */
		tkse_prc_sts(0, &PS, NULL);
		desr->__attr.__schedparam.sched_priority = PS.priority;
	}

	/*
	 * Create semaphore for join thread
	 */
	if( desr->__attr.__detachstate == PTHREAD_CREATE_JOINABLE ) 
	{
		if( (nRet = __internal_lock_init(&(desr->__lock)))  ) 
		{
			return nRet;
		}

		/* Use __internal_lock as semaphore , so empty it's resource first */
		__internal_lock(&(desr->__lock));
	}

	return 0;
}

/**********************************************************************/
/* Function name: __pthread_desr_destroy                              */
/* Description: Destroy a TCB                                         */
/* Return type: Always return 0                                       */
/* Argument - __pthread_desr* desr:                                   */
/**********************************************************************/
int __pthread_desr_destroy(__pthread_desr* desr)
{
	if( !desr ) 
	{
		return EINVAL;
	}

//	if( desr->__attr.__detachstate == PTHREAD_CREATE_JOINABLE ) {
	__internal_lock_destroy(&(desr->__lock));
//	}
	return 0;
}


/**********************************************************************/
/* Function name: __pthread_add_desr                                  */
/* Description: Add new thread control block,called by pthread_create */
/*                usually                                             */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - const pthread_t tid:  desr's owner, thread id           */
/* Argument - const __pthread_desr* desr: Thread control block pointer*/
/**********************************************************************/
int __pthread_add_desr(const pthread_t tid, const __pthread_desr* desr)
{
	if( !__g_threadhash || !desr ) 
	{
		return EINVAL;
	}

	if( !hashtable_insert(__g_threadhash, (void*)tid, (void*)desr) ) 	/* Return 0 is failed */
	{
		return EPERM;
	}

	return 0;
}

/**********************************************************************/
/* Function name: __pthread_del_desr                                  */
/* Description: Remove thread control block by thread id from hashtable,*/
/*                if thread is run under join type, it called by      */
/*                pthread_join, else by pthread_exit. But NOT FREE    */
/*                 __pthread_desr!                                    */
/* Return type: int   Return 0 if success else some EINVAL            */
/* Argument - const pthread_t tid:                                    */
/**********************************************************************/
int __pthread_del_desr(const pthread_t tid)
{
	if( !__g_threadhash ) 
	{
		return EINVAL;
	}

	hashtable_remove(__g_threadhash, (void*)tid);
	return 0;
}

/**********************************************************************/
/* Function name: __pthread_del_desr_2                                */
/* Description: Same as  __pthread_del_desr, just return the pointer  */
/*              that removed                                          */
/* Return type: __pthread_desr* corresponding thread control block.   */
/*             If not find, return NULL                               */
/* Argument - const pthread_t tid:                                    */
/**********************************************************************/
__pthread_desr* __pthread_del_desr_2(const pthread_t tid)
{
	__pthread_desr* desr = NULL;

	if( !__g_threadhash ) 
	{
		return NULL;
	}
	desr = hashtable_remove(__g_threadhash, (void*)tid);
	return desr;
}

/**********************************************************************/
/* Function name: __pthread_find_desr                                 */
/* Description: Get corresponding thread control block by thread id   */
/*              from hashtable                                        */
/* Return type: __pthread_desr* return corresponding TCB,             */
/*              if not find return NULL                               */
/* Argument - const pthread_t tid:                                    */
/**********************************************************************/
__pthread_desr* __pthread_find_desr(const pthread_t tid)
{
	__pthread_desr* desr = NULL;

	if( !__g_threadhash ) 
	{
		return NULL;
	}
	desr = hashtable_search(__g_threadhash, (void*)tid);
	return desr;
}

