/**
 * @file nes_posix_cond.c
 * @brief implement of condition operation
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <btron/util.h>
#include <stdlib.h>

#include "common.h"
#include "nes_posix_cond.h"
#include "./utility/queue.h"

/**********************************************************************/
/* Function name:   nes_posix_pthread_condattr_init                             */
/* Description:                                                       */
/*             Initializes a condition variable attributes object with*/
/*             default attributes.in fact, the default value of attr  */
/*             is PTHREAD_ PROCESS_PRIVATE                            */
/* Return type: 0           always return 0;                          */
/* Argument - pthread_condattr_t * attr:                              */
/**********************************************************************/

int nes_posix_pthread_condattr_init(pthread_condattr_t * attr)
{
	/*set the flg of condition can be deleted */
	attr->busy_check = 0;
	return 0;
}

/**********************************************************************/
/* Function name:  nes_posix_pthread_condattr_destroy                           */
/* Description:    destroy the attribute of pthread                   */
/*                 Initializes a condition variable attributes object */
/*                 with default attributes.in fact, the value of attr */
/*                 is NULL,so the function do nothing.                */
/* Return type: 0           always return 0;                          */
/* Argument - pthread_condattr_t * attr:                              */
/**********************************************************************/

int nes_posix_pthread_condattr_destroy(pthread_condattr_t *attr)
{
  return 0;
}

/**********************************************************************/
/* Function name:  nes_posix_pthread_cond_init                                  */
/* Description:    This function initializes a condition variable     */
/*                 Initializes a condition variable attributes object */
/*                 with default attributes.in fact, the value of attr */
/*                 is NULL,so the function do nothing.                */
/* Return type:  =0  successfully created condition variable          */
/*              EINVAL          'attr' is invalid or unknow error     */
/*              EAGAIN          insufficient resources (other than    */
/*                              memory,                               */
/*              ENOMEM          insufficient memory,                  */
/* Argument - cond : pointer to an instance of pthread_cond_t         */
/*            attr : specifies optional creation attributes           */
/**********************************************************************/

int nes_posix_pthread_cond_init(pthread_cond_t *cond, pthread_condattr_t *cond_attr)
{
	WERR wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value	

	if( !cond )
	{
		nRet = EINVAL;
		goto exit;
	}
	
	if(cond_attr)
	{
		cond->busy_check = cond_attr->busy_check;		
	} 
	else
	{
		cond->busy_check = 0;
	}	
		
		
	wCall = b_cre_flg (0, DELEXIT);
	// Create successfully
	if ( wCall > 0 )
	{
		//map errocode
		nRet = 0;
		// save cond
		cond ->iCond_id = wCall;
		// Init data queue
		cond ->myqueue = (struct queue*)malloc(sizeof(struct queue));
		queue_init((struct queue*)cond ->myqueue);		
		//initial internal lock
		nRet = nes_posix_pthread_mutex_init(&cond->lock, NULL);
		goto exit;

	}
		
	// Error code mapping
	switch(wCall)
	{
	case ER_NOSPC:			//insufficient memory
		nRet = ENOMEM;
		break;
	case ER_LIMIT:			// Reach  eventflag max count limit
		nRet = EAGAIN;		
		break;
	case ER_PAR:			// Invalid parameter
		nRet = EINVAL;
		break;
	default:			// Unknow error
		nRet = EINVAL;
	}
exit:
	return nRet;  
}

/**********************************************************************/
/* Function name: nes_posix_pthread_cond_destroy                                */
/* Description:   This function destroys a condition variable         */
/*                if no thread is using the cond Variable             */
/* Return type:   0  successfully destroy condition variable          */
/*              EINVAL          parameters is invalid                 */
/*              EBUSY          insufficient memory,                   */
/* Argument - cond: pointer to an instance of pthread_cond_t          */
/**********************************************************************/   
 
int nes_posix_pthread_cond_destroy(pthread_cond_t *cond)
{	
	ERR wCall = 0;		// T-Engine call result
	int nRet = 0;		// Linux type return value	

	if(!cond)
	{
		nRet = EINVAL;
		goto exit;
	}
		
	if(!cond ->iCond_id)
	{
		nRet = EINVAL;
		goto exit;
	}
	
	//delete the source of cond queuefree(cond ->myqueue)
	nes_posix_pthread_mutex_lock(&cond->lock);
	
	/*set the flg of condition can be deleted or not
	 *when value=1;if condition being used by thread,it cann't be deleted
	 *when value=0;if condition being used by thread,it can be deleted
	 */
	if( cond->busy_check ) 
	{
		//if condition is used by other thread,it cann't be deleted
		struct node* head = cond->myqueue->head;
		if( head )
		{
			nes_posix_pthread_mutex_unlock(&cond->lock);
			return EBUSY;
		}
	}	
 
	//condition can be deleted or the queue of cond is NULL
	nRet = queue_destroy((struct queue*)cond ->myqueue);
	//the return value of queue_destroy is unnecessary to judge
	
	/* 
	 * added by liu on 20060313
	 * if condtion is not initialed,don't destroy it
	 */
	if(cond ->myqueue)
	{
		free(cond ->myqueue);
	}
	
	/* Reset wait queue */
	cond ->myqueue = NULL;
	
	nes_posix_pthread_mutex_unlock(&cond->lock);
	
	//delete internal lock	
	nRet = nes_posix_pthread_mutex_destroy(&cond->lock);
	if (nRet != 0)
	{
		wCall = b_del_flg(cond->iCond_id);
		goto exit;
	}
	wCall = b_del_flg(cond->iCond_id);

	// code mapping
	switch(wCall)
	{
	case ER_OK:
		nRet = 0;		//delete flg successfully
		break;			
	case ER_ID:			// Invalid parameter
		nRet = EINVAL;
		break;
	case ER_NOEXS:			// Invalid parameter 
		nRet = EINVAL;		
		break;
	default:			// flg is using by other thread
		nRet = EBUSY;
	}
exit:
	return nRet;  
	
}

/**********************************************************************/
/* Function name: nes_posix_pthread_cond_wait                                   */
/* Description:   This function waits on a condition variable until   */
/*                awakened by a signal or broadcast                   */
/* Return type:   >0    :  wait condition successfully                */
/*                EINVAL:  prameter is wrong                          */
/*                ETIMEDOUT :other error                              */
/* Argument - cond: pointer to an instance of pthread_cond_t          */
/*            mutex:pointer to an instance of pthread_mutex_t         */
/**********************************************************************/
int nes_posix_pthread_cond_wait(pthread_cond_t *cond, pthread_mutex_t *mutex)
{
	WERR wCall;
	int nRet = 0;
	static int i = 1;
	int ifindptn = 0;
	node *pnode;
	
	/* because pthread_cond_wait always is used with
	 * pthread_mutex_lock and pthread_mutex_unlock 
	 * in Linux
	 */
	nes_posix_pthread_mutex_unlock(mutex);
	//Check the cond and mutex is invalid
	if(!cond)
	{
		nRet = EINVAL;
		goto exit;
	}
	if(!mutex)
	{
		nRet = EINVAL;
		goto exit;
	}	

	// Check whether the mutex is locked and owned by this thread.
	if (mutex->__m_kind != PTHREAD_MUTEX_TIMED_NP
	      && mutex->__m_kind != PTHREAD_MUTEX_ADAPTIVE_NP)	      
	{
		nRet = EINVAL;
		goto exit;
	}
	
	//check the condID is existing or not in the queue
	nes_posix_pthread_mutex_lock(&cond->lock);
	ifindptn = queue_find((struct queue*)cond->myqueue, nes_posix_pthread_self());
	nes_posix_pthread_mutex_unlock(&cond->lock);
		
	/* in the condition queue,if cond don't exist,
	 * push the cond into the condition queue.
	 * then execute the b_wai_flg()
	 */
	if (ifindptn == -1 )
	{
		/* transfer iSelf (thread id),i to *cond
		 * save the them to the cond
		 */
		// Create new data and put to queue
		pnode = (node*)malloc(sizeof(node));
		if( !pnode )
		{
			nRet= ENOMEM;
			goto exit;
		}
	
		pnode->itid = nes_posix_pthread_self();
		
		/* give the pattern value to the cond ,so when signal
		 * is executed, the phread (has the pattern value)
		 * is executed.
		 */
		pnode->iptn = i++;
		ifindptn = pnode->iptn;	
		
		
		/* if the condID is not existing in the queue,push
		 * it into the queue.
		 */
		nes_posix_pthread_mutex_lock(&cond->lock);
		queue_push((struct queue*)cond->myqueue, pnode);
		pnode = NULL;		
		nes_posix_pthread_mutex_unlock(&cond->lock);
	}
	
	wCall = b_wai_flg(cond->iCond_id, ifindptn, WF_AND, T_FOREVER);
	
	//at Linux, pthread_cond_wait always return 0
	if (wCall > 0)
	{
		nRet = 0;
		goto exit;		
	}
	else
	{
		// Error code mapping
		switch(wCall)
		{
		case ER_ID:			// Invalid parameter
			nRet = EINVAL;
			break;
		case ER_PAR:			// Invalid parameter
			nRet = EINVAL;
			break;	
		case ER_NOEXS:			// Invalid parameter 
			nRet = EINVAL;		
			break;
		default:			// flg is using by other thread
			nRet = ETIMEDOUT;       // abstime ellapsed before cond was signaled
		}

		nes_posix_pthread_mutex_lock(&cond->lock);	
		queue_search_del((struct queue*)cond->myqueue, nes_posix_pthread_self());
		nes_posix_pthread_mutex_unlock(&cond->lock);	
	}
exit:
	nes_posix_pthread_mutex_lock(mutex);
	return nRet;	
}

/**********************************************************************/
/* Function name: nes_posix_pthread_cond_timedwait                              */
/* Description:   This function waits on a condition variable either  */
/*                until awakened by a signal or broadcast; or until   */
/*                the time specified by abstime passes                */
/* Return type:   0    :  wait condition successfully                 */
/*                EINVAL: 'cond', 'mutex', or abstime is invalid      */
/*                EINVAL:  different mutexes for concurrent waits     */
/*                EINVAL:  mutex is not held by the calling thread    */
/*                EINTR;@ pthread_cond_timedwait was interrupted by  */
/*                         a signal                                   */
/*                ETIMEDOUT:abstime ellapsed before cond was signaled */
/* Argument - cond: pointer to an instance of pthread_cond_t          */
/*            mutex:pointer to an instance of pthread_mutex_t         */
/*            abstime: pointer to an instance of                      */
/*                     (const struct timespec)                        */
/**********************************************************************/

int nes_posix_pthread_cond_timedwait(pthread_cond_t *cond, pthread_mutex_t *mutex, const struct timespec *abstime)

{
	WERR wCall;
	int nRet = 0;
	static int i = 1;
	node *pnode;
	STIME currenttime;
	TMOUT tmo;
	int ifindptn = 0;
	
	/* because pthread_cond_wait always is used with
	 * pthread_mutex_lock and pthread_mutex_unlock 
	 * in Linux
	 */
	nes_posix_pthread_mutex_unlock(mutex);
	//Check the cond and mutex is invalid or not
	if(!cond)
	{
		nRet = EINVAL;
		goto exit;
	}
	if(!mutex)
	{
		nRet = EINVAL;
		goto exit;
	}
	if(!abstime)
	{
		nRet = EINVAL;
		goto exit;
	}
	
	/* check whether the mutex is locked and owned by this thread.
	 * check the cond and time varible are existing or not
	 */ 
	
	if (mutex->__m_kind != PTHREAD_MUTEX_TIMED_NP
	      && mutex->__m_kind != PTHREAD_MUTEX_ADAPTIVE_NP) 	      
	{
		nRet = EINVAL;
		goto exit;
	}
	
	//get now time
	b_get_tim(&currenttime, NULL);
	
	
	tmo = ( (abstime->tv_sec) * 1000 + (abstime->tv_nsec)/1000000) - currenttime * 1000;
	if(tmo < 0)
	{
		nRet = EINVAL;
		goto exit;	
	}
	//check the condID is existing or not in the queue
	nes_posix_pthread_mutex_lock(&cond->lock);
	ifindptn = queue_find((struct queue*)cond->myqueue, nes_posix_pthread_self());
	nes_posix_pthread_mutex_unlock(&cond->lock);

	/* in the condition queue,if cond don't exist,
	 * push the cond into the condition queue.
	 * then execute the b_wai_flg()
	 */
	if (ifindptn == -1 )
	{
		/* transfer iSelf (thread id),i to *cond
		 * save the them to the cond
		 */
		// Create new data and put to queue
		pnode = (node*)malloc(sizeof(node));
		pnode->itid = nes_posix_pthread_self();
		
		/* give the pattern value to the cond ,so when signal
		 * is executed, the phread (has the pattern value)
		 * is executed.
		 */
		pnode->iptn = i++;
		ifindptn = pnode->iptn;	
			
		/* if the condID is not existing in the queue,push
		 * it into the queue.
		 */
		nes_posix_pthread_mutex_lock(&cond->lock);	
		queue_push((struct queue*)cond->myqueue, pnode);
		nes_posix_pthread_mutex_unlock(&cond->lock);
	}
	wCall = b_wai_flg(cond->iCond_id, ifindptn, WF_AND, tmo);
	
	//the thread waiting for the conditon is waken successfully by other thread
	if (wCall > 0)
	{
		nRet = 0;	
		goto exit;		
	}
	else
	{
		// Error code mapping
		switch(wCall)
		{
		case ER_ID:			// Invalid parameter
		case ER_PAR:			// Invalid parameter
		case ER_NOEXS:			// Invalid parameter 
			nRet = EINVAL;		
			break;
		case ER_MINTR:
			nRet = EINTR;		// pthread_cond_timedwait was interrupted by a signal
			break;			// add by liu-cm 2006/03/03
		case ER_NONE:
			nRet = ETIMEDOUT;	// abstime ellapsed before cond was signaled
			//delete the info of caling thread from condtion struct(add on 20060310)
			//queue_search_del((struct queue*)cond->myqueue, pthread_self());
			break;
		default:			
			nRet = EINVAL;       
		}

		nes_posix_pthread_mutex_lock(&cond->lock);	
		queue_search_del((struct queue*)cond->myqueue, nes_posix_pthread_self());
		nes_posix_pthread_mutex_unlock(&cond->lock);
	}
exit:
	nes_posix_pthread_mutex_lock(mutex);
	return nRet;  
	
}

/**********************************************************************/
/* Function name: nes_posix_pthread_cond_signal                                 */
/* Description:   This function signals a condition variable, waking  */
/*                one waiting thread.(FIFO)                           */
/* Return type:   0     : successfully signaled condition             */
/*                EINVAL: parameter is invalid                        */
/* Argument - cond: pointer to an instance of pthread_cond_t          */
/**********************************************************************/
int nes_posix_pthread_cond_signal(pthread_cond_t *cond)
{	
	ERR wCall;
	int nRet = 0;
	int iptn = 0;
	node* pnode = NULL;
	//conditon queue is not existing	
	if (!cond->myqueue) 
	{
		nRet = EINVAL;
		goto exit;
	}
	else
	{
		//get the first inputed node(waiting thread)from the queue of condition 
		nes_posix_pthread_mutex_lock(&cond->lock);
		pnode = queue_pop((struct queue*)cond->myqueue);
		nes_posix_pthread_mutex_unlock(&cond->lock);
		//there is not waiting pthread in the condition queue
		if( !pnode )
		{
			goto exit;
		}
		else
		{
			//save the iptn
			iptn = pnode->iptn;
			free(pnode);
			pnode = NULL;				
		}
	}
	
	/* b_set_flg() the default value of flg is suit to multi-task to wake.
	 * to void it,set the thread's pattern (iptn) into the second pramater
	 * of b_set_flg.
	 */
	wCall = b_set_flg(cond->iCond_id, iptn);
	if (wCall == ER_OK )
	{
		nRet = 0;
		goto exit;		
	}
	else
	{
		// Error code mapping
		switch(wCall)
		{
		case ER_ID:			// Invalid parameter
		case ER_PAR:			// Invalid parameter
		case ER_NOEXS:			// Invalid parameter 
			nRet = EINVAL;		
			break;
		default:			
			nRet = EINVAL;       
		}
	}			
exit:
	return nRet;
	
}

/**********************************************************************/
/* Function name: nes_posix_pthread_cond_broadcast                              */
/* Description:   This function signals a condition variable, waking  */
/*                all waiting threads                                 */
/* Return type:   0     : successfully signaled condition             */
/*                EINVAL: parameter is invalid                        */
/* Argument - cond: pointer to an instance of pthread_cond_t          */
/**********************************************************************/
int nes_posix_pthread_cond_broadcast(pthread_cond_t *cond)
{
	ERR wCall;
	int nRet = 0;
	node* pnode = NULL;
	
	//conditon queue is not existing	
	if (!cond || !cond->myqueue) 
	{
		nRet = EINVAL;
		goto exit;
	}
	//first get the first inputed node(the infor waiting pthread)from the queue of condition 			
	nes_posix_pthread_mutex_lock(&cond->lock);
	pnode = queue_pop((struct queue*)cond->myqueue);
	//pop all the node(the infor waiting pthread) from the condtion queue
	while(pnode != NULL)
	{
		
		/* set the thread's pattern (iptn) into the second pramater
		 * of b_set_flg,so awake the pthread to execute
		 */
		wCall = b_set_flg(cond->iCond_id, pnode->iptn);
		//free the memory
		free(pnode);
		pnode = NULL;
		pnode = queue_pop((struct queue*)cond->myqueue);
	}
	nes_posix_pthread_mutex_unlock(&cond->lock);
exit:
	return nRet;
	
}

