/**
 * @file nes_posix_siginfo.h
 * @brief signal related type define
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __NES_POSIX_SIGINFO_H_
#define __NES_POSIX_SIGINFO_H_

#include <string.h>
#ifdef NES_SE   /* Use Standard Extension */
#include <tkse/stdtype.h>
#else           /* Use PMC Extension */
#include <unix/sys/types.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

/* Most things should be clean enough to redefine this at will, if care
is taken to make libc match.  */

#ifndef _NSIG
#define _NSIG		64
#endif

#define _NSIG_BPW	32
#define _NSIG_WORDS	(_NSIG / _NSIG_BPW)

#define SIGHUP		 1
#define SIGINT		 2
#define SIGQUIT		 3
#define SIGILL		 4
#define SIGTRAP		 5
#define SIGABRT		 6
#define SIGIOT		 6
#define SIGBUS		 7
#define SIGFPE		 8
#define SIGKILL		 9
#define SIGUSR1		10
#define SIGSEGV		11
#define SIGUSR2		12
#define SIGPIPE		13
#define SIGALRM		14
#define SIGTERM		15
#define SIGSTKFLT	16
#define SIGCHLD		17
#define SIGCONT		18
#define SIGSTOP		19
#define SIGTSTP		20
#define SIGTTIN		21
#define SIGTTOU		22
#define SIGURG		23
#define SIGXCPU		24
#define SIGXFSZ		25
#define SIGVTALRM	26
#define SIGPROF		27
#define SIGWINCH	28
#define SIGIO		29	/* #define SIGLOST		29 */
#define SIGPOLL		SIGIO
#define SIGPWR		30
#define SIGSYS		31
#define SIGUNUSED	31

/* These should not be considered constants from userland.  */
#define SIGRTMIN	32
#define SIGRTMAX	_NSIG

/* 
 * Copy from <asm-generic/signal.h>
 */ 
#ifndef SIG_BLOCK
#define SIG_BLOCK           0	/* for blocking signals */
#endif
#ifndef SIG_UNBLOCK
#define SIG_UNBLOCK       1	/* for unblocking signals */
#endif
#ifndef SIG_SETMASK
#define SIG_SETMASK      2	/* for setting the signal mask */
#endif

/*
 * SA_FLAGS define
 *
 * SA_ONSTACK indicates that a registered stack_t will be used.
 * SA_INTERRUPT is a no-op, but left due to historical reasons. Use the
 * SA_RESTART flag to get restarting signals (which were the default long ago)
 * SA_NOCLDSTOP flag to turn off SIGCHLD when children stop.
 * SA_RESETHAND clears the handler when the signal is delivered.
 * SA_NOCLDWAIT flag on SIGCHLD to inhibit zombies.
 * SA_NODEFER prevents the current signal from being masked in the handler.
 *
 * SA_ONESHOT and SA_NOMASK are the historical Linux names for the Single
 * Unix names RESETHAND and NODEFER respectively.
 */
#define SA_NOCLDSTOP	0x00000001u
#define SA_NOCLDWAIT	0x00000002u
#define SA_SIGINFO		0x00000004u
#define SA_ONSTACK		0x08000000u
#define SA_RESTART		0x10000000u
#define SA_NODEFER		0x40000000u
#define SA_RESETHAND		0x80000000u

#define SA_NOMASK		SA_NODEFER
#define SA_ONESHOT		SA_RESETHAND
#define SA_INTERRUPT		0x20000000u	/* dummy -- ignored */
#define SA_RESTORER		0x04000000u

/* 
 * Added by Wang Yage @ NEC Soft( Ji Nan) 
 * Default is not set this flag
 */
#define SA_NOTINTR		0x00000010u	/* Not interrupt main thread */

/*
* si_code values
* Digital reserves positive values for kernel-generated signals.
*/
#define SI_USER		0		/* sent by kill, sigsend, raise */
#define SI_KERNEL	0x80		/* sent by the kernel from somewhere */
#define SI_QUEUE	-1		/* sent by sigqueue */
#define SI_TIMER		-2		/* sent by timer expiration */
#define SI_MESGQ       -3		/* sent by real time mesq state change */
#define SI_ASYNCIO	-4		/* sent by AIO completion */
#define SI_SIGIO		-5		/* sent by queued SIGIO */
#define SI_TKILL		-6		/* sent by tkill system call */
#define SI_DETHREAD	-7		/* sent by execve() killing subsidiary threads */

typedef struct {
	unsigned long sig[_NSIG_WORDS];
} sigset_t;

typedef int timer_t;

typedef union sigval {
	int sival_int;
	void* sival_ptr;
} sigval_t;

typedef struct siginfo {
	int si_signo;
	int si_errno;
	int si_code;

	union {
		// int _pad[SI_PAD_SIZE];

		/* kill() */
		struct {
			pid_t _pid;		/* sender's pid */
			int _uid;	/* sender's uid */
		} _kill;

		/* POSIX.1b timers */
		struct {
			timer_t _tid;		/* timer id */
			int _overrun;		/* overrun count */
			// char _pad[sizeof( __ARCH_SI_UID_T) - sizeof(int)];
			sigval_t _sigval;	/* same as below */
			int _sys_private;       /* not to be passed to user */
		} _timer;

		/* POSIX.1b signals */
		struct {
			pid_t _pid;		/* sender's pid */
			// __ARCH_SI_UID_T _uid;	/* sender's uid */
			sigval_t _sigval;
		} _rt;

		/* SIGCHLD */
		struct {
			pid_t _pid;		/* which child */
			// __ARCH_SI_UID_T _uid;	/* sender's uid */
			int _status;		/* exit code */
			int _utime;
			int _stime;
		} _sigchld;

		/* SIGILL, SIGFPE, SIGSEGV, SIGBUS */
		struct {
			void* _addr; /* faulting insn/memory ref. */
		} _sigfault;

		/* SIGPOLL */
		struct {
			// __ARCH_SI_BAND_T _band;	/* POLL_IN, POLL_OUT, POLL_MSG */
			int _fd;
		} _sigpoll;
	} _sifields;

} siginfo_t;

/*
* Signal handler define 
*/
typedef void (*__sighandler_t)(int);
typedef void (*__sigrestore_t)(void);
typedef void (*__sigaction_handler_t)(int, struct siginfo *, void *);


#define SIG_DFL	(( __sighandler_t)1)	/* default signal handling */
#define SIG_IGN	(( __sighandler_t)2)	/* ignore signal */
#define SIG_ERR	(( __sighandler_t)-1)	/* error return from signal */

struct sigaction {
	union {
		__sighandler_t _sa_handler;
		__sigaction_handler_t _sa_sigaction;
	} _u;

# define sa_handler     _u._sa_handler
# define sa_sigaction   _u._sa_sigaction

	sigset_t sa_mask;
	unsigned long sa_flags;
	void (*sa_restorer)(void);
};

#ifdef __cplusplus
}
#endif

#endif //__MY_SIGINFO_H__
