#!/bin/sh
"exec" "python" "$0" "$@"

import sys, re, string, cgi

# token type definitions.
SPACE = 1
TEXT = 2

class AsciiData:
    "ascii text data class."

    def chomp (self, str):
        "delete LF from string"

        str = string.replace (str, '\n', '')
        return str

class AsciiDataStream:
    "ascii text data stream"

    def __init__ (self, stream = sys.stdin):
        self.stream =    stream
        self.read =      self.stream.read
        self.readline =  self.stream.readline
        self.readlines = self.stream.readlines
        self.fileno =    self.stream.fileno

    def chomp (self, str):
        "delete LF from string"

        str = string.replace (str, '\n', '')
        return str

class AsciiParagraphStream (AsciiDataStream):
    def __init__ (self, stream = sys.stdin):
        AsciiDataStream.__init__ (self, stream)        

        lines = []
        for line in self.readlines ():
            line = string.strip (line)
            lines.append (line)
        self.paragraph = string.join (lines)
        
    def getparagraph (self):
        return self.paragraph

class AsciiHeaderStream (AsciiParagraphStream):
    def __init__ (self, stream = sys.stdin):
        AsciiParagraphStream.__init__ (self, stream)        

    def getheader (self):
        self.header = self.getparagraph ()
        return self.header

class AsciiPreformatStream (AsciiDataStream):
    def __init__ (self, stream = sys.stdin):
        AsciiDataStream.__init__ (self, stream)        

        self.prefmt = self.read ()
            
    def getprefmt (self):
        return self.prefmt

class AsciiListStream (AsciiDataStream):
    def __init__ (self, stream = sys.stdin):
        AsciiDataStream.__init__ (self, stream)

        self.list = self.readlist ()
        
    def readlist (self):

        lbhead_obj = re.compile ("\s*[o*]\s+(.+)")
        rows = []
        buffer = ''

        for cell in self.readlines ():
            cell = self.chomp (cell)
            lmatch = lbhead_obj.match (cell)
            if lmatch:
                cell = lmatch.group (1)
                rows.append (cell)
                buffer = ''
            else:
                buffer = buffer + cell

        return rows

    def getlist (self):
        return self.list

class AsciiHeaderStream (AsciiParagraphStream):
    def __init__ (self, stream = sys.stdin):
        AsciiParagraphStream.__init__ (self, stream)        

    def getheader (self):
        self.header = self.getparagraph ()
        return self.header

class HTMLMaker (AsciiData):
    "making HTML base class."

    def __init__ (self, source):
        self.source = cgi.escape (source)
        self.tagformat = "<!-- ERROR [%s] ERROR -->"

    def __call__ (self):
        return self.tagformat % (self.source)

    def chomp (self, str):
        "delete LF from string"

        str = string.replace (str, '\n', '')
        return str

    def escape (self, s, quote = None):
        return cgi.escape (s, quote)

    def unescape(self, s, quote = None):
        "Replace special characters SGML entities by '&', '<' and '>'"
        s = string.replace (s, "&amp;", "&") # Must be done first!
        s = string.replace(s, "&lt;", "<")
        s = string.replace(s, "&gt;", ">")
        if quote:
            s = string.replace(s, "&quot;", '"')
        return s

class Header (HTMLMaker):
    "HTML header class."

    def __init__ (self, source):
        HTMLMaker.__init__ (self, source)

        self.source = self.chomp (self.source)
        self.level = '1'
        self.tagformat = "<h%s>%s</h%s>"

    def __call__ (self):
        return self.tagformat % (self.level , self.source, self.level)

    def setlevel (self, level):
        self.level = level

class Title (Header):
    "HTML title class."

    def __init__ (self, source):
        Header.__init__ (self, source)        
        obj = re.search ('___ (.+) ___', self.source)
        self.source = obj.group (1)

    def __call__ (self):
        return self.tagformat % (self.level , self.source, self.level)

class Paragraph (HTMLMaker):
    "a class of `P' tag."
    def __init__ (self, source):
        HTMLMaker.__init__ (self, source)
        self.tagformat = "<p>%s</p>"

class JParagraph (HTMLMaker):
    "a class of `P' tag. (Japanese type)"
    def __init__ (self, source):
        HTMLMaker.__init__ (self, source)
        self.tagformat = "<p>%s</p>"

        lines = []
        for line in string.split (self.source, '\n'):
            lines.append (string.strip (line))

        self.source = string.join (lines, '')

class Preformat (HTMLMaker):
    "a class of `PRE' tag."
    def __init__ (self, source):
        HTMLMaker.__init__ (self, source)
        self.tagformat = "<pre>%s</pre>"

class AsciiList (AsciiData):
    "list type text format."

    def __init__ (self, source):
        self.source = source
        
    def __call__ (self):
        return self.readlist ()

    def readlist (self):

        lbhead_obj = re.compile ("\s*[o*]\s+(.+)")
        rows = []
        buffer = ''

        for cell in string.split (self.source, '\n'):
            cell = self.chomp (cell)
            lmatch = lbhead_obj.match (cell)
            if lmatch:
                if len (buffer) > 0:
                    rows.append (buffer)
                buffer = ''
                buffer = lmatch.group (1)
            else:
                buffer = buffer + cell

        if len (buffer) > 0:
            rows.append (buffer)
        return rows

class List (HTMLMaker):
    "list type data HTML markup"

    def __init__ (self, source):
        HTMLMaker.__init__ (self, source)
        parser = AsciiList (self.source)
        self.source = parser ()
        
        BULLET = 1
        SEQUENCE = 2

        self.listtype = BULLET

        self.start_ultag = "<ul>"
        self.end_ultag = "</ul>"
        self.start_litag = "<li>"
        self.end_litag = "</li>"        

    def __call__ (self):
        return self.gethtml ()

    def gethtml (self):
        lines = []
        lines.append (self.start_ultag)
        for cell in self.source:
            lines.append ("<li>%s</li>" % (cell))

        lines.append (self.end_ultag)

        return string.join (lines, '\n')
