package traffic;

import java.util.*;
import traffic.object.*;

public class WorldModel implements Constants {
  public final HashMap idObjMap = new HashMap();
  private ArrayList m_movingObjectList = new ArrayList();
  private MovingObject[]  m_movingObjectArray;
  public final ArrayList ambulanceTeamList = new ArrayList();
  private int m_minX = Integer.MAX_VALUE;
  private int m_minY = Integer.MAX_VALUE;
  private int m_maxX = Integer.MIN_VALUE;
  private int m_maxY = Integer.MIN_VALUE;
  private int m_time = Integer.MIN_VALUE;;
  private double m_sec;
  private boolean m_isInitialized = false;

  public MovingObject[] movingObjectArray() { return m_movingObjectArray; }
  public int minX() { return m_minX; }
  public int maxX() { return m_maxX; }
  public int minY() { return m_minY; }
  public int maxY() { return m_maxY; }
  public int time() { return m_time; }
  public double sec() { return m_sec; }
  public double secTime() { return m_time * 60d + m_sec; }
  public boolean isInitialized() { return m_isInitialized; }

  public void setSec(double sec) { m_sec = sec; }
  public void progressSec() { m_sec += UNIT_SEC; }

  public void add(RealObject obj) {
    idObjMap.put(new Integer(obj.id), obj);

    if (obj instanceof MovingObject) {
      m_movingObjectList.add(obj);
      if (obj instanceof AmbulanceTeam)
	ambulanceTeamList.add(obj);
    }
  }

  public RealObject get(int id)
    { return (RealObject) idObjMap.get(new Integer(id)); }

  public void update(int[] data, int startIndex, int time) {
    m_time = time;
    for (int i = startIndex;  data[i] != TYPE_NULL;  i ++) {
      int type = data[i ++];
      int id   = data[i ++];
      RescueObject obj = get(id);
      if (obj == null) {
	obj = newRescueObject(type, id);
	if (obj instanceof RealObject)
	  add((RealObject) obj);
      }
      while (data[i] != PROPERTY_NULL)
	i += setProperty(data, i, obj);
    }
  }

  private RescueObject newRescueObject(int type, int id) {
    RescueObject obj;
    switch (type) {
    default: if(ASSERT)Util.myassert(false, "illeagle object type", type);
    case TYPE_WORLD:            obj = new World(id);           break;
    case TYPE_RIVER:            obj = new River(id);           break;
    case TYPE_RIVER_NODE:       obj = new RiverNode(id);       break;
    case TYPE_ROAD:             obj = new Road(id);            break;
    case TYPE_NODE:             obj = new Node(id);            break;
    case TYPE_BUILDING:         obj = new Building(id);        break;
    case TYPE_AMBULANCE_CENTER: obj = new AmbulanceCenter(id); break;
    case TYPE_FIRE_STATION:     obj = new FireStation(id);     break;
    case TYPE_POLICE_OFFICE:    obj = new PoliceOffice(id);    break;
    case TYPE_REFUGE:           obj = new Refuge(id);          break;
    case TYPE_CIVILIAN:         obj = new Civilian(id);        break;
    case TYPE_AMBULANCE_TEAM:   obj = new AmbulanceTeam(id);   break;
    case TYPE_FIRE_BRIGADE:     obj = new FireBrigade(id);     break;
    case TYPE_POLICE_FORCE:     obj = new PoliceForce(id);     break;
    case TYPE_CAR:              obj = new Car(id);             break;
    }
    return obj;
  }

  /** @return a size of the property which had finished to read */
  private int setProperty(int[] data, int index, RescueObject obj) {
    if (data[index] < 0x80  ||  0xff < data[index]) {
      // sets the integer-tyep-prpoerty
      int[] val = { data[index + 1] };
      obj.input(data[index], val);
      return 1 + 1;  // {property, value}
    } else {
      if (data[index] < 0xc0) {
	// sets the list-tyep-prpoerty
	int[] val = getListElmArray(data, index + 1);
	obj.input(data[index], val);
	return 1 + 1 + val.length;  // {property, length of list, value}
      } else {
	// sets the array-type-property
	int[] val = getIDsArray(data, index + 1);
	obj.input(data[index], val);
	return 1 + val.length + 1;  // {property, value, sentinel}
      }
    }
  }

  private int[] getListElmArray(int[] data, int index) {
    int[] result = new int[data[index] / 4];  // 4: sizeof(int)
    System.arraycopy(data, index + 1, result, 0, result.length);
    return result;
  }

  private int[] getIDsArray(int[] data, int index) {
    int length;
    for (length = 0;  data[index + length] != 0;  length ++) { }
    int[] result = new int[length];
    System.arraycopy(data, index, result, 0, result.length);
    return result;
  }

  public void shuffleMvObjsArray() {
    MovingObject tmp;
    MovingObject[] mvs = m_movingObjectArray;
    for (int i = mvs.length - 1;  i > 0;  i --) {
      tmp = mvs[i];
      int exchangeIndex = RANDOM.nextInt(i);
      mvs[i] = mvs[exchangeIndex];
      mvs[exchangeIndex] = tmp;
    }
  }

  public void initialize() {
    m_isInitialized = true;
    m_movingObjectArray = (MovingObject[]) m_movingObjectList.toArray(new MovingObject[m_movingObjectList.size()]);
    m_movingObjectList = null;
    preCulcLanesAndRoadList();
  }

  private void preCulcLanesAndRoadList() {
    Iterator it = idObjMap.values().iterator();
    while (it.hasNext()) {
      Object obj = it.next();
      if (obj instanceof PointObject) {
	PointObject po = (PointObject) obj;
 	po.outLanes();
 	po.inLanes();
	if (USE_VIEWER) {
	  if (m_minX > po.x()) m_minX = po.x();
	  if (m_maxX < po.x()) m_maxX = po.x();
	  if (m_minY > po.y()) m_minY = po.y();
	  if (m_maxY < po.y()) m_maxY = po.y();
	}
      } else if (obj instanceof Road) {
	((Road) obj).setAdjacentLanesOfSamePriorityRoad();
      }
    }

    it = idObjMap.values().iterator();
    while (it.hasNext()) {
      Object obj = it.next();
      if (obj instanceof Road) {
	Road rd = (Road) obj;
 	RoadCell.roadList((Lane) rd.lanesToHead().get(0));
 	RoadCell.roadList((Lane) rd.lanesToTail().get(0));
      }
    }
  }

  public void parseCommands(int[] data) {
    int i = 2;
    m_time = data[i ++];
    while (data[i] != HEADER_NULL) {
      int command = data[i ++];
      int remainderToNextCommand = data[i ++] / 4;  // 4: sizeof(int);
      if (remainderToNextCommand == 0)
	continue;
      switch (command) {
      default:  break;
      case AK_MOVE:    parseAK_MOVEs(data, i);    break;
      case AK_LOAD:    parseAK_LOADs(data, i);    break;
      case AK_UNLOAD:  parseAK_UNLOADs(data, i);  break;
      }
      i += remainderToNextCommand;
    }
  }

  private void parseAK_MOVEs(int[] data, int index) {
    while (data[index] != HEADER_NULL)
      index += parseAK_MOVE(data, index);
  }
  private int parseAK_MOVE(int[] data, int index) {
    int bodySize = 1 + 1 + data[index + 1] / 4;  // {senderID, length, {ID-list, sentinel}}
    int senderID = data[index ++];
    RealObject sender = get(senderID);
    if (!(sender instanceof MovingObject)) {
      printError(sender, "Wrong sender for moving: " + senderID);
      return bodySize;
    }
    MovingObject agent = (MovingObject) sender;

    agent.setLastMovingTime(m_time);

    int length = data[index ++] / 4 - 1;  // -1: sentinel
    Route submitedRoute = new Route(length);
    for (int i = 0;  i < length;  i ++)
      submitedRoute.add((MotionlessObject) get(data[index ++]));

    agent.setRoutePlan(submitedRoute.checkValidity(agent)
		       ? submitedRoute
		       : Route.singleObjRoute(agent.motionlessPosition()));
    return bodySize;
  }

  private void parseAK_LOADs(int[] data, int index) {
    while (data[index] != HEADER_NULL)
      index += parseAK_LOAD(data, index);
  }
  private int parseAK_LOAD(int[] data, int index) {
    int bodySize = 1 + 1 + 1;  // {senderID, length, targetID}
    int senderID = data[index ++];
    RealObject sender = get(senderID);
    if (!(sender instanceof AmbulanceTeam)) {
      printError(sender, "Wrong sender for loading: " + senderID);
      return bodySize;
    }
    AmbulanceTeam ambulance = (AmbulanceTeam) sender;
    int length = data[index ++] / 4;
    if(ASSERT)Util.myassert(length == 1, "A body size of the AK_LOAD command must be one", length);
    Object target = get(data[index ++]);
    if (!(target instanceof MovingObject)) {
      printError(sender, "Wrong loading target: " + target);
      return bodySize;
    }
    if (ambulance.isLoading()) {
      printError(sender, "This AmbulanceTeam has been loading an injured person already");
      return bodySize;
    }
    MovingObject mv = (MovingObject) target;
    if (ambulance.position() == mv.position()
	|| (!(mv.position() instanceof Building)
	    && (ambulance.position() instanceof Node  &&  ((Node) ambulance.position()).edges().contains(mv.position())
		|| ambulance.position() instanceof Road  &&  (((Road) ambulance.position()).head() == mv.position()
							      || ((Road) ambulance.position()).tail() == mv.position())))) {
      ambulance.setLoad(mv);
    } else {
      printError(sender, "Wrong position: This AmbulanceTeam's position (" + ambulance.position().id +
		 ") is not the same position of the target's (" + mv.position().id + ").");
    }
    return bodySize;
  }

  private void parseAK_UNLOADs(int[] data, int index) {
    while (data[index] != HEADER_NULL)
      index += parseAK_UNLOAD(data, index);
  }
  private int parseAK_UNLOAD(int[] data, int index) {
    int bodySize = 1 + 1;  // {senderID, length}
    int senderID = data[index ++];
    RealObject sender = get(senderID);
    if (!(sender instanceof AmbulanceTeam)) {
      printError(sender, "Wrong sender for unloading: " + senderID);
      return bodySize;
    }
    AmbulanceTeam ambulance = (AmbulanceTeam) sender;
    int length = data[index ++] / 4;
    if(ASSERT)Util.myassert(length == 0, "A body size of the AK_UNLOAD command must be zero.", length);
    if (ambulance.isLoading()) ambulance.setUnload();
    else                       printError(sender, "This AmbulanceTeam is not loading an injured person now.");
    return bodySize;
  }

  private void printError(RealObject obj, String message) {
    if (PRINT_REASON_WHY_AGENT_COMMAND_WAS_NOT_EXECUTED)
      System.err.println("[Wrong Agent Command] time:" + WORLD.time() + ", agentID:" + obj.id + "\n  " + message);
  }
}
