// RescueSystem.cxx
//
/////////////////////////////////////////////////////////////////////////////

#include "common.hxx"
#include "RescueSystem.hxx"

#ifndef NDEBUG
#  include "RescueSystem.inl"
#endif

#include "objects.hxx"

namespace Rescue
{
    /////////////////////////////////////////////////////////////////////////
    // RescueSystem
    
    RescueSystem::~RescueSystem()
    {
    }
    
    void RescueSystem::loop()
    {
        m_objectPool.clearFiredObjects();
        System::loop();
    }
    void RescueSystem::outputObjectForAgentsAtStart(const Object* o, Output& buffer)
    {
        if(o->type() == TYPE_BUILDING) {
            buffer.put(o->type());
            buffer.put(o->id());
            o->output(buffer, PROPERTY_X);
            o->output(buffer, PROPERTY_Y);
            o->output(buffer, PROPERTY_FLOORS);
            o->output(buffer, PROPERTY_BUILDING_ATTRIBUTES);
            //o->output(buffer, PROPERTY_IGNITION);
            o->output(buffer, PROPERTY_FIERYNESS);
            o->output(buffer, PROPERTY_BROKENNESS);
            o->output(buffer, PROPERTY_ENTRANCES);
            o->output(buffer, PROPERTY_BUILDING_CODE);
            o->output(buffer, PROPERTY_BUILDING_AREA_GROUND);
            o->output(buffer, PROPERTY_BUILDING_AREA_TOTAL);
            o->output(buffer, PROPERTY_BUILDING_APEXES);
            buffer.put(PROPERTY_NULL);
        } else if(dynamic_cast<const Humanoid*>(o) != 0) {
            buffer.put(o->type());
            buffer.put(o->id());
            
            if (config().notify_initial_position()) {
                o->output(buffer, PROPERTY_POSITION);
                o->output(buffer, PROPERTY_POSITION_EXTRA);
            }
            o->output(buffer, PROPERTY_DIRECTION);
            if (config().notify_position_history()) {
                o->output(buffer, PROPERTY_POSITION_HISTORY);
            }
            o->output(buffer, PROPERTY_STAMINA);
            o->output(buffer, PROPERTY_HP);
            o->output(buffer, PROPERTY_DAMAGE);
            o->output(buffer, PROPERTY_BURIEDNESS);
            
            if(dynamic_cast<const FireBrigade*>(o) != 0) {
                o->output(buffer, PROPERTY_WATER_QUANTITY);
                o->output(buffer, PROPERTY_STRETCHED_LENGTH);
            }
            
            buffer.put(PROPERTY_NULL);
        } else {
            o->outputObject(buffer);
        }
    }
    void RescueSystem::outputObjectForAgents(const Object* o, Output& buffer)
    {
        if (config().notify_unchangeable_informaion()) {
            if(o->type() == TYPE_BUILDING) {
                buffer.put(o->type());
                buffer.put(o->id());
                o->output(buffer, PROPERTY_X);
                o->output(buffer, PROPERTY_Y);
                o->output(buffer, PROPERTY_FLOORS);
                o->output(buffer, PROPERTY_BUILDING_ATTRIBUTES);
                //o->output(buffer, PROPERTY_IGNITION);
                o->output(buffer, PROPERTY_FIERYNESS);
                o->output(buffer, PROPERTY_BROKENNESS);
                o->output(buffer, PROPERTY_ENTRANCES);
                o->output(buffer, PROPERTY_BUILDING_CODE);
                o->output(buffer, PROPERTY_BUILDING_AREA_GROUND);
                o->output(buffer, PROPERTY_BUILDING_AREA_TOTAL);
                o->output(buffer, PROPERTY_BUILDING_APEXES);
                buffer.put(PROPERTY_NULL);
            } else if(dynamic_cast<const Humanoid*>(o) != 0) {
                buffer.put(o->type());
                buffer.put(o->id());
                
                o->output(buffer, PROPERTY_POSITION);
                o->output(buffer, PROPERTY_POSITION_EXTRA);
                o->output(buffer, PROPERTY_DIRECTION);
                if (config().notify_position_history()) {
                    o->output(buffer, PROPERTY_POSITION_HISTORY);
                }
                o->output(buffer, PROPERTY_STAMINA);
                o->output(buffer, PROPERTY_HP);
                o->output(buffer, PROPERTY_DAMAGE);
                o->output(buffer, PROPERTY_BURIEDNESS);
                
                if(dynamic_cast<const FireBrigade*>(o) != 0) {
                    o->output(buffer, PROPERTY_WATER_QUANTITY);
                    o->output(buffer, PROPERTY_STRETCHED_LENGTH);
                }
                
                buffer.put(PROPERTY_NULL);
            } else {
                o->outputObject(buffer);
            }
        } else {
            buffer.put(o->type());
            buffer.put(o->id());
            if(dynamic_cast<const PointObject*>(o) != 0) {
                // not changed, but notify for look-time notification.
                o->output(buffer, PROPERTY_X);
                o->output(buffer, PROPERTY_Y);
            }
            if(dynamic_cast<const MovingObject*>(o) != 0) {
                o->output(buffer, PROPERTY_POSITION);
                o->output(buffer, PROPERTY_POSITION_EXTRA);
            }
            
            if(dynamic_cast<const Road*>(o) != 0) {
                o->output(buffer, PROPERTY_BLOCK);
                o->output(buffer, PROPERTY_REPAIR_COST);
            } else if(dynamic_cast<const Building*>(o) != 0) {
                o->output(buffer, PROPERTY_FIERYNESS);
                o->output(buffer, PROPERTY_BROKENNESS);
            } else if(dynamic_cast<const Humanoid*>(o) != 0) {
                o->output(buffer, PROPERTY_STAMINA);
                o->output(buffer, PROPERTY_HP);
                o->output(buffer, PROPERTY_DAMAGE);
                o->output(buffer, PROPERTY_BURIEDNESS);
                
                o->output(buffer, PROPERTY_DIRECTION);
                o->output(buffer, PROPERTY_POSITION_HISTORY);
            }
            buffer.put(PROPERTY_NULL);
        }
    }
    void RescueSystem::outputFarBuildingForAgents(const Building* o, Output& buffer)
    {
        if (config().notify_only_fire_for_far_buildings()) {
            buffer.put(o->type());
            buffer.put(o->id());
            o->output(buffer, PROPERTY_FIERYNESS);
            buffer.put(PROPERTY_NULL);
        } else {
            outputObjectForAgents(o, buffer);
        }
    }
    void RescueSystem::sendToAgents()
    {
        //long theTime = ::timeGetTime();
        
        // TODO: ®
        // TODO: ֤ϥब˾ޤ
        const double squaredVision = (double)config().vision() * config().vision();
        for(unsigned int i=0; i < m_agentTypes.size(); i++) {
#if 0   // 
            receiveMessages(agentSocket());
#endif
            Agents::iterator it = m_pConnectedAgents[i].begin();
            for(; it != m_pConnectedAgents[i].end(); it++) {
                Agent* agent = *it;
                if(agent->needsSensoryInformation()) {
                    m_outputBuffer.clear();
                    m_outputBuffer.put(KA_SENSE);
                    Output::Cursor base = m_outputBuffer.put(~(S32)0);
                    ASSERT(dynamic_cast<RealObject*>(agent->asObject()) != 0);
                    RealObject* self = (RealObject*)agent->asObject();
                    m_outputBuffer.put(self->id());
                    m_outputBuffer.put(m_time);
                    // self
                    outputObjectForAgents(self, m_outputBuffer);
                    // map (near)
                    RescueObjectPool::Enumerator e = m_objectPool.objectsInRound(self->x(), self->y(), config().vision());
                    Object* o;
                    while(o = e.next(), o != 0)
                        outputObjectForAgents(o, m_outputBuffer);
                    // mapfar fire
                    if (m_time <= config().steps_far_fire_invisible()) {
                        // invisible.
                    } else if (m_time == config().steps_far_fire_invisible() + 1) {
                        Objects::const_iterator it2 = m_objectPool.objects().begin();
                        for(; it2 != m_objectPool.objects().end(); it2++) {
                            Building* b = dynamic_cast<Building*>(*it2);
                            if(b != 0) {
                                double x = (double)b->x() - self->x();
                                double y = (double)b->y() - self->y();
                                if(x * x + y * y > squaredVision)
                                    outputFarBuildingForAgents(b, m_outputBuffer);
                            }
                        }
                    } else {
                        Objects::const_iterator it2 = m_objectPool.firedObjects().begin();
                        for(; it2 != m_objectPool.firedObjects().end(); it2++) {
                            Building* b = dynamic_cast<Building*>(*it2);
                            if(b != 0) {
                                double x = (double)b->x() - self->x();
                                double y = (double)b->y() - self->y();
                                if(x * x + y * y > squaredVision)
                                    outputFarBuildingForAgents(b, m_outputBuffer);
                            }
                        }
                    }
                    m_outputBuffer.put(TYPE_NULL);
                    m_outputBuffer.setSize(base);
                    m_outputBuffer.put(HEADER_NULL);
                    agentSocket().send(m_outputBuffer, agent->clientAddress(), m_time);
                }
            }
        }
        
        //printf("sendToAgent: %ld\n", ::timeGetTime() - theTime);
        
        m_objectPool.clearFiredObjects();
    }
    
    void RescueSystem::processCommand(Agent* agent, Header header, S32 size, Input& input, const LongUDPSocket& from)
    {
        Humanoid* h = dynamic_cast<Humanoid*>(agent);
        if(h == 0 || h->hp() > 0) { // ʤɿͤǤʤͤǻǤʤ
            if(h != 0) {
                int denominator = config().block_stamina_denominator();
                if(denominator) {
                    if((double)rand() / ((double)RAND_MAX + 1) >= (double)h->stamina() / denominator) {
                        reportRejection(agent, "the action was blocked because of the low stamina.");
                        return; // blocking by low stamina.
                    }
                }
            }
            if(header != AK_SAY && header != AK_TELL) {
                bool block = false; // block the command ?
                if(header != AK_REST && h->buriedness() > 0) {
                    reportRejection(agent, "can only say/tell/rest because he/she is buried.");
                    block = true;
                }
                switch(header) {
                case AK_MOVE:
                    if(h == 0) {
                        reportRejection(agent, "can not move because he/she is not a movable agent.");
                        block = true;
                    }
                    break;
                case AK_EXTINGUISH:
                    {
                        // TODO: 礬Ƥʤ
                        FireBrigade* f = dynamic_cast<FireBrigade*>(agent);
                        if(f == 0) {
                            reportRejection(agent, "can not extinguish it because he/she is not a fire brigade.");
                            block = true;
                        } else {
#if 0 // ignore waterQuantity
                            S32 value = f->waterQuantity();
#endif
                            Input::Cursor cursor = m_inputBuffer.cursor();
                            S32 target;
                            while(target = m_inputBuffer.get(), target != 0) {
                                S32 direction = m_inputBuffer.get();
                                S32 positionX = m_inputBuffer.get();
                                S32 positionY = m_inputBuffer.get();
                                S32 quantity = m_inputBuffer.get();
                                
                                if (!config().ignore_nozzle_position()) {
                                    double xx = f->x() - positionX;
                                    double yy = f->y() - positionY;
                                    double zz = xx * xx + yy * yy;
                                    if(zz > (double)config().max_fire_hose_length() * config().max_fire_hose_length()) {
                                        reportRejection(agent, "can not stretch the fire hose. (2001 competion rule)");
                                        block = true;
                                    }
                                }
                                
                                Building* building = dynamic_cast<Building*>(m_objectPool.get(target));
                                if(building == 0) {
                                    reportRejection(agent, "can extinguish only a building.");
                                    block = true;
                                } else {
                                    double xx = positionX - building->x();
                                    double yy = positionY - building->y();
                                    double zz = xx * xx + yy * yy;
                                    if(zz > (double)config().max_extinguish_length() * config().max_extinguish_length()) {
                                        reportRejection(agent, "can not extinguish such a far building.");
                                        block = true;
                                    }

                                    if(building->ignition() > 0 && building->fieryness() <= 0) {
                                        reportRejection(agent, "can not extinguish an unburned building.");
                                        block = true;
                                    }
                                }
                                
                                if(quantity < 0) {
                                    reportRejection(agent, "can not extinguish it with <0 power.");
                                    block = true;
                                } else if(quantity > config().max_extinguish_power()) {
                                    char buffer[1024];
                                    sprintf(buffer, "can not extinguish it with >%ld power.", (long)config().max_extinguish_power());
                                    reportRejection(agent, buffer);
                                    block = true;
                                }
#if 0 // ignore waterQuantity
                                value -= quantity;
#endif
                            }
#if 0 // ignore waterQuantity
                            if(value < 0) { //TODO: ʬ夷
                                reportRejection(agent, "can not extinguish because the tank is empty.");
                                block = true;
                            } else {
                                f->setWaterQuantity(m_time, value);
                            }
#endif
                            m_inputBuffer.setCursor(cursor);
                        }
                    }
                    break;
                case AK_LOAD:
                    {
                        AmbulanceTeam* f = dynamic_cast<AmbulanceTeam*>(agent);
                        if(f == 0) {
                            reportRejection(agent, "can not load anything because he/she is not a ambulance team.");
                            block = true;
                        } else {
                            Input::Cursor cursor = m_inputBuffer.cursor();
                            S32 target = m_inputBuffer.get();
                            Humanoid* humanoid = dynamic_cast<Humanoid*>(m_objectPool.get(target));
                            if(humanoid == 0) {
                                reportRejection(agent, "can load only a humanoid.");
                                block = true;
                            } else {
                                if(humanoid->buriedness() > 0) {
                                    reportRejection(agent, "can not load the buried agent.");
                                    block = true;
                                }
                            }
                            m_inputBuffer.setCursor(cursor);
                        }
                    }
                    break;
                }
                if(!block) {
                    System::processCommand(agent, header, size, input, from);
                }
            } else {
                S32 size = m_inputBuffer.get();
                if(size > config().say_max_bytes()) {
                    reportRejection(agent, "can not say/tell such a long message.");
                } else {
                    S32 sizeBound = (size + 3) & ~3;
                    Byte* buffer = new Byte[sizeBound+1];
                    m_inputBuffer.get(sizeBound, buffer);
                    buffer[size] = '\0';
                    Id sender = agent->asObject()->id();
                    if(header == AK_SAY) {
                        //S32 target = m_inputBuffer.get();
                        say(sender, (char*)buffer); // TODO: ̵ꥭ㥹ȤƤ뤬Ȥ롣
                    } else {
                        tell(sender, (char*)buffer);    // TODO: ̵ꥭ㥹ȤƤ뤬Ȥ롣
                    }
                    delete[] buffer;
                }
            }
        }
    }
    
    void RescueSystem::resetAgents()
    {
        for(int i=0; i<AGENTTYPE_MAZ; i++) {
            Agents::const_iterator it = m_pConnectedAgents[i].begin();
            for(; it != m_pConnectedAgents[i].end(); it++) {
                Agent* a = *it;
                RealObject* ro = dynamic_cast<RealObject*>(a);
                if(ro != 0) {
                    a->resetCacheForVoices();
                }
            }
        }
    }
    void RescueSystem::buildCacheForVoices(Agent* a, int radius)
    {
        a->m_radiusOfCacheForVoices = radius;
        const double squaredRadius = (double)radius * radius;
        
        RealObject* center = dynamic_cast<RealObject*>(a);
        if(center) {
            for(int i=0; i<AGENTTYPE_MAZ; i++) {
                Agents::const_iterator it = m_pConnectedAgents[i].begin();
                for(; it != m_pConnectedAgents[i].end(); it++) {
                    RealObject* candidate = dynamic_cast<RealObject*>(*it);
                    if(candidate != 0) {
                        double dx = candidate->x() - center->x();
                        double dy = candidate->y() - center->y();
                        if(dx*dx + dy*dy <= squaredRadius)
                            a->m_cacheForVoices.push_back(candidate);
                    }
                }
            }
        }
    }
    void RescueSystem::say(Id from, char* message)
    {
        int radius = config().voice();
        Agent* a = dynamic_cast<Agent*>(m_objectPool.get(from));
        RealObject* agent = dynamic_cast<RealObject*>(a);
        if(!agent) {
            reportRejection(from, "can not say because he/she is virtual (not real) agent.");
        } else {
            if(a->m_radiusOfCacheForVoices != radius)
                buildCacheForVoices(a, radius);
            
            const Objects& targets = a->m_cacheForVoices;
            Objects::const_iterator it = targets.begin();
            for(; it != targets.end(); it++) {
                Object* target = *it;
                Agent* targetAsAgent = dynamic_cast<Agent*>(target);
                if(targetAsAgent) {
                    m_outputBuffer.clear();
                    m_outputBuffer.put(KA_HEAR);
                    Output::Cursor base = m_outputBuffer.put(~(S32)0);
                    m_outputBuffer.put(target->id());
                    m_outputBuffer.put(from);
                    m_outputBuffer.putString(message);
                    m_outputBuffer.setSize(base);
                    m_outputBuffer.put(HEADER_NULL);
                    //printf("say %s\n", message);
                    agentSocket().send(m_outputBuffer, targetAsAgent->clientAddress(), m_time);
                }
            }
        }
    }
    
    void RescueSystem::tell(Id from, char* message)
    {
        RealObject* agent = dynamic_cast<RealObject*>(dynamic_cast<Agent*>(m_objectPool.get(from)));
        if(agent == 0) {
            reportRejection(from, "can not tell because he/she is virtual (not real) agent.");
            return;
        }
        
        TypeId agentType = agent->type();
        int recepients[4] = {AGENTTYPE_MAZ, AGENTTYPE_MAZ, AGENTTYPE_MAZ, AGENTTYPE_MAZ};
        switch (agentType){
        case TYPE_CIVILIAN:
            return;
        case TYPE_FIRE_BRIGADE:
            recepients[0]=AGENTTYPE_FIRE_COMPANY;
            recepients[1]=AGENTTYPE_FIRE_STATION;
            break;
        case TYPE_FIRE_STATION:
            recepients[0]=AGENTTYPE_FIRE_COMPANY;
            recepients[1]=AGENTTYPE_FIRE_STATION;
            recepients[2]=AGENTTYPE_AMBULANCE_CENTER;
            recepients[3]=AGENTTYPE_POLICE_OFFICE;
            break;
        case TYPE_AMBULANCE_TEAM:
            recepients[0]=AGENTTYPE_AMBULANCE_TEAM;
            recepients[1]=AGENTTYPE_AMBULANCE_CENTER;
            break;
        case TYPE_AMBULANCE_CENTER:
            recepients[0]=AGENTTYPE_AMBULANCE_TEAM;
            recepients[1]=AGENTTYPE_FIRE_STATION;
            recepients[2]=AGENTTYPE_AMBULANCE_CENTER;
            recepients[3]=AGENTTYPE_POLICE_OFFICE;
            break;
        case TYPE_POLICE_FORCE:
            recepients[0]=AGENTTYPE_POLICE_FORCE;
            recepients[1]=AGENTTYPE_POLICE_OFFICE;
            break;
        case TYPE_POLICE_OFFICE:
            recepients[0]=AGENTTYPE_POLICE_FORCE;
            recepients[1]=AGENTTYPE_FIRE_STATION;
            recepients[2]=AGENTTYPE_AMBULANCE_CENTER;
            recepients[3]=AGENTTYPE_POLICE_OFFICE;
            break;
        default:
            return;
        }
        
        for (int i=0; i<COUNT_OF(recepients); i++){
            if (recepients[i] != AGENTTYPE_MAZ) {
                Agents::const_iterator it = m_pConnectedAgents[recepients[i]].begin();
                for(; it != m_pConnectedAgents[recepients[i]].end(); it++) {
                    Agent* a = *it;
                    m_outputBuffer.clear();
                    m_outputBuffer.put(KA_HEAR);
                    Output::Cursor base = m_outputBuffer.put(~(S32)0);
                    m_outputBuffer.put(a->asObject()->id());
                    m_outputBuffer.put(from);
                    m_outputBuffer.putString(message);
                    m_outputBuffer.setSize(base);
                    m_outputBuffer.put(HEADER_NULL);
                    //printf("say %s\n", message);
                    agentSocket().send(m_outputBuffer, a->clientAddress(), m_time);
                }
            }
        }
    }
    
    void RescueSystem::reportRejection(Agent* agent, const char* why) {
        reportRejection(agent->asObject()->id(), why);
    }
    void RescueSystem::reportRejection(Id agentId, const char* why) {
        printf("reject: step=%ld agentID=%ld  %s\n", (long)m_time, (long)agentId, why);
    }
    
    
    /////////////////////////////////////////////////////////////////////////
} // namespace Rescue
