// -*- Mode:C++ -*-
//Header:
//File: echosample.cc
//Author: NODA, Itsuki
//Date: 2001/09/09
//

//ModifyHistory:
// 2001/09/09: Start to create this file
//EndModifyHistory:

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
/** @name echosample
   {\tt echosample} echo a contents of a frame message.
 */
//@{
/** @name Usage
    #% echosample [options]#
    \\
    {\bf [Options:]}
       \begin{description}
         \item[-port Port]
	 	specifies proxy's port number.
	 	Default is `16600'.
	 \item[-verbose]
	 	makes to show states of processing.
	 \item[-help]
	 	shows the help messages.
       \end{description}
 */
//@}
//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

#include "itk/itk.h"
#include "FrameSexp.h"

using namespace Itk ;

//======================================================================
// Global Values

UInt Port = 6600 ;

Socket::Protocol Proto = Socket::TCP ;

UInt BufSize = 10000 ;

UInt TimeoutMSec = 0 ;

VbsLevel VbsLocal = 2 ;

//======================================================================
// EchoAgent

class EchoAgent {

      //------------------------------------------------------------
      // EchoAgent:: member
   public:
      FrameSexp::Heap inheap ;
      FrameSexp::Heap outheap ;

      FrameSexp::Scanner * scanner ;

      ServerSocket socket ;
      ServerSocket tmpsocket ;

      Buffer inbuf ;
      Buffer outbuf ;

      //------------------------------------------------------------
      // EchoAgent:: constructor
   public:
      EchoAgent() { init() ; } ;

      //------------------------------------------------------------
      // EchoAgent:: init
   public:
      void init() {
	 if(!socket.init(Proto,Port)) {
	    ITK_ERR("Can not open socket port:"
		    << Socket::protocolName(Proto) << "/" << Port) ;
	    throw 0 ;
	 }
	 ITK_VBS(VbsLocal,"Open server socket:" << socket) ;

	 scanner = new FrameSexp::Scanner(&inheap) ;

	 inbuf.alloc(BufSize) ;
	 outbuf.alloc(BufSize) ;
      } ;
	 
      //------------------------------------------------------------
      // EchoAgent:: mainloop
   public:
      void mainloop() ;
} ;

void EchoAgent::mainloop() {

   while(True) {

      //------------------------------
      // receive message

      /* wait request */
      if(!socket.accept(tmpsocket)) {
	 ITK_ERR("Error: can not accept :" << socket) ;
	 return ;
      }

      /* read a socket */
      inbuf.clear() ;
      tmpsocket.receive(inbuf) ;
      ITK_VBS(VbsLocal,"Recv " << tmpsocket << ":" << inbuf.body()) ;

      //--------------------------------------------------
      // processing

      /* scan a frame */
#ifdef USE_SCANNER_DIRECTLY
      				/* use agent local scanner 
				 * it needs to manage scanner status
				 * manually.
				 */
      scanner->reset(inbuf) ;
      FrameSexp * inframe = scanner->scan(True) ;
#else
      				/* use heap's scanner
				 * it need not manage scanner status
				 * (management is done inside of heap.)
				 * it is easier to use
				 */
      FrameSexp * inframe = inheap.scan(inbuf,True) ;
#endif

      /* setup output */
      outheap.clear() ;

      outbuf.clear() ;
      ostrstream * ostr = outbuf.newOStrStream();

      /* syntax check */
      if(!inframe->chkSyntax()) {
	 ITK_WRN("Illegal Frame:" << inbuf.body()) ;
	 FrameSexp * outframe = outheap.newFrame("error") ;
	 outframe->setSlot(outheap.newSymbol(":content"),
			   outheap.newSymbol(inbuf.body()),
			   &outheap);
	 *ostr << outframe ;
      } 
      /* make a echo frame */
      else {
	 FrameSexp * outframe = outheap.newFrame("echo") ;
	 outframe->setSlot(outheap.newSymbol(":sender"),
			   outheap.newSymbol("echosample"),
			   &outheap) ;
	 outframe->setSlot(outheap.newSymbol(":content"),
			   outheap.dup(inframe),
			   &outheap) ;
	 *ostr << outframe ;
      }

      /* finalize output */
      outbuf.deleteOStrStream(ostr) ;

      tmpsocket.send(outbuf) ;
      ITK_VBS(VbsLocal,"Send " << tmpsocket << ":" << outbuf.body()) ;

      tmpsocket.kill() ;
   }
}

//======================================================================
// get Args

void getArgs(int argc, char** argv) {
   for(UInt i = 1 ; i < (UInt)argc ; i++) {
      if       (!strcmp(argv[i],"-port") && i+1 < (UInt)argc) {
	 i++ ; Port = atol(argv[i]) ;
      } else if(!strcmp(argv[i],"-verbose")) {
	 ITK_SET_VBS(VbsLocal) ;
      } else if(!strcmp(argv[i],"-timeout") && i+1 < (UInt)argc) {
	 i++ ; TimeoutMSec = atol(argv[i]) ;
      } else {
	 cout << "Usage:" << endl 
	      << "\t" << "% " << argv[0] << " [options]" << endl
	      << "\t    options:" << endl 
	      << "\t\t" << "-port PortNumber [" << Port << "]" << endl 
	      << "\t\t" << "-timeout TimeOutMSec [" 
	      << TimeoutMSec << "]" << endl 
	      << "\t\t" << "-verbose" << endl 
	      << "\t\t" << "-help" << endl 
	    ;
      }
   }
}

//======================================================================
// main

int main(int argc, char** argv) {
   getArgs(argc,argv) ;

   EchoAgent agent ;
   agent.mainloop() ;
} 









