// -*- Mode:C++ -*-
//Header:
//File: FrameSexp.cc
//Author: NODA, Itsuki
//Date: 2001/06/29
//

//ModifyHistory:
// 2001/06/29: Start to create this file
//EndModifyHistory:

#include "FrameSexp.h"

namespace Itk {

   //======================================================================
   // class FrameSexp's special symbols (class-variables)

   ITK_FRAME_DEF_KEYSYM_BODY(Set) ;
   ITK_FRAME_DEF_KEYSYM_BODY(List) ;
   ITK_FRAME_DEF_KEYSYM_BODY(Not) ;
   ITK_FRAME_DEF_KEYSYM_BODY(Wh) ;

   //======================================================================
   // class FrameSexp's method

   //------------------------------------------------------------
   // syntax check
   //      if depth > 0, then check upper than the depth.
   //      if depth < 0, it checks whole frames.
   //      but if depth < 0 and too deep fram (more than 2^32), 
   //      then it stops.

   Bool FrameSexp::chkSyntax(Int depth) const {
      if     (isAtom()) return True ;	// atom is frame
      else if(isNil()) return True ;	// nil is frame
      else if(isCons()) {
	 if(depth == 0) return True ;	// not check cons at bottom

	 SimpleSexp2 * head = car() ;
	 SimpleSexp2 * args = cdr() ;
	 if      (isNotFrame() || 
		  isWhFrame()) { 	// for one arg keyword
	    if(args->length() != 1) {
	       ITK_WRN(ITK_HERE <<
		       "keyword " << head << " should have just one arg." << 
		       " [" << this << "]") ;
	       return False ;
	    }
	    return static_cast<FrameSexp*>(args->car())->chkSyntax(depth - 1) ;
	 } 
	 else if (isSetFrame() ||
		  isListFrame()) {	// for multi arg keyword
	    for(args = args ;
		!args->isNil() ;
		args = args->cdr()) {
	       if(!args->isCons()) {
		  ITK_WRN(ITK_HERE <<
			  "A frame must not be terminated by non-nil." <<
			  " [" << this << "]") ;
		  return False ;
	       }
	       if(!static_cast<FrameSexp*>(args->car())->chkSyntax(depth - 1))
		  return False ;		// inner error
	    } 
	 }
	 else { 				// general case
	    for(args = args ;
		!args->isNil() ;
		args = args->cdr()) {

	       if(!args->isCons()) {
		  ITK_WRN(ITK_HERE <<
			  "A frame must not be terminated by non-nil." <<
			  " [" << this << "]") ;
		  return False ;
	       }

	       SimpleSexp2 * slotname = args->car() ;
	       if(!slotname->isAtom()) {
		  ITK_WRN(ITK_HERE <<
			  "A slotname should be an atom. :" << slotname <<
			  " [" << this << "]") ;
		  return False ;
	       }
	       if(!slotname->scanAsConstSymbol().hasPrefix(slotnamePrefix)) {
		  ITK_WRN(ITK_HERE <<
			  "A slotname should start by '" << slotnamePrefix <<
			  "'. : " << slotname <<
			  " [" << this << "]") ;
		  return False ;
	       }
	    
	       args = args->cdr() ;

	       if(!args->isCons()) {
		  ITK_WRN(ITK_HERE <<
			  "A frame must have slot-value pairs." <<
			  " [" << this << "]") ;
		  return False ;
	       }

	       FrameSexp * slotvalue = static_cast<FrameSexp*>(args->car()) ;
	       if(!slotvalue->chkSyntax(depth - 1)) {
		  return False ;
	       }
	    }
	 }
	 return True ;  // when all args is fine, this is a frame.
      }
      else {
	 ITK_WRN(ITK_HERE << "unknown syntax error." << 
		 " [" << this << "]") ;
	 return False ;
      }
   }

   //----------------------------------------------------------------------
   // FrameSexp:: formated output

   void FrameSexp:: outputWithMark(ostream & ostr, 
				   Int indent,
				   char * beginframe,
				   char * endframe,
				   char * headprefix,
				   char * headsuffix,
				   char * slotprefix,
				   char * slotvaluesep,
				   char * valuesuffix) {
      // if this is an atom, then output and return.

      Int indent1 = indent + 2 ;
      Int indent2 = indent1 + 2 ;

      if(!isFrame()) {
	 ostr << this ;
      } 
      else if(isSetFrame() || isListFrame()) {
	 ostr << endl ;
	 ostr << setw(indent) << setfill(' ') << "" ;
	 ostr << beginframe << endl ;

	 ostr << setw(indent1) << setfill(' ') << "" ;
	 ostr << headprefix << head() << headsuffix << endl ;

	 for(SimpleSexp2 * s = rest() ; !s->isNil() ; s = s->nthcdr(1)) {
	    ostr << setw(indent1) << setfill(' ') << "" ;
	    ostr << slotprefix << "" << slotvaluesep ;
	    FrameSexp * value = static_cast<FrameSexp *>(s->first()) ;
	    value->outputWithMark(ostr,indent2,
				  beginframe,endframe,
				  headprefix,headsuffix,
				  slotprefix,slotvaluesep,valuesuffix) ;
	    if(value->isFrame()) {
	       ostr << setw(indent1) << setfill(' ') << "" ;
	    }
	    ostr << valuesuffix << endl ;
	 }
      
	 ostr << setw(indent) << setfill(' ') << "" ;
	 ostr << endframe << endl ;
      } 
      else if(isNotFrame() || isWhFrame()) {
	 ostr << endl ;
	 ostr << setw(indent) << setfill(' ') << "" ;
	 ostr << beginframe << endl ;

	 ostr << setw(indent1) << setfill(' ') << "" ;
	 ostr << headprefix << head() << headsuffix << endl ;

	 SimpleSexp2 * s = second() ;
	 ostr << setw(indent1) << setfill(' ') << "" ;
	 ostr << slotprefix << "" << slotvaluesep ;
	 FrameSexp * value = static_cast<FrameSexp *>(s->first()) ;
	 value->outputWithMark(ostr,indent2,
			       beginframe,endframe,
			       headprefix,headsuffix,
			       slotprefix,slotvaluesep,valuesuffix) ;
	 if(value->isFrame()) {
	    ostr << setw(indent1) << setfill(' ') << "" ;
	 }
	 ostr << valuesuffix << endl ;
      
	 ostr << setw(indent) << setfill(' ') << "" ;
	 ostr << endframe << endl ;
      } 
      else {
	 ostr << endl ;

	 ostr << setw(indent) << setfill(' ') << "" ;
	 ostr << beginframe << endl ;

	 ostr << setw(indent1) << setfill(' ') << "" ;
	 ostr << headprefix << head() << headsuffix << endl ;

	 for(SimpleSexp2 * s = rest() ; !s->isNil() ; s = s->nthcdr(2)) {
	    ostr << setw(indent1) << setfill(' ') << "" ;
	    ostr << slotprefix << s->first() << slotvaluesep ;
	    FrameSexp * value = static_cast<FrameSexp *>(s->second()) ;
	    value->outputWithMark(ostr,indent2,
				  beginframe,endframe,
				  headprefix,headsuffix,
				  slotprefix,slotvaluesep,valuesuffix) ;
	    if(value->isFrame()) {
	       ostr << setw(indent1) << setfill(' ') << "" ;
	    }
	    ostr << valuesuffix << endl ;
	 }
      
	 ostr << setw(indent) << setfill(' ') << "" ;
	 ostr << endframe << endl ;
      }
   } ;
   
   void FrameSexp::outputLaTeX(ostream & ostr, Int indent) {
      return outputWithMark(ostr,indent,
			    "\\begin{tabular}[c]{|l|l|} \\hline",
			    "\\end{tabular}",
			    "\\multicolumn{2}{|c|}{",  "} \\\\ \\hline",
			    "", " & ", " \\\\ \\hline") ;
   } ;
   
   void FrameSexp::outputHTML(ostream & ostr, Int indent) {
      return outputWithMark(ostr,indent,
			    "<table border=\"1\">",
			    "</table>",
			    "<tr> <td colspan=\"2\"> ",  " </td> </tr>",
			    "<tr> <td> ", " </td><td> ", " </td> </tr>") ;
   } ;
}

//======================================================================
// for test
#if 0

using namespace Itk ;

int main() {
   FrameSexp * fs =
      FrameSexp::scan("(foo :bar (foo1 :bar () :baz 7) :boo 3
		        :a (:list a b c d e))") ;
   ITK_DBG(fs) ;

   ITK_DBG(fs->chkSyntax()) ;
   
   ITK_DBG(fs->slotvalue(":bar")) ;
   ITK_DBG(fs->slotvalue(SString(":bar"))->slotvalue(":baz")) ;

   FrameSexp::Heap heap ;

   FrameSexp * fs2 = heap.newFrame("foo2") ;
   ITK_DBG(fs2) ;
   
   fs2->setSlotValue(heap.newSymbol("bar2"),
		     heap.newInt(3),
		     &heap) ;
   ITK_DBG(fs2) ;

   fs2->setSlotValue(heap.newSymbol("baz2"),
		     heap.newFlt(2.718),
		     &heap) ;
   ITK_DBG(fs2) ;

   nop() ;
   fs2->setSlotValue(heap.newSymbol("bar2"),
		     heap.newFlt(3.14),
		     &heap) ;
   ITK_DBG(fs2) ;
   
   ITK_DBG(fs2->slotvalue(heap.newSymbol("bar2"))) ;


} ;

#endif
