// -*- Mode: c++ -*-
//Header:
//File: Thread.h
//Author: NODA, Itsuki
//Date: 1999/05/18
//

//ModifyHistory:
// 1999/05/18: Start to create this file
// 1999/11/07: introduce Itk namespace
// 1999/11/07: Divide each class to separate file
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#ifndef _itk_Thread_h_
#define _itk_Thread_h_
//////////////////////////////////////////////////////////////////////

//======================================================================
// includes

extern "C" {
#ifdef THREADED
#include <pthread.h>
#endif
}


#include "itk/btype.h"
#include "itk/utility.h"
#include "itk/WithDescriber.h"
#include "itk/MutexCondVar.h"
#include "itk/SString.h"
#include "itk/Buffer.h"

namespace Itk {
   
//======================================================================
// Thread

   //extern void* defaultStartFunc(void* p) ; // move into class Thread

   /*--------------------*/
   /**
    *
    */

   class Thread {

	 //------------------------------------------------------------
	 // Thread:: class static definitions
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static void* defaultStartFunc(void* p) ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static void* run(void* p) ;

	 //----------------------------------------
	 // Thread:: main thread
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static Thread * mainThread ;

	 //----------------------------------------
	 // Thread:: facility to get thread itself
      public:
	 static Bool keyToThreadAssignedP ;
	 static Mutex keyToThreadMutex ;
#ifdef THREADED
	 static pthread_key_t keyToThread ; /* key to get Thread 
					     * by thread specific key */
#endif

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static Bool keyToThreadAssign() {

	    Bool r = False ;
#ifdef THREADED
	    keyToThreadMutex.lock() ;
	    {
	       if(!keyToThreadAssignedP) {
		  pthread_key_create(&keyToThread,ITK_NULLPTR) ;
		  keyToThreadAssignedP = True;
		  r = True ;
	       }
	    }
	    keyToThreadMutex.unlock() ;
#endif
	    return r ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void registerThisThread() {
#ifdef THREADED
	    keyToThreadAssign() ;
	    pthread_setspecific(keyToThread,static_cast<void*>(this)) ;
#endif	    
	 } ;
	    
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static Thread * thisThread() {
#ifdef THREADED
	    if(keyToThreadAssignedP) {
	       return static_cast<Thread*>(pthread_getspecific(keyToThread)) ;
	    } else {
	       return static_cast<Thread*>(ITK_NULLPTR) ;
	    }
#else
	    return ITK_NULLPTR ;
#endif
	 } ;

	 //------------------------------------------------------------
	 // Thread:: members
      public:
#ifdef THREADED
	 pthread_t pid ;
#endif	 
	 SString name_ ;
	 ITK_DEF_ACCESS(SString & name(), { return name_ ; }) ;

	 MutexCondVar starter ;
	 void* (*func)(void*) ;
	 void* arg ;
	 
	 //------------------------------------------------------------
	 // Thread:: constructor

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Thread() { init(True) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Thread(Bool assignp) {  init(assignp) ; } ; // for main thread

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Thread(void*(*func)(void*),void* arg) { init(func,arg) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Thread(const char* namestr, Bool assignp = True) { 
	    init(SubString(const_cast<char*>(namestr)),assignp) ; 
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Thread(const SubString & namestr, Bool assignp = True) { 
	    init(namestr,assignp) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Thread(const char* namestr,
		void*(*func)(void*),void* arg) { 
	    init(SubString(const_cast<char*>(namestr)),func,arg) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Thread(const SubString & namestr,
		void*(*func)(void*),void* arg) { 
	    init(namestr,func,arg) ; } ;

	 //------------------------------------------------------------
	 // Thread:: destructor
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 ~Thread() { destroy() ; } ;

	 //------------------------------------------------------------
	 // Thread:: init
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void init(Bool assignp = True) {
	    assignTempNameIfNeeded() ;

	    if(assignp) {
	       startByWaitUntilStarter() ;
	    } else { // for initialize main thread
#ifdef THREADED
	       pid = pthread_self() ;
#endif
	       registerThisThread() ;
	    }

	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void init(void*(*f)(void*), void* a) {
	    assignTempNameIfNeeded() ;

	    func = f ;
	    arg = a ;
#ifdef THREADED
//	    pthread_create(&pid,ITK_NULLPTR,func,arg) ;
	    pthread_create(&pid,ITK_NULLPTR,run,this) ;  // using run
#endif
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void init(const SubString & namestr, Bool assignp = True) {
	    name().copy(namestr,True) ;
	    init(assignp) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void init(const SubString & namestr,
		   void*(*f)(void*), void* a) {
	    name().copy(namestr,True) ;
	    init(f,a) ;
	 } ;

	 //------------------------------------------------------------
	 // Thread:: destroy
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void destroy() {
	    delete name().head ;
	 } ;

	 //------------------------------------------------------------
	 // Thread:: assign temporal name if need
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 const static Int tmpNameStrMax = 32 ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool assignTempNameIfNeeded() {
	    if(name().isNullPtr()) {
	       name().alloc(tmpNameStrMax) ;
	       Buffer buf(name()) ;
	       buf.clear() ;
#ifdef THREADED	       
	       buf.printf("pid%lX",(Int)pid) ;
#else
	       buf.printf("pid0");
#endif
	       buf.terminate() ;

	       return True ;
	    } else {
	       return False ;
	    } 
	 } ;

	 //------------------------------------------------------------
	 // Thread:: start
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void start(void*(*f)(void*),void* a) {
	    starter.lock();
	    func = f ;
	    arg = a ;
	    starter.signal() ;
	    starter.unlock() ;
	 } ;
	 
	 //------------------------------------------------------------
	 // Thread:: starter facilities
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void startByWaitUntilStarter() {
	    starter.lock() ;
	    init(defaultStartFunc,this) ;
	    starter.wait() ;	// wait until the new Thread is ready
	    starter.unlock() ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void* waitUntilStarter() {
	    starter.lock() ;
	    starter.signal() ;	// inform the Thread is ready
	    starter.wait() ;	// wait starter 
	    starter.unlock() ;

	    return (*func)(arg) ;
	 } ;
   } ;
};

//////////////////////////////////////////////////////////////////////
#endif




