// -*- Mode:C++ -*-
//Header:
//File: fsringT.h
//Author: NODA, Itsuki
//Date: 1999/05/18
//

//ModifyHistory:
// 1999/05/18: Start to create this file
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#ifndef _itk_RingT_h_
#define _itk_RingT_h_
////////////////////////////////////////////////////////////////////////

#include "itk/btype.h"
#include "itk/RcylHeapT.h"

namespace Itk {

//======================================================================
// RingT<Content> : cyclic double linked list
//
   /*--------------------*/
   /**
    *  RingT is a class for elements of cyclic double linked list.
    *
    *  Each RingT has the links to previous and next elements.
    *  The link should be cyclic, so we call this "Ring".
    *  RingT provides operations to merge and divide rings at
    *  specified positions.
    * 
    *  A single RingT ring is double self-linked elemnets.
    */

   template <class Content>
      class RingT : public WithDescriber {
	 public:
	    //--------------------------------------------------
	    // local typedef

	    typedef RingT<Content> Ring ;
 
	    //--------------------------------------------------
	    // member

	    Content entity ;
	    Ring *prev ;
	    Ring *next ;

	    //--------------------------------------------------
	    // constructor

	    /*--------------------*/
	    /**
	     *  Construct an empty ring that links null pointer.
	     */
	 public:
	    RingT<Content>() {
	       prev = ITK_NULLPTR ;
	       next = ITK_NULLPTR ;
	    } ;

	    /*--------------------*/
	    /**
	     *  Construct a cyclic ring that has a content.
	     */
	 public:
	    RingT<Content>(const Content& val, Bool circlep = True) {
	       entity = val ;
	       if(circlep) {
		  linkSelf() ;
	       } else {
		  prev = ITK_NULLPTR ;
		  next = ITK_NULLPTR ;
	       }
	    } ;

	    /*--------------------*/
	    /**
	     *  Copy a ring.
	     */
	 public:
	    RingT<Content>(const RingT<Content> & org) {
	       copy(org) ;
	    } ;

	    //--------------------------------------------------
	    // copy
	    /*--------------------*/
	    /**
	     *  Sharrow Copy.
	     */
	 public:
	    Ring & copy(const Ring & org) {
	       entity = org.entity ;
	       prev = org.prev ;
	       next = org.next ;
	       return *this ;
	    } ;

	    //--------------------------------------------------
	    // set value
	    /*--------------------*/
	    /**
	     *  set the content.
	     */
	 public:
	    Ring * set(const Content & val) {
	       entity = val ;
	       return this ;
	    } ;

	    //--------------------------------------------------
	    // link next
	    /*--------------------*/
	    /**
	     *  change the next-link to newnext.
	     */
	 public:
	    Ring * linkNext(Ring* newnext) {
	       next = newnext ;
	       newnext->prev = this ;
	       return this ;
	    } ;

	    //--------------------------------------------------
	    // link prev
	    /*--------------------*/
	    /**
	     *  change the previous-link to newprev.
	     */
	 public:
	    Ring * linkPrev(Ring* newprev) {
	       return newprev->linkNext(this) ;
	    } ;

	    //--------------------------------------------------
	    // link self (single ring)
	    /*--------------------*/
	    /**
	     *  make itself a self-cycle ring.
	     */
	 public:
	    Ring * linkSelf() { return linkNext(this) ; } ;

	    //--------------------------------------------------
	    // make single ring
	    /*--------------------*/
	    /**
	     *  set a content and make it a single ring.
	     */
	 public:
	    Ring * setSingle(const Content & val) {
	       set(val) ;
	       return linkSelf() ;
	    } ;

	    //--------------------------------------------------
	    // single ring check
	    /*--------------------*/
	    /** 
	     *  check this is single ring.
	     */
	 public:
	    Bool isSingle() { return next == this && prev == this ; } ;

	    //--------------------------------------------------
	    // insert next
	    /*--------------------*/
	    /**
	     *  insert a ring into next of the ring.
	     */
	 public:
	    Ring* insertNext(Ring* newnext) {
	       Ring * oldnext = next ;
	       Ring * oldnewnextprev = newnext->prev ;

	       linkNext(newnext) ;
	       oldnext->linkPrev(oldnewnextprev) ;
	       return this ;
	    } ;

	    /*--------------------*/
	    /**
	     *  insert a ring to the first position (same as insertNext.)
	     */
	 public:
	    Ring* insertFirst(Ring* ring) { return insertNext(ring) ; } ;

	    //--------------------------------------------------
	    // insert prev
	    /*--------------------*/
	    /**
	     *  insert a ring into the previous of the ring.
	     */
	 public:
	    Ring* insertPrev(Ring* newprev) {
	       Ring * oldprev = prev ;
	       Ring * oldnewprevnext = newprev->next ;
	
	       linkPrev(newprev) ;
	       oldprev->linkNext(oldnewprevnext) ;

	       return this ;
	    } ;

	    /*--------------------*/
	    /**
	     *  insert a ring to the last position (same as insertPrev.)
	     */
	 public:
	    Ring* insertLast(Ring* ring) { return insertPrev(ring) ; } ;

	    //--------------------------------------------------
	    // add insert new data (allocate new memory)
	    /*--------------------*/
	    /**
	     * make a new ring for the content and insert it to 
	     * the first position
	     */
	 public:
	    Ring* putFirst(const Content& dt) {
	       Ring * newring = new Ring(dt,True) ;
	       insertNext(newring) ;
	       return newring ;
	    } ;

	    /*--------------------*/
	    /**
	     * make a new ring for the content and insedrt it to 
	     * the last positoin.
	     */
	 public:
	    Ring* putLast(const Content& dt) {
	       Ring* newring = new Ring(dt,True) ;
	       insertPrev(newring) ;
	       return newring ;
	    } ;

	    //--------------------------------------------------
	    // remove self 
	    /*--------------------*/
	    /**
	     *  remove itself from the rest of ring.
	     */
	 public:
	    void removeSelf() {
	       Ring *pre = prev ;
	       Ring *nxt = next ;
	       pre->next = nxt ;
	       nxt->prev = pre ;
	    } ;

	    //--------------------------------------------------
	    // remove one
	    /*--------------------*/
	    /**
	     *  remove the specified ring from the ring.
	     */
	 public:
     
	    void remove(Ring* del) { del->removeSelf() ; } ;

	    //--------------------------------------------------
	    // split (split operation is same as insert operation)
	    /*--------------------*/
	    /**
	     *  split the ring at the next link.
	     */
	 public:
	    Ring * splitNext(Ring * newnext) {
	       return insertNext(newnext) ;
	    } ;
	 
	    /*--------------------*/
	    /**
	     *  split the ring at the previous link.
	     */
	 public:
	    Ring * splitPrev(Ring * newprev) {
	       return insertPrev(newprev) ;
	    } ;

	    //--------------------------------------------------
	    // end check
	    /*--------------------*/
	    /**
	     *  check the tail is the end.
	     */
	 public:
	    Bool isEnd(const Ring* tail) const { return tail == this ; } ;

	    //--------------------------------------------------
	    // reverse
	    /*--------------------*/
	    /**
	     * make a reversed ring.
	     */
	 public:
	    void reverse() ;

	    //--------------------------------------------------
	    // length
	    /*--------------------*/
	    /**
	     * calculate the length
	     */
	 public:
	    UInt length() const ;

	    //--------------------------------------------------
	    // describe
	    /*--------------------*/
	    /**
	     *  show the description.
	     */
	 public:
	    virtual void describe(ostream& ostr, 
				  const Bool detailp = True) const {
	       ostr << "#Ring[<" << this << ">:<" << prev << ">=*=<" 
		    << next << ">]" ;
	       if(detailp) {
		  const Ring *r = this ;
		  ostr << endl << "  [Content]" ;
		  do {
		     ostr << endl << "\t" << *r << ": " << r->entity ;
		     r = r->next ;
		  } while (!isEnd(r)) ;
		  ostr << endl << "\t--- end ---" << endl ;
	       }
	    } ;
      } ;      
   
//--------------------------------------------------
// reverse

   template <class Content>
      void RingT<Content>::reverse() {
      Ring r = this ;
      Ring nr ;
      do {
	 nr = r->next ;
	 r->next= r->prev ;
	 r->prev = nr ;
	 r = nr ;
      } while (!isEnd(r)) ;
      return this ;
   }

//--------------------------------------------------
// length

   template <class Content>
      UInt RingT<Content>::length() const {
      const Ring* r = this ;
      UInt l = 0 ;
      do {
	 r = r->next ;
	 l++ ;
      } while (!isEnd(r)) ;
      return l ;
   }

//======================================================================
// RingRcylHeap<Content> : ring heap
//

#ifndef ITK_DFLT_RINGRCYLHEAP_NAME
#define ITK_DFLT_RINGRCYLHEAP_NAME		"RingRcylHeap"
#endif

   /*--------------------*/
   /**
    *
    */

   template <class Content>
      class RingRcylHeapT : public RcylHeapT< RingT<Content> > {
	 public:
	    //--------------------------------------------------
	    // typedef 

	    typedef RingT<Content> Ring ;
	    typedef RingRcylHeapT<Content> RingRcylHeap ;
      
	    //--------------------------------------------------
	    // constructor
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    RingRcylHeapT<Content>(char *nm = ITK_DFLT_RINGRCYLHEAP_NAME,
				   UInt sz = ITK_DFLT_MEMARRAY_SIZE,
				   UInt ef = ITK_DFLT_EXPAND_FACTOR) {
	       init(nm,sz,ef) ;
	    } ;

	    //--------------------------------------------------
	    // get
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Ring * getPtr() {
	       Ring * r = RcylHeap::getPtr() ;
	       r->linkSelf() ;
	       return r ;
	    } ;
      
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Ring & get() { return *getPtr() ; } ;

	    //--------------------------------------------------
	    // get & set
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Ring * getPtr(const Content& val) {
	       Ring * r = getPtr() ;
	       return r->set(val) ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Ring & get(const Content& val) { return *getPtr(val) ; } ;

	    //--------------------------------------------------
	    // add ring at first position
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Ring * addRingFirst(Ring* root) {
	       Ring * r = getPtr() ;
	       root->insertNext(r) ;
	       return r ;
	    } ;
      
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Ring * addRingFirst(Ring* root, const Content& val) {
	       Ring * r = getPtr(val) ;
	       root->insertNext(r) ;
	       return r ;
	    } ;

	    //--------------------------------------------------
	    // add ring at last position
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Ring * addRingLast(Ring* root) {
	       Ring * r = getPtr() ;
	       root->insertPrev(r) ;
	       return r ;
	    } ;
      
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Ring * addRingLast(Ring* root, const Content& val) {
	       Ring * r = getPtr(val) ;
	       root->insertPrev(r) ;
	       return r ;
	    } ;
      
      } ;

} ;

////////////////////////////////////////////////////////////////////////
#endif


	 

      
      
      

