// -*- Mode:C++ -*-
//Header:
//File: Socket.h
//Author: NODA, Itsuki
//Date: 1999/05/19
//

//ModifyHistory:
// 1999/05/19: Start to create this file
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#ifndef _itk_Socket_h_
#define _itk_Socket_h_
////////////////////////////////////////////////////////////////////////

//======================================================================
// Includes
//

#ifdef FreeBSD
#include <sys/types.h>
#endif

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/ioctl.h>

#include "itk/btype.h"
#include "itk/utility.h"
#include "itk/SString.h"
#include "itk/Buffer.h"
#include "itk/syserrortab.h"

//======================================================================
// Socket Protocols
//

namespace Itk {

   //======================================================================
   // inet host address printer
   //

   /*--------------------*/
   /**
    *
    */

   inline ostream& operator<<(ostream& ostr, const struct in_addr& addr) {
      const unsigned char* a = (const unsigned char*)&addr ;
      ostr << (UInt)a[0] << "." << (UInt)a[1] << "."
	   << (UInt)a[2] << "." << (UInt)a[3] ;
      return ostr ;
   }

   /*--------------------*/
   /**
    *
    */

   inline ostream& operator<<(ostream& ostr, const struct sockaddr_in& addr) {
      ostr << addr.sin_addr << ":" << ntohs(addr.sin_port) ;
      return ostr ;
   }

   //======================================================================
   // Socket
   //

   /*--------------------*/
   /**
    *
    */

   class Socket : public WithDescriber {
	 
	 //--------------------------------------------------
	 // Socket Protocol Type
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 enum Protocol {
	    NoProto,
	    UDP,
	    TCP,
	    UnknownProto
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static const char * protocolName(Protocol type) {
	    switch(type) {
	       case NoProto : return "none" ;
	       case UDP : return "udp" ;
	       case TCP : return "tcp" ;
	       default: return "unknown" ;
	    } 
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static Protocol protocolIDByName(const SString & name) {
	    if(name == "none") return NoProto ;
	    if(name == "udp") return UDP ;
	    if(name == "tcp") return TCP ;
	    return UnknownProto ;
	 } ;

	 //--------------------------------------------------
	 // member
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Protocol protocol ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 int socketfd ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 struct sockaddr_in remoteAddr ; 

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 struct sockaddr_in localAddr ; 

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal birthTime ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal lastSendTime ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal lastRecvTime ;

	 //--------------------------------------------------
	 // constructor

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Socket() { 
	    protocol = NoProto ;
	    socketfd = 0 ;
	    memset(&remoteAddr,0,sizeof(remoteAddr)) ;
	    memset(&localAddr,0,sizeof(localAddr)) ;
	    birthTime.update() ;
	    lastSendTime.update() ;
	    lastRecvTime.update() ;
	 } ;

	 //--------------------------------------------------
	 // init & setup

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool setupRemote(char* host, const UInt port) ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool setupLocal(const Protocol proto,
			 const UInt port) ;

	 //--------------------------------------------------
	 // info
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 UInt localPort() const { 
	    return (UInt)ntohs(localAddr.sin_port) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 UInt remotePort() const { 
	    return (UInt)ntohs(remoteAddr.sin_port) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 struct in_addr localHostInAddress() const {
	    return localAddr.sin_addr ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 struct in_addr remoteHostInAddress() const {
	    return remoteAddr.sin_addr ; } ;

	 //--------------------------------------------------
	 // protocol 

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isUDP() const { return protocol == UDP ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isTCP() const { return protocol == TCP ; } ;

	 //--------------------------------------------------
	 // blocking mode

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool setBlocking(const Bool blockingp = True) ;

	 //--------------------------------------------------
	 // send / receive

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool send(const Buffer& buf) ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool receive(Buffer& buf) {
	    return receive(buf, buf.restLength()) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool receive(Buffer& buf, UInt const size) ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool send(void * data, const UInt size) ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool receive(void * data, const UInt size) ;

	 //--------------------------------------------------
	 // read / write

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool read(Buffer& buf) {
	    return read(buf, buf.restLength()) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool read(Buffer& buf, UInt const size) ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool write(const Buffer& buf) ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool read(void * data, const UInt size) ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool write(void * data, const UInt size) ;

	 //--------------------------------------------------
	 // discard input

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void discard(const UInt size) ;

	 //--------------------------------------------------
	 // copy
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void copy(const Socket& src) {
	    socketfd = src.socketfd ;
	    protocol = src.protocol ;
	    remoteAddr = src.remoteAddr ;
	    localAddr = src.localAddr ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool assignNewFD() {
	    return setupLocal(protocol,0) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool assignNewFD(Protocol proto) {
	    protocol = proto ;
	    return assignNewFD() ;
	 } ;

	 //--------------------------------------------------
	 // alive / kill
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isAlive() const { return protocol != NoProto ; } ;
      
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void kill() {
	    close(socketfd) ;
	    protocol = NoProto ; 
	 } ;

	 //--------------------------------------------------
	 // descripe
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 virtual void describe(ostream& ostr, Bool detailp = True) const {
	    ostr << "#Socket[" << socketfd 
		 << ":" << protocolName(protocol) 
		 << "|" << localPort() 
		 << " <-> " << remoteHostInAddress() << ":" << remotePort() 
		 << "]" ;
	    if(detailp) {
	       ostr << endl ;
	    }
	 } ;

   } ;

   //======================================================================
   // Server Socket
   //

#define ITK_TCP_SOCKET_LISTEN_QUEUE_MAX	8

   /*--------------------*/
   /**
    *
    */

   class ServerSocket : public Socket {
      public:
      
	 //--------------------------------------------------
	 // constructor

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 ServerSocket(const Protocol proto = NoProto,
		      const UInt port = 0) {
	    init(proto, port) ;
	 } ;

	 //--------------------------------------------------
	 // init & setup

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool init(const Protocol proto, UInt port = 0) {
	    if(proto != NoProto) {
	       Bool r = setupLocal(proto, port) ;
	       if(proto == TCP) {
		  listen(socketfd,ITK_TCP_SOCKET_LISTEN_QUEUE_MAX) ;
	       } ;
	       return r ;
	    } else {
	       protocol = proto ;
	       return False ;
	    }
	 } ;

	 //--------------------------------------------------
	 // accept
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool accept(ServerSocket& child) ;

	 //--------------------------------------------------
	 // copy
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void copy(const ServerSocket& ref) {
	    Socket::copy(ref) ;
	 } ;

	 //--------------------------------------------------
	 // clone (copy original but assign new port and socketfd (for UDP)
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void clone(const ServerSocket& org) {
	    init(org.protocol,0) ;
	    remoteAddr = org.remoteAddr ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 ServerSocket* clone() const {
	    ServerSocket *r = new ServerSocket() ;
	    r->clone(*this) ;
	    return r ;
	 } ;

	 //--------------------------------------------------
	 // descripe
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 virtual void describe(ostream& ostr, 
			       const Bool detailp = True) const {
	    ostr << "#ServerSocket[" << socketfd 
		 << ":" << protocolName(protocol) 
		 << "/" << localPort() 
		 << " <- " << remoteHostInAddress() << ":" << remotePort() 
		 << "]" ;
	    if(detailp) {
	       ostr << endl ;
	    }
	 } ;

   } ;

   //======================================================================
   // Client Socket
   //
   /*--------------------*/
   /**
    *
    */

   class ClientSocket : public Socket {
      public:

	 char* _remoteHost ;
      
	 //--------------------------------------------------
	 // constructor

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 ClientSocket() { _remoteHost = ITK_NULLPTR ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 ClientSocket(char* host, 
		      const Protocol proto = NoProto,
		      const UInt port = 0) {
	    init(host, proto, port) ;
	 } ;

	 //--------------------------------------------------
	 // init & setup
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool init(const char* host,
		   const Protocol proto, 
		   UInt port = 0) {
	    _remoteHost = strdup(host) ;
	    if(!setupLocal(proto, 0)) return False ;
	    if(!setupRemote(_remoteHost, port)) return False ;
	    if(isTCP()) {
	       if(connect(socketfd,(struct sockaddr*)&(remoteAddr),
			  sizeof(remoteAddr))) {
		  cerr << "Error: can not connect" << *this ;
		  return False ;
	       }
	    }
	    return True ;
	 } ;

	 //--------------------------------------------------
	 // descripe
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 virtual void describe(ostream& ostr, 
			       const Bool detailp = True) const {
	    ostr << "#ClientSocket[" << socketfd 
		 << ":" << protocolName(protocol) 
		 << "/" << localPort() 
		 << " -> " << _remoteHost << ":" << remotePort() 
		 << "]" ;
	    if(detailp) {
	       ostr << endl ;
	    }
	 } ;
   } ;

   //======================================================================
   // Socket Selector 
   //
   /*--------------------*/
   /**
    *
    */

   class SocketReadSelector : public WithDescriber {
      public:
	 fd_set readfds ;
	 int fdmax ;

	 //--------------------------------------------------
	 // constructor

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SocketReadSelector() { clear() ; } ;

	 //--------------------------------------------------
	 // clear
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void clear() { 
	    FD_ZERO(&readfds) ; 
	    fdmax = 0 ;
	 } ;

	 //--------------------------------------------------
	 // clear one
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void clear(const int fd) {
	    FD_CLR(fd,&readfds) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void clear(const Socket& sock) {
	    if(sock.isAlive()) set(sock.socketfd) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void clear(const Socket* sock) { clear(*sock) ; } ;

	 //--------------------------------------------------
	 // set
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void set(const int fd) {
	    FD_SET(fd,&readfds) ;
	    fdmax = Max(fdmax,fd) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void set(const Socket& sock) { 
	    if(sock.isAlive()) set(sock.socketfd) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void set(const Socket* sock) { set(*sock) ; } ;

	 //--------------------------------------------------
	 // isSet
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isSet(const int fd) const { return FD_ISSET(fd,&readfds) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isSet(const Socket& sock) const { 
	    return sock.isAlive() && isSet(sock.socketfd) ; 
	 } ;

	 //--------------------------------------------------
	 // select
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int select() {
	    return ::select(fdmax+1,&readfds,
			    ITK_NULLPTR,ITK_NULLPTR,ITK_NULLPTR) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int select(const TimeVal& tv) {
	    TimeVal tv_work = tv ;
	    return ::select(fdmax+1,&readfds,
			    ITK_NULLPTR,ITK_NULLPTR,&tv_work.tval) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int select(UInt sec, UInt msec = 0, UInt usec = 0) {
	    TimeVal tval(sec,msec,usec) ;
	    return select(tval) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int selectUntil(const TimeVal& deadline) {
	    TimeVal now ; now.update() ;
	    if(deadline < now) return select(0) ;
	    TimeVal diff = deadline - now ;
	    return select(diff) ;
	 } ;

	 //--------------------------------------------------
	 // select
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 virtual void describe(ostream& ostr, Bool detailp = True) const {
	    ostr << "#Selector[<" << this << ">" << ":fdmax=" << fdmax 
		 << "]" ;
	    if(detailp) {
	       ostr << endl << "   [mask]= " ;
	       for(Int fd = 0 ; fd < fdmax ; fd++) {
		  if(isSet(fd)) ostr << "+" ;
		  else          ostr << "." ;
	       }
	       ostr << endl ;
	    }
	 } ;
   } ;

} ;

////////////////////////////////////////////////////////////////////////
#endif




