// -*- Mode:C++ -*-
//Header:
//File: SimpleSexp.cc
//Author: NODA, Itsuki
//Date: 2000/08/08
//

//ModifyHistory:
// 2000/08/08: Start to create this file
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#include "itk/SimpleSexp.h"

namespace Itk {

   SimpleSexp::Heap SimpleSexp::heap ;

   SimpleSexp SimpleSexp::_Nil(T_Nil,SubString("()")) ;
   SimpleSexp * SimpleSexp::Nil = &_Nil ;

   //======================================================================
   // SimpleSexp:: scan
   
   //--------------------------------------------------
   // SimpleSexp:: scan toplevel

   SimpleSexp * SimpleSexp::scan(GenericInput & gi, Heap * hp) {
      if(isNull(hp)) hp = &(SimpleSexp::heap) ;
      hp->reset() ;
      return scanSexp(gi,hp) ;
   } ;

   //--------------------------------------------------
   // SimpleSexp:: scan sexp

   SimpleSexp * SimpleSexp::scanSexp(GenericInput & gi, Heap * hp) {
      skipSpaceComment(gi) ;
      if(gi.isEnd()) return ITK_NULLPTR ;
      
      if(gi.peek() == ITK_SEXP_CONS_BEGIN_CHAR) {
	 return scanCons(gi, hp) ;
      } else {
	 return scanAtom(gi, hp) ;
      }
   } ;

   //--------------------------------------------------
   // SimpleSexp:: scan atom

   SimpleSexp * SimpleSexp::scanAtom(GenericInput & gi, Heap * hp) {
      SimpleSexp * sx = hp->getPtr() ;

      sx->tag = T_Atom ;
      sx->str.head = gi.cpos() ;

      if(gi.peek() == ITK_SEXP_STRING_BEGIN_CHAR) { 	/* string */
	 gi.get() ;			/* skip first '"' */
	 while(!gi.isEnd()) {
	    char c = gi.get() ;
	    if(c == ITK_SEXP_STRING_ESCAPE_CHAR) { gi.get() ; continue ; }
	    if(c == ITK_SEXP_STRING_END_CHAR) { break ; } 
	 }
      } else { 			/* normal symbol */
	 while(!gi.isEnd()) {
	    char c = gi.get() ;
	    if(isspace(c) || 
	       c == ITK_SEXP_CONS_BEGIN_CHAR || 
	       c == ITK_SEXP_CONS_END_CHAR) {
	       gi.unget() ;
	       break ; 
	    } 
	 }
      }
      
      sx->str.tail = gi.cpos() ;

      return sx ;
   } ;

   //--------------------------------------------------
   // SimpleSexp:: scan cons

   SimpleSexp * SimpleSexp::scanCons(GenericInput & gi, Heap * hp) {
      char * head = gi.cpos() ;

      gi.get() ;	/* skip "(" */
      
      skipSpaceComment(gi) ;

      return scanConsBody(head,gi,hp) ;
   } ;
   
   //--------------------------------------------------
   // SimpleSexp:: scan cons body

   SimpleSexp * SimpleSexp::scanConsBody(char * head, GenericInput & gi,
					 Heap * hp) {
      SimpleSexp * sx = hp->getPtr() ;

      if (gi.isEnd()) {			/* illegal EOF in cons */
	 sx->tag = T_Error ;
      } else if (gi.peek() == ITK_SEXP_CONS_END_CHAR) {	/* end Cons */
	 sx->tag = T_Nil ;
	 gi.get() ;
      } else {				/* read car */
	 sx->tag = T_Cons ;
	 sx->car = scanSexp(gi,hp) ;
	 skipSpaceComment(gi) ;
	 sx->cdr = scanConsBody(gi.cpos(),gi,hp) ;
      }

      sx->str.head = head ;
      sx->str.tail = gi.cpos() ;

      return sx ;
   } ;

   //--------------------------------------------------
   // SimpleSexp:: skip space & comment

   Bool SimpleSexp::skipSpaceComment(GenericInput & gi) {
      Bool r = skipSpace(gi) ;
      if(skipComment(gi)) {
	 skipSpaceComment(gi) ;
	 return True ;
      } else {
	 return r ;
      } 
   } ;

   //----------------------------------------
   // SimpleSexp:: skip space 

   Bool SimpleSexp::skipSpace(GenericInput & gi) {
      Bool r = False ;
      while( !gi.isEnd() && isspace(gi.peek())) {
	 gi.ignore() ;
	 r = True ;
      } ;
      return r ;
   } ;

   //----------------------------------------
   // SimpleSexp:: skip comment

   Bool SimpleSexp::skipComment(GenericInput & gi) {
      if( !gi.isEnd() && gi.peek() == ITK_SEXP_COMMENT_BEGIN_CHAR) {
	 while(!gi.isEnd()) {
	    char c = gi.peek() ;
	    gi.ignore();
	    if(c == ITK_SEXP_COMMENT_END_CHAR) break ;
	 }
	 return True ;
      } else {
	 return False ;
      }
   } ;

   //======================================================================
   // SimpleSexp:: scan As Number

   Int SimpleSexp::scanAsInt() const {
      Int v ; 
      scanAsIntTo(v) ;
      return v ;
   } ;

   Bool SimpleSexp::scanAsIntTo(Int & val) const {
      if(isAtom()) {
	 char buf[128] ;
	 str.copyTo(buf) ;
	 char *endptr ;
	 val = strtol(buf,&endptr,0) ;
	 if(isNull(*endptr)) return True ;
      }
      return False ;
   } ;

   Flt SimpleSexp::scanAsFlt() const {
      Flt v ; 
      scanAsFltTo(v) ;
      return v ;
   } ;

   Bool SimpleSexp::scanAsFltTo(Flt & val) const {
      if(isAtom()) {
	 char buf[128] ;
	 str.copyTo(buf) ;
	 char *endptr ;
	 val = strtod(buf,&endptr) ;
	 if(isNull(*endptr)) return True ;
      }
      return False ;
   } ;

   SubString SimpleSexp::scanAsSymbol() const { return str.dup() ; } ;

   SubString SimpleSexp::scanAsString(Bool naked, Bool copyp) const {
      SubString r ;
      if(copyp) r.copy(str,True) ;
      else      r = str ;
      if(isString()) {
	 r.ignoreChar() ; // chop the first char
	 r.chop() ;       // chop the last char
      }
      return r ;
   } ;

   //--------------------------------------------------
   // SimpleSexp:: copy to buffer

   void SimpleSexp::copyTo(Buffer & buffer) const {
      buffer.put(str) ;
   }

   //======================================================================
   // SimpleSexp:: list operations

   UInt SimpleSexp::length() const {
      UInt l = 0 ;
      for(const SimpleSexp * p = this ; p->isCons() ; p = p->cdr) {
	 l++ ;
      }
      return l ;
   } ;

   SimpleSexp * SimpleSexp::nthcdr(UInt n) {
      SimpleSexp * r = this ;
      for(UInt i = 0 ; i < n ; i++) { 
	 if(isNullPtr() || !isCons()) {
	    r = ITK_NULLPTR ;
	    break ;
	 }
	 r = r->cdr ; 
      } 
      return r ;
   } ;

   SimpleSexp * SimpleSexp::nth(UInt n) {
      SimpleSexp * r = nthcdr(n) ;
      if(r->isCons()) return r->car ;
      else return ITK_NULLPTR;
   } ;

   //======================================================================
   // SimpleSexp:: table operations

   //------------------------------------------------------------
   // SimpleSexp:: assoc

   SimpleSexp * SimpleSexp::assoc(const SimpleSexp * const key) const {
      SimpleSexp * l = const_cast<SimpleSexp*>(this) ;
      SimpleSexp * e = ITK_NULLPTR;
      for( ; l->isCons() ; l = l->rest()) {
	 e = l->first() ;
	 if(e->first()->equal(key)) break ;
      }
      if (l->isCons()) return e ;
      else return Nil ;
   } ;

   //------------------------------------------------------------
   // SimpleSexp:: assoc by string

   SimpleSexp * SimpleSexp::assoc(const SubString & key) const {
      SimpleSexp * l = const_cast<SimpleSexp*>(this) ;
      SimpleSexp * e = ITK_NULLPTR;
      for( ; l->isCons() ; l = l->rest()) {
	 e = l->first() ;
	 if(e->first()->equal(key)) break ;
      }
      if (l->isCons()) return e ;
      else return Nil ;
   } ;

   //------------------------------------------------------------
   // SimpleSexp:: assoc Value

   SimpleSexp * SimpleSexp::assocValue(const SimpleSexp * const key) const {
      SimpleSexp * r = assoc(key) ;
      if(r->isCons()) return r->second() ;
      else return Nil ;
   } ;

   //------------------------------------------------------------
   // SimpleSexp:: assoc Value by String

   SimpleSexp * SimpleSexp::assocValue(const SubString& key) const {
      SimpleSexp * r = assoc(key) ;
      if(r->isCons()) return r->second() ;
      else return Nil ;
   } ;

   //------------------------------------------------------------
   // SimpleSexp:: passoc (assoc property list)

   SimpleSexp * SimpleSexp::passoc(const SimpleSexp * const key) const {
      SimpleSexp * l = const_cast<SimpleSexp*>(this) ;
      for( ; l->isCons() ; l = l->rest()->rest()) {
	 if(l->first()->equal(key)) break ;
      }
      if (l->isCons()) return l->second() ;
      else return Nil ;
   } ;

   //------------------------------------------------------------
   // SimpleSexp:: passoc (assoc property list) by string

   SimpleSexp * SimpleSexp::passoc(const SubString & key) const {
      SimpleSexp * l = const_cast<SimpleSexp*>(this) ;
      for( ; l->isCons() ; l = l->rest()->rest()) {
	 if(l->first()->equal(key)) break ;
      }
      if (l->isCons()) return l->second() ;
      else return Nil ;
   } ;

   //======================================================================
   // SimpleSexp:: describe

   void SimpleSexp::describe(ostream& o, Bool detailp) const {
      if(detailp) {
	 o << "#SimpleSexp[tag=" ;
	 switch(tag) {
	    case T_Atom : {
	       o << "atom(" << ( atype==AT_None   ? "any" :
				 atype==AT_Symbol ? "sym" :
				 atype==AT_Int    ? "int" :
				 atype==AT_Flt    ? "flt" : "err" )
		 << ")" ;
	       break ;
	    }
	    case T_Nil  : o << "nil" ; break ;
	    case T_Cons : o << "cons" ; break ;
	    case T_Error : o << "error" ; break ;
	    default      : o << "other" ; break ;
	 }
	 o << ",str=\"" << str << "\"]" << endl ;
	 if(tag == T_Cons) {
	    o << "\tcar=" ;
	    car->describe(o,detailp) ;
	    o << "\tcdr=" ;
	    cdr->describe(o,detailp) ;
	 }
	 else if (tag == T_Atom) {
	    if(atype == AT_Int) {
	       o << "\tival=" << ival << endl ;
	    } else if(atype == AT_Flt) {
	       o << "\tfval=" << fval << endl ;
	    }
	 }
      } else {
	 switch(tag) {
	    case T_Atom : {
	       switch(atype) {
		  case AT_None :
		  case AT_Symbol :
		  case AT_Error : o << str ;  break ;
		  case AT_Int :   o << ival ; break ;
		  case AT_Flt :   o << fval ; break ;
		  default : o << str ; break ;
	       }
	       break ;
	    } ;
	    case T_Nil :
	    case T_Cons : {
	       o << ITK_SEXP_CONS_BEGIN_CHAR ;
	       for(const SimpleSexp * sx = this ; sx->tag == T_Cons ; 
		   sx = sx->cdr) {
		  if(sx != this) o << ITK_SEXP_CONS_SEP_CHAR ;
		  sx->car->describe(o,detailp) ;
	       }
	       o << ITK_SEXP_CONS_END_CHAR ;
	       break ;
	    } ;
	    case T_Error : 
	    default : {
	       o << "#SimpleSexpError[" << str << "]" ;
	    }
	 }
      }
   } ;

   //======================================================================
   // SimpleSexp::Heap

   //------------------------------------------------------------
   // SimpleSexp::Heap:: append

   SimpleSexp * SimpleSexp::Heap::append(SimpleSexp * list1, 
					 SimpleSexp * list2) {
      if(!list1->isCons()) 
	 return list2 ;
      else
	 return cons(list1->car,append(list1->cdr,list2)) ;
   } ;

   //------------------------------------------------------------
   // SimpleSexp::Heap:: append

   SimpleSexp * SimpleSexp::Heap::addLast(SimpleSexp * lst, 
					  SimpleSexp * lastelm) {
      SimpleSexp * l = lst ;
      for(; l->cdr->isCons() ; l = l->cdr) {} ;
      l->cdr = list(lastelm) ;
      return lst ;
   } ;
}
