// -*- Mode: c++ -*-
//Header:
//File: SString.h
//Author: NODA, Itsuki
//Date: 1999/05/18
//

//ModifyHistory:
// 1999/05/18: Start to create this file
// 1999/11/07: introduce Itk namespace
// 1999/11/07: Divide each class to separate file
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#ifndef _itk_SString_h_
#define _itk_SString_h_
//////////////////////////////////////////////////////////////////////
#include "itk/btype.h"
#include "itk/utility.h"
#include <stdio.h>
#include <stdarg.h>


namespace Itk {

   //======================================================================
   // SString Class
   /*--------------------*/
   /**
    * class for string and substring of char.
    * The class include head and tail pointer of the string.
    */

   class SubString {
      public:
	 char* head ;
	 char* tail ;

	 //--------------------------------------------------
	 // static member

	 //static const SubString null() { return SubString() ; } ;
	 //static const SubString zero() { return SubString("") } ;

	 /*--------------------*/
	 /**
	  * indicates null string.
	  */
      public:
	 static SubString null ;

	 //--------------------------------------------------
	 // constructor
	 /*--------------------*/
	 /**
	  * allocate new area of the size and construct a SubString.
	  */
      public:
	 SubString(const UInt size) { alloc(size) ; } ;

	 /*--------------------*/
	 /**
	  * construct a SubString that has specified head and tail.
	  */
      public:
	 SubString(char* h, char* t) { set(h,t) ; } ;

	 /*--------------------*/
	 /**
	  * construct a SubString that starts the specified head and
	  * has specified length.
	  */
      public:
	 SubString(char* h, const UInt l) { set(h,l) ; } ;

	 /*--------------------*/
	 /**
	  * construct a SubString of the specifiled c-string.
	  */
      public:
	 SubString(char* str = ITK_NULLPTR) { set(str) ; } ;

	 /*--------------------*/
	 /**
	  * construct a copy of the specifiled SubString.
	  */
      public:
	 SubString(const SubString& src) { set(src) ; } ;

	 /*--------------------*/
	 /**
	  * construct a SubString that is a sub-string of the original string,
	  * head and tail. 
	  */
      public:
	 SubString(const SubString& src, 
		   const UInt hidx, const UInt tidx) {
	    set(src,hidx,tidx) ;
	 } ;

	 //--------------------------------------------------
	 // operator=
	 /*--------------------*/
	 /**
	  * set the value of the original SubString s.
	  * see (set(const SubString &)).
	  */
      public:
	 SubString& operator= (const SubString& s) { 
	    set(s) ; return *this ;
	 } ;

	 /*--------------------*/
	 /**
	  * set the value of the original char *s.
	  * see (set(char* s)).
	  */
      public:
	 SubString& operator= (char* s) {
	    set(s) ; return *this ;
	 } ;

	 /*--------------------*/
	 /**
	  * set the value of the original char *s.
	  * see (set(const char* s)).
	  */
      public:
	 SubString& operator= (const char * s) {
	    set(s) ; return *this ;
	 } ;

	 //--------------------------------------------------
	 // operator==
	 /*--------------------*/
	 /**
	  * compare the head and tail pointer of the SubString.
	  */
      public:
	 Bool eq(const SubString& r) const ;

	 /*--------------------*/
	 /**
	  * compare the contents of the SubString.
	  */
      public:
	 Bool equal(const SubString& r) const ;

	 /*--------------------*/
	 /**
	  * compare the contents of the SubString. 
	  * (see equal(const SubString&r))
	  */
      public:
	 Bool operator== (const SubString& r) const ;

	 /*--------------------*/
	 /**
	  * compare the contents of the SubString with c-string ref.
	  */
      public:
	 Bool operator== (const char* ref) const ;

	 //--------------------------------------------------
	 // less  // used in map
	 /*--------------------*/
	 /**
	  * compare the contents of two SubString.
	  */
      public:
	 static Bool less(const SubString & str0, 
			  const SubString & str1) ;
	 /*--------------------*/
	 /**
	  * compare the contents of two SubString.
	  */
      public:
	 static Bool greater(const SubString & str0, 
			     const SubString & str1) ;

	 /*--------------------*/
	 /**
	  * compare the contents of two SubString.
	  */
      public:
	 static Bool lessequal(const SubString & str0, 
			       const SubString & str1) ;

	 /*--------------------*/
	 /**
	  * compare the contents of two SubString.
	  */
      public:
	 static Bool greaterequal(const SubString & str0, 
				  const SubString & str1) ;

	 /*--------------------*/
	 /**
	  * compare the contents of two SubString.
	  */
      public:
	 Bool operator<(const SubString & ref) const ;

	 /*--------------------*/
	 /**
	  * compare the contents of two SubString.
	  */
      public:
	 Bool operator<=(const SubString & ref) const ;

	 /*--------------------*/
	 /**
	  * compare the contents of two SubString.
	  */
      public:
	 Bool operator>(const SubString & ref) const ;

	 /*--------------------*/
	 /**
	  * compare the contents of two SubString.
	  */
      public:
	 Bool operator>=(const SubString & ref) const ;

	 //<<[add by kshinoda]

	 /*--------------------*/
	 /**
	  * compare the contents of two SubString.
	  */
      public:
	 Bool operator<(const char* ref) const;

	 /*--------------------*/
	 /**
	  * compare the contents of two SubString.
	  */
      public:
	 Bool operator<=(const char* ref) const;

	 /*--------------------*/
	 /**
	  * compare the contents of two SubString.
	  */
      public:
	 Bool operator>(const char* ref) const;

	 /*--------------------*/
	 /**
	  * compare the contents of two SubString.
	  */
      public:
	 Bool operator>=(const char* ref) const;

	 //>>[add by kshinoda]

	 //--------------------------------------------------
	 // prefix check
	 /*--------------------*/
	 /**
	  * check the SubString is a prefix of the specifiled SubString ref.
	  */
      public:
	 Bool isPrefixOf(const SubString& ref) const ;

	 /*--------------------*/
	 /**
	  * check the SubString is a prefix of the specifiled const char *ref.
	  */
      public:
	 Bool isPrefixOf(const char* const ref) const ;

	 /*--------------------*/
	 /**
	  * check the SubString has the specifiled prefix.
	  */
      public:
	 Bool hasPrefix(const SubString& prefix) const ;

	 /*--------------------*/
	 /**
	  * check the SubString has the specifiled prefix.
	  */
      public:
	 Bool hasPrefix(const char* const ref) const ;

	 /*--------------------*/
	 /**
	  * check the prefix and pop it.
	  * 
	  * If this SubString begins the specified prefix,
	  * then pop the prefix part and move the head pointer to the
	  * next of it.
	  */
      public:
	 Bool checkPopPrefix(const SubString & prefix) ;

	 /*--------------------*/
	 /**
	  * check the prefix and pop it.
	  */
      public:
	 Bool checkPopPrefix(const char * const prefix) ;

	 //--------------------------------------------------
	 // index (return -1 if not exist)
	 /*--------------------*/
	 /**
	  * check the existence of the char c in this SubString.
	  *
	  * It returns the position of the first c from the head pointer.
	  * If the char does not exist, it returns -1.
	  */
      public:

	 Int index(const char c) const ;

	 /**
	  * check the existence of the char c in this SubString.
	  *
	  * It returns Bool value instead of index value.
	  */
      public:
	 Bool has(const char c) const ;

	 //--------------------------------------------------
	 // set
	 /*--------------------*/
	 /**
	  * set head and tail pointer.
	  */
      public:
	 void set(char* h, char* t) { head = h ;  tail = t ; } ;

	 /*--------------------*/
	 /**
	  * set head and the length.
	  */
      public:
	 void set(char* h, const UInt l) { 
	    head = h ; setLength(l) ;
	 } ;

	 /*--------------------*/
	 /**
	  * set the head and tail by char* str.
	  */
      public:
	 void set(char* str = ITK_NULLPTR) { 
	    if(isNull(str)) set((char*)ITK_NULLPTR,(char*)ITK_NULLPTR) ;
	    else set(str,strlen(str)) ;
	 } ;

	 /*--------------------*/
	 /**
	  * set the head and tail by const char* str.
	  *
	  * It allocates the new area and copy contents from the str.
	  */
      public:
	 void set(const char * str) { set(::strdup(str)) ; } ;

	 /*--------------------*/
	 /**
	  * set the head and tail by SubString.
	  *
	  * After this operation, this SubString and the src SubString
	  * share the same head and tail pointer.
	  */
      public:
	 void set(const SubString& src) { set(src.head,src.tail) ; } ;

	 /*--------------------*/
	 /**
	  * set the head and tail by SubString offseted by hidx and tidx.
	  *
	  * After this operation, this SubString becomes a substring of
	  * the src SubString.
	  */
      public:
	 void set(const SubString& src, const UInt hidx, const UInt tidx) {
	    head = &(src.head[hidx]) ;
	    tail = &(src.head[tidx]) ;
	 } ;

	 /*--------------------*/
	 /**
	  * set the head and tail by SubString offseted by hidx.
	  *
	  * After this operation, this SubString becomes a substring of
	  * the src SubString. 
	  * In this case, tails of both SubStrings are the same.
	  */
      public:
	 void set(const SubString& src, const UInt hidx) {
	    head = &(src.head[hidx]) ;
	    tail = src.tail ;
	 } ;

	 //--------------------------------------------------
	 // setNull & check Null
	 /*--------------------*/
	 /**
	  * set this SubString to the null string. 
	  * 
	  * The null string is that both of head and tail are the null pointer.
	  */
      public:
	 void setNull() { set((char*)ITK_NULLPTR, (char*)ITK_NULLPTR) ; } ;

	 /*--------------------*/
	 /**
	  * check the head of this SubString is null pointer.
	  *
	  * (see isNullStr()).
	  */
      public:
	 Bool isNullPtr() const { return isNull(head) ; } ;

	 //--------------------------------------------------
	 // alloc
	 /*--------------------*/
	 /**
	  * allocate the area of the length l.
	  *
	  * If forcep==True, then it allocates new are even if
	  * this SubString already has the enough space.
	  */
      public:
	 Bool alloc(const UInt l, Bool forcep = True) ;

	 //--------------------------------------------------
	 // length
	 /*--------------------*/
	 /**
	  * get the length of this SubString.
	  *
	  * It is same as (tail-head).
	  */
      public:
	 UInt length() const { return (tail - head) ; } ;

	 /*--------------------*/
	 /**
	  * get the size of this SubString.
	  *
	  * It is the same as length()+1. 
	  * But, this is not safe that the SubString is not allocated by
	  * null-terminated c-string.
	  */
      public:
	 UInt size() const { return length() + 1 ; } ;

	 /*--------------------*/
	 /**
	  * set the length of this SubString.
	  *
	  * It changes tail pointer to be head+1.
	  */
      public:
	 void setLength(const UInt l) ;
      
	 /*--------------------*/
	 /**
	  * return strlen() value of this SubString.
	  */
      public:
	 UInt strlength() const ;

	 /*--------------------*/
	 /**
	  * adjust the length of this SubString to be strlen(head).
	  */
      public:
	 void adjustLength() ;

	 //--------------------------------------------------
	 // copy
	 /*--------------------*/
	 /**
	  * copy the content from src SubString.
	  *
	  * If allocp==True, then it allocates the new area.
	  */
      public:
	 Bool copy(const SubString& src, Bool allocp = False) ;

	 /*--------------------*/
	 /**
	  * copy the content from c-string between char *h  and *t.
	  *
	  * If allocp==True, then it allocates the new area.
	  */
      public:
	 Bool copy(char* h, char* t, Bool allocp = False) ;

	 /*--------------------*/
	 /**
	  * copy the content from c-string between char *h  and *t.
	  *
	  * If t==null, then it copy until the null character appear.
	  */
      public:
	 Bool copy(const char * h, const char * t = ITK_NULLPTR) ;

	 //--------------------------------------------------
	 // dup
	 /*--------------------*/
	 /**
	  * duplicate this SubString.
	  *
	  * It create a new SubString and copy the contents of this SubString.
	  */
      public:
	 SubString dup() const ;

	 /*--------------------*/
	 /**
	  * duplicate this SubString.
	  *
	  * It create a new c-string and copy the contents of this SubString.
	  */
      public:
	 char * strdup() const ;
      
	 //--------------------------------------------------
	 // copy to char*
	 /*--------------------*/
	 /**
	  * copy the content of this SubString to the dst.
	  *
	  * If maxlen is specified, then it copy maxlen characters in maximum.
	  */
      public:
	 Bool copyTo(char* dst, Int maxlen = -1) const ;

	 //--------------------------------------------------
	 // converter

	 // operator const char* () const { return head ; } ;

	 /*--------------------*/
	 /**
	  * convert this SubString to c-str.
	  */
      public:
	 operator char* () const { return head ; } ;
      
	 //--------------------------------------------------
	 // terminate
	 /*--------------------*/
	 /**
	  * terminate the tail.
	  *
	  * It put the null char to the tail.
	  */
      public:
	 void terminate() { *tail = ITK_NULLCHR ; } ;

	 /*--------------------*/
	 /**
	  * check this SubString is terminated.
	  * (see isTerminated()).
	  */
      public:
	 Bool terminatedp() const { return isTerminated() ; } ;

	 /*--------------------*/
	 /**
	  * check this SubString is terminated.
	  * 
	  * It checks the SubString contains the null character.
	  */
      public:
	 Bool isTerminated() const { return strlength() <= length() ; } ; 

	 /*--------------------*/
	 /**
	  * check this SubString is null string.
	  * 
	  * It returns True if this SubString's head and tail is the same. 
	  */
      public:
	 Bool isNullStr() const { return head == tail ; } ;
      
	 //--------------------------------------------------
	 // basic operation (get)

	 /*--------------------*/
	 /**
	  * return the first char.
	  */
      public:
	 char& first() const { return *head ; } ;

	 /*--------------------*/
	 /**
	  * return the last (tail-1) char.
	  */
      public:
	 char& last() const { return *(tail - 1) ; } ;

	 /*--------------------*/
	 /**
	  * check the first character of this SubString is c.
	  */
      public:
	 Bool isFirst(const char c) const { return first() == c ; } ;
      
	 /*--------------------*/
	 /**
	  * if the head is equal or after the tail.
	  *
	  * It is useful for checking overrun as a buffer.
	  */
      public:
	 Bool tailp() const { return head >= tail ; } ;

	 /*--------------------*/
	 /**
	  * if the head is equal or after the tail.
	  * (same as tailp()).
	  */
      public:
	 Bool nullp() const { return head >= tail ; } ;

	 /*--------------------*/
	 /**
	  * get the first char and shift the head.
	  */
      public:
	 char popChar() ;

	 /*--------------------*/
	 /**
	  * get the first char and shift the head.
	  * (same as popChar()).
	  */
      public:
	 char getChar() { return popChar() ; } ; 

	 /*--------------------*/
	 /**
	  * chop the tail.
	  */
      public:
	 void chop() { tail-- ; } ;

	 /*--------------------*/
	 /**
	  * unshift the head.
	  */
      public:
	 void ungetChar() { head-- ; } ;

	 /*--------------------*/
	 /**
	  * ignore the first char and shift the head.
	  */
      public:
	 void ignoreChar() { head++ ; } ;

	 /*--------------------*/
	 /**
	  * ignore N chars and shift the head.
	  */
      public:
	 Bool skipNChar(UInt n = 1) ;

	 /*--------------------*/
	 /**
	  * skip during the characters are in char list.
	  */
      public:
	 UInt skipWhile(const SubString & charlist) ;

	 /*--------------------*/
	 /**
	  * skip until the characters are in char list.
	  */
      public:
	 UInt skipUntil(const SubString & charlist) ;

	 /*--------------------*/
	 /**
	  * get a substring of the length.
	  *
	  * It returns a substring that starts the head and has the length.
	  */
      public:
	 SubString sub(UInt length) const { return sub(0, length) ; }

	 /*--------------------*/
	 /**
	  * get a substring of the specifiled offset and length.
	  */
      public:
	 SubString sub(UInt offset, UInt length) const ;

	 /*--------------------*/
	 /**
	  * pop a SubString of the length from the head of this SubString.
	  *
	  * It shift the head pointer to the tail of the popped SubString.
	  */
      public:
	 SubString popSubString(UInt length) { 
	    return popSubString(0,length) ; 
	 } ;

	 /*--------------------*/
	 /**
	  * pop a SubString of the length from the offset of this SubString.
	  *
	  * It shift the head pointer to the tail of the popped SubString.
	  */
      public:
	 SubString popSubString(UInt offset, UInt length) ;

	 /*--------------------*/
	 /**
	  * get a SubString of the length from the tail of this SubString.
	  */
      public:
	 SubString shiftedSubString(UInt offset) const {
	    return sub(offset,length()-offset) ;
	 } ;
      
	 //--------------------------------------------------
	 // basic operation (put)
	 /*--------------------*/
	 /**
	  * push a char c before the head, and shift.
	  */
      public:
	 void pushChar(const char c) ;

	 /*--------------------*/
	 /**
	  * put a char c after the tail.
	  */
      public:
	 void putChar(const char c) ;

	 /*--------------------*/
	 /**
	  * put str SubString after the tail.
	  */
      public:
	 void put(const SubString& str) ;

	 /*--------------------*/
	 /**
	  * put str char* after the tail.
	  */
      public:
	 void put(const char* str) { 
	    put(SubString(const_cast<char*>(str))) ; } ;

	 /*--------------------*/
	 /**
	  * put str char* (with length len) after the tail.
	  */
      public:
	 void put(const char* str, UInt len) { 
	    put(SubString(const_cast<char*>(str),len)) ; } ;

	 //--------------------------------------------------
	 // printf1
	 /*--------------------*/
	 /**
	  * printf values into this SubString using the format.
	  */
      public:
	 Int printf(const char * format, ...) ;

	 /*--------------------*/
	 /**
	  * printf the format (just a c-string) into this SubString.
	  */
      public:
	 Int printf1(const char* format) ;

	 /*--------------------*/
	 /**
	  * printf char* values into this SubString using the format.
	  */
      public:
	 Int printf1(const char* format, const char * x) ;

	 /*--------------------*/
	 /**
	  * printf Int values into this SubString using the format.
	  */
      public:
	 Int printf1(const char* format, Int x) ;

	 /*--------------------*/
	 /**
	  * printf UInt values into this SubString using the format.
	  */
      public:
	 Int printf1(const char* format, UInt x) ;

	 /*--------------------*/
	 /**
	  * printf Flt values into this SubString using the format.
	  */
      public:
	 Int printf1(const char* format, Flt x) ;
	 
	 //--------------------------------------------------
	 // concatination
	 /*--------------------*/
	 /**
	  * append a SubString rest into this SubString.
	  */
      public:
	 SubString append(const SubString & rest) const ;

	 /*--------------------*/
	 /**
	  * same as append.
	  */
      public:
	 SubString operator+ (const SubString & rest) const {
	    return append(rest) ;
	 } ;

	 /*--------------------*/
	 /**
	  * same as append.
	  */
      public:
	 SubString operator+ (const char* rest) const {
	    return append(SubString((char*)rest)) ;
	 } ;

	 //--------------------------------------------------
	 // scan by char set
	 /*--------------------*/
	 /**
	  * scan this SubString using charSet.
	  * 
	  * It return a SubString from the head that contain only the charSet.
	  */
      public:
	 SubString scanInCharSet(const SubString & charSet) ;

	 /*--------------------*/
	 /**
	  * scan this SubString using charSet.
	  */
      public:
	 SubString scanInCharSet(char * const charSet) {
	    return scanInCharSet(SubString((char*)charSet)) ;
	 } ;

	 /*--------------------*/
	 /**
	  * skip this SubString using charSet.
	  * 
	  * It returns True if it skips at least one character.
	  */
      public:
	 Bool skipInCharSet(const SubString & charSet) ;
	    
	 /*--------------------*/
	 /**
	  * skip this SubString using charSet.
	  */
      public:
	 Bool skipInCharSet(char * const charSet) {
	    return skipInCharSet(SubString((char*)charSet)) ;
	 } ;

	 //--------------------------------------------------
	 // scan by delimiter
	 /*--------------------*/
	 /**
	  * scan this SubString until one of delimiter appears.
	  *
	  * It returns a SubString until the delimiter.
	  */
      public:
	 SubString scanByDelimiter(const SubString & delimiter) ;

	 /*--------------------*/
	 /**
	  * scan this SubString until one of delimiter appears.
	  */
      public:
	 SubString scanByDelimiter(char * const delimiter) {
	    return scanByDelimiter(SubString((char*)delimiter)) ;
	 } ;

	 //--------------------------------------------------
	 // scan as number value
	 /*--------------------*/
	 /**
	  * scan this SubString as Flt value.
	  *
	  * It shift the head after the scanned string.
	  */
      public:
	 Flt scanAsFlt() ;

	 /*--------------------*/
	 /**
	  * scan this SubString as Int value.
	  *
	  * It shift the head after the scanned string.
	  */
      public:
	 Int scanAsInt() ;

	 /*--------------------*/
	 /**
	  * scan this SubString as UInt value.
	  *
	  * It shift the head after the scanned string.
	  */
      public:
	 UInt scanAsUInt() ;

	 //--------------------------------------------------
	 // conversion
	 /*--------------------*/
	 /**
	  * pop one Int value from  this SubString.
	  */
      public:
	 Bool popIntVal(Int& i) ;

	 /*--------------------*/
	 /**
	  * pop one UInt value from  this SubString.
	  */
      public:
	 Bool popUIntVal(UInt& i) ;

	 /*--------------------*/
	 /**
	  * pop one Flt value from  this SubString.
	  */
      public:
	 Bool popFloatVal(Flt& v) ;

	 //--------------------------------------------------
	 // describe
	 /*--------------------*/
	 /**
	  * describe this SubString.
	  */
      public:
	 void describe(ostream& o, Bool detailp = True) const {
	    if(detailp) {
	       o << "#SubString[<" << (void*)head << ">-<" << (void*)tail
		 << ">(len=" << length() << "):" ;
	    } 
	    if(isNullPtr()) {
	       o << "(null)" ;
	    } else {
	       for(char* s = head ; s < tail ; s++) {
		  o << *s ;
	       }
	    }
	    if(detailp) {
	       o << "]" << endl ;
	    }
	 } ;

	 friend ostream& operator<< (ostream& o, const SubString& s) {
	    s.describe(o,False) ;
	    return o ;
	 } ;
   } ;

   //======================================================================
   // SString Class
   /*--------------------*/
   /**
    * SString is the same as SubString.
    */

   typedef SubString SString ;

} ;


//////////////////////////////////////////////////////////////////////
#endif
