// -*- Mode:C++ -*-
//Header:
//File: RingBufferT.h
//Author: NODA, Itsuki
//Date: 1999/09/06
//

//ModifyHistory:
// 1999/09/06: Start to create this file
// 1999/09/06: Split cc file
// 1999/11/08: separate into files for each class
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#ifndef _itk_RingBufferT_h_
#define _itk_RingBufferT_h_
////////////////////////////////////////////////////////////////////////
#include "itk/btype.h"
#include "itk/Exception.h"
#include "itk/WithDescriber.h"

namespace Itk {
//======================================================================
// Ring Buffer
//

#ifndef ITK_DFLT_RINGBUFFER_NAME
#define ITK_DFLT_RINGBUFFER_NAME		"RingBuffer"
#endif
#ifndef ITK_DFLT_RINGBUFFER_SIZE
#define ITK_DFLT_RINGBUFFER_SIZE  ITK_DFLT_MEMARRAY_SIZE
#endif

//------------------------------------------------------------
// class template

   /*--------------------*/
   /**
    *
    */

   template <class Content>
      class RingBufferT : public WithDescriber {
	 public:
	    //--------------------------------------------------
	    // local typedef

	    typedef RingBufferT<Content> RingBuffer ;

	    //--------------------------------------------------
	    // member

	    char* name ;
	    Content* table ;
	    UInt size ;
	    UInt head ;
	    UInt tail ;
	    UInt expandFactor ;

	    //--------------------------------------------------
	    // exception
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    class XptIllegalExpand : public Exception {
	       public:
		  RingBuffer *ptr ;
		  // RingBuffer  copy ; 
	    
		  virtual const char * name() const { 
		     return "IllegalExpand" ; } ;
      
		  static Exception * root(char* const f, UInt const l, 
					  char* const d, 
					  const RingBuffer* p) {
		     static XptIllegalExpand _root ;
		     _root.set(f,l,d) ;
		     _root.ptr = (RingBuffer*)p ;
		     // _root.copy = *p ;
		     return &_root ;
		  } ;

		  virtual void describe_aux(ostream& ostr) const {
		     ostr << ":ring=<" << ptr << ">";
		     if(!isNull(ptr)) ostr << *ptr ;
		  } ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    class XptUnderflow : public Exception {
	       public:
		  RingBuffer *ptr ;
		  //RingBuffer  copy ;
	    
		  virtual const char * name() const { 
		     return "RingbufferUnderflow" ; 
		  } ;

		  static Exception * root(char* const f, UInt const l, 
					  char* const d, 
					  const RingBuffer* p) {
		     static XptUnderflow _root ;
		     _root.set(f,l,d) ;
		     _root.ptr = (RingBuffer*)p ;
		     //_root.copy = *p ;
		     return &_root ;
		  } ;

		  virtual void describe_aux(ostream& ostr) const {
		     ostr << ":ring=<" << ptr << ">";
		     if(!isNull(ptr)) ostr << *ptr ;
		  } ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    class XptOverflow : public Exception {
	       public:
		  RingBuffer *ptr ;
		  // RingBuffer  copy ;
	    
		  virtual const char * name() const { 
		     return "RingbufferOverflow" ; 
		  } ;

		  static Exception * root(char* const f, UInt const l, 
					  char* const d, 
					  const RingBuffer* p) {
		     static XptOverflow _root ;
		     _root.set(f,l,d) ;
		     _root.ptr = (RingBuffer*)p ;
		     //_root.copy = *p ;
		     return &_root ;
		  } ;

		  virtual void describe_aux(ostream& ostr) const {
		     ostr << ":ring=<" << ptr << ">";
		     if(!isNull(ptr)) ostr << *ptr ;
		  } ;
	    } ;

	    //--------------------------------------------------
	    // constructor
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    RingBufferT<Content>(char *nm = ITK_DFLT_RINGBUFFER_NAME,
				 UInt sz = ITK_DFLT_RINGBUFFER_SIZE,
				 UInt ef = ITK_DFLT_EXPAND_FACTOR) {
	       init(nm,sz,ef) ;
	    } ;

	    //--------------------------------------------------
	    // initialize
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    void init(char *nm = ITK_DFLT_RINGBUFFER_NAME,
		      UInt sz = ITK_DFLT_RINGBUFFER_SIZE,
		      UInt ef = ITK_DFLT_EXPAND_FACTOR) {
	       name = nm ;
	       size = 0 ;
	       expandFactor = ef ;
	       head = 0 ;
	       tail = 0 ;
	       table = ITK_NULLPTR ;
	       if(sz > 0) expand(sz) ;
	    } ;

	    //--------------------------------------------------
	    // copy
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    RingBuffer & copy(const RingBuffer& org) {
	       name = org.name ;
	       table = org.table ;
	       size = org.size ;
	       head = org.head ;
	       tail = org.tail ;
	       expandFactor = org.expandFactor ;
	       return *this ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    RingBuffer & operator= (const RingBuffer& org) {
	       return copy(org) ;
	    } ;

	    //--------------------------------------------------
	    // access
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Content & nth(const UInt i) const { return table[i % size] ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Content & operator[] (const UInt i) const { return nth(i) ; } ;

	    //--------------------------------------------------
	    // expand
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    void expand() {
	       if(size > 0) return expand(size * expandFactor) ;
	       else return expand(ITK_DFLT_RINGBUFFER_SIZE) ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    void expand(UInt newsize) ;

	    //--------------------------------------------------
	    // reduce counter
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    void reduce_counter() {
	       head = head % size ;
	       tail = tail % size ;
	    } ;

	    //--------------------------------------------------
	    // get 
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Content & get() {
	       if(!isEmpty()) {
		  return nth(tail++) ;
	       } else {
		  throw ITK_XPT1(XptUnderflow,name,this) ;
	       }
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Content get(const Content & failval) {
	       try {
		  return get() ;
	       } catch(XptUnderflow* xpt) {
		  return failval ;
	       } ;
	    } ;

	    //--------------------------------------------------
	    // put
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Content & put(const Content& elm) {
	       if(isFull()) expand() ;

	       Content & e = nth(head++) ;
	       e = elm ;
	       return e ;
	    } ;

	    //--------------------------------------------------
	    // reserve entry for put
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Content & reserve() {
	       if(isFull()) expand() ;

	       Content & e = nth(head++) ;
	       return e ;
	    } ;

	    //--------------------------------------------------
	    // clear : make empty
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    void clear() { tail = head ; } ;

	    //--------------------------------------------------
	    // empty/full check
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Bool isEmpty() const { return head <= tail ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Bool isFull() const { return head >= tail + size ; } ;

	    //--------------------------------------------------
	    // describe
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    virtual void describe(ostream& ostr, 
				  const Bool detailp = True) const {
	       ostr << "#RingBuffer[" << name << "|<" << this 
		    << ">:" << head << "-" << tail << "/" << size << "]" ;

	       if(detailp) {
		  ostr << " (expand factor = " << expandFactor 
		       << ")" << endl ;
		  ostr << " [Content]" ;
		  for(UInt i = tail ; i < head ; i++) {
		     ostr << endl << setw(8) << i << ":" << nth(i) ;
		  }
		  ostr << endl << "  === end ===  " << endl ;
	       }
	    } ;
      } ;
//--------------------------------------------------
// expand

   /*--------------------*/
   /**
    *
    */

   template <class Content>
      void RingBufferT<Content>::expand(UInt newsize) {
      if(newsize == 0 || newsize <= size) 
	 throw(ITK_XPT1(XptIllegalExpand,name,this)) ;

      UInt oldsize = size ;
      Content* oldtable = table ;
      Content* newtable = new Content[newsize] ;

      if(isNull(newtable)) 
	 throw(ITK_XPT(XptFailMemoryAlloc,"at expand RingBuffer")) ;
	 
      for(UInt i = tail ; i < head ; i++) 
	 newtable[i % newsize] = oldtable[i % oldsize] ;

      table = newtable ;
      size = newsize ;

      if(!isNull(oldtable)) delete oldtable ;
   }

} ;


////////////////////////////////////////////////////////////////////////
#endif
