// -*- Mode:C++ -*-
//Header:
//File: MemArrayT.h
//Author: NODA, Itsuki
//Date: 1999/09/06
//

//ModifyHistory:
// 1999/09/06: Start to create this file
// 1999/09/06: Split cc file
// 1999/11/08: separate into files for each class
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#ifndef _itk_MemArrayT_h_
#define _itk_MemArrayT_h_
////////////////////////////////////////////////////////////////////////
#include "itk/btype.h"
#include "itk/Exception.h"
#include "itk/WithDescriber.h"

namespace Itk {

//======================================================================
// MemArray<Content>
//	Memory Array

#ifndef ITK_DFLT_MEMARRAY_NAME
#define ITK_DFLT_MEMARRAY_NAME	"MemArray"
#endif
#ifndef ITK_DFLT_MEMARRAY_SIZE
#define ITK_DFLT_MEMARRAY_SIZE	8
#endif
#ifndef ITK_DFLT_EXPAND_FACTOR
#define ITK_DFLT_EXPAND_FACTOR	2
#endif

//------------------------------------------------------------
// class template

   /*--------------------*/
   /**
    *
    */

   template <class Content>
      class MemArrayT : public WithDescriber {
	 public:

	    //--------------------------------------------------
	    // local type def

	    typedef MemArrayT<Content> MemArray ;

	    //--------------------------------------------------
	    // member

	    char* name ;
	    Content* table ;
	    UInt edge ;
	    UInt size ;
	    UInt expandFactor ;

	    const static UInt npos = (UInt)(-1) ; // not a position (for find)
	    static Bool isNPos(UInt pos) { return pos == npos ; } ;

	    //--------------------------------------------------
	    // exception
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    class XptIllegalExpand : public Exception {
	       public:
		  MemArray* ptr ;
		  // MemArray  copy ; /* this is incomplete type */

		  virtual const char * name() const { 
		     return "IllegalExpand" ; } ;
      
		  static Exception * root(char* const f, UInt const l, 
					  char* const d, 
					  const MemArray* p) {
		     static XptIllegalExpand _root ;
		     _root.set(f,l,d) ;
		     _root.ptr = (MemArray*)p ;
		     // _root.copy = *p ; /* this is incomplete type */
		     return &_root ;
		  } ;

		  virtual void describe_aux(ostream& ostr) const {
		     ostr << ":array=" << (Ptr)ptr ;
		     if(!isNull(ptr)) ostr << *ptr ;
		  } ;
	    } ;

	    //--------------------------------------------------
	    // constructor
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    MemArrayT<Content>(char *nm = ITK_DFLT_MEMARRAY_NAME,
			       UInt sz = ITK_DFLT_MEMARRAY_SIZE,
			       UInt ef = ITK_DFLT_EXPAND_FACTOR) {
	       init(nm,sz,ef) ;
	    } ;
			  
	    //--------------------------------------------------
	    // initialize
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    void init(char *nm = ITK_DFLT_MEMARRAY_NAME,
		      UInt sz = ITK_DFLT_MEMARRAY_SIZE,
		      UInt ef = ITK_DFLT_EXPAND_FACTOR) {
	       name = nm ;
	       size = 0 ;
	       expandFactor = ef ;
	       edge = 0 ;
	       table = ITK_NULLPTR ;
	       if(sz > 0) expand(sz) ;
	    } ;

	    //--------------------------------------------------
	    // copy
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    MemArray & copy(const MemArray & org) {
	       name = org.name ;
	       table = org.table ;
	       edge = org.edge ;
	       size = org.size ;
	       expandFactor = expandFactor ;

	       return *this ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    MemArray & operator=(const MemArray& org) { 
	       return copy(org) ; 
	    } ;

	    //--------------------------------------------------
	    // access
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Content& operator[] (const UInt i) const { return table[i] ; } ;

	    //--------------------------------------------------
	    // expand
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    void expand() {
	       if(size > 0) return expand(size * expandFactor) ;
	       else return expand(ITK_DFLT_MEMARRAY_SIZE) ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    void expand(UInt newsize) ;

	    //--------------------------------------------------
	    // find
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Content * findPtr(const Content & content) ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    UInt findPos(const Content & content) ;

	    //--------------------------------------------------
	    // shift (fragile)
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    void shiftUp(const UInt n, const UInt l) ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    void shiftDown(const UInt n, const UInt l) ;

	    //--------------------------------------------------
	    // remove by content
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Bool remove(const Content & content) ;

	    //--------------------------------------------------
	    // iterator
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    class Iterator : public WithDescriber {
	       public:
		  MemArray * array ;
		  UInt index ;

		  Iterator() { array = ITK_NULLPTR ; index = 0 ; } ;
		  Iterator(MemArray * a, UInt ix = 0) { 
		     array = a ; index = ix ; 
		  } ;

		  Iterator & operator++ () { index++ ; return *this ; } ;
		  Iterator & operator++ (int) { index++ ; return *this ; } ;
		  
		  Content & operator* () const { 
		     return (*array)[index] ;
		  } ;

		  Bool operator< (const Iterator & ref) const {
		     return (index < ref.index) ;
		  } ;

		  virtual void describe(ostream& ostr, 
					const Bool detailp = True) const {
		     ostr << "#Iterator[" << *array << ":" << index << "]" ;

		     if(detailp) {
			ostr << endl ;
		     } ;
		  } ;
	    }  ;
	    

	    //------------------------------
	    // begin / end

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Iterator begin() { return Iterator(this) ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Iterator end() { return Iterator(this,edge) ; } ;

	    //--------------------------------------------------
	    // reverse iterator
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    class RevIterator : public Iterator {
	       public:
		  RevIterator() { array = ITK_NULLPTR ; index = 0 ; } ;
		  RevIterator(MemArray * a, UInt ix = 0) {
		     array = a ; 
		     index = ix ;
		  } ;

		  Content & operator*() const {
		     return (*array)[array->edge - index - 1] ;
		  } ;

		  virtual void describe(ostream& ostr, 
					const Bool detailp = True) const {
		     ostr << "#RevIterator[" << *array << ":" << index << "]" ;

		     if(detailp) {
			ostr << endl ;
		     } ;
		  } ;
	    } ;

	    //------------------------------
	    // rbegin / rend
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    RevIterator rbegin() { return RevIterator(this) ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    RevIterator rend() { return RevIterator(this,edge) ; } ;

	    //--------------------------------------------------
	    // describe
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    virtual void describe(ostream& ostr, 
				  const Bool detailp = True) const {
	       ostr << "#MemArray[" << name << "|" << this 
		    << ":" << edge << "/" << size << "]";

	       if(detailp) {
		  ostr << " (expand factor = " << expandFactor << ")" 
		       << endl ;
		  ostr << " [Content]" ;
		  for(UInt i = 0 ; i < edge ; i++) {
		     ostr << endl << setw(8) << i << " : " << table[i] ;
		  }
		  ostr << endl << "  === end ===  " << endl ;
	       }
	    }  ;

      } ;

//--------------------------------------------------
// expand

   /*--------------------*/
   /**
    *
    */

   template <class Content>
      void MemArrayT<Content>::expand(UInt newsize) {
      if(newsize == 0 || newsize < size) 
	 throw(ITK_XPT1(XptIllegalExpand,name,this)) ;

      Content* oldtable = table ;

      table = new Content[newsize] ;
      if(isNull(table)) {
	 table = oldtable ;
	 throw(ITK_XPT(XptFailMemoryAlloc,"at expand MemArray")) ;
      }

      for(UInt i = 0 ; i < size ; i++) 
	 table[i] = oldtable[i] ;

      size = newsize ;

      if(!isNull(table)) delete oldtable ;
   } 

//--------------------------------------------------
// find pointer

   /*--------------------*/
   /**
    *
    */

   template <class Content>
      Content * MemArrayT<Content>::findPtr(const Content& content) {
      for(UInt i = 0 ; i < edge ; i++) {
	 if(table[i] == content) return &(table[i]) ;
      }
      return ITK_NULLPTR ;
   }

//--------------------------------------------------
// find position

   /*--------------------*/
   /**
    *
    */

   template <class Content>
      UInt MemArrayT<Content>::findPos(const Content& content) {
      for(UInt i = 0 ; i < edge ; i++) {
	 if(table[i] == content) return i ;
      }
      return npos ;
   }

//--------------------------------------------------
// shiftDown (fragile)

   /*--------------------*/
   /**
    *
    */

   template <class Content>
      void MemArrayT<Content>::shiftDown(const UInt n, const UInt l) {
      for(UInt i = n ; i < edge - l ; i++) 
	 table[i] = table[i+l] ;
      edge -= l ;
   }

//--------------------------------------------------
// shiftUp (fragile)

   /*--------------------*/
   /**
    *
    */

   template <class Content>
      void MemArrayT<Content>::shiftUp(const UInt n, const UInt l) {
      for(UInt i = edge ; i > n ; i--) 
	 table[i+l-1] = table[i-1] ;
      edge += l ;
   }

//--------------------------------------------------
// remove by content

   /*--------------------*/
   /**
    *
    */

   template <class Content>
      Bool MemArrayT<Content>::remove(const Content& content) {
      for(UInt i = 0 ; i < edge ; i++) {
	 if(table[i] == content) {
	    shiftDown(i,1) ;
	    return True ;
	 }
      }
      return False ;
   }
   
} ;


////////////////////////////////////////////////////////////////////////
#endif
