// -*- Mode:C++ -*-
//Header:
//File: IndexTableT.h
//Author: NODA, Itsuki
//Date: 1999/09/06
//

//ModifyHistory:
// 1999/09/06: Start to create this file
// 1999/09/06: Split cc file
// 1999/11/08: separate into files for each class
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#ifndef _itk_IndexTableT_h_
#define _itk_IndexTableT_h_
////////////////////////////////////////////////////////////////////////
#include "itk/btype.h"
#include "itk/SString.h"
#include "itk/Exception.h"
#include "itk/MemArrayT.h"
#include "itk/RcylHeapT.h"
#include "itk/RingT.h"

namespace Itk {
//======================================================================
// index table

#ifndef ITK_DFLT_INDEXTABLE_NAME
#define ITK_DFLT_INDEXTABLE_NAME "IndexTable"
#endif
#ifndef ITK_DFLT_INDEXTABLE_SIZE
#define ITK_DFLT_INDEXTABLE_SIZE ITK_DFLT_MEMARRAY_SIZE
#endif

   /*--------------------*/
   /**
    *
    */

   template <class Index, class Content>
      class IndexTableT : public WithDescriber, public Mutex {
	 public:
	    //--------------------------------------------------
	    // local typedef

	    typedef IndexTableT<Index,Content> IndexTable ;

	    //--------------------------------------------------
	    // local class
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    class Entry : public WithDescriber {
	       public:
		  Index index ;
		  Content  content ;

		  virtual void describe(ostream& ostr,
					const Bool detailp = True) const {
		     ostr << "#IndexTableEntry[<" << this << ">"
			  << "|" << index << ":" << content << "]" ;
		     if(detailp) ostr << endl ;
		  } ;
	    } ;

	    typedef RcylHeapT<Entry> EntryHeap ;
	    typedef RingT<Entry *> EntryRing ;
	    typedef RingRcylHeapT<Entry *> EntryRingHeap ;

	    //--------------------------------------------------
	    // member

	    char* name ;
	    EntryHeap heap ;
	    EntryRing root ;
	    EntryRingHeap ringheap ;
	    
	    //--------------------------------------------------
	    // constructor
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    IndexTableT<Index,Content>(char* nm = ITK_DFLT_INDEXTABLE_NAME,
				       UInt sz = ITK_DFLT_INDEXTABLE_SIZE) {
	       init(nm,sz) ;
	    } ;

	    //--------------------------------------------------
	    // initialize
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    void init(char* nm = ITK_DFLT_INDEXTABLE_NAME,
		      UInt sz = ITK_DFLT_INDEXTABLE_SIZE) {
	       name = nm ;
	       heap.init(nm,sz) ;
	       ringheap.init(nm,sz) ;
	       root.linkSelf() ;
	       root.entity = newEntry() ;
	    } ;

	    //--------------------------------------------------
	    // set nulldata (return value of find if not found)
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    void setNullData(const Content & null) { 
	       root.entity->content = null ; 
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Bool notFound(const Content & data) { 
	       return data == root.entity->content ; 
	    } ;

	    //--------------------------------------------------
	    // new entry
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Entry * newEntry() { return heap.getPtr() ; } ;

	    //--------------------------------------------------
	    // add entry to ring
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    EntryRing * addEntry(Entry* entry) {
	       EntryRing * r = ringheap.getPtr(entry) ;
	       root.insertLast(r) ;
	       return r ;
	    } ;

	    //--------------------------------------------------
	    // remove ring
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Entry * removeRing(EntryRing * ring) {
	       ring->removeSelf() ;
	       ringheap.putPtr(ring) ;
	       return ring->entity ;
	    } ;

	    //--------------------------------------------------
	    // discard entry
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    void discardEntry(Entry * entry) { heap.putPtr(entry) ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    void discardRing(EntryRing * ring) {
	       removeRing(ring) ;
	       discardEntry(ring->entity) ;
	    } ;

	    //--------------------------------------------------
	    // assign new entry
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Entry * assign(const Index & index) { 
	       return assignEntry(index) ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Entry * assignEntry(const Index & index) {
	       Entry * entry = newEntry() ;
	       entry->index = index ;
	       addEntry(entry) ;
	       return entry ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Entry * assign(const Index & index, const Content & content) {
	       return assignEntry(index,content) ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Entry * assignEntry(const Index & index, 
				const Content & content) {
	       Entry * entry = assignEntry(index) ;
	       entry->content = content ;
	       return entry ;
	    } ;

	    //--------------------------------------------------
	    // find (not assign new entry)
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    EntryRing * findRing(const Index & index) const {
	       for(EntryRing * r = (EntryRing*)(root.next) ; 
		   !root.isEnd(r) ; r = r->next) {
		  if(MatchIndex(index,r->entity->index)) return r ;
	       }
	       return ITK_NULLPTR ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Entry * findEntry(const Index & index) const {
	       EntryRing * r = findRing(index) ;
	       if(!isNull(r)) return r->entity ;
	       else return ITK_NULLPTR ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Content & find(const Index & index) const {
	       Entry * e = findEntry(index) ;
	       if(!isNull(e)) return e->content ;
	       else return root.entity->content ;
	    } ;

	    //--------------------------------------------------
	    // get : find or assign
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Entry* getEntry(const Index & index) {
	       Entry * r = findEntry(index) ;
	       if(!isNull(r)) {
		  return r ;
	       } else {
		  return assignEntry(index) ;
	       }
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Content & get(const Index & index) {
	       return getEntry(index)->content ;
	    } ;

	    //--------------------------------------------------
	    // remove entry
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Bool remove(const Index & index) {
	       return removeEntry(index) ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Bool removeEntry(const Index & index) {
	       EntryRing * r = findRing(index) ;
	       if(!isNull(r)) {
		  discardRing(r) ;
		  return True ;
	       } else {
		  return False ;
	       } 
	    } ;

	    //--------------------------------------------------
	    // count, size
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    UInt count() const { return root.length() - 1 ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    UInt size() const { return heap.size ; } ;

	    //--------------------------------------------------
	    // first ring / check end ring
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    EntryRing * firstRing() const { return root.next ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    EntryRing * lastRing() const { return root.prev ; } ;
      
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Bool isEnd(EntryRing * ring) const { 
	       return root.isEnd(ring) ; } ;

	    //--------------------------------------------------
	    // Iterator
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    class Iterator : public WithDescriber {
	       public:
		  IndexTable * table ;
		  EntryRing * ring ;

		  Iterator() { table = ITK_NULLPTR ; ring = ITK_NULLPTR ; } ;
		  Iterator(IndexTable * tb) {
		     table = tb ;
		     ring = tb->firstRing() ;
		  } ;
		  Iterator(IndexTable * tb, EntryRing* r) {
		     table = tb ;
		     ring = r ;
		  } ;
		     
		  Iterator & operator++ () { 
		     ring = ring->next ;
		     return *this ; 
		  } ;
		  Iterator & operator++ (int) { return ++(*this) ; } ;

		  const Content & content() const {
		     return ring->entity->content ;
		  } ;
		  
		  Content & content() {
		     return ring->entity->content ;
		  } ;

		  const Index & index() const { 
		     return ring->entity->index ; 
		  } ;
		  Index & index() {
		     return ring->entity->index ;
		  } ;

		  const Content & operator* () const { 
		     return content() ;
		  } ;

		  Content & operator* () {
		     return ring->entity->content ;
		  } ;

		  Bool operator< (const Iterator & ref) const {
		     return !(ref.ring->isEnd(ring)) ;
		  } ;

		  virtual void describe(ostream& ostr, 
					const Bool detailp = True) const {
		     ostr << "#Iterator[" << *table << ":" << *ring << "]" ;

		     if(detailp) {
			ostr << endl ;
		     } ;
		  } ;
	    }  ;

	    //------------------------------
	    // begin / end
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Iterator begin() { return Iterator(this) ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Iterator end() { return Iterator(this,&root) ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Bool isBegin(const Iterator & i) const {
	       return i.ring == firstRing() ;
	    } ;

	    //--------------------------------------------------
	    // reverse iterator
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    class RevIterator : public Iterator {
	       public:
		  RevIterator() { 
		     table = ITK_NULLPTR ; 
		     ring = ITK_NULLPTR ; 
		  } ;
		  RevIterator(IndexTable * tb) {
		     table = tb ;
		     ring = tb->lastRing() ;
		  } ;
		  RevIterator(IndexTable * tb, EntryRing* r) {
		     table = tb ;
		     ring = r ;
		  } ;
		     
		  Iterator & operator++ () { 
		     ring = ring->prev ;
		     return *this ; 
		  } ;
		  Iterator & operator++ (int) { return ++(*this) ; } ;
		  
		  virtual void describe(ostream& ostr, 
					const Bool detailp = True) const {
		     ostr << "#RevIterator[" << *table << ":" << *ring << "]" ;

		     if(detailp) {
			ostr << endl ;
		     } ;
		  } ;
	    }  ;
	    
	    //------------------------------
	    // rbegin / rend
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    RevIterator rbegin() { return RevIterator(this) ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    RevIterator rend() { return RevIterator(this,&root) ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Bool isRBegin(const Iterator & i) const {
	       return i.ring == lastRing() ;
	    } ;

	    //--------------------------------------------------
	    // describe
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    virtual void describe(ostream& ostr,
				  const Bool detailp = True) const {
	       ostr << "#IndexTable[<" << this << ">" 
		    << "|" << name << "]" ;

	       if(detailp) {
		  ostr << endl << "  [Entry]" << root ;
		  for(EntryRing * r = root.next ; 
		      !root.isEnd(r); 
		      r = r->next ) {
		     ostr << endl << "    " << *(r->entity) ;
		  }
		  ostr << endl << "    --- end ---" ;
		  ostr << endl << "  [Heap] = " << heap ;
		  ostr << endl ;
	       }
	    } ;
      } ;

   /*--------------------*/
   /**
    *
    */

   inline Bool MatchIndex(const void * x, const void * y) { 
      return x == y ;} ;

   /*--------------------*/
   /**
    *
    */

   inline Bool MatchIndex(const Int x, const Int y) { return x == y ; } ;

   /*--------------------*/
   /**
    *
    */

   inline Bool MatchIndex(const UInt x, const UInt y) { return x == y ;};

   /*--------------------*/
   /**
    *
    */

   inline Bool MatchIndex(const char * x, const char * y) {
      return !strcmp(x,y) ;
   } ;

   /*--------------------*/
   /**
    *
    */

   inline Bool MatchIndex(const SubString & x, const SubString & y) {
      return x == y ;
   } ;


}


////////////////////////////////////////////////////////////////////////
#endif
