

--
-- Copyright (C) 2023  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--





with snd4ada;

with gl, gl.binding, gl.pointers;
with glu, glu.binding, glu.pointers;
with glext, glext.binding, glext.pointers;

-------------------------------------------------------------
with System;
with Interfaces.C;
use  type interfaces.c.unsigned;
with Interfaces.C.Pointers;
with interfaces.c.strings;


----------------------------------------------------------------

with glfw3;		use glfw3;
with zoomwheel; use zoomwheel;

----------------------------------------------------------------

with matutils;
with stex;

with ada.numerics.generic_elementary_functions;

with Ada.Strings.Fixed;
with Ada.Strings.Unbounded;
use Ada.Strings.Unbounded;
with Ada.Strings.Unbounded.Text_IO;
use Ada.Strings.Unbounded.Text_IO;



with unchecked_deallocation;

with text_io;
with pngloader;
with gametypes;
with matutils;

with pictobj;
with pictobj2;

with shader;  use shader;



with Ada.Command_Line;
with Ada.Directories;
with SysUtils;

with gameutils; use gameutils;

----------------------------------------------------------------



procedure tomb6 is -- 4x6x4 with Medusa head

	dirname: constant string := ada.directories.current_directory;
	onMac : boolean := (ada.strings.fixed.index( dirname, "MacOS", 1 ) > 1);
	onLinux : boolean := (ada.strings.fixed.index( dirname, "gnu", 1 ) > 1);


	prerestoreBuf, restoreBuf: unbounded_string;

	sel: array(1..9999) of integer;
	dir: array(1..9999) of character;
	nmoves: integer := 0;
	playscript : boolean := false;

	package myint_io is new text_io.integer_io(integer);


------- begin game specific code -------------------

mx, mz : constant integer := 4;
my     : constant integer := 6;


ncubes  : constant integer := mx*my*mz; -- total space volume: 96 cubelets

subtype rngm is integer range 1..ncubes;


empty : array(1..mx,1..my,1..mz) of boolean;


-- overall object drawing centroid (move to origin):
xxcc : constant float := float(mx)/2.0;
yycc : constant float := float(my)/2.0;
zzcc : constant float := float(mz)/2.0;

------- end game specific code -------------------






use text_io;
use gametypes;
use pngloader;
use matutils;
use gametypes.fmath;

use interfaces.c;
use interfaces.c.strings;
use glext;
use glext.pointers;
use glext.binding;
use gl;
use gl.binding;
use gl.pointers;



	origskin : boolean := true; -- false => Vadasz colors





procedure initEmptyHi is
begin

	for ix in 1..mx loop
		for iz in 1..mz loop
			for iy in 1..my loop
				empty(ix,iy,iz) := false;
			end loop;
		end loop;
	end loop;


	-- middle 2 layers are hollow:
	for ix in 2..3 loop
		for iz in 2..3 loop
			for iy in 2..3 loop
				empty(ix,iy,iz) := true;
			end loop;
		end loop;
	end loop;

	-- edges of top 2 layers are blank:
	for iy in 5..6 loop
		for ix in 1..4 loop
			empty(ix,iy,1):=true;
			empty(ix,iy,4):=true;
		end loop;
		for iz in 1..4 loop
			empty(1,iy,iz):=true;
			empty(4,iy,iz):=true;
		end loop;
	end loop;

end initEmptyHi;



procedure initEmptyMid is
begin

	for ix in 1..mx loop
		for iz in 1..mz loop
			for iy in 1..my loop
				empty(ix,iy,iz) := false;
			end loop;
		end loop;
	end loop;


	-- 8 extreme corners are hollow:
	empty(1,1,1):=true;
	empty(1,1,4):=true;
	empty(4,1,1):=true;
	empty(4,1,4):=true;

	empty(1,6,1):=true;
	empty(1,6,4):=true;
	empty(4,6,1):=true;
	empty(4,6,4):=true;


	-- edges of middle 2 layers are blank:
	for iy in 3..4 loop
		for ix in 1..mx loop
			empty(ix,iy,1):=true;
			empty(ix,iy,4):=true;
		end loop;
		for iz in 1..mz loop
			empty(1,iy,iz):=true;
			empty(4,iy,iz):=true;
		end loop;
	end loop;

end initEmptyMid;


procedure initEmptyGoal is
begin

	for ix in 1..mx loop
		for iz in 1..mz loop
			for iy in 1..my loop
				empty(ix,iy,iz) := false;
			end loop;
		end loop;
	end loop;

	-- top, bottom layers are blank:
	for ix in 1..mx loop
	for iz in 1..mz loop
		empty(ix,6,iz):=true;
		empty(ix,1,iz):=true;
	end loop;
	end loop;

end initEmptyGoal;
















windelay : constant float := 1.0;
testime : float := 0.0;

procedure test4winner is
	oktop, okbot, okmid : boolean := true;
begin

	for iy in 5..6 loop
		for ix in 1..mx loop
		for iz in 1..mz loop
			oktop := oktop and empty(ix,iy,iz);
		end loop;
		end loop;
	end loop;

	for iy in 1..2 loop
		for ix in 1..mx loop
		for iz in 1..mz loop
			okbot := okbot and empty(ix,iy,iz);
		end loop;
		end loop;
	end loop;


	for iy in 1..1 loop
		for ix in 1..mx loop
		for iz in 1..mz loop
			okmid := okmid and empty(ix,iy,iz);
		end loop;
		end loop;
	end loop;

	for iy in 6..6 loop
		for ix in 1..mx loop
		for iz in 1..mz loop
			okmid := okmid and empty(ix,iy,iz);
		end loop;
		end loop;
	end loop;

	winner := oktop or okmid or okbot;

	if winner then
		testime := float(glfwGetTime);
	end if;

end test4winner;









-- define 9 puzzle pieces here, and initialize:
mp : constant integer := 9;
subtype rngp is integer range 1..mp;
subtype fparray is gfparray(rngp);
subtype iparray is giparray(rngp);




movedelay : constant float := 0.25; -- seconds to move a block
movebegin : float := 0.0;
oxc,oyc,ozc : integer;
grinding : array(rngp) of boolean := ( others=>false );
anyGrinding: boolean := false;


nun_texid : gluint;

num_texid : array(rngp) of gluint;
box_texid : gluint;


centroidGoal : constant iparray :=
-- piece-centroids initialization:
( 
		1=>(1,4,1), --ulp
		2=>(3,4,1), --urp
		3=>(1,4,3), --ula
		4=>(3,4,3), --ura

		5=>(1,2,1), --llp
		6=>(3,2,1), --lrp
		7=>(1,2,3), --lla
		8=>(3,2,3), --lra

		9=>(2,3,2)  --to be entombed in center
);




centroidHi : constant iparray :=
-- piece-centroids initialization:
( 
		1=>(1,3,1), --ulp
		2=>(3,3,1), --urp
		3=>(1,3,3), --ula
		4=>(3,3,3), --ura

		5=>(1,1,1), --llp
		6=>(3,1,1), --lrp
		7=>(1,1,3), --lla
		8=>(3,1,3), --lra

		9=>(2,5,2)  --to be entombed in center
);


centroidMid : constant iparray :=
-- piece-centroids initialization:
( 
		1=>(3,1,3), --lra
		2=>(1,1,3), --lla
		3=>(3,1,1), --lrp
		4=>(1,1,1), --llp

		5=>(3,5,3), --ura
		6=>(1,5,3), --ula
		7=>(3,5,1), --urp
		8=>(1,5,1), --ulp

		9=>(2,3,2)  --to be entombed in center
);

--Note that for each piece, its full extent is:
-- (p.x,p.y,p.z)...(p.x+1,p.y+1,p.z+1)

centroid : iparray;


--offsets from piece-centroid to space-centroid:
space : constant iparray :=
( 
		1=>(1,0,1), --ulp
		2=>(0,0,1), --urp
		3=>(1,0,0), --ula
		4=>(0,0,0), --ura

		5=>(1,1,1), --llp
		6=>(0,1,1), --lrp
		7=>(1,1,0), --lla
		8=>(0,1,0), --lra

		9=>(0,0,0)  --unused
);










mxskin: constant integer := 4;
skin: integer := mxskin-1;

procedure setNextSkin is
	macpath: string := "../../../data/"; -- osx
	xinpath: string := "../../data/"; -- windows or linux
	nopath: string := "./data/"; --below current dir
	path: unbounded_string;
begin


	if onmac then
		append(path,macpath);
	elsif ada.directories.exists(nopath) then
		append(path,nopath);
	else
		append(path,xinpath);
	end if;



	skin := (1+skin) mod mxskin; --0..3
	myassert( (skin < mxskin) and (skin>=0) );

	if skin=0 then --Original colors

		nun_texid    := loadPng(mirror,to_string(path)&"none.png");

		num_texid(1) := loadPng(mirror,to_string(path)&"one.png");
		num_texid(2) := loadPng(mirror,to_string(path)&"two.png");
		num_texid(3) := loadPng(mirror,to_string(path)&"three.png");
		num_texid(4) := loadPng(mirror,to_string(path)&"four.png");
		num_texid(5) := loadPng(mirror,to_string(path)&"five.png");
		num_texid(6) := loadPng(mirror,to_string(path)&"six.png");
		num_texid(7) := loadPng(mirror,to_string(path)&"seven.png");
		num_texid(8) := loadPng(mirror,to_string(path)&"eight.png");

	elsif skin=1 then --wornRock

		nun_texid    := loadPng(mirror,to_string(path)&"worn.png");

		num_texid(1) := loadPng(mirror,to_string(path)&"worn1.png");
		num_texid(2) := loadPng(mirror,to_string(path)&"worn2.png");
		num_texid(3) := loadPng(mirror,to_string(path)&"worn3.png");
		num_texid(4) := loadPng(mirror,to_string(path)&"worn4.png");

		num_texid(5) := loadPng(mirror,to_string(path)&"worn5.png");
		num_texid(6) := loadPng(mirror,to_string(path)&"worn6.png");
		num_texid(7) := loadPng(mirror,to_string(path)&"worn7.png");
		num_texid(8) := loadPng(mirror,to_string(path)&"worn8.png");

	elsif skin=2 then --wood

		nun_texid    := loadPng(mirror,to_string(path)&"wood.png");

		num_texid(1) := loadPng(mirror,to_string(path)&"w1.png");
		num_texid(2) := loadPng(mirror,to_string(path)&"w2.png");
		num_texid(3) := loadPng(mirror,to_string(path)&"w3.png");
		num_texid(4) := loadPng(mirror,to_string(path)&"w4.png");

		num_texid(5) := loadPng(mirror,to_string(path)&"w5.png");
		num_texid(6) := loadPng(mirror,to_string(path)&"w6.png");
		num_texid(7) := loadPng(mirror,to_string(path)&"w7.png");
		num_texid(8) := loadPng(mirror,to_string(path)&"w8.png");

	elsif skin=3 then --granite

		nun_texid    := loadPng(mirror,to_string(path)&"gran.png");

		num_texid(1) := loadPng(mirror,to_string(path)&"g1.png");
		num_texid(2) := loadPng(mirror,to_string(path)&"g2.png");
		num_texid(3) := loadPng(mirror,to_string(path)&"g3.png");
		num_texid(4) := loadPng(mirror,to_string(path)&"g4.png");

		num_texid(5) := loadPng(mirror,to_string(path)&"g5.png");
		num_texid(6) := loadPng(mirror,to_string(path)&"g6.png");
		num_texid(7) := loadPng(mirror,to_string(path)&"g7.png");
		num_texid(8) := loadPng(mirror,to_string(path)&"g8.png");

	end if;

end setNextSkin;














procedure checkfile is
	ifil: file_type;
	i: integer;
	ch,junk: character;
	use myint_io;
begin

	set_unbounded_string(prerestoreBuf, "");

	if ada.directories.Exists("t6restore.txt") then

		text_io.open(ifil, in_file, "t6restore.txt");

		nmoves:=0;
		loop
			get(ifil, i);
			exit when i<0 or i>10;
			get(ifil, junk);
			get(ifil, ch);
			nmoves:=nmoves+1;
			sel(nmoves) := i;
			dir(nmoves) := ch;
			append(prerestoreBuf, integer'image(i) & " " & ch &" ");
			exit when nmoves>9_999;
		end loop;

		text_io.close(ifil);

		--put(integer'image(nmoves)); new_line;
		--if nmoves<10 then
		--for i in 1..nmoves loop
		--	put( integer'image( sel(i) ) );
		--	put(" "&dir(i));
		--	new_line;
		--end loop;
		--end if;

		playscript:=true;
	end if;


end checkfile;








--goalMode : boolean := false;
normalMode : boolean := true;

procedure first_prep is -- main program setup
	macpath: string := "../../../data/"; -- osx
	xinpath: string := "../../data/"; -- windows or linux
	nopath: string := "./data/"; --below current dir
	path: unbounded_string;
	fontcol: constant vec4 := (0.0,0.0,0.0,1.0); --black

	fontfile: constant string := "data/NotoSans-Regular.ttf";
	annex: constant string := "../../";
	nerr: integer;
begin


	if onmac then
		append(path,macpath);
	elsif ada.directories.exists(nopath) then
		append(path,nopath);
	else
		append(path,xinpath);
	end if;

	snd4ada.initSnds;

	brick := snd4ada.initSnd(
		Interfaces.C.Strings.New_String(to_string(path)&"concrete_third.wav"));
	
	whoosh := snd4ada.initSnd(
		Interfaces.C.Strings.New_String(to_string(path)&"whoosh_4th.wav"));
	
	fanfare := snd4ada.initSnd(
		Interfaces.C.Strings.New_String(to_string(path)&"fanfare.wav"));
	
	shriek := snd4ada.initSnd(
		Interfaces.C.Strings.New_String(to_string(path)&"medusa.wav"));
	
	clap := snd4ada.initSnd(
		Interfaces.C.Strings.New_String(to_string(path)&"applause.wav"));
	

	if brick<0 or whoosh<0 or fanfare<0 or shriek<0 or clap<0 then
		put_line("snd4ada.initSnds ERROR");
		raise program_error;
	end if;


------- begin GLFW prep ---------------------------------------------------


	gameutils.InitGlfw( wwid,whit,fwid,fhit,"Reliquary6 - Medusa");
	zoomwheel.enable(mainWin);

	-- prepare font -------------
	if ada.directories.exists(fontfile) then
		stex.InitFont ( fontfile );
	else
		stex.InitFont ( annex&fontfile );
	end if;

	stex.setColor( fontcol );
	stex.reSize(wwid,whit);

	glViewport(0,0,Fwid,Fhit);



	glgenvertexarrays(1, vertexarrayid'address );
	glbindvertexarray(vertexarrayid);

	glactivetexture(gl_texture0); -- moved here 5nov14 (outside main loop)

	glgenbuffers(1, vertbuff'address);
	glgenbuffers(1, rgbbuff'address);
	glgenbuffers(1, uvbuff'address);
	glgenbuffers(1, elembuff'address);


	glenable(gl_depth_test);
	gldepthfunc( gl_lequal );
	glenable( gl_cull_face );


	-- theoretically reduces aliasing (can't tell for sure):
	glEnable(GL_MULTISAMPLE);
	glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
	glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);

	glClearColor(0.4,0.4,0.4,1.0);

	nerr:=dumpGLerrorQueue("main 1"); 
	--prevents misleading messages in pngloader or loadshaders



	box_texid := loadPng(mirror,to_string(path)&"fence.png");

	skin:=2;
	setNextSkin;

	num_texid(9) := loadPng(mirror,to_string(path)&"medusa3.png"); --pixabay.com

	wall_texid := loadPng(mirror,to_string(path)&"granite.png");


	pgmTexShadID := loadshaders(to_string(path)&"texobj.vs",to_string(path)&"texobj.fs");
	matrixid := glgetuniformlocation(pgmtexshadid, pmvp);
	uniftex  := glgetuniformlocation(pgmtexshadid, pmyts);

	pgmTexID := loadshaders(to_string(path)&"otexobj.vs",to_string(path)&"otexobj.fs");
	matid := glgetuniformlocation(pgmtexid, pmvp);
	unitex  := glgetuniformlocation(pgmtexid, pmyts);


	-- note:  deep copies of structures are automatic in Ada
	--        (unless we use pointer assignments)
--	if goalMode then
--		centroid:=centroidGoal;
--		initEmptyGoal;
--	elsif normalMode then
--		centroid:=centroidMid;
--		initEmptyMid;
--	else
--		centroid:=centroidHi;
--		initEmptyHi;
--	end if;

	if normalMode then
		centroid:=centroidMid;
		initEmptyMid;
	else --goalmode
		centroid:=centroidGoal;
		initEmptyGoal;
	end if;

end first_prep;















-- the enclosing chainlink box:
procedure drawBigCube( po: pictobj2.pictangle ) is

	xr, zr : constant float := 2.02;
	    yr : constant float := 3.03;

	xc : constant float := xr - xxcc;
	yc : constant float := yr - yycc;
	zc : constant float := zr - zzcc;

	r,g,b : constant float := 0.5;
begin
	pictobj2.setrect( po, xc,yc,zc, xr,yr,zr, r,g,b);
	pictobj2.draw( po, vertbuff, uvbuff, elembuff );
end drawBigCube;











-- this is the 9th puzzle piece...double size: 2x2x2
procedure draw9( po: pictobj.pictangle; nowTime: float ) is
	tt, r,g,b, xc,yc,zc, xo,yo,zo, xt,yt,zt : float;
	ix,iy,iz: integer;
begin

	ix:=centroid(9)(1);
	iy:=centroid(9)(2);
	iz:=centroid(9)(3);

	xc:=float(ix)-xxcc;
	yc:=float(iy)-yycc;
	zc:=float(iz)-zzcc;

	xo:=float(oxc)-xxcc;
	yo:=float(oyc)-yycc;
	zo:=float(ozc)-zzcc;

	r:=0.6; g:=0.6; b:=0.6;

	if grinding(9) then
		tt := (nowTime-movebegin)/movedelay;
		if tt>1.0 then
			pictobj.setrect( po, xc,yc,zc, 0.95,0.95,0.95, r,g,b);
			grinding(9):=false;
		else
			xt := xo + tt*(xc-xo);
			yt := yo + tt*(yc-yo);
			zt := zo + tt*(zc-zo);
			pictobj.setrect( po, xt,yt,zt, 0.95,0.95,0.95, r,g,b);
		end if;
	else
		pictobj.setrect( po, xc,yc,zc, 0.95,0.95,0.95, r,g,b);
	end if;


	--pictobj.draw( po, vertbuff, rgbbuff, uvbuff, elembuff );
	pictobj.draw( po, vertbuff, uvbuff, elembuff );

end draw9;










-- these are individual 1x1 parts of each of corner pieces,
-- where the 8 corners have 7 cubelets each:
procedure drawCube( 
	po: pictobj.pictangle; ii, 
	cx,cy,cz, i,j,k, ixc,iyc,izc : integer; 
	nowTime: float ) is

	-- (xxcc,yycc,zzcc)==(2,3,2) : centroid of whole puzzle...

	-- (ixc,iyc,izc) = centroid of the current puzzle piece = PP,
	-- (i,j,k) = offset from PP-centroid to current cubelet
	-- (cx,cy,cz) = offset from PP-centroid to corner cubelet

	xc : float := float(ixc+i) -0.5 - xxcc;
	yc : float := float(iyc+j) -0.5 - yycc;
	zc : float := float(izc+k) -0.5 - zzcc;

	xo : float := float(oxc+i) -0.5 - xxcc;
	yo : float := float(oyc+j) -0.5 - yycc;
	zo : float := float(ozc+k) -0.5 - zzcc;

----------- begin addendum -------------------------

	-- cubelet centroid before offset to puzzle center:
	xcc : float := float(ixc+i) -0.5;
	ycc : float := float(iyc+j) -0.5;
	zcc : float := float(izc+k) -0.5;

	fr : float := 0.06; -- move cubelet-center toward corner cubelet
	dx : float := fr*(float(ixc+cx) - xcc);
	dy : float := fr*(float(iyc+cy) - ycc);
	dz : float := fr*(float(izc+cz) - zcc);

	-- this is to slightly separate the puzzle pieces
	-- but not their constituent cubelets.

----------- end addendum -------------------------

	sc: float := 0.47; --0.47; --reduce radius from 0.5

	r,g,b, tt, xt,yt,zt : float;

begin

	if ii=1 then
		r:=0.2; g:=1.0; b:=0.2;
	elsif ii=2 then
		r:=1.0; g:=1.0; b:=0.2;
	elsif ii=3 then
		r:=0.2; g:=1.0; b:=1.0;
	elsif ii=4 then
		r:=1.0; g:=1.0; b:=1.0;
	elsif ii=5 then
		r:=0.2; g:=0.2; b:=0.2;
	elsif ii=6 then
		r:=1.0; g:=0.2; b:=0.3;
	elsif ii=7 then
		r:=0.2; g:=0.2; b:=1.0;
	elsif ii=8 then
		r:=1.0; g:=0.2; b:=1.0;
	elsif ii=9 then
		r:=0.6; g:=0.6; b:=0.6;
	end if;

	if grinding(ii) then
		tt := (nowTime-movebegin)/movedelay;
		if tt>1.0 then
			pictobj.setrect( po, xc+dx,yc+dy,zc+dz, sc,sc,sc, r,g,b);
			grinding(ii):=false;
		else
			xt := xo + tt*(xc-xo);
			yt := yo + tt*(yc-yo);
			zt := zo + tt*(zc-zo);
			pictobj.setrect( po, xt+dx,yt+dy,zt+dz, sc,sc,sc, r,g,b);
		end if;
	else
		pictobj.setrect( po, xc+dx,yc+dy,zc+dz, sc,sc,sc, r,g,b);
	end if;

	pictobj.draw( po, vertbuff, rgbbuff, uvbuff, elembuff );

end drawCube;














function trynear(i: rngp) return boolean is -- +Z
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid, oldVoidWasEmpty : boolean;
begin
	ix:=centroid(i)(1);
	iy:=centroid(i)(2);
	iz:=centroid(i)(3);

	sx:=space(i)(1);
	sy:=space(i)(2);
	sz:=space(i)(3);

	leadingVoid := (sz=1);

	oldVoidWasEmpty := empty(ix+sx,iy+sy,iz+sz);

	-- next centroid:
	iz:=iz+1;

	-- next leading test location
	iz:=iz+1;

if iz>mz then
	return false;

elsif leadingVoid  and i/=9  then

	if    sx=0 and sy=0 then

		if
			oldVoidWasEmpty and
			empty( ix+1,iy+0,iz ) and
			empty( ix+1,iy+1,iz ) and
			empty( ix+0,iy+1,iz )
		then
			return true;
		else
			return false;
		end if;


	elsif sx=0 and sy=1 then

		if
			oldVoidWasEmpty and
			empty( ix+0,iy+0,iz ) and
			empty( ix+1,iy+0,iz ) and
			empty( ix+1,iy+1,iz )
		then
			return true;
		else
			return false;
		end if;


	elsif sx=1 and sy=0 then

		if
			oldVoidWasEmpty and
			empty( ix+0,iy+0,iz ) and
			empty( ix+1,iy+1,iz ) and
			empty( ix+0,iy+1,iz )
		then
			return true;
		else
			return false;
		end if;


	elsif sx=1 and sy=1 then

		if
			oldVoidWasEmpty and
			empty( ix+0,iy+0,iz ) and
			empty( ix+1,iy+0,iz ) and
			empty( ix+0,iy+1,iz )
		then
			return true;
		else
			return false;
		end if;

	else
		return false;

	end if;

else -- not leadingVoid or i=9

		if
			empty( ix+0,iy+0,iz ) and
			empty( ix+1,iy+0,iz ) and
			empty( ix+1,iy+1,iz ) and
			empty( ix+0,iy+1,iz )
		then
			return true;
		else
			return false;
		end if;

end if; -- leadingVoid

end trynear;









function tryaway(i: rngp) return boolean is -- -Z
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid : boolean;
	oldVoidWasEmpty : boolean;
begin
	ix:=centroid(i)(1);
	iy:=centroid(i)(2);
	iz:=centroid(i)(3);

	sx:=space(i)(1);
	sy:=space(i)(2);
	sz:=space(i)(3);

	leadingVoid := (sz=0);

	oldVoidWasEmpty := empty(ix+sx,iy+sy,iz+sz);

	-- next centroid:
	iz:=iz-1;

	-- next leading test location
	iz:=iz+0;


if iz<1 then
	return false;

elsif leadingVoid  and i/=9  then

	if    sx=0 and sy=0 then

		if
			oldVoidWasEmpty and
			empty( ix+1,iy+0,iz ) and
			empty( ix+1,iy+1,iz ) and
			empty( ix+0,iy+1,iz )
		then
			return true;
		else
			return false;
		end if;


	elsif sx=0 and sy=1 then

		if
			oldVoidWasEmpty and
			empty( ix+0,iy+0,iz ) and
			empty( ix+1,iy+0,iz ) and
			empty( ix+1,iy+1,iz )
		then
			return true;
		else
			return false;
		end if;


	elsif sx=1 and sy=0 then

		if
			oldVoidWasEmpty and
			empty( ix+0,iy+0,iz ) and
			empty( ix+1,iy+1,iz ) and
			empty( ix+0,iy+1,iz )
		then
			return true;
		else
			return false;
		end if;


	elsif sx=1 and sy=1 then

		if
			oldVoidWasEmpty and
			empty( ix+0,iy+0,iz ) and
			empty( ix+1,iy+0,iz ) and
			empty( ix+0,iy+1,iz )
		then
			return true;
		else
			return false;
		end if;

	else
		return false;

	end if;

else -- not leadingVoid or i=9

		if
			empty( ix+0,iy+0,iz ) and
			empty( ix+1,iy+0,iz ) and
			empty( ix+1,iy+1,iz ) and
			empty( ix+0,iy+1,iz )
		then
			return true;
		else
			return false;
		end if;

end if; -- leadingVoid

end tryaway;














function tryup(i: rngp) return boolean is -- +Y
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid : boolean;
	oldVoidWasEmpty : boolean;
begin
	ix:=centroid(i)(1);
	iy:=centroid(i)(2);
	iz:=centroid(i)(3);

	sx:=space(i)(1);
	sy:=space(i)(2);
	sz:=space(i)(3);

	leadingVoid := (sy=1);

	oldVoidWasEmpty := empty(ix+sx,iy+sy,iz+sz);

	-- next centroid:
	iy:=iy+1;

	-- next leading test location
	iy:=iy+1;


if iy>my then
	return false;

elsif leadingVoid  and i/=9  then

	if    sx=0 and sz=0 then

		if
			oldVoidWasEmpty and
			empty( ix+1,iy,iz+0 ) and
			empty( ix+1,iy,iz+1 ) and
			empty( ix+0,iy,iz+1 )
		then
			return true;
		else
			return false;
		end if;


	elsif sx=0 and sz=1 then

		if
			oldVoidWasEmpty and
			empty( ix+0,iy,iz+0 ) and
			empty( ix+1,iy,iz+0 ) and
			empty( ix+1,iy,iz+1 )
		then
			return true;
		else
			return false;
		end if;


	elsif sx=1 and sz=0 then

		if
			oldVoidWasEmpty and
			empty( ix+0,iy,iz+0 ) and
			empty( ix+1,iy,iz+1 ) and
			empty( ix+0,iy,iz+1 )
		then
			return true;
		else
			return false;
		end if;


	elsif sx=1 and sz=1 then

		if
			oldVoidWasEmpty and
			empty( ix+0,iy,iz+0 ) and
			empty( ix+1,iy,iz+0 ) and
			empty( ix+0,iy,iz+1 )
		then
			return true;
		else
			return false;
		end if;

	else
		return false;

	end if;

else -- not leadingVoid or i=9

		if
			empty( ix+0,iy,iz+0 ) and
			empty( ix+1,iy,iz+0 ) and
			empty( ix+1,iy,iz+1 ) and
			empty( ix+0,iy,iz+1 )
		then
			return true;
		else
			return false;
		end if;

end if; -- leadingVoid

end tryup;












function trydown(i: rngp) return boolean is -- -Y
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid : boolean;
	oldVoidWasEmpty : boolean;
begin
	ix:=centroid(i)(1);
	iy:=centroid(i)(2);
	iz:=centroid(i)(3);

	sx:=space(i)(1);
	sy:=space(i)(2);
	sz:=space(i)(3);


	leadingVoid := (sy=0);

	oldVoidWasEmpty := empty(ix+sx,iy+sy,iz+sz);

	-- next centroid:
	iy:=iy-1;

	-- next leading test location
	iy:=iy+0;


if iy<1 then

	return false;

elsif leadingVoid  and i/=9 then

	if    sx=0 and sz=0 then


		if
			oldVoidWasEmpty and
			empty( ix+1,iy,iz+0 ) and
			empty( ix+1,iy,iz+1 ) and
			empty( ix+0,iy,iz+1 )
		then
			return true;
		else
			return false;
		end if;


	elsif sx=0 and sz=1 then


		if
			oldVoidWasEmpty and
			empty( ix+0,iy,iz+0 ) and
			empty( ix+1,iy,iz+0 ) and
			empty( ix+1,iy,iz+1 )
		then
			return true;
		else
			return false;
		end if;


	elsif sx=1 and sz=0 then


		if
			oldVoidWasEmpty and
			empty( ix+0,iy,iz+0 ) and
			empty( ix+1,iy,iz+1 ) and
			empty( ix+0,iy,iz+1 )
		then
			return true;
		else
			return false;
		end if;


	elsif sx=1 and sz=1 then


		if
			oldVoidWasEmpty and
			empty( ix+0,iy,iz+0 ) and
			empty( ix+1,iy,iz+0 ) and
			empty( ix+0,iy,iz+1 )
		then
			return true;
		else
			return false;
		end if;

	else
		return false;

	end if;

else -- not leadingVoid or i=9

		if
			empty( ix+0,iy,iz+0 ) and
			empty( ix+1,iy,iz+0 ) and
			empty( ix+1,iy,iz+1 ) and
			empty( ix+0,iy,iz+1 )
		then
			return true;
		else
			return false;
		end if;

end if; -- leadingVoid

end trydown;

















function tryright(i: rngp) return boolean is -- +X
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid : boolean;
	oldVoidWasEmpty : boolean;
begin
	ix:=centroid(i)(1);
	iy:=centroid(i)(2);
	iz:=centroid(i)(3);

	sx:=space(i)(1);
	sy:=space(i)(2);
	sz:=space(i)(3);


	leadingVoid := (sx=1);

	oldVoidWasEmpty := empty(ix+sx,iy+sy,iz+sz);

	-- next centroid:
	ix:=ix+1;

	-- next leading test location
	ix:=ix+1;


if ix>mx then
	return false;

elsif leadingVoid  and i/=9  then

	if    sy=0 and sz=0 then

		if
			oldVoidWasEmpty and
			empty( ix,iy+1,iz+0 ) and
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+1 )
		then
			return true;
		else
			return false;
		end if;


	elsif sy=0 and sz=1 then

		if
			oldVoidWasEmpty and
			empty( ix,iy+0,iz+0 ) and
			empty( ix,iy+1,iz+0 ) and
			empty( ix,iy+1,iz+1 )
		then
			return true;
		else
			return false;
		end if;


	elsif sy=1 and sz=0 then

		if
			oldVoidWasEmpty and
			empty( ix,iy+0,iz+0 ) and
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+1 )
		then
			return true;
		else
			return false;
		end if;


	elsif sy=1 and sz=1 then

		if
			oldVoidWasEmpty and
			empty( ix,iy+0,iz+0 ) and
			empty( ix,iy+1,iz+0 ) and
			empty( ix,iy+0,iz+1 )
		then
			return true;
		else
			return false;
		end if;

	else
		return false;

	end if;

else -- not leadingVoid or i=9

		if
			empty( ix,iy+0,iz+0 ) and
			empty( ix,iy+1,iz+0 ) and
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+1 )
		then
			return true;
		else
			return false;
		end if;

end if; -- leadingVoid

end tryright;
















function tryleft(i: rngp) return boolean is -- -X
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid : boolean;
	oldVoidWasEmpty : boolean;
begin
	ix:=centroid(i)(1);
	iy:=centroid(i)(2);
	iz:=centroid(i)(3);

	sx:=space(i)(1);
	sy:=space(i)(2);
	sz:=space(i)(3);


	leadingVoid := (sx=0);

	oldVoidWasEmpty := empty(ix+sx,iy+sy,iz+sz);

	-- next centroid:
	ix:=ix-1;

	-- next leading test location
	ix:=ix+0;


if ix<1 then
	return false;

elsif leadingVoid  and i/=9  then

	if    sy=0 and sz=0 then

		if
			oldVoidWasEmpty and
			empty( ix,iy+1,iz+0 ) and
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+1 )
		then
			return true;
		else
			return false;
		end if;


	elsif sy=0 and sz=1 then

		if
			oldVoidWasEmpty and
			empty( ix,iy+0,iz+0 ) and
			empty( ix,iy+1,iz+0 ) and
			empty( ix,iy+1,iz+1 )
		then
			return true;
		else
			return false;
		end if;


	elsif sy=1 and sz=0 then

		if
			oldVoidWasEmpty and
			empty( ix,iy+0,iz+0 ) and
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+1 )
		then
			return true;
		else
			return false;
		end if;


	elsif sy=1 and sz=1 then

		if
			oldVoidWasEmpty and
			empty( ix,iy+0,iz+0 ) and
			empty( ix,iy+1,iz+0 ) and
			empty( ix,iy+0,iz+1 )
		then
			return true;
		else
			return false;
		end if;

	else
		return false;

	end if;

else -- not leadingVoid or i=9

		if
			empty( ix,iy+0,iz+0 ) and
			empty( ix,iy+1,iz+0 ) and
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+1 )
		then
			return true;
		else
			return false;
		end if;

end if; -- leadingVoid

end tryleft;


---------------------------------------------------------------------------




procedure movenear( kk: in out integer; nowTime: float ) is -- +Z
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid, wasEmpty : boolean;
begin

if not trynear(kk) then -- pick one that might make sense
	for i in 1..9 loop
		if trynear(i) then
			kk:=i;
			exit;
		end if;
	end loop;
end if;


if trynear(kk) then

	append(restoreBuf, integer'image(kk) & " n ");

	oxc:=centroid(kk)(1);
	oyc:=centroid(kk)(2);
	ozc:=centroid(kk)(3);

	sx:=space(kk)(1);
	sy:=space(kk)(2);
	sz:=space(kk)(3);

	ix:=oxc;
	iy:=oyc;
	iz:=ozc;

	leadingVoid := (sz=1);


	if kk=9 then -- (easy)

		-- trailing face:
		empty( ix+0,iy+0,iz+0 ):=true;
		empty( ix+1,iy+0,iz+0 ):=true;
		empty( ix+1,iy+1,iz+0 ):=true;
		empty( ix+0,iy+1,iz+0 ):=true;

		-- make the move:
		centroid(kk)(3):=centroid(kk)(3)+1;
		iz:=centroid(kk)(3);

		-- leading face:
		empty( ix+0,iy+0,iz+1 ):=false;
		empty( ix+1,iy+0,iz+1 ):=false;
		empty( ix+1,iy+1,iz+1 ):=false;
		empty( ix+0,iy+1,iz+1 ):=false;

	elsif leadingVoid then

		-- next leading void position:
		wasEmpty := empty( ix+sx,iy+sy,iz+1+sz );

		-- trailing face:
		empty( ix+0,iy+0,iz+0 ):=true;
		empty( ix+1,iy+0,iz+0 ):=true;
		empty( ix+1,iy+1,iz+0 ):=true;
		empty( ix+0,iy+1,iz+0 ):=true;

		-- make the move:
		centroid(kk)(3):=centroid(kk)(3)+1;
		iz:=centroid(kk)(3);

		-- leading face:
		empty( ix+0,iy+0,iz+1 ):=false;
		empty( ix+1,iy+0,iz+1 ):=false;
		empty( ix+1,iy+1,iz+1 ):=false;
		empty( ix+0,iy+1,iz+1 ):=false;

		-- new leading void setting...
		-- this corrects one of the previous 4 lines:
		empty( ix+sx,iy+sy,iz+sz ):=wasEmpty;

	else -- trailingVoid

		-- old trailing void status:
		wasEmpty := empty( ix+sx,iy+sy,iz+sz );

		-- trailing face:
		empty( ix+0,iy+0,iz+0 ):=true;
		empty( ix+1,iy+0,iz+0 ):=true;
		empty( ix+1,iy+1,iz+0 ):=true;
		empty( ix+0,iy+1,iz+0 ):=true;

		-- this corrects one of the previous 4 lines:
		empty( ix+sx,iy+sy,iz+sz ):=wasEmpty;

		-- make the move:
		centroid(kk)(3):=centroid(kk)(3)+1;
		iz:=centroid(kk)(3);

		-- leading face:
		empty( ix+0,iy+0,iz+1 ):=false;
		empty( ix+1,iy+0,iz+1 ):=false;
		empty( ix+1,iy+1,iz+1 ):=false;
		empty( ix+0,iy+1,iz+1 ):=false;

		-- new trailing void is empty:
		empty( ix+sx,iy+sy,iz+sz ):=true;

	end if;

	test4winner;

	if not mute then 
	snd4ada.playSnd(brick); 
	end if;

	movebegin:=nowTime;
	grinding(kk):=true;

end if; --trynear

end movenear;
-----------------------------------------------------------



procedure moveaway( kk: in out integer; nowTime: float ) is -- -Z
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid, wasEmpty : boolean;
begin

if not tryaway(kk) then -- pick one that might make sense
	for i in 1..9 loop
		if tryaway(i) then
			kk:=i;
			exit;
		end if;
	end loop;
end if;


if tryaway(kk) then

	append(restoreBuf, integer'image(kk) & " a ");

	oxc:=centroid(kk)(1);
	oyc:=centroid(kk)(2);
	ozc:=centroid(kk)(3);

	sx:=space(kk)(1);
	sy:=space(kk)(2);
	sz:=space(kk)(3);

	ix:=oxc;
	iy:=oyc;
	iz:=ozc;

	leadingVoid := (sz=0);



	if kk=9 then -- (easy)

		-- trailing face:
		empty( ix+0,iy+0,iz+1 ):=true;
		empty( ix+1,iy+0,iz+1 ):=true;
		empty( ix+1,iy+1,iz+1 ):=true;
		empty( ix+0,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(3):=centroid(kk)(3)-1;
		iz:=centroid(kk)(3);

		-- leading face:
		empty( ix+0,iy+0,iz+0 ):=false;
		empty( ix+1,iy+0,iz+0 ):=false;
		empty( ix+1,iy+1,iz+0 ):=false;
		empty( ix+0,iy+1,iz+0 ):=false;

	elsif leadingVoid then

		-- next leading void position:
		wasEmpty := empty( ix+sx,iy+sy,iz-1+sz );

		-- trailing face:
		empty( ix+0,iy+0,iz+1 ):=true;
		empty( ix+1,iy+0,iz+1 ):=true;
		empty( ix+1,iy+1,iz+1 ):=true;
		empty( ix+0,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(3):=centroid(kk)(3)-1;
		iz:=centroid(kk)(3);

		-- leading face:
		empty( ix+0,iy+0,iz+0 ):=false;
		empty( ix+1,iy+0,iz+0 ):=false;
		empty( ix+1,iy+1,iz+0 ):=false;
		empty( ix+0,iy+1,iz+0 ):=false;

		-- new leading void setting...
		-- this corrects one of the previous 4 lines:
		empty( ix+sx,iy+sy,iz+sz ):=wasEmpty;

	else -- trailingVoid

		-- old trailing void status:
		wasEmpty := empty( ix+sx,iy+sy,iz+sz );

		-- trailing face:
		empty( ix+0,iy+0,iz+1 ):=true;
		empty( ix+1,iy+0,iz+1 ):=true;
		empty( ix+1,iy+1,iz+1 ):=true;
		empty( ix+0,iy+1,iz+1 ):=true;

		-- this corrects one of the previous 4 lines:
		empty( ix+sx,iy+sy,iz+sz ):=wasEmpty;

		-- make the move:
		centroid(kk)(3):=centroid(kk)(3)-1;
		iz:=centroid(kk)(3);

		--leading face:
		empty( ix+0,iy+0,iz+0 ):=false;
		empty( ix+1,iy+0,iz+0 ):=false;
		empty( ix+1,iy+1,iz+0 ):=false;
		empty( ix+0,iy+1,iz+0 ):=false;

		-- new trailing void is empty:
		empty( ix+sx,iy+sy,iz+sz ):=true;

	end if;

	test4winner;

	if not mute then 
	snd4ada.playSnd(brick); 
	end if;

	movebegin:=nowTime;
	grinding(kk):=true;

end if; --tryaway


end moveaway;




procedure moveleft( kk: in out integer; nowTime: float ) is -- -X
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid, wasEmpty : boolean;
begin

if not tryleft(kk) then -- pick one that might make sense
	for i in 1..9 loop
		if tryleft(i) then
			kk:=i;
			exit;
		end if;
	end loop;
end if;


if tryleft(kk) then

	append(restoreBuf, integer'image(kk) & " l ");

	oxc:=centroid(kk)(1);
	oyc:=centroid(kk)(2);
	ozc:=centroid(kk)(3);

	sx:=space(kk)(1);
	sy:=space(kk)(2);
	sz:=space(kk)(3);

	ix:=oxc;
	iy:=oyc;
	iz:=ozc;

	leadingVoid := (sx=0);



	if kk=9 then -- (easy)

		-- trailing face:
		empty( ix+1,iy+0,iz+0 ):=true;
		empty( ix+1,iy+0,iz+1 ):=true;
		empty( ix+1,iy+1,iz+0 ):=true;
		empty( ix+1,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(1):=centroid(kk)(1)-1;
		ix:=centroid(kk)(1);

		-- leading face:
		empty( ix+0,iy+0,iz+0 ):=false;
		empty( ix+0,iy+0,iz+1 ):=false;
		empty( ix+0,iy+1,iz+0 ):=false;
		empty( ix+0,iy+1,iz+1 ):=false;

	elsif leadingVoid then

		-- next leading void position:
		wasEmpty := empty( ix-1+sx,iy+sy,iz+sz );

		-- trailing face:
		empty( ix+1,iy+0,iz+0 ):=true;
		empty( ix+1,iy+0,iz+1 ):=true;
		empty( ix+1,iy+1,iz+0 ):=true;
		empty( ix+1,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(1):=centroid(kk)(1)-1;
		ix:=centroid(kk)(1);

		-- leading face:
		empty( ix+0,iy+0,iz+0 ):=false;
		empty( ix+0,iy+0,iz+1 ):=false;
		empty( ix+0,iy+1,iz+0 ):=false;
		empty( ix+0,iy+1,iz+1 ):=false;

		-- new leading void setting...
		-- this corrects one of the previous 4 lines:
		empty( ix+sx,iy+sy,iz+sz ):=wasEmpty;

	else -- trailingVoid

		-- old trailing void status:
		wasEmpty := empty( ix+sx,iy+sy,iz+sz );

		-- trailing face:
		empty( ix+1,iy+0,iz+0 ):=true;
		empty( ix+1,iy+0,iz+1 ):=true;
		empty( ix+1,iy+1,iz+0 ):=true;
		empty( ix+1,iy+1,iz+1 ):=true;

		-- this corrects one of the previous 4 lines:
		empty( ix+sx,iy+sy,iz+sz ):=wasEmpty;

		-- make the move:
		centroid(kk)(1):=centroid(kk)(1)-1;
		ix:=centroid(kk)(1);

		-- leading face:
		empty( ix+0,iy+0,iz+0 ):=false;
		empty( ix+0,iy+0,iz+1 ):=false;
		empty( ix+0,iy+1,iz+0 ):=false;
		empty( ix+0,iy+1,iz+1 ):=false;

		-- new trailing void is empty:
		empty( ix+sx,iy+sy,iz+sz ):=true;

	end if;

	test4winner;

	if not mute then 
	snd4ada.playSnd(brick); 
	end if;

	movebegin:=nowTime;
	grinding(kk):=true;

end if; --tryleft


end moveleft;




procedure moveright( kk: in out integer; nowTime: float ) is -- +X
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid, wasEmpty : boolean;
begin

if not tryright(kk) then -- pick one that might make sense
	for i in 1..9 loop
		if tryright(i) then
			kk:=i;
			exit;
		end if;
	end loop;
end if;


if tryright(kk) then

	append(restoreBuf, integer'image(kk) & " r ");

	oxc:=centroid(kk)(1);
	oyc:=centroid(kk)(2);
	ozc:=centroid(kk)(3);

	sx:=space(kk)(1);
	sy:=space(kk)(2);
	sz:=space(kk)(3);

	ix:=oxc;
	iy:=oyc;
	iz:=ozc;

	leadingVoid := (sx=1);



	if kk=9 then -- (easy)

		empty( ix+0,iy+0,iz+0 ):=true;
		empty( ix+0,iy+0,iz+1 ):=true;
		empty( ix+0,iy+1,iz+0 ):=true;
		empty( ix+0,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(1):=centroid(kk)(1)+1;
		ix:=centroid(kk)(1);

		empty( ix+1,iy+0,iz+0 ):=false;
		empty( ix+1,iy+0,iz+1 ):=false;
		empty( ix+1,iy+1,iz+0 ):=false;
		empty( ix+1,iy+1,iz+1 ):=false;

	elsif leadingVoid then

		-- next leading void position:
		wasEmpty := empty( ix+1+sx,iy+sy,iz+sz );

		empty( ix+0,iy+0,iz+0 ):=true;
		empty( ix+0,iy+0,iz+1 ):=true;
		empty( ix+0,iy+1,iz+0 ):=true;
		empty( ix+0,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(1):=centroid(kk)(1)+1;
		ix:=centroid(kk)(1);

		empty( ix+1,iy+0,iz+0 ):=false;
		empty( ix+1,iy+0,iz+1 ):=false;
		empty( ix+1,iy+1,iz+0 ):=false;
		empty( ix+1,iy+1,iz+1 ):=false;

		-- new leading void setting...
		-- this corrects one of the previous 4 lines:
		empty( ix+sx,iy+sy,iz+sz ):=wasEmpty;

	else -- trailingVoid

		-- old trailing void status:
		wasEmpty := empty( ix+sx,iy+sy,iz+sz );

		empty( ix+0,iy+0,iz+0 ):=true;
		empty( ix+0,iy+0,iz+1 ):=true;
		empty( ix+0,iy+1,iz+0 ):=true;
		empty( ix+0,iy+1,iz+1 ):=true;

		-- this corrects one of the previous 4 lines:
		empty( ix+sx,iy+sy,iz+sz ):=wasEmpty;

		-- make the move:
		centroid(kk)(1):=centroid(kk)(1)+1;
		ix:=centroid(kk)(1);

		empty( ix+1,iy+0,iz+0 ):=false;
		empty( ix+1,iy+0,iz+1 ):=false;
		empty( ix+1,iy+1,iz+0 ):=false;
		empty( ix+1,iy+1,iz+1 ):=false;

		-- new trailing void is empty:
		empty( ix+sx,iy+sy,iz+sz ):=true;

	end if;

	test4winner;

	if not mute then 
	snd4ada.playSnd(brick); 
	end if;

	movebegin:=nowTime;
	grinding(kk):=true;

end if; --tryright


end moveright;






procedure movedown( kk: in out integer; nowTime: float ) is -- -Y
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid, wasEmpty : boolean;
begin

if not trydown(kk) then -- pick one that might make sense
	for i in 1..9 loop
		if trydown(i) then
			kk:=i;
			exit;
		end if;
	end loop;
end if;


if trydown(kk) then

	append(restoreBuf, integer'image(kk) & " d ");

	oxc:=centroid(kk)(1);
	oyc:=centroid(kk)(2);
	ozc:=centroid(kk)(3);

	sx:=space(kk)(1);
	sy:=space(kk)(2);
	sz:=space(kk)(3);

	ix:=oxc;
	iy:=oyc;
	iz:=ozc;

	leadingVoid := (sy=0);



	if kk=9 then -- (easy)

		empty( ix+0,iy+1,iz+0 ):=true;
		empty( ix+0,iy+1,iz+1 ):=true;
		empty( ix+1,iy+1,iz+0 ):=true;
		empty( ix+1,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(2):=centroid(kk)(2)-1;
		iy:=centroid(kk)(2);

		empty( ix+0,iy+0,iz+0 ):=false;
		empty( ix+0,iy+0,iz+1 ):=false;
		empty( ix+1,iy+0,iz+0 ):=false;
		empty( ix+1,iy+0,iz+1 ):=false;

	elsif leadingVoid then

		-- next leading void position:
		wasEmpty := empty( ix+sx,iy-1+sy,iz+sz );

		empty( ix+0,iy+1,iz+0 ):=true;
		empty( ix+0,iy+1,iz+1 ):=true;
		empty( ix+1,iy+1,iz+0 ):=true;
		empty( ix+1,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(2):=centroid(kk)(2)-1;
		iy:=centroid(kk)(2);

		empty( ix+0,iy+0,iz+0 ):=false;
		empty( ix+0,iy+0,iz+1 ):=false;
		empty( ix+1,iy+0,iz+0 ):=false;
		empty( ix+1,iy+0,iz+1 ):=false;

		-- new leading void setting...
		-- this corrects one of the previous 4 lines:
		empty( ix+sx,iy+sy,iz+sz ):=wasEmpty;

	else -- trailingVoid

		-- old trailing void status:
		wasEmpty := empty( ix+sx,iy+sy,iz+sz );

		empty( ix+0,iy+1,iz+0 ):=true;
		empty( ix+0,iy+1,iz+1 ):=true;
		empty( ix+1,iy+1,iz+0 ):=true;
		empty( ix+1,iy+1,iz+1 ):=true;

		-- this corrects one of the previous 4 lines:
		empty( ix+sx,iy+sy,iz+sz ):=wasEmpty;

		-- make the move:
		centroid(kk)(2):=centroid(kk)(2)-1;
		iy:=centroid(kk)(2);

		empty( ix+0,iy+0,iz+0 ):=false;
		empty( ix+0,iy+0,iz+1 ):=false;
		empty( ix+1,iy+0,iz+0 ):=false;
		empty( ix+1,iy+0,iz+1 ):=false;

		-- new trailing void is empty:
		empty( ix+sx,iy+sy,iz+sz ):=true;

	end if;

	test4winner;

	if not mute then 
	snd4ada.playSnd(brick); 
	end if;

	movebegin:=nowTime;
	grinding(kk):=true;

end if; --trydown


end movedown;



procedure moveup( kk: in out integer; nowTime: float ) is -- +Y
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid, wasEmpty : boolean;
begin

if not tryup(kk) then -- pick one that might make sense
	for i in 1..9 loop
		if tryup(i) then
			kk:=i;
			exit;
		end if;
	end loop;
end if;


if tryup(kk) then

	append(restoreBuf, integer'image(kk) & " u ");

	oxc:=centroid(kk)(1);
	oyc:=centroid(kk)(2);
	ozc:=centroid(kk)(3);

	sx:=space(kk)(1);
	sy:=space(kk)(2);
	sz:=space(kk)(3);

	ix:=oxc;
	iy:=oyc;
	iz:=ozc;

	leadingVoid := (sy=1);



	if kk=9 then -- (easy)

		-- trailing face:
		empty( ix+0,iy+0,iz+0 ):=true;
		empty( ix+0,iy+0,iz+1 ):=true;
		empty( ix+1,iy+0,iz+0 ):=true;
		empty( ix+1,iy+0,iz+1 ):=true;

		-- make the move:
		centroid(kk)(2):=centroid(kk)(2)+1;
		iy:=centroid(kk)(2);

		-- leading face:
		empty( ix+0,iy+1,iz+0 ):=false;
		empty( ix+0,iy+1,iz+1 ):=false;
		empty( ix+1,iy+1,iz+0 ):=false;
		empty( ix+1,iy+1,iz+1 ):=false;

	elsif leadingVoid then

		-- next leading void position:
		wasEmpty := empty( ix+sx,iy+1+sy,iz+sz );

		-- trailing face:
		empty( ix+0,iy+0,iz+0 ):=true;
		empty( ix+0,iy+0,iz+1 ):=true;
		empty( ix+1,iy+0,iz+0 ):=true;
		empty( ix+1,iy+0,iz+1 ):=true;

		-- make the move:
		centroid(kk)(2):=centroid(kk)(2)+1;
		iy:=centroid(kk)(2);

		-- leading face:
		empty( ix+0,iy+1,iz+0 ):=false;
		empty( ix+0,iy+1,iz+1 ):=false;
		empty( ix+1,iy+1,iz+0 ):=false;
		empty( ix+1,iy+1,iz+1 ):=false;

		-- new leading void setting...
		-- this corrects one of the previous 4 lines:
		empty( ix+sx,iy+sy,iz+sz ):=wasEmpty;

	else -- trailingVoid

		-- old trailing void status:
		wasEmpty := empty( ix+sx,iy+sy,iz+sz );

		-- trailing face:
		empty( ix+0,iy+0,iz+0 ):=true;
		empty( ix+0,iy+0,iz+1 ):=true;
		empty( ix+1,iy+0,iz+0 ):=true;
		empty( ix+1,iy+0,iz+1 ):=true;

		-- this corrects one of the previous 4 lines:
		empty( ix+sx,iy+sy,iz+sz ):=wasEmpty;

		-- make the move:
		centroid(kk)(2):=centroid(kk)(2)+1;
		iy:=centroid(kk)(2);

		-- leading face:
		empty( ix+0,iy+1,iz+0 ):=false;
		empty( ix+0,iy+1,iz+1 ):=false;
		empty( ix+1,iy+1,iz+0 ):=false;
		empty( ix+1,iy+1,iz+1 ):=false;

		-- new trailing void is empty:
		empty( ix+sx,iy+sy,iz+sz ):=true;

	end if;

	test4winner;

	if not mute then 
	snd4ada.playSnd(brick); 
	end if;

	movebegin:=nowTime;
	grinding(kk):=true;

end if; --tryup


end moveup;














	playedonce, userexit : boolean := false;
	wintime, currentTime : float;

	v4, vcc : vec4;
	--mzndc : float;


	--keydlay : constant float := 0.25;

	otitle : constant interfaces.c.char_array 
		:= value( new_string("Ada7") );
	vtitle : constant interfaces.c.char_array 
		:= value( new_string("Vadasz") );



	ix,iy,iz, sx,sy,sz, cx,cy,cz : integer := 1;

	xize: float := 0.25;


	-- I shall first attempt to treat OS-X the same as GNU-Linux
	-- given that like-binaries will be together in same directory:
	--dirname: constant string := ada.directories.current_directory;
	--nexname: constant string := dirname & "/reliquarium";
	--Ok: boolean;

	ixsel: integer:=1;
	pik: boolean := false;

	hc: boolean := true; --HighContrast

	rfile: file_type;
	char: character;


	procedure shutdown is
	begin

		text_io.create(rfile,out_file,"t6restore.txt");
		put( rfile, restoreBuf );
		put( rfile, " -1 ");
		new_line(rfile);
		text_io.close(rfile);
		put_line(" Saved config to t6restore.txt");

		glext.binding.glDeleteProgram(pgmtexshadid);

		glext.binding.glDeleteBuffers(1, vertbuff'address);
		glext.binding.glDeleteBuffers(1, rgbbuff'address);
		glext.binding.glDeleteBuffers(1, elembuff'address);
		glext.binding.glDeleteBuffers(1, uvbuff'address);

		glext.binding.glDeleteVertexArrays(1, vertexarrayid'address);

		stex.CloseFont;

		snd4ada.termSnds;

		glfwdestroywindow(mainWin);
		glfwTerminate;

	end shutdown;



	procedure replay is
	begin

		for i in 1..nmoves loop
			ixsel:= sel(i);
			char := dir(i);

			currentTime := float(glfwGetTime);

			case char is
				when 'n' => movenear(ixsel,currentTime);
				when 'a' => moveaway(ixsel,currentTime);
				when 'u' => moveup(ixsel,currentTime);
				when 'd' => movedown(ixsel,currentTime);
				when 'l' => moveleft(ixsel,currentTime);
				when 'r' => moveright(ixsel,currentTime);
				when others => null;
			end case;
		end loop;

	end replay;



----------------- main program begin ==========================
begin --tomb

	checkfile; --see if restore file exists; sets preRestoreBuf


normalMode:=false; --shows goal cfg
	first_prep;  -- main program setup

	show_axes:=true;



	-- rotate into preferred initial orientation:
	degRotate( mm, 10.0, 1.0, 0.0, 0.0 );




	-- note:  only mm changes:
	updateMVP( float(wwid), float(whit) );






	-- main event loop begin: ------------------------------------------
   while not userexit loop

------- begin response to keys ------------------------------------------

		currentTime := float(glfwGetTime);
		--deltaT := currentTime - oldTimeKb;


		glfwPollEvents;



		--check for <esc>,<q>:
		if glfwgetkey( mainWin, glfw_key_escape ) = Glfw_Press then
				userexit:=true;

		elsif glfwgetkey( mainWin, glfw_key_q ) = Glfw_Press then
				userexit:=true;

		end if;

		exit when glfwWindowShouldClose(mainWin) /= 0; --19oct21


		if not normalMode then
			if glfwgetkey( mainWin, glfw_key_equal ) = Glfw_Press then
				normalMode:=true;
				centroid:=centroidMid;
				initEmptyMid;
				--centroid:=centroidHi;
				--initEmptyHi;
				if playscript then
					replay;
				end if;
		
				set_unbounded_string(restoreBuf, to_string(prerestoreBuf) );

			end if;
		end if;





		anyGrinding:=false;
		for i in rngp loop
			anyGrinding := anyGrinding or grinding(i);
		end loop;

		if not anyGrinding then

			if normalMode then
				gameutils.getKeyInputs(
					mainWin,
					rngp'first, rngp'last,
					ixsel,pik);
			end if;

			if goUp then
				goUp:=false;
				moveUp(ixsel,currentTime);
			elsif goDown then
				goDown:=false;
				moveDown(ixsel,currentTime);
			elsif goLeft then
				goLeft:=false;
				moveLeft(ixsel,currentTime);
			elsif goRight then
				goRight:=false;
				moveRight(ixsel,currentTime);
			elsif goAway then
				goAway:=false;
				moveAway(ixsel,currentTime);
			elsif goNear then
				goNear:=false;
				moveNear(ixsel,currentTime);
			elsif doRestart then
				set_unbounded_string(restoreBuf, "");
				doRestart:=false;
				centroid:=centroidMid;
				initEmptyMid;
			elsif tryNextSkin then
				tryNextSkin:=false;
				setNextSkin;
			end if;


			gameutils.handle_mouse(
				pik, 
				rngp'first, rngp'last, 
				ixsel,
				centroid,
				xxcc,yycc,zzcc
				);
		end if;





----///////////////////// end response to key/mouse


		updateMVP( float(wwid), float(whit) );


-------- here we should handle resized window ----------------------


		glfwGetWindowSize( mainWin, Nwid'access, Nhit'access );
		if( (Nwid /= wwid) or (Nhit /= whit) ) then
			wwid:=Nwid;  whit:=Nhit;

			glfwGetFramebufferSize(mainwin, fwid'access, fhit'access);
			glViewport(0,0,Fwid,Fhit);

		end if;



--------- begin drawing ===============================================

		glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);


	if not help then --normal draw 23jun23



------ draw here ----------------------

		-- now, the 8 puzzle pieces:
		glUseProgram( pgmTexShadID );
		gluniformmatrix4fv( matrixid, 1, gl_false, mvp(1,1)'address );
		gluniform1i(uniftex,0);

		for ii in rngp loop -- 8 corners + 1 head


			if numerals then
				glBindTexture(gl_texture_2d, num_texid(ii) );
			else
				glBindTexture(gl_texture_2d, nun_texid );
			end if;



			ix:=centroid(ii)(1);
			iy:=centroid(ii)(2);
			iz:=centroid(ii)(3);

			if ii = rngp'last then -- crystal skull

				--draw9(ppo); --medusa here
				null;

			else -- one of 8 corner pieces

				sx:=space(ii)(1);
				sy:=space(ii)(2);
				sz:=space(ii)(3);

				cx:=1-sx;
				cy:=1-sy;
				cz:=1-sz; --corner cubelet indexes


				for i in 0..1 loop
				for j in 0..1 loop
				for k in 0..1 loop
					-- this is a 7-cubelet-corner:
					if i=sx and j=sy and k=sz then
						null;
					else
						drawCube(ppo, ii, cx,cy,cz, i,j,k, ix, iy, iz, currentTime );
					end if;
				end loop;
				end loop;
				end loop;

			end if;

		end loop; --ii


		glUseProgram( pgmTexID );
		gluniformmatrix4fv( matid, 1, gl_false, mvp(1,1)'address );
		gluniform1i(unitex,0);

		-- first, the bounding chainlink box:
		if not winner then
			glBindTexture(gl_texture_2d, box_texid ); 
			drawBigCube(ppo2);
		end if;

		-- draw medusa's head here 
		-- (using the current shaders that are more suitable)
		glBindTexture(gl_texture_2d, num_texid(9) ); 
		draw9(ppo, currentTime);






		if show_axes then

			v4 := (+2.2, -3.1, -2.0, 1.0);
			matXvec(mvp, v4, vcc);
			stex.print3d("+X", vcc(1), vcc(2), vcc(3), vcc(4), 1.0, 1.0);

			v4 := (-2.2, +3.3, -2.2, 1.0);
			matXvec(mvp, v4, vcc);
			stex.print3d("+Y", vcc(1), vcc(2), vcc(3), vcc(4), 1.0, 1.0);

			v4 := (-2.0, -3.1, +2.0, 1.0);
			matXvec(mvp, v4, vcc);
			stex.print3d("+Z", vcc(1), vcc(2), vcc(3), vcc(4), 1.0, 1.0);

			v4 := (-2.0, -3.0, -2.0, 1.0);
			matXvec(mvp, v4, vcc);
			stex.print3d("O", vcc(1), vcc(2), vcc(3), vcc(4), 1.0, 1.0);

		end if;



		xize:=0.25;

		-- fixed 2d location text:

		stex.print2d("(spc)=>Restart", 0.02, 0.05, xize );
		stex.print2d("(i,o,mousewheel)=>Zoom", 0.36, 0.05, xize);
		stex.print2d("(esc)=>exit",    0.78, 0.05, xize );
		stex.print2d("1-8=>pick",      0.02, 0.02, xize );
		stex.print2d("m=>Medusa",        0.20, 0.02, xize );
		stex.print2d("n=>nums",        0.40, 0.02, xize ); --8oct21
		stex.print2d("c=>nextColor",        0.58, 0.02, xize );
		stex.print2d("h=>help",        0.85, 0.02, xize );


	end if; --not help (normal draw) 23jun23



		if help then

			xize:=0.4; hc:=false;
			stex.print2d("Reckless raiders have plundered a reliquary",0.02,0.90,xize,hc);

			stex.print2d("...and exposed the head of Medusa !",0.02,0.85,xize,hc);
			stex.print2d("Return the head to its proper place", 0.02, 0.80, xize,hc);

			stex.print2d("Completely hidden within its Cubical Tomb.", 0.02, 0.75, xize,hc);

			stex.print2d("With X rightward, Y upward, and Z outward,", 0.02, 0.70, xize,hc);
			stex.print2d("use the keys:  u/d  l/r  f/b", 0.02, 0.65, xize,hc);
			stex.print2d("(Up/Down Left/Right Forward/Backward)", 0.02, 0.60, xize,hc);

			stex.print2d("Select a block by number or mouse-click...", 0.02, 0.50, xize,hc);
			stex.print2d("Laptop users can place cursor on block", 0.02, 0.45, xize,hc);
			stex.print2d("then hit <enter>-key to select", 0.02, 0.40, xize,hc);

			stex.print2d("Zoom keys:   i/o = In/Out", 0.02, 0.35, xize,hc);

			stex.print2d("V-key to mute moves", 0.02, 0.30, xize,hc);


			xize:=0.25;
		else
			if normalMode then
			stex.print2d("Easy:   Hide Medusa within her cubical Tomb.", 0.15,0.96, xize );
			stex.print2d("WARNING:  Never look directly at Medusa!", 0.15,0.92, xize );
			else
			stex.print2d("This is the goal configuration...Press  =  to begin",0.10,0.95,xize);
			end if;

		end if;

		--if winner and (currentTime-testime>windelay) then
		if winner then
			stex.print2d("Correct!", 0.19, 0.83, 1.0 );

			if not playedonce then
		 		snd4ada.playSnd(shriek); --shriek
				playedonce:=true;
				wintime:=currentTime;
			end if;

			if( currentTime-wintime > 5.0 ) then 
				winner:=false;
			end if;

		else
			playedonce:=false;
		end if;


--------- end drawing =================================================



		glflush;
		glfwSwapBuffers( mainWin );


   end loop; ---------------- main event loop end -------------------

	shutdown;


exception
	when others=>
		shutdown;

end tomb6;



