

--
-- Copyright (C) 2022  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--


-- 4tomb.adb ( 4x4x4 )
-- 19jan16: changing corners 7..8 to be simple, 
-- vertical legs : 1x2x1
-- to a table-shaped enclosure for the skull
-- to see if the result is a) solvable, and b) challenging.
--
-- Note:  centroids of 7,8 reference lower rearward cell
--
-- 4mar16:  WAIT!!!
-- ...discovered that I can make 7,8 into L-shaped pieces with
-- rearward projections and still solvable;  BUT starting
-- position must switch 7, 8.
--
-- 6mar16:  WAIT !!!
-- ...let's add two 1x1 blocks along the bottom front to fill in the space.
--


with snd4ada;

with gl, gl.binding, gl.pointers;
with glu, glu.binding, glu.pointers;
with glext, glext.binding, glext.pointers;

-------------------------------------------------------------
with System;
with Interfaces.C;
use  type interfaces.c.unsigned;
with Interfaces.C.Pointers;
with interfaces.c.strings;


----------------------------------------------------------------

with glfw3;		use glfw3;
with zoomwheel; use zoomwheel;

----------------------------------------------------------------

with matutils;
with stex;

with ada.numerics.generic_elementary_functions;

with Ada.Strings.Fixed;
with Ada.Strings.Unbounded;
use Ada.Strings.Unbounded;


with unchecked_deallocation;

with text_io;
with pngloader;
with gametypes;
with matutils;

with pictobj;
with pictobj2;

with shader;  use shader;



with Ada.Command_Line;
with Ada.Directories;
with SysUtils;

with gameutils; use gameutils;

----------------------------------------------------------------




procedure tomb4 is -- 4x4x4 with 2x2x2 skull

	dirname: constant string := ada.directories.current_directory;
	onMac : boolean := (ada.strings.fixed.index( dirname, "MacOS", 1 ) > 1);
	onLinux : boolean := (ada.strings.fixed.index( dirname, "gnu", 1 ) > 1);


------- begin game specific code -------------------

mx, mz : constant integer := 4;
my     : constant integer := 4;


ncubes  : constant integer := mx*my*mz; -- 96

subtype rngm is integer range 1..ncubes;


empty : array(1..mx,1..my,1..mz) of boolean;



-- overall object drawing centroid (move to origin):
xxcc : constant float := float(mx)/2.0;
yycc : constant float := float(my)/2.0;
zzcc : constant float := float(mz)/2.0;

------- end game specific code -------------------



use text_io;
use gametypes;
use pngloader;
use matutils;
use gametypes.fmath;

use interfaces.c;
use interfaces.c.strings;
use glext;
use glext.pointers;
use glext.binding;
use gl;
use gl.binding;
use gl.pointers;


	origskin : boolean := true; -- false => Vadasz colors







------------------------------------------------------------------





procedure initEmptyMid is
begin

	for ix in 1..mx loop
		for iz in 1..mz loop
			for iy in 1..my loop
				empty(ix,iy,iz) := false;
			end loop;
		end loop;
	end loop;


	-- middle of each face is blank:

	-- bottom face:
	for ix in 2..3 loop
	for iz in 2..3 loop
		empty(ix,1,iz):=true;
		empty(ix,1,iz):=true;
	end loop;
	end loop;

	-- top face:
	for ix in 2..3 loop
	for iz in 2..3 loop
		empty(ix,4,iz):=true;
		empty(ix,4,iz):=true;
	end loop;
	end loop;

	-- front face:
	for ix in 2..3 loop
	for iy in 2..3 loop
		empty(ix,iy,mz):=true;
		empty(ix,iy,mz):=true;
	end loop;
	end loop;

	-- rear face:
	for ix in 2..3 loop
	for iy in 2..3 loop
		empty(ix,iy,1):=true;
		empty(ix,iy,1):=true;
	end loop;
	end loop;

	-- left face:
	for iz in 2..3 loop
	for iy in 2..3 loop
		empty(1,iy,iz):=true;
		empty(1,iy,iz):=true;
	end loop;
	end loop;

	-- right face:
	for iz in 2..3 loop
	for iy in 2..3 loop
		empty(mx,iy,iz):=true;
		empty(mx,iy,iz):=true;
	end loop;
	end loop;


end initEmptyMid;









-- normal staring configuration:
procedure initEmptyHi is -- skull is on TOP
begin

	for ix in 1..mx loop
		for iz in 1..mz loop
			for iy in 1..my loop
				empty(ix,iy,iz) := false;
			end loop;
		end loop;
	end loop;


	-- edges of top 2 layers is blank:
	for ix in 1..mx loop
	for iz in 1..mz loop
	if 
		(ix/=2 and ix/=3) or (iz/=2 and iz/=3)
	then
		empty(ix,4,iz):=true;
		empty(ix,3,iz):=true;
	end if;
	end loop;
	end loop;


end initEmptyHi;








-- define 11 puzzle pieces here, and initialize:
mp : constant integer := 10;
subtype rngp is integer range 0..mp;
subtype fparray is gfparray(rngp);
subtype iparray is giparray(rngp);



movedelay : constant float := 0.25; -- seconds to move a block
movebegin : float := 0.0;
oxc,oyc,ozc : integer;
grinding : array(rngp) of boolean := ( others=>false );
anyGrinding: boolean := false;

nun_texid : gluint;
num_texid : array(rngp) of gluint;
box_texid : gluint;



-- normal starting config:
centroidHi : constant iparray :=
-- piece-centroids initialization:
( 
		0=>(1,1,1),
		1=>(3,1,1),
		2=>(3,1,3),
		3=>(1,1,3),

		4=>(1,1,3),
		5=>(3,1,3),

		6=>(3,1,1),

		7=>(4,1,2), -- 1x1 block
		8=>(1,1,2), -- 1x1 block

		9=>(2,1,1),

		10=>(2,3,2)  --to be entombed in center
); --------------------------------------------------------------

-- solved configuration:
centroidMid : constant iparray :=
-- piece-centroids initialization:
( 
		0=>(1,3,1), --ulp (UpperLeftPosterior)
		1=>(3,3,1), --urp
		2=>(1,3,3), --ula
		3=>(3,3,3), --ura (UpperRightAnterior)

		4=>(1,1,1), --llp
		5=>(3,1,1), --lrp

		6=>(1,1,3), --lla

		7=>(2,1,4), -- 1x1 block
		8=>(3,1,4), -- 1x1 block

		9=>(4,1,3), --lra

		10=>(2,2,2)  --[skull=s] to be entombed in center
);



--Note that for each normal piece, its full envelope is:
-- (p.x,p.y,p.z)...(p.x+1,p.y+1,p.z+1)
--
-- so each coord is in 1..3
--
-- For 6,9: x is in 1..4
--
-- For 7,8: x,y,z is in 1..4

centroid : iparray;



--offsets from piece-centroid to space-centroid:
space : constant iparray :=
( 
		0=>(1,0,1), --ulp
		1=>(0,0,1), --urp
		2=>(1,0,0), --ula
		3=>(0,0,0), --ura

		4=>(1,1,1), --llp
		5=>(0,1,1), --lrp

----------- unused below here --------------

		6=>(1,1,0), --lla

		7=>(0,0,0),
		8=>(0,0,0),

		9=>(0,1,0), --lra

		10=>(0,0,0)  --unused
);






normalMode : boolean := true;
windelay : constant float := 1.0;
testime : float := 0.0;


procedure test4winner is
begin

	winner := (centroid=centroidMid);

	if winner then
		testime := float(glfwGetTime);
	end if;

end test4winner;









mxskin: constant integer := 4;
skin: integer := mxskin-1;

procedure setNextSkin is
	macpath: string := "../../../data/"; -- osx
	xinpath: string := "../../data/"; -- windows or linux
	nopath: string := "./data/"; --below current dir
	path: unbounded_string;
begin

	if onmac then
		append(path,macpath);
	elsif ada.directories.exists(nopath) then
		append(path,nopath);
	else
		append(path,xinpath);
	end if;



	skin := (1+skin) mod mxskin; --0..3
	myassert( (skin < mxskin) and (skin>=0) );

	if skin=0 then --Original colors

		nun_texid    := loadPng(mirror,to_string(path)&"none.png");

		num_texid(0) := loadPng(mirror,to_string(path)&"zero.png");
		num_texid(1) := loadPng(mirror,to_string(path)&"one.png");
		num_texid(2) := loadPng(mirror,to_string(path)&"two.png");
		num_texid(3) := loadPng(mirror,to_string(path)&"three.png");
		num_texid(4) := loadPng(mirror,to_string(path)&"four.png");
		num_texid(5) := loadPng(mirror,to_string(path)&"five.png");
		num_texid(6) := loadPng(mirror,to_string(path)&"six.png");
		num_texid(7) := loadPng(mirror,to_string(path)&"seven.png");
		num_texid(8) := loadPng(mirror,to_string(path)&"eight.png");
		num_texid(9) := loadPng(mirror,to_string(path)&"nine.png");

	elsif skin=1 then --wornRock

		nun_texid    := loadPng(mirror,to_string(path)&"worn.png");

		num_texid(0) := loadPng(mirror,to_string(path)&"worn0.png");
		num_texid(1) := loadPng(mirror,to_string(path)&"worn1.png");
		num_texid(2) := loadPng(mirror,to_string(path)&"worn2.png");
		num_texid(3) := loadPng(mirror,to_string(path)&"worn3.png");
		num_texid(4) := loadPng(mirror,to_string(path)&"worn4.png");

		num_texid(5) := loadPng(mirror,to_string(path)&"worn5.png");
		num_texid(6) := loadPng(mirror,to_string(path)&"worn6.png");
		num_texid(7) := loadPng(mirror,to_string(path)&"worn7.png");
		num_texid(8) := loadPng(mirror,to_string(path)&"worn8.png");
		num_texid(9) := loadPng(mirror,to_string(path)&"worn9.png");

	elsif skin=2 then --wood

		nun_texid    := loadPng(mirror,to_string(path)&"wood.png");

		num_texid(0) := loadPng(mirror,to_string(path)&"w0.png");
		num_texid(1) := loadPng(mirror,to_string(path)&"w1.png");
		num_texid(2) := loadPng(mirror,to_string(path)&"w2.png");
		num_texid(3) := loadPng(mirror,to_string(path)&"w3.png");
		num_texid(4) := loadPng(mirror,to_string(path)&"w4.png");

		num_texid(5) := loadPng(mirror,to_string(path)&"w5.png");
		num_texid(6) := loadPng(mirror,to_string(path)&"w6.png");
		num_texid(7) := loadPng(mirror,to_string(path)&"w7.png");
		num_texid(8) := loadPng(mirror,to_string(path)&"w8.png");
		num_texid(9) := loadPng(mirror,to_string(path)&"w9.png");

	elsif skin=3 then --granite

		nun_texid    := loadPng(mirror,to_string(path)&"gran.png");

		num_texid(0) := loadPng(mirror,to_string(path)&"g0.png");
		num_texid(1) := loadPng(mirror,to_string(path)&"g1.png");
		num_texid(2) := loadPng(mirror,to_string(path)&"g2.png");
		num_texid(3) := loadPng(mirror,to_string(path)&"g3.png");
		num_texid(4) := loadPng(mirror,to_string(path)&"g4.png");

		num_texid(5) := loadPng(mirror,to_string(path)&"g5.png");
		num_texid(6) := loadPng(mirror,to_string(path)&"g6.png");
		num_texid(7) := loadPng(mirror,to_string(path)&"g7.png");
		num_texid(8) := loadPng(mirror,to_string(path)&"g8.png");
		num_texid(9) := loadPng(mirror,to_string(path)&"g9.png");

	end if;

end setNextSkin;








procedure first_prep is -- main program setup
	macpath: string := "../../../data/"; -- osx
	xinpath: string := "../../data/"; -- windows or linux
	nopath: string := "./data/"; --below current dir
	path: unbounded_string;
	fontcol: constant vec4 := (0.0,0.0,0.0,1.0); --black

	fontfile: constant string := "data/NotoSans-Regular.ttf";
	annex: constant string := "../../";
	nerr: integer;
begin


	if onmac then
		append(path,macpath);
	elsif ada.directories.exists(nopath) then
		append(path,nopath);
	else
		append(path,xinpath);
	end if;

	snd4ada.initSnds;

	brick := snd4ada.initSnd(
		Interfaces.C.Strings.New_String(to_string(path)&"concrete_third.wav"));
	
	whoosh := snd4ada.initSnd(
		Interfaces.C.Strings.New_String(to_string(path)&"whoosh_4th.wav"));
	
	fanfare := snd4ada.initSnd(
		Interfaces.C.Strings.New_String(to_string(path)&"fanfare.wav"));
	
	shriek := snd4ada.initSnd(
		Interfaces.C.Strings.New_String(to_string(path)&"medusa.wav"));
	
	clap := snd4ada.initSnd(
		Interfaces.C.Strings.New_String(to_string(path)&"applause.wav"));
	
	if brick<0 or whoosh<0 or fanfare<0 or shriek<0 or clap<0 then
		put_line("snd4ada.initSnds ERROR");
		raise program_error;
	end if;


------- begin GLFW prep ---------------------------------------------------

	gameutils.InitGlfw( wwid,whit,fwid,fhit,"Reliquary4 - CrystalSkull");
	zoomwheel.enable(mainWin);

	-- prepare font -------------
	if ada.directories.exists(fontfile) then
		stex.InitFont ( fontfile );
	else
		stex.InitFont ( annex&fontfile );
	end if;

	stex.setColor( fontcol );
	stex.reSize(wwid,whit);


	glViewport(0,0,Fwid,Fhit);



	glgenvertexarrays(1, vertexarrayid'address );
	glbindvertexarray(vertexarrayid);

	glactivetexture(gl_texture0); -- moved here 5nov14 (outside main loop)

	glgenbuffers(1, vertbuff'address);
	glgenbuffers(1, rgbbuff'address);
	glgenbuffers(1, uvbuff'address);
	glgenbuffers(1, elembuff'address);


	glenable(gl_depth_test);
	gldepthfunc( gl_lequal );
	glenable( gl_cull_face );


	-- theoretically reduces aliasing (can't tell for sure):
	glEnable(GL_MULTISAMPLE);
	glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
	glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);

	glClearColor(0.4,0.4,0.4,1.0);

	nerr:=dumpGLerrorQueue("main 1"); 
	--prevents misleading messages in pngloader or loadshaders



	box_texid := loadPng(mirror,to_string(path)&"fence.png");
	setNextSkin;
	num_texid(10) := loadPng(mirror,to_string(path)&"tskull.png");

	wall_texid := loadPng(mirror,to_string(path)&"granite.png");


	pgmTexShadID := loadshaders(to_string(path)&"texobj.vs",to_string(path)&"texobj.fs");
	matrixid := glgetuniformlocation(pgmtexshadid, pmvp);
	uniftex  := glgetuniformlocation(pgmtexshadid, pmyts);

	pgmTexID := loadshaders(to_string(path)&"otexobj.vs",to_string(path)&"otexobj.fs");
	matid := glgetuniformlocation(pgmtexid, pmvp);
	unitex  := glgetuniformlocation(pgmtexid, pmyts);


	-- note:  deep copies of structures are automatic in Ada
	--        (unless we use pointer assignments)
	if normalMode then
	centroid:=centroidHi;
	initEmptyHi;
	else
	centroid:=centroidMid;
	initEmptyMid;
	end if;

end first_prep;















-- the enclosing chainlink box:
procedure drawBigCube( po: pictobj2.pictangle ) is

	xr, zr : constant float := 2.02;
	    yr : constant float := 2.02;

	xc : constant float := xr - xxcc;
	yc : constant float := yr - yycc;
	zc : constant float := zr - zzcc;

	r,g,b : constant float := 0.5;
begin
	pictobj2.setrect( po, xc,yc,zc, xr,yr,zr, r,g,b);
	pictobj2.draw( po, vertbuff, uvbuff, elembuff );
end drawBigCube;












-- this is the 10th puzzle piece...a skull double size: 2x2x2
procedure drawSkull( po: pictobj.pictangle; nowTime: float ) is
	tt, r,g,b, xc,yc,zc, xo,yo,zo, xt,yt,zt : float;
	ix,iy,iz: integer;
begin

	ix:=centroid(10)(1);
	iy:=centroid(10)(2);
	iz:=centroid(10)(3);

	xc:=float(ix)-xxcc;
	yc:=float(iy)-yycc;
	zc:=float(iz)-zzcc;

	xo:=float(oxc)-xxcc;
	yo:=float(oyc)-yycc;
	zo:=float(ozc)-zzcc;

	r:=0.6; g:=0.6; b:=0.6;

	if grinding(10) then
		tt := (nowTime-movebegin)/movedelay;
		if tt>1.0 then
			pictobj.setrect( po, xc,yc,zc, 0.95,0.95,0.95, r,g,b);
			grinding(10):=false;
		else
			xt := xo + tt*(xc-xo);
			yt := yo + tt*(yc-yo);
			zt := zo + tt*(zc-zo);
			pictobj.setrect( po, xt,yt,zt, 0.95,0.95,0.95, r,g,b);
		end if;
	else
		pictobj.setrect( po, xc,yc,zc, 0.95,0.95,0.95, r,g,b);
	end if;


	--pictobj.draw( po, vertbuff, rgbbuff, uvbuff, elembuff );
	pictobj.draw( po, vertbuff, uvbuff, elembuff );

end drawSkull;










-- these are individual 1x1 parts of each of corner pieces,
-- where the 6/8 corners have 4 cubelets each, corners 7,8 have 3 cubelets each.
procedure drawCube( 
	po: pictobj.pictangle; ii, 
	cx,cy,cz, i,j,k, ixc,iyc,izc : integer; nowTime: float ) is

	-- (xxcc,yycc,zzcc)==(2,3,2) : centroid of whole puzzle...

	-- (ixc,iyc,izc) = centroid of the current puzzle piece = PP,
	-- (i,j,k) = offset from PP-centroid to current cubelet
	-- (cx,cy,cz) = offset from PP-centroid to corner cubelet


	xc : float := float(ixc+i) -0.5 - xxcc;
	yc : float := float(iyc+j) -0.5 - yycc;
	zc : float := float(izc+k) -0.5 - zzcc;

	xo : float := float(oxc+i) -0.5 - xxcc;
	yo : float := float(oyc+j) -0.5 - yycc;
	zo : float := float(ozc+k) -0.5 - zzcc;

----------- begin addendum -------------------------

	-- cubelet centroid before offset to puzzle center:
	xcc : float := float(ixc+i) -0.5;
	ycc : float := float(iyc+j) -0.5;
	zcc : float := float(izc+k) -0.5;

	fr : float := 0.06; -- move cubelet-center toward corner cubelet
	dx : float := fr*(float(ixc+cx) - xcc);
	dy : float := fr*(float(iyc+cy) - ycc);
	dz : float := fr*(float(izc+cz) - zcc);

	-- this is to slightly separate the puzzle pieces
	-- but not their constituent cubelets.

----------- end addendum -------------------------


	sc: float := 0.47; --reduce radius from 0.5


	r,g,b, tt, xt,yt,zt : float;

begin

	if    ii=0 then
		r:=0.2; g:=1.0; b:=0.2; --green

	elsif ii=1 then
		r:=1.0; g:=1.0; b:=0.2; -- yellow

	elsif ii=2 then
		r:=0.2; g:=1.0; b:=1.0; --cyan

	elsif ii=3 then
		r:=1.0; g:=1.0; b:=1.0; --white

	elsif ii=4 then
		r:=0.2; g:=0.2; b:=0.2; --gray

	elsif ii=5 then
		r:=1.0; g:=0.2; b:=0.3; --red

	elsif ii=6 then --front lower left
		r:=0.2; g:=0.2; b:=1.0; -- blue

	elsif ii=7 then -- 1x1
		r:=0.2; g:=0.2; b:=1.0; --blue

	elsif ii=8 then -- 1x1
		r:=1.0; g:=0.2; b:=1.0; --magenta

	elsif ii=9 then -- front lower right
		r:=1.0; g:=0.2; b:=1.0; --magenta


	elsif ii=10 then -- skull
		r:=0.6; g:=0.6; b:=0.6; -- light gray
	end if;


	if grinding(ii) then
		tt := (nowTime-movebegin)/movedelay;
		if tt>1.0 then
			pictobj.setrect( po, xc+dx,yc+dy,zc+dz, sc,sc,sc, r,g,b);
			grinding(ii):=false;
		else
			xt := xo + tt*(xc-xo);
			yt := yo + tt*(yc-yo);
			zt := zo + tt*(zc-zo);
			pictobj.setrect( po, xt+dx,yt+dy,zt+dz, sc,sc,sc, r,g,b);
		end if;
	else
		pictobj.setrect( po, xc+dx,yc+dy,zc+dz, sc,sc,sc, r,g,b);
	end if;


	pictobj.draw( po, vertbuff, rgbbuff, uvbuff, elembuff );

end drawCube;






--------------------------------------------------------------------------

type vtype is array(0..1,0..1,0..1) of boolean;
wake, probe : vtype;

function trynear(ii: rngp) return boolean is -- +Z
	ix,iy,iz : integer;
	ox,oy,oz : integer;
	sx,sy,sz : integer;
	ok, leadingVoid : boolean;
begin
	ox:=centroid(ii)(1);
	oy:=centroid(ii)(2);
	oz:=centroid(ii)(3);

	sx:=space(ii)(1);
	sy:=space(ii)(2);
	sz:=space(ii)(3);

	leadingVoid := (sz=1);


	for i in 0..1 loop
	for j in 0..1 loop
	for k in 0..1 loop
	if leadingVoid then -- (sz=1)

		if    i=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then probe(i,j,k):=true;
		else probe(i,j,k):=false;
		end if;

	else -- trailing void (sz=0)

		if   i/=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then probe(i,j,k):=true;
		else probe(i,j,k):=false;
		end if;

	end if;
	end loop;
	end loop;
	end loop;



	-- next centroid:
	ix:=ox;
	iy:=oy;
	iz:=oz+1;

	-- next test location
	--iz:=iz+1;

	if iz>mz then
		return false;

	elsif ii<=5 and iz+1<=mz then

		ok:=true;
		for i in 0..1 loop
		for j in 0..1 loop
		for k in 0..1 loop
			if probe(i,j,k) and not empty(ix+i,iy+j,iz+k) then
				ok:=false;
			end if;
		end loop;
		end loop;
		end loop;

		return ok;

	elsif ii=10 and iz+1<=mz then

		if
			empty( ix+0,iy+0,iz+1 ) and
			empty( ix+1,iy+0,iz+1 ) and
			empty( ix+1,iy+1,iz+1 ) and
			empty( ix+0,iy+1,iz+1 )
		then
			return true;
		else
			return false;
		end if;


	elsif ii=6 and iz+1<=mz then

		if
			empty( ix,iy+0,iz+1 ) and
			empty( ix,iy+1,iz+1 )
		then
			return true;
		else
			return false;
		end if;

	elsif ii=9 and iz+1<=mz then

		if
			empty( ix,iy+0,iz+1 ) and
			empty( ix,iy+1,iz+1 )
		then
			return true;
		else
			return false;
		end if;


	elsif ii=7 or ii=8 then
		if empty(ix,iy,iz) then
			return true;
		else
			return false;
		end if;


	else
		return false;

	end if;

end trynear;









function tryaway(ii: rngp) return boolean is -- -Z
	ix,iy,iz : integer;
	ox,oy,oz : integer;
	sx,sy,sz : integer;
	ok, leadingVoid : boolean;
begin
	ox:=centroid(ii)(1);
	oy:=centroid(ii)(2);
	oz:=centroid(ii)(3);

	sx:=space(ii)(1);
	sy:=space(ii)(2);
	sz:=space(ii)(3);

	leadingVoid := (sz=0);



	for i in 0..1 loop
	for j in 0..1 loop
	for k in 0..1 loop
	if leadingVoid then -- (sz=0)

		if    i=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then probe(i,j,k):=true;
		else probe(i,j,k):=false;
		end if;

	else -- trailing void (sz=1)

		if   i/=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then probe(i,j,k):=true;
		else probe(i,j,k):=false;
		end if;

	end if;
	end loop;
	end loop;
	end loop;




	-- next centroid:
	ix:=ox;
	iy:=oy;
	iz:=oz-1;

	-- next test location
	--iz:=iz+0;


	if iz<1 then
		return false;

	elsif ii<=5  then

		ok:=true;
		for i in 0..1 loop
		for j in 0..1 loop
		for k in 0..1 loop
			if probe(i,j,k) and not empty(ix+i,iy+j,iz+k) then
				ok:=false;
			end if;
		end loop;
		end loop;
		end loop;

		return ok;

	elsif ii=10 then

		if
			empty( ix+0,iy+0,iz ) and
			empty( ix+1,iy+0,iz ) and
			empty( ix+1,iy+1,iz ) and
			empty( ix+0,iy+1,iz )
		then
			return true;
		else
			return false;
		end if;


	elsif ii=6 then

		if
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+0 )
		then
			return true;
		else
			return false;
		end if;

	elsif ii=9 then

		if
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+0 )
		then
			return true;
		else
			return false;
		end if;


	elsif ii=7 or ii=8 then
		if empty(ix,iy,iz) then
			return true;
		else
			return false;
		end if;


	else
		return false;


	end if;

end tryaway;














function tryup(ii: rngp) return boolean is -- +Y
	ix,iy,iz : integer;
	ox,oy,oz : integer;
	sx,sy,sz : integer;
	ok, leadingVoid : boolean;
begin
	ox:=centroid(ii)(1);
	oy:=centroid(ii)(2);
	oz:=centroid(ii)(3);

	sx:=space(ii)(1);
	sy:=space(ii)(2);
	sz:=space(ii)(3);

	leadingVoid := (sy=1);



	for i in 0..1 loop
	for j in 0..1 loop
	for k in 0..1 loop
	if leadingVoid then -- (sy=1)

		if    i=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then probe(i,j,k):=true;
		else probe(i,j,k):=false;
		end if;

	else -- trailing void (sy=0)

		if   i/=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then probe(i,j,k):=true;
		else probe(i,j,k):=false;
		end if;

	end if;
	end loop;
	end loop;
	end loop;




	-- next centroid:
	ix:=ox;
	iy:=oy+1;
	iz:=oz;

	-- next test location
	--iy:=iy+1;


	if iy+1>my and ii/=7 and ii/=8 then
		return false;

	elsif ii<=5  then

		ok:=true;
		for i in 0..1 loop
		for j in 0..1 loop
		for k in 0..1 loop
			if probe(i,j,k) and not empty(ix+i,iy+j,iz+k) then
				ok:=false;
			end if;
		end loop;
		end loop;
		end loop;

		return ok;

	elsif ii=10  then

		if
			empty( ix+0,iy+1,iz+0 ) and
			empty( ix+1,iy+1,iz+0 ) and
			empty( ix+1,iy+1,iz+1 ) and
			empty( ix+0,iy+1,iz+1 )
		then
			return true;
		else
			return false;
		end if;



	elsif ii=6 then

		if
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+0 )
		then
			return true;
		else
			return false;
		end if;

	elsif ii=9 then

		if
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+0 )
		then
			return true;
		else
			return false;
		end if;



	elsif ii=7 or ii=8 then
		if empty(ix,iy,iz) then
			return true;
		else
			return false;
		end if;



	else
		return false;


	end if;

end tryup;












function trydown(ii: rngp) return boolean is -- -Y
	ix,iy,iz : integer;
	ox,oy,oz : integer;
	sx,sy,sz : integer;
	ok, leadingVoid : boolean;
begin
	ox:=centroid(ii)(1);
	oy:=centroid(ii)(2);
	oz:=centroid(ii)(3);

	sx:=space(ii)(1);
	sy:=space(ii)(2);
	sz:=space(ii)(3);

	leadingVoid := (sy=0);



	for i in 0..1 loop
	for j in 0..1 loop
	for k in 0..1 loop
	if leadingVoid then -- (sy=0)

		if    i=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then probe(i,j,k):=true;
		else probe(i,j,k):=false;
		end if;

	else -- trailing void (sy=1)

		if   i/=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then probe(i,j,k):=true;
		else probe(i,j,k):=false;
		end if;

	end if;
	end loop;
	end loop;
	end loop;





	-- next centroid:
	ix:=ox;
	iy:=oy-1;
	iz:=oz;

	-- next test location
	--iy:=iy+0;


	if iy<1 then

		return false;

	elsif ii<=5  then

		ok:=true;
		for i in 0..1 loop
		for j in 0..1 loop
		for k in 0..1 loop
			if probe(i,j,k) and not empty(ix+i,iy+j,iz+k) then
				ok:=false;
			end if;
		end loop;
		end loop;
		end loop;

		return ok;

	elsif ii=10 then

		if
			empty( ix+0,iy,iz+0 ) and
			empty( ix+1,iy,iz+0 ) and
			empty( ix+1,iy,iz+1 ) and
			empty( ix+0,iy,iz+1 )
		then
			return true;
		else
			return false;
		end if;



	elsif ii=6 then

		if
			empty( ix,iy+0,iz+1 ) and
			empty( ix,iy+0,iz+0 )
		then
			return true;
		else
			return false;
		end if;

	elsif ii=9 then

		if
			empty( ix,iy+0,iz+1 ) and
			empty( ix,iy+0,iz+0 )
		then
			return true;
		else
			return false;
		end if;


	elsif ii=7 or ii=8 then
		if empty(ix,iy,iz) then
			return true;
		else
			return false;
		end if;



	else
		return false;


	end if; -- leadingVoid

end trydown;

















function tryright(ii: rngp) return boolean is -- +X
	ix,iy,iz : integer;
	ox,oy,oz : integer;
	sx,sy,sz : integer;
	ok, leadingVoid : boolean;
begin
	ox:=centroid(ii)(1);
	oy:=centroid(ii)(2);
	oz:=centroid(ii)(3);

	sx:=space(ii)(1);
	sy:=space(ii)(2);
	sz:=space(ii)(3);

	leadingVoid := (sx=1);



	for i in 0..1 loop
	for j in 0..1 loop
	for k in 0..1 loop
	if leadingVoid then -- (sx=1)

		if    i=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then probe(i,j,k):=true;
		else probe(i,j,k):=false;
		end if;

	else -- trailing void (sx=0)

		if   i/=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then probe(i,j,k):=true;
		else probe(i,j,k):=false;
		end if;

	end if;
	end loop;
	end loop;
	end loop;





	-- next centroid:
	ix:=ox+1;
	iy:=oy;
	iz:=oz;

	-- next test location
	--ix:=ix+1;


	if ix>mx then
		return false;

	elsif ii<=5 and ix+1<=mx then

		ok:=true;
		for i in 0..1 loop
		for j in 0..1 loop
		for k in 0..1 loop
			if probe(i,j,k) and not empty(ix+i,iy+j,iz+k) then
				ok:=false;
			end if;
		end loop;
		end loop;
		end loop;

		return ok;

	elsif ii=10 and ix+1<=mx then

		if
			empty( ix+1,iy+0,iz+0 ) and
			empty( ix+1,iy+1,iz+0 ) and
			empty( ix+1,iy+1,iz+1 ) and
			empty( ix+1,iy+0,iz+1 )
		then
			return true;
		else
			return false;
		end if;



	elsif ii=6 then

		if
			empty( ix,iy+0,iz+1 ) and
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+0 )
		then
			return true;
		else
			return false;
		end if;

	elsif ii=9 then

		if
			empty( ix,iy+0,iz+1 ) and
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+0 )
		then
			return true;
		else
			return false;
		end if;


	elsif ii=7 or ii=8 then
		if empty(ix,iy,iz) then
			return true;
		else
			return false;
		end if;



	else
		return false;


	end if;

end tryright;
















function tryleft(ii: rngp) return boolean is -- -X
	ix,iy,iz : integer;
	ox,oy,oz : integer;
	sx,sy,sz : integer;
	ok, leadingVoid : boolean;
begin
	ox:=centroid(ii)(1);
	oy:=centroid(ii)(2);
	oz:=centroid(ii)(3);

	sx:=space(ii)(1);
	sy:=space(ii)(2);
	sz:=space(ii)(3);

	leadingVoid := (sx=0);



	for i in 0..1 loop
	for j in 0..1 loop
	for k in 0..1 loop
	if leadingVoid then -- (sx=0)

		if    i=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then probe(i,j,k):=true;
		else probe(i,j,k):=false;
		end if;

	else -- trailing void (sx=1)

		if   i/=sx and j/=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then probe(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then probe(i,j,k):=true;
		else probe(i,j,k):=false;
		end if;

	end if;
	end loop;
	end loop;
	end loop;





	-- next centroid:
	ix:=ox-1;
	iy:=oy;
	iz:=oz;

	-- next test location
	--ix:=ix+0;


	if ix<1 then
		return false;

	elsif ii<=5  then

		ok:=true;
		for i in 0..1 loop
		for j in 0..1 loop
		for k in 0..1 loop
			if probe(i,j,k) and not empty(ix+i,iy+j,iz+k) then
				ok:=false;
			end if;
		end loop;
		end loop;
		end loop;

		return ok;

	elsif ii=10 then

		if
			empty( ix,iy+0,iz+0 ) and
			empty( ix,iy+1,iz+0 ) and
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+1 )
		then
			return true;
		else
			return false;
		end if;



	elsif ii=6 then

		if
			empty( ix,iy+0,iz+1 ) and
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+0 )
		then
			return true;
		else
			return false;
		end if;

	elsif ii=9 then

		if
			empty( ix,iy+0,iz+1 ) and
			empty( ix,iy+1,iz+1 ) and
			empty( ix,iy+0,iz+0 )
		then
			return true;
		else
			return false;
		end if;



	elsif ii=7 or ii=8 then
		if empty(ix,iy,iz) then
			return true;
		else
			return false;
		end if;



	else
		return false;


	end if;

end tryleft;


---------------------------------------------------------------------------




procedure movenear( kk: in out integer; nowTime: float ) is -- +Z
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid : boolean;
begin


if not trynear(kk) then -- pick one that might make sense
	for i in 0..10 loop
		if trynear(i) then
			kk:=i;
			exit;
		end if;
	end loop;
end if;


if trynear(kk) then

	oxc:=centroid(kk)(1);
	oyc:=centroid(kk)(2);
	ozc:=centroid(kk)(3);

	sx:=space(kk)(1);
	sy:=space(kk)(2);
	sz:=space(kk)(3);

	ix:=centroid(kk)(1);
	iy:=centroid(kk)(2);
	iz:=centroid(kk)(3);

	leadingVoid := (sz=1);

	for i in 0..1 loop
	for j in 0..1 loop
	for k in 0..1 loop
	if leadingVoid then -- (sz=1)

		if   i/=sx and j/=sy and k/=sz then 
			probe(i,j,k):=false;
			wake(i,j,k):=true;
		elsif i=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=false;
		else 
			probe(i,j,k):=false;
			wake(i,j,k):=false;
		end if;

	else -- trailing void (sz=0)

		if   i/=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=false;
		elsif i=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then 
			probe(i,j,k):=false;
			wake(i,j,k):=true;
		else 
			probe(i,j,k):=false;
			wake(i,j,k):=false;
		end if;

	end if;
	end loop;
	end loop;
	end loop;




	if kk=10 then -- (easy)

		-- trailing face:
		empty( ix+0,iy+0,iz+0 ):=true;
		empty( ix+1,iy+0,iz+0 ):=true;
		empty( ix+1,iy+1,iz+0 ):=true;
		empty( ix+0,iy+1,iz+0 ):=true;

		-- make the move:
		centroid(kk)(3):=centroid(kk)(3)+1;
		iz:=centroid(kk)(3);

		-- leading face:
		empty( ix+0,iy+0,iz+1 ):=false;
		empty( ix+1,iy+0,iz+1 ):=false;
		empty( ix+1,iy+1,iz+1 ):=false;
		empty( ix+0,iy+1,iz+1 ):=false;



	elsif kk=6 then

		-- trailing face:
		empty( ix,iy+0,iz+0 ):=true;
		empty( ix,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(3):=centroid(kk)(3)+1;
		iz:=centroid(kk)(3);

		-- leading face:
		empty( ix,iy+0,iz+1 ):=false;
		empty( ix,iy+1,iz+1 ):=false;

	elsif kk=9 then

		-- trailing face:
		empty( ix,iy+0,iz+0 ):=true;
		empty( ix,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(3):=centroid(kk)(3)+1;
		iz:=centroid(kk)(3);

		-- leading face:
		empty( ix,iy+0,iz+1 ):=false;
		empty( ix,iy+1,iz+1 ):=false;


	elsif kk=7 or kk=8 then
		empty(ix,iy,iz):=true;
		centroid(kk)(3):=centroid(kk)(3)+1;
		iz:=centroid(kk)(3);
		empty(ix,iy,iz):=false;


	else

		-- trailing face:
		for i in 0..1 loop
			for j in 0..1 loop
				for k in 0..1 loop
					if wake(i,j,k) then
						empty(ix+i,iy+j,iz+k):=true;
					end if;
				end loop;
			end loop;
		end loop;

		-- make the move:
		centroid(kk)(3):=centroid(kk)(3)+1;
		iz:=centroid(kk)(3);

		-- leading face:
		for i in 0..1 loop
			for j in 0..1 loop
				for k in 0..1 loop
					if probe(i,j,k) then
						empty(ix+i,iy+j,iz+k):=false;
					end if;
				end loop;
			end loop;
		end loop;

	end if;

	test4winner;

	if not mute then 
	snd4ada.playSnd(whoosh); 
	end if;

	movebegin:=nowTime;
	grinding(kk):=true;

end if; --trynear

end movenear;
-----------------------------------------------------------



procedure moveaway( kk: in out integer; nowTime: float ) is -- -Z
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid : boolean;
begin

if not tryaway(kk) then -- pick one that might make sense
	for i in 0..10 loop
		if tryaway(i) then
			kk:=i;
			exit;
		end if;
	end loop;
end if;



if tryaway(kk) then

	oxc:=centroid(kk)(1);
	oyc:=centroid(kk)(2);
	ozc:=centroid(kk)(3);

	sx:=space(kk)(1);
	sy:=space(kk)(2);
	sz:=space(kk)(3);

	ix:=centroid(kk)(1);
	iy:=centroid(kk)(2);
	iz:=centroid(kk)(3);

	leadingVoid := (sz=0);


	for i in 0..1 loop
	for j in 0..1 loop
	for k in 0..1 loop
	if leadingVoid then -- (sz=0)

		if   i/=sx and j/=sy and k/=sz then 
			probe(i,j,k):=false;
			wake(i,j,k):=true;
		elsif i=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=false;
		else 
			probe(i,j,k):=false;
			wake(i,j,k):=false;
		end if;

	else -- trailing void (sz=1)

		if   i/=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=false;
		elsif i=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then 
			probe(i,j,k):=false;
			wake(i,j,k):=true;
		else 
			probe(i,j,k):=false;
			wake(i,j,k):=false;
		end if;

	end if;
	end loop;
	end loop;
	end loop;




	if kk=10 then -- (easy)

		-- trailing face:
		empty( ix+0,iy+0,iz+1 ):=true;
		empty( ix+1,iy+0,iz+1 ):=true;
		empty( ix+1,iy+1,iz+1 ):=true;
		empty( ix+0,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(3):=centroid(kk)(3)-1;
		iz:=centroid(kk)(3);

		-- leading face:
		empty( ix+0,iy+0,iz+0 ):=false;
		empty( ix+1,iy+0,iz+0 ):=false;
		empty( ix+1,iy+1,iz+0 ):=false;
		empty( ix+0,iy+1,iz+0 ):=false;



	elsif kk=6 then

		-- trailing face:
		empty( ix,iy+0,iz+1 ):=true;
		empty( ix,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(3):=centroid(kk)(3)-1;
		iz:=centroid(kk)(3);

		-- leading face:
		empty( ix,iy+0,iz+0 ):=false;
		empty( ix,iy+1,iz+1 ):=false;

	elsif kk=9 then

		-- trailing face:
		empty( ix,iy+0,iz+1 ):=true;
		empty( ix,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(3):=centroid(kk)(3)-1;
		iz:=centroid(kk)(3);

		-- leading face:
		empty( ix,iy+0,iz+0 ):=false;
		empty( ix,iy+1,iz+1 ):=false;


	elsif kk=7 or kk=8 then
		empty(ix,iy,iz):=true;
		centroid(kk)(3):=centroid(kk)(3)-1;
		iz:=centroid(kk)(3);
		empty(ix,iy,iz):=false;


	else

		-- trailing face:
		for i in 0..1 loop
			for j in 0..1 loop
				for k in 0..1 loop
					if wake(i,j,k) then
						empty(ix+i,iy+j,iz+k):=true;
					end if;
				end loop;
			end loop;
		end loop;

		-- make the move:
		centroid(kk)(3):=centroid(kk)(3)-1;
		iz:=centroid(kk)(3);

		-- leading face:
		for i in 0..1 loop
			for j in 0..1 loop
				for k in 0..1 loop
					if probe(i,j,k) then
						empty(ix+i,iy+j,iz+k):=false;
					end if;
				end loop;
			end loop;
		end loop;

	end if;


	test4winner;

	if not mute then 
	snd4ada.playSnd(whoosh); 
	end if;

	movebegin:=nowTime;
	grinding(kk):=true;

end if; --tryaway


end moveaway;




procedure moveleft( kk: in out integer; nowTime: float ) is -- -X
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid : boolean;
begin

if not tryleft(kk) then -- pick one that might make sense
	for i in 0..10 loop
		if tryleft(i) then
			kk:=i;
			exit;
		end if;
	end loop;
end if;



if tryleft(kk) then

	oxc:=centroid(kk)(1);
	oyc:=centroid(kk)(2);
	ozc:=centroid(kk)(3);

	sx:=space(kk)(1);
	sy:=space(kk)(2);
	sz:=space(kk)(3);

	ix:=centroid(kk)(1);
	iy:=centroid(kk)(2);
	iz:=centroid(kk)(3);

	leadingVoid := (sx=0);


	for i in 0..1 loop
	for j in 0..1 loop
	for k in 0..1 loop
	if leadingVoid then -- (sx=0)

		if   i/=sx and j/=sy and k/=sz then 
			probe(i,j,k):=false;
			wake(i,j,k):=true;
		elsif i=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=false;
		elsif i/=sx and j=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		else 
			probe(i,j,k):=false;
			wake(i,j,k):=false;
		end if;

	else -- trailing void (sx=1)

		if   i/=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=false;
		elsif i=sx and j/=sy and k/=sz then 
			probe(i,j,k):=false;
			wake(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		else 
			probe(i,j,k):=false;
			wake(i,j,k):=false;
		end if;

	end if;
	end loop;
	end loop;
	end loop;




	if kk=10 then -- (easy)

		-- trailing face:
		empty( ix+1,iy+0,iz+0 ):=true;
		empty( ix+1,iy+0,iz+1 ):=true;
		empty( ix+1,iy+1,iz+0 ):=true;
		empty( ix+1,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(1):=centroid(kk)(1)-1;
		ix:=centroid(kk)(1);

		-- leading face:
		empty( ix+0,iy+0,iz+0 ):=false;
		empty( ix+0,iy+0,iz+1 ):=false;
		empty( ix+0,iy+1,iz+0 ):=false;
		empty( ix+0,iy+1,iz+1 ):=false;



	elsif kk=6 then

		-- trailing face:
		empty( ix,iy+0,iz+0 ):=true;
		empty( ix,iy+0,iz+1 ):=true;
		empty( ix,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(1):=centroid(kk)(1)-1;
		ix:=centroid(kk)(1);

		-- leading face:
		empty( ix,iy+0,iz+0 ):=false;
		empty( ix,iy+0,iz+1 ):=false;
		empty( ix,iy+1,iz+1 ):=false;

	elsif kk=9 then

		-- trailing face:
		empty( ix,iy+0,iz+0 ):=true;
		empty( ix,iy+0,iz+1 ):=true;
		empty( ix,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(1):=centroid(kk)(1)-1;
		ix:=centroid(kk)(1);

		-- leading face:
		empty( ix,iy+0,iz+0 ):=false;
		empty( ix,iy+0,iz+1 ):=false;
		empty( ix,iy+1,iz+1 ):=false;



	elsif kk=7 or kk=8 then
		empty(ix,iy,iz):=true;
		centroid(kk)(1):=centroid(kk)(1)-1;
		ix:=centroid(kk)(1);
		empty(ix,iy,iz):=false;



	else

		-- trailing face:
		for i in 0..1 loop
			for j in 0..1 loop
				for k in 0..1 loop
					if wake(i,j,k) then
						empty(ix+i,iy+j,iz+k):=true;
					end if;
				end loop;
			end loop;
		end loop;

		-- make the move:
		centroid(kk)(1):=centroid(kk)(1)-1;
		ix:=centroid(kk)(1);

		-- leading face:
		for i in 0..1 loop
			for j in 0..1 loop
				for k in 0..1 loop
					if probe(i,j,k) then
						empty(ix+i,iy+j,iz+k):=false;
					end if;
				end loop;
			end loop;
		end loop;

	end if;


	test4winner;

	if not mute then 
	snd4ada.playSnd(whoosh); 
	end if;

	movebegin:=nowTime;
	grinding(kk):=true;

end if; --tryleft


end moveleft;




procedure moveright( kk: in out integer; nowTime: float ) is -- +X
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid : boolean;
begin

if not tryright(kk) then -- pick one that might make sense
	for i in 0..10 loop
		if tryright(i) then
			kk:=i;
			exit;
		end if;
	end loop;
end if;



if tryright(kk) then

	oxc:=centroid(kk)(1);
	oyc:=centroid(kk)(2);
	ozc:=centroid(kk)(3);

	sx:=space(kk)(1);
	sy:=space(kk)(2);
	sz:=space(kk)(3);

	ix:=centroid(kk)(1);
	iy:=centroid(kk)(2);
	iz:=centroid(kk)(3);

	leadingVoid := (sx=1);


	for i in 0..1 loop
	for j in 0..1 loop
	for k in 0..1 loop
	if leadingVoid then -- (sx=1)

		if   i/=sx and j/=sy and k/=sz then 
			probe(i,j,k):=false;
			wake(i,j,k):=true;
		elsif i=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=false;
		elsif i/=sx and j=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		else 
			probe(i,j,k):=false;
			wake(i,j,k):=false;
		end if;

	else -- trailing void (sx=0)

		if   i/=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=false;
		elsif i=sx and j/=sy and k/=sz then 
			probe(i,j,k):=false;
			wake(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		else 
			probe(i,j,k):=false;
			wake(i,j,k):=false;
		end if;

	end if;
	end loop;
	end loop;
	end loop;




	if kk=10 then -- (easy)

		empty( ix+0,iy+0,iz+0 ):=true;
		empty( ix+0,iy+0,iz+1 ):=true;
		empty( ix+0,iy+1,iz+0 ):=true;
		empty( ix+0,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(1):=centroid(kk)(1)+1;
		ix:=centroid(kk)(1);

		empty( ix+1,iy+0,iz+0 ):=false;
		empty( ix+1,iy+0,iz+1 ):=false;
		empty( ix+1,iy+1,iz+0 ):=false;
		empty( ix+1,iy+1,iz+1 ):=false;



	elsif kk=6 then

		-- trailing face:
		empty( ix,iy+0,iz+0 ):=true;
		empty( ix,iy+0,iz+1 ):=true;
		empty( ix,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(1):=centroid(kk)(1)+1;
		ix:=centroid(kk)(1);

		-- leading face:
		empty( ix,iy+0,iz+0 ):=false;
		empty( ix,iy+0,iz+1 ):=false;
		empty( ix,iy+1,iz+1 ):=false;

	elsif kk=9 then

		-- trailing face:
		empty( ix,iy+0,iz+0 ):=true;
		empty( ix,iy+0,iz+1 ):=true;
		empty( ix,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(1):=centroid(kk)(1)+1;
		ix:=centroid(kk)(1);

		-- leading face:
		empty( ix,iy+0,iz+0 ):=false;
		empty( ix,iy+0,iz+1 ):=false;
		empty( ix,iy+1,iz+1 ):=false;



	elsif kk=7 or kk=8 then
		empty(ix,iy,iz):=true;
		centroid(kk)(1):=centroid(kk)(1)+1;
		ix:=centroid(kk)(1);
		empty(ix,iy,iz):=false;



	else

		-- trailing face:
		for i in 0..1 loop
			for j in 0..1 loop
				for k in 0..1 loop
					if wake(i,j,k) then
						empty(ix+i,iy+j,iz+k):=true;
					end if;
				end loop;
			end loop;
		end loop;

		-- make the move:
		centroid(kk)(1):=centroid(kk)(1)+1;
		ix:=centroid(kk)(1);

		-- leading face:
		for i in 0..1 loop
			for j in 0..1 loop
				for k in 0..1 loop
					if probe(i,j,k) then
						empty(ix+i,iy+j,iz+k):=false;
					end if;
				end loop;
			end loop;
		end loop;

	end if;



	test4winner;

	if not mute then 
	snd4ada.playSnd(whoosh); 
	end if;

	movebegin:=nowTime;
	grinding(kk):=true;

end if; --tryright


end moveright;






procedure movedown( kk: in out integer; nowTime: float ) is -- -Y
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid : boolean;
begin
 
if not trydown(kk) then -- pick one that might make sense
	for i in 0..10 loop
		if trydown(i) then
			kk:=i;
			exit;
		end if;
	end loop;
end if;



if trydown(kk) then

	oxc:=centroid(kk)(1);
	oyc:=centroid(kk)(2);
	ozc:=centroid(kk)(3);

	sx:=space(kk)(1);
	sy:=space(kk)(2);
	sz:=space(kk)(3);

	ix:=centroid(kk)(1);
	iy:=centroid(kk)(2);
	iz:=centroid(kk)(3);

	leadingVoid := (sy=0);


	for i in 0..1 loop
	for j in 0..1 loop
	for k in 0..1 loop
	if leadingVoid then -- (sy=0)

		if   i/=sx and j/=sy and k/=sz then 
			probe(i,j,k):=false;
			wake(i,j,k):=true;
		elsif i=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=false;
		elsif i/=sx and j/=sy and k=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		else 
			probe(i,j,k):=false;
			wake(i,j,k):=false;
		end if;

	else -- trailing void (sy=1)

		if   i/=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=false;
		elsif i=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then 
			probe(i,j,k):=false;
			wake(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		else 
			probe(i,j,k):=false;
			wake(i,j,k):=false;
		end if;

	end if;
	end loop;
	end loop;
	end loop;




	if kk=10 then -- (easy)

		empty( ix+0,iy+1,iz+0 ):=true;
		empty( ix+0,iy+1,iz+1 ):=true;
		empty( ix+1,iy+1,iz+0 ):=true;
		empty( ix+1,iy+1,iz+1 ):=true;

		-- make the move:
		centroid(kk)(2):=centroid(kk)(2)-1;
		iy:=centroid(kk)(2);

		empty( ix+0,iy+0,iz+0 ):=false;
		empty( ix+0,iy+0,iz+1 ):=false;
		empty( ix+1,iy+0,iz+0 ):=false;
		empty( ix+1,iy+0,iz+1 ):=false;


	elsif kk=6 then

		empty( ix,iy+1,iz+1 ):=true;
		empty( ix,iy+0,iz+0 ):=true;

		-- make the move:
		centroid(kk)(2):=centroid(kk)(2)-1;
		iy:=centroid(kk)(2);

		empty( ix,iy+0,iz+1 ):=false;
		empty( ix,iy+0,iz+0 ):=false;

	elsif kk=9 then

		empty( ix,iy+1,iz+1 ):=true;
		empty( ix,iy+0,iz+0 ):=true;

		-- make the move:
		centroid(kk)(2):=centroid(kk)(2)-1;
		iy:=centroid(kk)(2);

		empty( ix,iy+0,iz+1 ):=false;
		empty( ix,iy+0,iz+0 ):=false;



	elsif kk=7 or kk=8 then
		empty(ix,iy,iz):=true;
		centroid(kk)(2):=centroid(kk)(2)-1;
		iy:=centroid(kk)(2);
		empty(ix,iy,iz):=false;



	else

		-- trailing face:
		for i in 0..1 loop
			for j in 0..1 loop
				for k in 0..1 loop
					if wake(i,j,k) then
						empty(ix+i,iy+j,iz+k):=true;
					end if;
				end loop;
			end loop;
		end loop;

		-- make the move:
		centroid(kk)(2):=centroid(kk)(2)-1;
		iy:=centroid(kk)(2);

		-- leading face:
		for i in 0..1 loop
			for j in 0..1 loop
				for k in 0..1 loop
					if probe(i,j,k) then
						empty(ix+i,iy+j,iz+k):=false;
					end if;
				end loop;
			end loop;
		end loop;

	end if;




	test4winner;

	if not mute then 
	snd4ada.playSnd(whoosh); 
	end if;

	movebegin:=nowTime;
	grinding(kk):=true;

end if; --trydown


end movedown;



procedure moveup( kk: in out integer; nowTime: float ) is -- +Y
	ix,iy,iz : integer;
	sx,sy,sz : integer;
	leadingVoid : boolean;
begin
 
if not tryup(kk) then -- pick one that might make sense
	for i in 0..10 loop
		if tryup(i) then
			kk:=i;
			exit;
		end if;
	end loop;
end if;



if tryup(kk) then

	oxc:=centroid(kk)(1);
	oyc:=centroid(kk)(2);
	ozc:=centroid(kk)(3);

	sx:=space(kk)(1);
	sy:=space(kk)(2);
	sz:=space(kk)(3);

	ix:=centroid(kk)(1);
	iy:=centroid(kk)(2);
	iz:=centroid(kk)(3);

	leadingVoid := (sy=1);



	for i in 0..1 loop
	for j in 0..1 loop
	for k in 0..1 loop
	if leadingVoid then -- (sy=1)

		if   i/=sx and j/=sy and k/=sz then 
			probe(i,j,k):=false;
			wake(i,j,k):=true;
		elsif i=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=false;
		elsif i/=sx and j/=sy and k=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		else 
			probe(i,j,k):=false;
			wake(i,j,k):=false;
		end if;

	else -- trailing void (sy=0)

		if   i/=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=false;
		elsif i=sx and j/=sy and k/=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		elsif i/=sx and j=sy and k/=sz then 
			probe(i,j,k):=false;
			wake(i,j,k):=true;
		elsif i/=sx and j/=sy and k=sz then 
			probe(i,j,k):=true;
			wake(i,j,k):=true;
		else 
			probe(i,j,k):=false;
			wake(i,j,k):=false;
		end if;

	end if;
	end loop;
	end loop;
	end loop;




	if kk=10 then -- (easy)

		-- trailing face:
		empty( ix+0,iy+0,iz+0 ):=true;
		empty( ix+0,iy+0,iz+1 ):=true;
		empty( ix+1,iy+0,iz+0 ):=true;
		empty( ix+1,iy+0,iz+1 ):=true;

		-- make the move:
		centroid(kk)(2):=centroid(kk)(2)+1;
		iy:=centroid(kk)(2);

		-- leading face:
		empty( ix+0,iy+1,iz+0 ):=false;
		empty( ix+0,iy+1,iz+1 ):=false;
		empty( ix+1,iy+1,iz+0 ):=false;
		empty( ix+1,iy+1,iz+1 ):=false;


	elsif kk=6 then

		empty( ix,iy+0,iz+1 ):=true;
		empty( ix,iy+0,iz+0 ):=true;

		-- make the move:
		centroid(kk)(2):=centroid(kk)(2)+1;
		iy:=centroid(kk)(2);

		empty( ix,iy+1,iz+1 ):=false;
		empty( ix,iy+0,iz+0 ):=false;

	elsif kk=9 then

		empty( ix,iy+0,iz+1 ):=true;
		empty( ix,iy+0,iz+0 ):=true;

		-- make the move:
		centroid(kk)(2):=centroid(kk)(2)+1;
		iy:=centroid(kk)(2);

		empty( ix,iy+1,iz+1 ):=false;
		empty( ix,iy+0,iz+0 ):=false;



	elsif kk=7 or kk=8 then
		empty(ix,iy,iz):=true;
		centroid(kk)(2):=centroid(kk)(2)+1;
		iy:=centroid(kk)(2);
		empty(ix,iy,iz):=false;



	else

		-- trailing face:
		for i in 0..1 loop
			for j in 0..1 loop
				for k in 0..1 loop
					if wake(i,j,k) then
						empty(ix+i,iy+j,iz+k):=true;
					end if;
				end loop;
			end loop;
		end loop;

		-- make the move:
		centroid(kk)(2):=centroid(kk)(2)+1;
		iy:=centroid(kk)(2);

		-- leading face:
		for i in 0..1 loop
			for j in 0..1 loop
				for k in 0..1 loop
					if probe(i,j,k) then
						empty(ix+i,iy+j,iz+k):=false;
					end if;
				end loop;
			end loop;
		end loop;

	end if;




	test4winner;

	if not mute then 
	snd4ada.playSnd(whoosh); 
	end if;

	movebegin:=nowTime;
	grinding(kk):=true;

end if; --tryup


end moveup;


















	playedonce, userexit : boolean := false;
	wintime, currentTime : float;

	v4, vcc : vec4;


	--keydlay : constant float := 0.25;

	otitle : constant interfaces.c.char_array 
		:= value( new_string("Ada7") );
	vtitle : constant interfaces.c.char_array 
		:= value( new_string("Vadasz") );



	xize: float := 0.25;

	ix,iy,iz, sx,sy,sz, cx,cy,cz : integer := 1;


	ixsel: integer:=1;
	pik: boolean := false;

	hc: boolean := true; --HighContrast

----------------- main program begin ==========================
begin --tomb

	first_prep;  -- main program setup

	show_axes:=true;


	-- rotate into preferred initial orientation:
	degRotate( mm, 20.0, 1.0, 0.0, 0.0 );



	-- note:  only mm changes:
	updateMVP( float(wwid), float(whit) );




	-- main event loop begin: ------------------------------------------
   while not userexit loop

------- begin response to keys ------------------------------------------

		currentTime := float(glfwGetTime);
		--deltaT := currentTime - oldTimeKb;


		glfwPollEvents;



		--check for <esc>,<q>:
		if glfwgetkey( mainWin, glfw_key_escape ) = Glfw_Press then
				userexit:=true;

		elsif glfwgetkey( mainWin, glfw_key_q ) = Glfw_Press then
				userexit:=true;

		end if;

		exit when glfwWindowShouldClose(mainWin) /= 0; --19oct21



		anyGrinding:=false;
		for i in rngp loop
			anyGrinding := anyGrinding or grinding(i);
		end loop;

		if not anyGrinding then
			gameutils.getKeyInputs(
				mainWin,
				rngp'first, rngp'last,
				ixsel,pik);

			if goUp then
				goUp:=false;
				moveUp(ixsel,currentTime);
			elsif goDown then
				goDown:=false;
				moveDown(ixsel,currentTime);
			elsif goLeft then
				goLeft:=false;
				moveLeft(ixsel,currentTime);
			elsif goRight then
				goRight:=false;
				moveRight(ixsel,currentTime);
			elsif goAway then
				goAway:=false;
				moveAway(ixsel,currentTime);
			elsif goNear then
				goNear:=false;
				moveNear(ixsel,currentTime);
			elsif doRestart then
				doRestart:=false;
				centroid:=centroidHi;
				initEmptyHi;
			elsif tryNextSkin then
				tryNextSkin:=false;
				setNextSkin;
			end if;


			gameutils.handle_mouse(
				pik, 
				rngp'first, rngp'last, 
				ixsel,
				centroid,
				xxcc,yycc,zzcc
				);
		end if;




----///////////////////// end response to key/mouse


		updateMVP( float(wwid), float(whit) );


-------- here we should handle resized window ----------------------


		glfwGetWindowSize( mainWin, Nwid'access, Nhit'access );
		if( (Nwid /= wwid) or (Nhit /= whit) ) then
			wwid:=Nwid;  whit:=Nhit;

			glfwGetFramebufferSize(mainwin, fwid'access, fhit'access);
			glViewport(0,0,Fwid,Fhit);

		end if;



--------- begin drawing ===============================================

		glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);



------ draw here ----------------------

	-- now, the 10 puzzle pieces:
	glUseProgram( pgmTexShadID );
	gluniformmatrix4fv( matrixid, 1, gl_false, mvp(1,1)'address );
	gluniform1i(uniftex,0);

	for ii in rngp loop -- 8 corners + 2 edge + 1 egg


		if numerals then
			glBindTexture(gl_texture_2d, num_texid(ii) );
		else
			glBindTexture(gl_texture_2d, nun_texid );
		end if;



		ix:=centroid(ii)(1);
		iy:=centroid(ii)(2);
		iz:=centroid(ii)(3);


		if ii = rngp'last then --egg

			--drawSkull(ppo, currentTime);
			null;

		elsif ii>=0 and ii<=5 then

			sx:=space(ii)(1);
			sy:=space(ii)(2);
			sz:=space(ii)(3);

			cx:=1-sx;
			cy:=1-sy;
			cz:=1-sz; --corner cubelet indexes


			for i in 0..1 loop
			for j in 0..1 loop
			for k in 0..1 loop
				-- this is the 4-cube-corner:
				if i/=sx and j/=sy and k/=sz then
					drawCube(ppo, ii, cx,cy,cz, i,j,k, ix, iy, iz, currentTime );
				elsif i=sx and j/=sy and k/=sz then
					drawCube(ppo, ii, cx,cy,cz, i,j,k, ix, iy, iz, currentTime );
				elsif i/=sx and j=sy and k/=sz then
					drawCube(ppo, ii, cx,cy,cz, i,j,k, ix, iy, iz, currentTime );
				elsif i/=sx and j/=sy and k=sz then
					drawCube(ppo, ii, cx,cy,cz, i,j,k, ix, iy, iz, currentTime );
				else -- empty space
					null;
				end if;
			end loop;
			end loop;
			end loop;

		elsif ii=6 or ii=9 then -- ?? is (0,0,1) the corner ??
			drawCube(ppo, ii, 0,0,1, 0,0,0, ix,iy,iz, currentTime); --bottom rearward
			drawCube(ppo, ii, 0,0,1, 0,0,1, ix,iy,iz, currentTime);
			drawCube(ppo, ii, 0,0,1, 0,1,1, ix,iy,iz, currentTime);

		else -- 7,8 (1x1)
			drawCube(ppo, ii, 0,0,0, 0,0,0, ix,iy,iz, currentTime); --bottom rearward
		end if;


	end loop; --ii



	glUseProgram( pgmTexID );
	gluniformmatrix4fv( matid, 1, gl_false, mvp(1,1)'address );
	gluniform1i(unitex,0);

	if not winner then
		glBindTexture(gl_texture_2d, box_texid ); 
		drawBigCube(ppo2); --bounding chainlink box
	end if;

	glBindTexture(gl_texture_2d, num_texid(rngp'last) );
	drawSkull(ppo, currentTime);





		if show_axes then

			v4 := (+2.0, -2.0, -2.0, 1.0);
			matXvec(mvp, v4, vcc);
			stex.print3d("+X", vcc(1), vcc(2), vcc(3), vcc(4), 1.0, 1.0);

			v4 := (-2.2, +2.0, -2.2, 1.0);
			matXvec(mvp, v4, vcc);
			stex.print3d("+Y", vcc(1), vcc(2), vcc(3), vcc(4), 1.0, 1.0);

			v4 := (-2.0, -2.0, +2.0, 1.0);
			matXvec(mvp, v4, vcc);
			stex.print3d("+Z", vcc(1), vcc(2), vcc(3), vcc(4), 1.0, 1.0);

			v4 := (-2.0, -2.0, -2.0, 1.0);
			matXvec(mvp, v4, vcc);
			stex.print3d("O", vcc(1), vcc(2), vcc(3), vcc(4), 1.0, 1.0);

		end if;



		xize:=0.25;

		-- fixed 2d location text:

		stex.print2d("(spc)=>Restart", 0.02, 0.05, xize );
		stex.print2d("(i,o,mousewheel)=>Zoom", 0.36, 0.05, xize);
		stex.print2d("(esc)=>exit",    0.78, 0.05, xize );
		stex.print2d("0-9=>pick",      0.02, 0.02, xize );
		stex.print2d("s=>skull",        0.20, 0.02, xize );
		stex.print2d("n=>nums",        0.40, 0.02, xize ); --8oct21
		stex.print2d("c=>nextColor",        0.58, 0.02, xize );
		stex.print2d("h=>help",        0.85, 0.02, xize );





		if help then
			xize:=0.4; hc:=false;
			stex.print2d("Reckless raiders have plundered a reliquary",0.02,0.90,xize,hc);
			stex.print2d("...and displaced an ancient Crystal Skull !",0.02,0.85,xize,hc);

			stex.print2d("Return the Skull to its proper position in the", 0.02, 0.80, xize,hc);
			stex.print2d("center of its [numerically ordered] Cubical Frame.", 0.02, 0.75, xize,hc);

			stex.print2d("With X rightward, Y upward, and Z outward,", 0.02, 0.70, xize,hc);
			stex.print2d("use the keys:  u/d  l/r  f/b", 0.02, 0.65, xize,hc);
			stex.print2d("(Up/Down Left/Right Forward/Backward)", 0.02, 0.60, xize,hc);

			stex.print2d("Select a block by number or mouse-click...", 0.02, 0.50, xize,hc);
			stex.print2d("Laptop users can place cursor on block", 0.02, 0.45, xize,hc);
			stex.print2d("then hit <enter>-key to select", 0.02, 0.40, xize,hc);

			stex.print2d("Zoom keys:   i/o = In/Out", 0.02, 0.35, xize,hc);

			stex.print2d("V-key to mute moves", 0.02, 0.30, xize,hc);

			xize:=0.25;
		else
			stex.print2d("Hard:   Move the skull into the center of its cubical tomb.",0.10,0.95,xize);
		end if;

		if winner and (currentTime-testime>windelay) then
			stex.print2d("Correct!", 0.19, 0.83, 1.0 );

			if not playedonce then
		 		snd4ada.playSnd(clap); --clap
				playedonce:=true;
				wintime:=currentTime;
			end if;

			if( currentTime-wintime > 5.0 ) then 
				winner:=false;
			end if;

		else
			playedonce:=false;
		end if;


--------- end drawing =================================================



		glflush;
		glfwSwapBuffers( mainWin );


   end loop; ---------------- main event loop end -------------------




	glext.binding.glDeleteProgram(pgmtexshadid);

	glext.binding.glDeleteBuffers(1, vertbuff'address);
	glext.binding.glDeleteBuffers(1, rgbbuff'address);
	glext.binding.glDeleteBuffers(1, elembuff'address);
	glext.binding.glDeleteBuffers(1, uvbuff'address);

	glext.binding.glDeleteVertexArrays(1, vertexarrayid'address);

	stex.CloseFont;

	snd4ada.termSnds;

	glfwdestroywindow(mainWin);
	glfwTerminate;




end tomb4;

