/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.sql;

import java.io.IOException;
import java.io.StringReader;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;

import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.sqlcs.ddl.SqlAlterTableAdd;
import net.morilib.db.sqlcs.ddl.SqlAlterTableDrop;
import net.morilib.db.sqlcs.ddl.SqlAlterTableModify;
import net.morilib.db.sqlcs.ddl.SqlAlterTableRenameColumn;
import net.morilib.db.sqlcs.ddl.SqlColumnAttribute;
import net.morilib.db.sqlcs.ddl.SqlColumnDefinition;
import net.morilib.db.sqlcs.ddl.SqlColumnType;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;
import net.morilib.db.sqlcs.ddl.SqlDropTable;
import net.morilib.db.sqlcs.ddl.SqlTruncateTable;
import net.morilib.db.sqlcs.ddl.SqlTypeDate;
import net.morilib.db.sqlcs.ddl.SqlTypeNumeric;
import net.morilib.db.sqlcs.ddl.SqlTypeVarchar;
import net.morilib.db.sqlcs.dml.SqlBinaryOperator;
import net.morilib.db.sqlcs.dml.SqlDelete;
import net.morilib.db.sqlcs.dml.SqlExpression;
import net.morilib.db.sqlcs.dml.SqlInsertSelect;
import net.morilib.db.sqlcs.dml.SqlInsertValues;
import net.morilib.db.sqlcs.dml.SqlJoin;
import net.morilib.db.sqlcs.dml.SqlJoinType;
import net.morilib.db.sqlcs.dml.SqlOrderBy;
import net.morilib.db.sqlcs.dml.SqlRelation;
import net.morilib.db.sqlcs.dml.SqlSelect;
import net.morilib.db.sqlcs.dml.SqlSelectDistinct;
import net.morilib.db.sqlcs.dml.SqlSubqueryRelation;
import net.morilib.db.sqlcs.dml.SqlTable;
import net.morilib.db.sqlcs.dml.SqlTableColumn;
import net.morilib.db.sqlcs.dml.SqlUpdate;

public class DbSqlParser {

	SqlExpression _join_on(
			DbSqlLexer lex) throws IOException, SQLException {
		// ON expression [JOIN ...]
		if(lex.eq(DbSqlReserved.ON)) {
			return new DbSqlExprParser().parse(lex);
		}
		return null;
	}

	SqlRelation _join(SqlRelation r,
			DbSqlLexer lex) throws IOException, SQLException {
		SqlRelation s;

		if(lex.eq(DbSqlReserved.INNER)) {
			lex.eq(DbSqlReserved.JOIN);
			s = _relation(lex);
			return new SqlJoin(SqlJoinType.INNER, r, s, _join_on(lex));
		} else if(lex.eq(DbSqlReserved.LEFT)) {
			lex.eq(DbSqlReserved.OUTER);
			lex.eq(DbSqlReserved.JOIN);
			s = _relation(lex);
			return new SqlJoin(SqlJoinType.LEFT, r, s, _join_on(lex));
		} else if(lex.eq(DbSqlReserved.RIGHT)) {
			lex.eq(DbSqlReserved.OUTER);
			lex.eq(DbSqlReserved.JOIN);
			s = _relation(lex);
//			return new SqlJoin(SqlJoinType.RIGHT, r, s, _join_on(lex));
			throw ErrorBundle.getDefault(10022, lex.get().toString());
		} else if(lex.eq(DbSqlReserved.FULL)) {
			lex.eq(DbSqlReserved.OUTER);
			lex.eq(DbSqlReserved.JOIN);
			s = _relation(lex);
//			return new SqlJoin(SqlJoinType.FULL, r, s, _join_on(lex));
			throw ErrorBundle.getDefault(10022, lex.get().toString());
		} else if(lex.eq(DbSqlReserved.OUTER)) {
			lex.eq(DbSqlReserved.JOIN);
			s = _relation(lex);
			return new SqlJoin(SqlJoinType.LEFT, r, s, _join_on(lex));
		} else if(lex.eq(DbSqlReserved.NATURAL)) {
			lex.eq(DbSqlReserved.JOIN);
			s = _relation(lex);
//			return new SqlJoin(SqlJoinType.NATURAL, r, s, null);
			throw ErrorBundle.getDefault(10022, lex.get().toString());
		} else if(lex.eq(DbSqlReserved.CROSS)) {
			lex.eq(DbSqlReserved.JOIN);
			s = _relation(lex);
//			return new SqlJoin(SqlJoinType.CROSS, r, s, null);
			throw ErrorBundle.getDefault(10022, lex.get().toString());
		} else if(lex.eq(DbSqlReserved.JOIN)) {
			s = _relation(lex);
			return new SqlJoin(SqlJoinType.INNER, r, s, _join_on(lex));
		} else {
			return r;
		}
	}

	SqlRelation _relation(
			DbSqlLexer lex) throws IOException, SQLException {
		SqlSelect q;
		String s;

		if((s = lex.getsym()) != null) {
			lex.eqsym("AS");
			return new SqlTable(s, lex.getsym());
		} else if(lex.eqchar('(')) {
			lex.eatsym("SELECT");
			q = _select(lex);
			lex.eatchar(')');
			lex.eqsym("AS");
			return new SqlSubqueryRelation(q, lex.getsym());
		} else {
			throw ErrorBundle.getDefault(10023);
		}
	}

	SqlSelect _select(
			DbSqlLexer lex) throws IOException, SQLException {
		List<SqlTableColumn> s = new ArrayList<SqlTableColumn>();
		SqlExpression e, w = null, h = null;
		List<SqlRelation> f = null;
		List<SqlOrderBy> o = null;
		List<String> g = null;
		SqlSelectDistinct d;
		SqlRelation r, q;
		String n;

		d = lex.eqsym("DISTINCT") ?
				SqlSelectDistinct.DISTINCT : SqlSelectDistinct.DEFAULT;

		// columns
		if(lex.eqchar('*')) {
			s.add(SqlTableColumn.WILDCARD);
		} else {
			do {
				e = new DbSqlExprParser().parse(lex);
				lex.eqsym("AS");
				s.add(new SqlTableColumn(e, lex.getsym()));
			} while(lex.eqchar(','));
		}

		// FROM
		if(lex.eq(DbSqlReserved.FROM)) {
			f = new ArrayList<SqlRelation>();
			do {
				r = _relation(lex);
				for(q = null; (q = _join(r, lex)) != r; r = q);
				f.add(r);
			} while(lex.eqchar(','));
		}

		// WHERE
		if(lex.eq(DbSqlReserved.WHERE)) {
			w = new DbSqlExprParser().parse(lex);
		}

		// GROUP BY
		if(lex.eq(DbSqlReserved.GROUP)) {
			lex.eqsym("BY");
			g = new ArrayList<String>();
			do {
				g.add(lex.getsym());
			} while(lex.eqchar(','));
		}

		// HAVING
		if(lex.eq(DbSqlReserved.HAVING)) {
			h = new DbSqlExprParser().parse(lex);
		}

		// ORDER BY
		if(lex.eq(DbSqlReserved.ORDER)) {
			lex.eqsym("BY");
			o = new ArrayList<SqlOrderBy>();
			do {
				n = lex.getsym();
				if(lex.eqsym("ASC")) {
					o.add(new SqlOrderBy(n, true));
				} else if(lex.eqsym("DESC")) {
					o.add(new SqlOrderBy(n, false));
				} else {
					o.add(new SqlOrderBy(n, true));
				}
			} while(lex.eqchar(','));
		}
		return new SqlSelect(s, d, f, w, g, h, o);
	}

	EnumSet<SqlColumnAttribute> _ctoption(
			EnumSet<SqlColumnAttribute> r,
			DbSqlLexer lex)  throws IOException, SQLException {
		if(lex.eqsym("NOT")) {
			lex.eatsym("NULL");
			r.add(SqlColumnAttribute.NOT_NULL);
			return _ctoption(r, lex);
		} else if(lex.eqsym("PRIMARY")) {
			lex.eatsym("KEY");
			r.add(SqlColumnAttribute.NOT_NULL);
			r.add(SqlColumnAttribute.PRIMARY_KEY);
			return _ctoption(r, lex);
		} else {
			return r;
		}
	}

	SqlColumnType _cttype(
			DbSqlLexer lex)  throws IOException, SQLException {
		int j, k;

		if(lex.eqsym("VARCHAR")) {
			if(lex.eqchar('(')) {
				j = lex.eatsmallint();
				lex.eatchar(')');
				return new SqlTypeVarchar(j);
			} else {
				return new SqlTypeVarchar(SqlTypeVarchar.UNLIMITED);
			}
		} else if(lex.eqsym("NUMERIC")) {
			if(lex.eqchar('(')) {
				j = lex.eqsym("UNLIMITED") ?
						SqlTypeNumeric.UNLIMITED : lex.eatsmallint();
				k = lex.eqchar(',') ? lex.eatsmallint() : 0;
				lex.eatchar(')');
			} else {
				j = SqlTypeNumeric.UNLIMITED;
				k = 0;
			}
			return new SqlTypeNumeric(j, k);
		} else if(lex.eqsym("DATE")) {
			return new SqlTypeDate();
		} else {
			throw ErrorBundle.getDefault(10024, lex.get().toString());
		}
	}

	SqlColumnDefinition _ctcolumn(
			DbSqlLexer lex)  throws IOException, SQLException {
		EnumSet<SqlColumnAttribute> a;
		SqlColumnType t;
		String s;

		s = lex.eatsym();
		t = _cttype(lex);
		a = _ctoption(EnumSet.noneOf(SqlColumnAttribute.class), lex);
		return new SqlColumnDefinition(s, t, a);
	}

	SqlCreateTable _create_table(
			DbSqlLexer lex)  throws IOException, SQLException {
		List<SqlColumnDefinition> t;
		String n;

		t = new ArrayList<SqlColumnDefinition>();
		n = lex.getsym();
		lex.eatchar('(');
		do {
			t.add(_ctcolumn(lex));
		} while(lex.eqchar(','));
		lex.eatchar(')');
		return new SqlCreateTable(n, t);
	}

	Object _create(DbSqlLexer lex)  throws IOException, SQLException {
		if(lex.eqsym("TABLE")) {
			return _create_table(lex);
		} else {
			throw ErrorBundle.getDefault(10025, lex.get().toString());
		}
	}

	Object _insert(DbSqlLexer lex) throws IOException, SQLException {
		List<SqlExpression> m;
		List<String> l = null;
		SqlSelect q;
		String t;

		lex.eatsym("INTO");
		t = lex.eatsym();

		if(lex.eqchar('(')) {
			l = new ArrayList<String>();
			do {
				l.add(lex.getsym());
			} while(lex.eqchar(','));
			lex.eatchar(')');
		}

		if(lex.eqsym("VALUES")) {
			lex.eatchar('(');
			m = new ArrayList<SqlExpression>();
			do {
				m.add(new DbSqlExprParser().parse(lex));
			} while(lex.eqchar(','));
			lex.eatchar(')');
			return new SqlInsertValues(t, l, m);
		} else if(lex.eqsym("SELECT")) {
			q = _select(lex);
			return new SqlInsertSelect(t, l, q);
		} else {
			throw ErrorBundle.getDefault(10026, lex.get().toString());
		}
	}

	Object _update(DbSqlLexer lex) throws IOException, SQLException {
		List<SqlExpression> l = new ArrayList<SqlExpression>();
		List<String> n = new ArrayList<String>();
		SqlExpression w = null;
		String t, s;

		t = lex.eatsym();
		lex.eatsym("SET");
		do {
			s = lex.eatsym();
			lex.eat(SqlBinaryOperator.EQ);
			n.add(s);
			l.add(new DbSqlExprParser().parse(lex));
		} while(lex.eqchar(','));

		// WHERE
		if(lex.eq(DbSqlReserved.WHERE)) {
			w = new DbSqlExprParser().parse(lex);
		}
		return new SqlUpdate(t, n, l, w);
	}

	Object _delete(DbSqlLexer lex) throws IOException, SQLException {
		SqlExpression w = null;
		String t;

		lex.eat(DbSqlReserved.FROM);
		t = lex.eatsym();

		// WHERE
		if(lex.eq(DbSqlReserved.WHERE)) {
			w = new DbSqlExprParser().parse(lex);
		}
		return new SqlDelete(t, w);
	}

	Object _drop(DbSqlLexer lex) throws IOException, SQLException {
		String t;

		if(lex.eqsym("TABLE")) {
			t = lex.eatsym();
			return new SqlDropTable(t);
		} else {
			throw ErrorBundle.getDefault(10027, lex.get().toString());
		}
	}

	Object _truncate(DbSqlLexer lex) throws IOException, SQLException {
		String t;

		if(lex.eqsym("TABLE")) {
			t = lex.eatsym();
			return new SqlTruncateTable(t);
		} else {
			throw ErrorBundle.getDefault(10028, lex.get().toString());
		}
	}

	private void _chkattr(
			List<SqlColumnDefinition> t) throws SQLException {
		for(SqlColumnDefinition d : t) {
			if(d.getAttributes().contains(
					SqlColumnAttribute.PRIMARY_KEY)) {
				throw ErrorBundle.getDefault(10029);
			} else if(d.getAttributes().contains(
					SqlColumnAttribute.NOT_NULL)) {
				throw ErrorBundle.getDefault(10030);
			}
		}
	}

	Object _alter_table(
			DbSqlLexer lex)  throws IOException, SQLException {
		List<SqlColumnDefinition> t;
		List<String> l;
		String n, s, v;

		n = lex.eatsym();
		if(lex.eqsym("ADD")) {
			t = new ArrayList<SqlColumnDefinition>();
			lex.eatchar('(');
			do {
				t.add(_ctcolumn(lex));
			} while(lex.eqchar(','));
			lex.eatchar(')');
			_chkattr(t);
			return new SqlAlterTableAdd(n, t);
		} else if(lex.eqsym("MODIFY")) {
			t = new ArrayList<SqlColumnDefinition>();
			lex.eatchar('(');
			do {
				t.add(_ctcolumn(lex));
			} while(lex.eqchar(','));
			lex.eatchar(')');
			_chkattr(t);
			return new SqlAlterTableModify(n, t);
		} else if(lex.eqsym("DROP")) {
			l = new ArrayList<String>();
			lex.eatchar('(');
			do {
				l.add(lex.eatsym());
			} while(lex.eqchar(','));
			lex.eatchar(')');
			return new SqlAlterTableDrop(n, l);
//		} else if(!lex.eqsym("RENAME")) {
//			throw new SQLException();
		} else if(lex.eqsym("COLUMN")) {
			s = lex.eatsym();
			lex.eatsym("TO");
			v = lex.eatsym();
			return new SqlAlterTableRenameColumn(n, s, v);
		} else {
//			lex.eatsym("TO");
//			v = lex.eatsym();
//			return new SqlAlterTableRename(n, v);
			throw ErrorBundle.getDefault(10031, lex.get().toString());
		}
	}

	Object _alter(DbSqlLexer lex)  throws IOException, SQLException {
		if(lex.eqsym("TABLE")) {
			return _alter_table(lex);
		} else {
			throw ErrorBundle.getDefault(10032, lex.get().toString());
		}
	}

	Object _top(DbSqlLexer lex) throws IOException, SQLException {
		if(lex.eqsym("CREATE")) {
			return _create(lex);
		} else if(lex.eqsym("INSERT")) {
			return _insert(lex);
		} else if(lex.eqsym("UPDATE")) {
			return _update(lex);
		} else if(lex.eqsym("DELETE")) {
			return _delete(lex);
		} else if(lex.eqsym("DROP")) {
			return _drop(lex);
		} else if(lex.eqsym("TRUNCATE")) {
			return _truncate(lex);
		} else if(lex.eqsym("ALTER")) {
			return _alter(lex);
		} else {
			return new DbSqlSetParser().parse(lex);
		}
	}

	/**
	 * 
	 * @param lex
	 * @return
	 * @throws IOException
	 * @throws SQLException
	 */
	public Object parse(
			DbSqlLexer lex) throws IOException, SQLException {
		Object o = _top(lex);

		if(!lex.isEnd()) {
			throw ErrorBundle.getDefault(10033);
		}
		return o;
	}

	/**
	 * 
	 * @param sql
	 * @return
	 * @throws IOException
	 * @throws SQLException
	 */
	public static Object parse(String sql) throws IOException, SQLException {
		DbSqlLexer lex = new DbSqlLexer(new StringReader(sql));

		return new DbSqlParser().parse(lex);
	}

}
