/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.util {
	import flash.display.DisplayObject;
	import flash.display.DisplayObjectContainer;
	
	import jp.co.fujitsu.reffi.client.flex.controller.BaseController;
	
	import mx.core.Application;
	import mx.core.UIComponent;
	
	/**
	 * <p>[概 要]</p>
	 * アプリケーションにロードされている構成要素を検索するクラスです.
	 * 
	 * <p>[詳 細]</p>
	 * AVM上に読み込んでいる、画面コンポーネントやコントローラを検索します。<br>
	 * 画面コンポーネントを探す場合、検索キーとしてname属性値、もしくはid属性値を指定します。<br>
	 * 
	 * <p>[備 考]</p>
	 * name属性値でコンポーネントを検索する場合、画面上のコンテナの子を再帰的に走査します。<br>
	 * 読み込まれているコンテナによっては子要素にアクセス出来ない（セキュリティエラーが発生する）
	 * ものが有ります（例 GoogleMapのMapオブジェクト）。<br>
	 * このエラーは、ClientConfigクラスのexcludeFromComponentSearchプロパティに走査対象外クラス型を
	 * 追加することで回避できます。<br>
	 * <listing version="3.0">
        package mapdemo.controller {
        
            import com.google.maps.Map;
            import com.google.maps.MapEvent;
            
            import jp.co.fujitsu.reffi.client.flex.controller.BaseController;
            import jp.co.fujitsu.reffi.client.flex.controller.ClientConfig;
            import jp.co.fujitsu.reffi.client.flex.controller.EventBinder;
            
            import mapdemo.action.MapReadyAction;
            
            public class MapDemoController extends BaseController {

                // アプリケーション初期化時に走査対象外クラスとしてcom.google.maps.Mapを追加        
                override protected function initialize(clientConfig:ClientConfig):void {
                    clientConfig.excludeFromComponentSearch.push(Class(Map));
                }

                override protected function bind(eventBinder:EventBinder):void {
                    eventBinder.addEventBinding("mapPanel.map", MapEvent.MAP_READY, Class(MapReadyAction));
                        :
                        :
	 * </listing>
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi 
	 */
	public class CompositionSearchUtil {
		
		/** 自Document範囲を表す定数です。 */
		public static const DOCUMENT_RANGE:int = 1;
		
		/** SystemManager配下の、自Documentを除いた他Document範囲を表す定数です。 */
		public static const OTHER_DOCUMENTS_RANGE:int = 2;
		
		/** 自SystemManager範囲を表す定数です。 */
		public static const ALL_RANGE:int = 3;
		

		/**
		 * <p>[概 要]</p>
		 * 引数namePatternと合致するname属性を持つコンポーネントを検索します.
		 * 
		 * <p>[詳 細]</p>
		 * 第一引数documentからnamePatternをname属性値として持つコンポーネントを再帰的に検索します。<br>
		 * document領域内に存在しなかった場合、SystemManagerが持つ他の子Documentを同様に検索します。<br>
		 * 
		 * 検索途中、該当するコンポーネントが発見された時点でそのコンポーネントを返却します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @param document 検索対象document
		 * @param namePattern 取得したいコンポーネントのname属性値
		 * @param searchRange 検索範囲
		 */
		public static function searchComponentByName(document:Object,
														namePattern:String,
														searchRange:int=ALL_RANGE):DisplayObject 
		{
			if (!(document is DisplayObjectContainer)) {
				return null;
			}

			var array:Array = new Array();

			// 自分の所属するDocumentを検索
			if((searchRange && DOCUMENT_RANGE) == DOCUMENT_RANGE) {
				searchComponentByNameRecursively(DisplayObjectContainer(document), namePattern, array, false);
				if(array.length != 0) {
					return array[0];
				}
			}
		
			// 自分のDocument以外のDocumentを検索
			if((searchRange & OTHER_DOCUMENTS_RANGE) == OTHER_DOCUMENTS_RANGE) {
				var application:Application = searchApplication(document);
				if (application == null) {
					return null;
				}
				// 引数SystemManagerが保持する子Document（MXML単位）からコンポーネント検索
				var numChildren:int = application.systemManager.numChildren;
				for(var i:int=0; i<numChildren; i++) {
					var child:DisplayObjectContainer = application.systemManager.getChildAt(i) as DisplayObjectContainer;
					// 自分が属するDocumentは検索済み
					if (child == null || child == document) {
						continue;
					}
					searchComponentByNameRecursively(child, namePattern, array, false);
					if(array.length != 0) {
						return array[0];
					}
				}
			}
			
			return null;
		}

		/**
		 * <p>[概 要]</p>
		 * 引数namePatternと合致するname属性を持つコンポーネント群を検索します.
		 * 
		 * <p>[詳 細]</p>
		 * 第一引数document、SystemManagerが持つ他の子Documentの順番で
		 * namePatternをname属性値として持つコンポーネントを再帰的に検索します。<p>
		 * 
		 * ロードされている全コンポーネントの内、namePatternに合致する全コンポーネントの配列を返却します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * 
		 * @example
		 * AVM上のname属性の付いた全DisplayObjectを取得する
		 * <listing version="3.0">
		 * var allNamedComponents:Array = getComponentsByName(".&#42;");
		 * </listing>
		 * 
		 * 「btn」という文字列を含むname属性値が付与されている全DisplayObjectを取得する
		 * <listing version="3.0">
		 * var allNamedButtons:Array = getComponentsByName(".&#42;btn.&#42;");
		 * </listing>
		 * 
		 * @param namePattern
		 * @param searchRange コンポーネントを探す範囲（デフォルト：ALL_RANGE）
		 * @return 
		 */
		public static function searchComponentsByName(document:Object, 
														 namePattern:String, 
														 searchRange:int=ALL_RANGE):Array 
		{
			var array:Array = new Array();
			
			if (!(document is DisplayObjectContainer)) {
				return array;
			}

			// 自分の所属するDocumentを検索
			if((searchRange && DOCUMENT_RANGE) == DOCUMENT_RANGE) {
				searchComponentByNameRecursively(DisplayObjectContainer(document), namePattern, array, true);
			}
			
			// 自分のDocument以外のDocumentを検索
			if((searchRange & OTHER_DOCUMENTS_RANGE) == OTHER_DOCUMENTS_RANGE) {
				var application:Application = searchApplication(document);
				if (application == null) {
					return array;
				}
				// 引数SystemManagerが保持する子Document（MXML単位）からコンポーネント検索
				var numChildren:int = application.systemManager.numChildren;
				for(var i:int=0; i<numChildren; i++) {
					var child:DisplayObjectContainer = application.systemManager.getChildAt(i) as DisplayObjectContainer;
					// 自分が属するDocumentは検索済み
					if (child == null || child == document) {
						continue;
					}
					searchComponentByNameRecursively(child, namePattern, array, true);
				}
			}
			
			return array;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 引数namePatternと合致するname属性を持つコンポーネントを検索します.
		 * 
		 * <p>[詳 細]</p>
		 * 指定したdocumentからnamePatternをname属性値として持つコンポーネントを検索します。<p>
		 * 検索は指定したdocumentの親を辿る形となり、document自体も対象となります。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * 
		 * @example
		 * AVM上のname属性の付いたDisplayObjectを取得する
		 * <listing version="3.0">
		 * var nameComponent:DisplayObject = searchComponentByNameParentTrace(target, ".&#42;");
		 * </listing>
		 * 
		 * 「btn」という文字列を含むname属性値が付与されているDisplayObjectを親を辿って取得する
		 * <listing version="3.0">
		 * var nameComponent:DisplayObject = searchComponentByNameParentTrace(target, ".&#42;btn.&#42;");
		 * </listing>
		 *
		 * @param document 検索対象となるドキュメント
		 * @param namePattern 検索する名前
		 * @return 検索に該当したコンポーネント、該当がない場合はnull
		 */
		public static function searchComponentByNameParentTrace(document:Object, namePattern:String):DisplayObject {
			// 指定されたドキュメントがDisplayObjectContainerでない場合は終了
			if (!(document is DisplayObjectContainer)) {
				return null;
			}
			var array:Array = new Array();
			// 指定されたdocumentを検索
			searchComponentByNameRecursively(DisplayObjectContainer(document), namePattern, array, false);
			if (array.length != 0) {
				return array[0];
			}
			// 指定されたdocumentに親がない場合は終了
			if (DisplayObjectContainer(document).parent == null) {
				return null;
			}
			return searchComponentByNameParentTrace(DisplayObjectContainer(document).parent, namePattern);
		}

		/**
		 * <p>[概 要]</p>
		 * 引数namePatternと合致するname属性を持つコンポーネント群を再帰的に検索します.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 *
		 */
		private static function searchComponentByNameRecursively(container:DisplayObjectContainer, 
															namePattern:String, 
															array:Array, 
															multi:Boolean):void 
		{
			// containerがコンポーネント検索除外クラス配列に入っているクラス型の場合
			for each(var excludeClass:Class in BaseController.clientConfig.excludeFromComponentSearch) {
				if(container is excludeClass) {
					return;
				}
			}
			
			// 第一引数が保持するrawChildrenの数を取得
			var numChildren:int = container.numChildren;
			// rawChildren数分name属性検索
			for (var i:int; i<numChildren; i++) {
				var child:DisplayObject = container.getChildAt(i);
				// 子コンポーネントのname属性が第二引数nameと等しい場合
				var regexp:RegExp = new RegExp(namePattern);
				if(regexp.test(child.name)) {
					array.push(child);
					if(multi == false) {
						break;
					}
				}
				
				// 子コンポーネントが第二引数nameと等しくなく、
				// コンテナオブジェクトである場合は再帰開始
				if(child is DisplayObjectContainer) {
					searchComponentByNameRecursively(DisplayObjectContainer(child), namePattern, array, multi);
				}
				
				// 呼び出し先スタックでコンポーネントが発見されている場合、
				// 呼び出し元スタックループを抜ける
				if(array.length != 0) {
					if(multi == false) {
						break;
					}
				}
			}
		}

		/**
		 * <p>[概 要]</p>
		 * 引数idと合致するid属性を持つコンポーネントを検索します.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param id 取得するコンポーネントid属性値
		 * @return idを基に取得した画面コンポーネント
		 */
		public static function searchComponentById(document:Object, 
													  id:String, 
													  searchRange:int=ALL_RANGE):DisplayObject
		{
			
			if (!(document is DisplayObjectContainer)) {
				return null;
			} 

			// 自分のDocumentを検索
			if((searchRange & DOCUMENT_RANGE) == DOCUMENT_RANGE) {
				if(document.hasOwnProperty(id)){
					return document[id];
				}
			}
			
			// 自分のDocument以外のDocumentを検索
			if((searchRange & OTHER_DOCUMENTS_RANGE) == OTHER_DOCUMENTS_RANGE) {
				var application:Application = searchApplication(document);
				if (application == null) {
					return null;
				}
				// 引数SystemManagerが保持する子Document（MXML単位）からコンポーネント検索
				var numChildren:int = application.systemManager.numChildren;
				for(var i:int=0; i<numChildren; i++) {
					var child:DisplayObject = application.systemManager.getChildAt(i) as DisplayObject;
					// 自分が属するDocumentは検索済み
					if(child == null || child == document) {
						continue;
					}
					if(child.hasOwnProperty(id)){
						return child[id];
					}
				}
			}
			
			return null;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 引数idと合致するid属性を持つコンポーネント群を検索します.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param id 取得するコンポーネントid属性値
		 * @return idを基に取得した画面コンポーネント
		 */
		public static function searchComponentsById(document:Object, 
													   id:String, 
													   searchRange:int=ALL_RANGE):Array
		{
			var array:Array = new Array();
			
			if (!(document is DisplayObjectContainer)) {
				return array;
			} 

			// 自分のDocumentを検索
			if((searchRange & DOCUMENT_RANGE) == DOCUMENT_RANGE) {
				if(document.hasOwnProperty(id)){
					array.push(document[id]);
				}
			}
			
			// 自分のDocument以外のDocumentを検索
			if((searchRange & OTHER_DOCUMENTS_RANGE) == OTHER_DOCUMENTS_RANGE) {
				var application:Application = searchApplication(document);
				if (application == null) {
					return array;
				}
				// 引数SystemManagerが保持する子Document（MXML単位）からコンポーネント検索
				var numChildren:int = application.systemManager.numChildren;
				for(var i:int=0; i<numChildren; i++) {
					var child:DisplayObject = application.systemManager.getChildAt(i) as DisplayObject;
					// 自分が属するDocumentは検索済み
					if(child == null || child == document) {
						continue;
					}
					if(child.hasOwnProperty(id)){
						array.push(child[id]);
					}
				}
			}
			
			return array;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 引数displayObjectの直近の親UIComponentを検索します.
		 * 
		 * <p>[詳 細]</p>
		 * displayObjectのparentプロパティを辿ってUIComponentである親を継続走査します。<br>
		 * 引数指定されたのがparentを持たないstage等であった場合、Application.applicationが返却されます。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @param displayObject 
		 */
		public static function searchParentUIComponent(displayObject:DisplayObject):UIComponent {
			var parent:Object = displayObject.parent;
			var ret:UIComponent = null;
			while (true) {
				if (parent == null) {
					ret = Application.application as UIComponent;
					break;
				}
				if (parent is UIComponent) {
					ret = parent as UIComponent;
					break;
				}
				if (!(parent is DisplayObject)) {
					break;
				}
				parent = DisplayObject(parent).parent;
			}
			return ret;
		}

		/**
		 * <p>[概 要]</p>
		 * 引数displayObjectが所属するDocumentを検索します.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 *
		 * @param displayObject 
		 */
		public static function searchDocument(displayObject:DisplayObject):DisplayObject {
			var ret:DisplayObject = null;
			
			if(displayObject is UIComponent) {
				ret = UIComponent(displayObject).document as DisplayObject;
			} else {
				var uiComponent:UIComponent = searchParentUIComponent(displayObject);
				
				if(uiComponent) {
					ret = uiComponent.document as DisplayObject;
				}
			}
			
			return ret;
		}	
					
		/**
		 * <p>[概 要]</p>
		 * 引数documentが所属するApplicationを検索します.
		 * 
		 * <p>[詳 細]</p>
		 * 引数documentのparentApplicationを取得して返却します。<br>
		 * documentがApplicationであった場合は引数documentを返却します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @param document Applicationを探すためのdocument
		 */
		public static function searchApplication(document:Object):Application {
			if (document is UIComponent && UIComponent(document).parentApplication != null) {
				return Application(UIComponent(document).parentApplication);
			}
			
			if(document is Application) {
				return Application(document);
			}
			
			return null;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 引数documentが所属するApplicationを管理するコントローラを検索します.
		 * 
		 * <p>[詳 細]</p>
		 * コントローラに管理されている、AVM上にロードされているコントローラマップから、
		 * 引数documentを管理しているコントローラインスタンスを返却します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @parma document コントローラを探すためのdocument
		 */
		public static function searchController(document:Object):BaseController {
			if (!(document is DisplayObject)) {
				return null;
			}
			var application:Application = searchApplication(document);
			var controller:BaseController = BaseController.getController(application);
			
			return controller;
		}
	}
}
