/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.server.flex.manager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import flex.messaging.MessageBroker;
import flex.messaging.MessageDestination;
import flex.messaging.services.MessageService;

/**
 * <p>[概 要]</p>
 * メッセージ宛先管理。
 * 
 * <p>[詳 細]</p>
 * messaging-config.xmlに定義している宛先以外で、動的に宛先を登録、削除します。
 * 
 * <p>[備 考]</p>
 * 本クラスは現在開発、検証中のため、試験的な提供となります。
 * 
 * @author Project Reffi
 */
public enum MessageManager {
	/** インスタンス */
	instance;
	/** messaging-config.xmlのserviceタグ属性に記述されているデフォルトのサービスID */
	public static final String DEFAULT_SERVICE_ID = "message-service";
	/** ロガー */
	private static final Log LOG = LogFactory.getLog(MessageManager.class);
	
	/**
	 * <p>[概 要]</p>
	 * メッセージ受信開始
	 * <p>[詳 細]</p>
	 * 指定した宛先でメッセージ受信を開始します。
	 * サービスIDは{@link DEFAULT_SERVICE_ID}が使用されます。
	 * <p>[備 考]</p>
	 * 
	 * @param destination 受信を開始する宛先
	 */
	public void start(String destination) {
		start(destination, DEFAULT_SERVICE_ID);
	}

	/**
	 * <p>[概 要]</p>
	 * メッセージ受信開始
	 * <p>[詳 細]</p>
	 * 指定した宛先とサービスIDで、メッセージ受信を開始します。
	 * メッセージ受信は指定したサービスID上で管理されます。
	 * <p>[備 考]</p>
	 * 
	 * @param destination 受信を開始する宛先
	 * @param serviceId 宛先を管理するサービスID
	 */
	public void start(String destination, String serviceId) {
		if (exists(destination, serviceId)) {
			LOG.warn("destination exists (" + destination + ", " + serviceId + ")");
			return;
		}
		MessageService messageService = getMessageService(serviceId);
		MessageDestination messageDestination = (MessageDestination)messageService.createDestination(destination);
		if (messageService.isStarted()) {
			messageDestination.start();
			LOG.info("destination started (" + destination + ", " + serviceId + ")");
		} else {
			LOG.warn("destination not started (" + destination + ", " + serviceId + ")");
		}
	}

	/**
	 * <p>[概 要]</p>
	 * メッセージ受信停止
	 * <p>[詳 細]</p>
	 * 指定した宛先を停止します。
	 * サービスIDは{@link DEFAULT_SERVICE_ID}が使用されます。
	 * <p>[備 考]</p>
	 * 
	 * @param destination 受信を停止する宛先
	 */
	public void stop(String destination) {
		stop(destination, DEFAULT_SERVICE_ID);
	}
	
	/**
	 * <p>[概 要]</p>
	 * メッセージ受信停止
	 * <p>[詳 細]</p>
	 * 指定したサービスID上で管理されている宛先を停止します。
	 * <p>[備 考]</p>
	 * 
	 * @param destination 受信を停止する宛先
	 * @param serviceId 宛先を管理するサービスID
	 */
	public void stop(String destination, String serviceId) {
		if (!exists(destination, serviceId)) {
			LOG.warn("destination not exists (" + destination + ", " + serviceId + ")");
			return;
		}
		MessageService messageService = getMessageService(serviceId);
		MessageDestination messageDestination = (MessageDestination)messageService.getDestination(destination);
		if (messageDestination.isStarted()) {
			messageDestination.stop();
			LOG.info("destination stopped (" + destination + ", " + serviceId + ")");
		} else {
			LOG.warn("destination not stopped (" + destination + ", " + serviceId + ")");
		}
	}
	
	/**
	 * <p>[概 要]</p>
	 * メッセージ受信停止と登録削除
	 * <p>[詳 細]</p>
	 * 指定した宛先のメッセージ受信を停止し、管理上から削除します。
	 * サービスIDは{@link DEFAULT_SERVICE_ID}が使用されます。
	 * <p>[備 考]</p>
	 * 削除するため、再利用はできません。
	 * 
	 * @param destination 受信を停止、削除する宛先
	 */
	public void stopAndRemove(String destination) {
		stopAndRemove(destination, DEFAULT_SERVICE_ID);
	}
	
	/**
	 * <p>[概 要]</p>
	 * メッセージ受信停止と登録削除
	 * <p>[詳 細]</p>
	 * 指定したサービスID上で管理される宛先のメッセージ受信を停止し、管理上から削除します。
	 * <p>[備 考]</p>
	 * 削除するため、再利用はできません。
	 * 
	 * @param destination 受信を停止、削除する宛先
	 * @param serviceId 宛先を管理するサービスID
	 */
	public void stopAndRemove(String destination, String serviceId) {
		stop(destination, serviceId);
		MessageService messageService = getMessageService(serviceId);
		messageService.removeDestination(destination);
	}
	
	/**
	 * <p>[概 要]</p>
	 * 宛先の登録確認
	 * <p>[詳 細]</p>
	 * 指定した宛先が登録されているか確認をします。
	 * サービスIDは{@link DEFAULT_SERVICE_ID}が使用されます。
	 * <p>[備 考]</p>
	 * 
	 * @param destination 確認をする宛先
	 * @return 既に登録されいればtrue、それ以外はfalse
	 */
	public boolean exists(String destination) {
		return exists(destination, DEFAULT_SERVICE_ID);
	}
	
	/**
	 * <p>[概 要]</p>
	 * 宛先の登録確認
	 * <p>[詳 細]</p>
	 * 指定したサービスID管理にて、宛先が登録されているか確認をします。
	 * <p>[備 考]</p>
	 * 
	 * @param destination 確認をする宛先
	 * @param serviceId 確認をするサービスID
	 * @return 既に登録されいればtrue、それ以外はfalse
	 */
	public boolean exists(String destination, String serviceId) {
		MessageService messageService = getMessageService(serviceId);
		return (messageService.getDestination(destination) != null);
	}
	
	private MessageService getMessageService(String serviceId) {
		MessageBroker messageBroker = MessageBroker.getMessageBroker(null);
		if (messageBroker == null) {
			return null;
		}
		return (MessageService)messageBroker.getService(serviceId);
	}
}
