/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.logconsumer;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.Date;

import jp.co.fujitsu.reffi.client.nexaweb.controller.ClientConfig;

import com.nexaweb.util.Log;
import com.nexaweb.util.LogConsumer;
import com.nexaweb.util.LogEvent;

/**
 * <p>[概 要] </p>
 * Nexaweb LogConsumer I/Fを実装する為のアダプタクラスです。
 * 
 * <p>[詳 細] </p>
 * <ol>
 *     <li>LogConsumer I/Fが提供するinit、destroyの仮実装</li>
 *     <li>時刻ヘッダのカスタマイズ機能</li>
 * </ol>
 * を提供します。
 * 
 * <p>[備 考] </p>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi 
 */
public abstract class LogConsumerAdapter implements LogConsumer {

	/** ログヘッダに含まれる日付、時間フォーマットです。（デフォルト：HH:mm:ss.SSS） */
	private String dateFormat = "HH:mm:ss.SSS";
	
	/**
	 * <p>[概 要] </p>
	 * ログヘッダに含まれる日付、時間フォーマットを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * dateFormatフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ログヘッダに含まれる日付、時間フォーマット
	 */
	public String getDateFormat() {
		return dateFormat;
	}

	/**
	 * <p>[概 要] </p>
	 * ログヘッダに含まれる日付、時間フォーマットを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * dateFormatフィールドに引数dateFormatを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param dateFormat ログヘッダに含まれる日付、時間フォーマット
	 */
	public void setDateFormat(String dateFormat) {
		this.dateFormat = dateFormat;
	}

	/**
	 * <p>[概 要] </p>
	 * LogConsumer初期化メソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * LogConsumerI/Fを実装します。<br>
	 * 引数Propertiesから日付、時間フォーマットを取得します。（client.log.dateformatキー）<br>
	 * 取得したフォーマットはdateFormatフィールドに設定されます。
	 * 
	 * <p>[備 考] </p>
	 * Properties以外のobjは初期化パラメータとして認識されません。
	 * 
	 * @param obj 初期化パラメータ
	 */
	public void init(Object obj) {
		if(obj instanceof ClientConfig) {
			ClientConfig config = (ClientConfig)obj;
			String dateFormat = config.getLogDateFormat();
			setDateFormat(dateFormat);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * LogConsumer終了メソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * LogConsumerI/Fを実装します。<br>
	 * デフォルト処理は有りません。
	 * 
	 * <p>[備 考] </p>
	 * 実装は具象LogConsumerに委譲されます。
	 * 
	 */
	public void destroy() {
		
	}

	/**
	 * <p>[概 要] </p>
	 * 具象コンシューマで共通的に使用可能なログ文字列を生成します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * 
	 * <p>[備 考] </p>
	 * date formatを変更出来ないLogEvent#getLogString()の代わりとして使用出来ます。
	 * 
	 * @param evt ログイベント
	 * @return ログメッセージ文字列
	 */
	protected String createLogString(LogEvent evt) {
		StringBuilder sb = new StringBuilder();
		
		short level = evt.getType();
		long timestamp = evt.getTimestamp();
		
		String message = "";
		if(evt.getThrowable()!=null) {
			StringWriter sw = new StringWriter();
			PrintWriter pw = new PrintWriter(sw);
			evt.getThrowable().printStackTrace(pw);
			sw.flush(); 
			pw.flush();
			try{
				sw.close(); 
				pw.close();
			}catch(IOException e) {
				e.printStackTrace();
			}

			message = sw.toString();
		}else{
			message = evt.getMessage();
		}
		
		String header = getHeader(level, timestamp);
		sb.append("[");
		sb.append(header);
		sb.append("]");
		sb.append(" ");
		sb.append(message);

		
		return sb.toString();
	}
	
	/**
	 * <p>[概 要] </p>
	 * ログヘッダ文字列を返却します。
	 * 
	 * <p>[詳 細] </p>
	 * 引数levelからレベルの文字列表現を、引数timestampから日付、時刻の文字列表現
	 * を求め、levelstring-datetimeの形式で返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param level ログレベル
	 * @param timestamp 時刻文字列変換用long値
	 * @return ログヘッダ文字列
	 */
	protected String getHeader(short level, long timestamp) {
		StringBuilder sb = new StringBuilder();
		String levelstring = getLevelString(level);
		String datetime = getDateTime(timestamp);
		
		sb.append(levelstring);
		sb.append("-");
		sb.append(datetime);
		
		return sb.toString();
	}
	
	/**
	 * <p>[概 要] </p>
	 * 引数timestampの日付、時間文字列表現を返却します。
	 * 
	 * <p>[詳 細] </p>
	 * dateFormatフィールドを引数としたSimpleDateFormatオブジェクトで、
	 * 引数timestampをフォーマット、戻り値を返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param timestamp 時刻文字列変換用long値
	 * @return 文字列変換されたtimestamp
	 */
	protected String getDateTime(long timestamp) {
		Date date = new Date(timestamp);
		SimpleDateFormat sdf = new SimpleDateFormat(getDateFormat());
		String datetime = sdf.format(date);

		return datetime;
	}
	
	/**
	 * <p>[概 要] </p>
	 * Log.[レベル]の文字列表現を返却します。
	 * 
	 * <p>[詳 細] </p>
	 * short値であるLog.[レベル]を文字列表記に変換して返却します。
	 * 
	 * <ul>
	 *     <li>1 : "DEBUG"</li>
	 *     <li>2 : "INFO"</li>
	 *     <li>3 : "ERROR"</li>
	 *     <li>4 : "EXCEPTION"</li>
	 * </ul>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param level ログレベル（LogEvent#getType()値）
	 * @return 文字列表現されたログレベル
	 */
	protected String getLevelString(short level) {
		String ret;
		
		switch(level) {
			case Log.DEBUG:
				ret = "DEBUG";
				break;
			case Log.INFO:
				ret = "INFO";
				break;
			case Log.ERROR:
				ret = "ERROR";
				break;
			case Log.EXCEPTION:
				ret = "EXCEPTION";
				break;
			default:
				ret = "UNKNOWN LEVEL";
		}
		
		return ret;
	}
}
