/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.server.messaging;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <p>[概 要]</p>
 * メッセージ受信マネージャ。
 * 
 * <p>[詳 細]</p>
 * 複数のメッセージ受信クラスを管理します。
 * <p>
 * RIAプロダクトで多く利用されるクライアント/サーバ間のメッセージ送受信を管理できるクラスと
 * なっており、{@link AbstractMessageReceiver}を実装したクラスを登録することで
 * クライアントからのメッセージ受信を管理することが可能となります。
 * 
 * <p>[備 考]</p>
 *
 * @author Project Reffi
 */
public enum ReceiverManager {
	/** インスタンス */
	instance;
	/** メッセージ受信クラスマップ */
	private Map<String, List<AbstractMessageReceiver>> _receiverMap = new HashMap<String, List<AbstractMessageReceiver>>();
	
	synchronized public List<AbstractMessageReceiver> getReceivers(String name) {
		if (this._receiverMap.containsKey(name)) {
			return this._receiverMap.get(name);
		}
		return null;
	}
	
	/**
	 * <p>[概 要]</p>
	 * メッセージ受信クラス登録
	 * 
	 * <p>[詳 細]</p>
	 * 指定した名前(トピック)でメッセージ受信クラスを登録します。
	 * 既に同じ名前で登録されている場合は追加登録となり、以前に登録した
	 * ものは削除されません。
	 * 
	 * <p>[備 考]</p>
	 * メッセージ受信クラスは開発者にて実装する必要があります。
	 * 
	 * @param topic 登録するメッセージ受信名
	 * @param messageReceiver 実装されたメッセージ受信クラス
	 */
	synchronized public void addReceiver(String topic, AbstractMessageReceiver messageReceiver) {
		List<AbstractMessageReceiver> messageReceivers = getReceivers(topic);
		if (messageReceivers == null) {
			messageReceivers = new ArrayList<AbstractMessageReceiver>();
		}
		if (!messageReceivers.contains(messageReceiver)) {
			messageReceivers.add(messageReceiver);
			messageReceiver.start(topic);
		}
		this._receiverMap.put(topic, messageReceivers);
	}
}
