// -*-c++-*-

/*!
  \file view_config_dialog.cpp
  \brief View Config Dialog class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "view_config_dialog.h"

#include "main_data.h"
#include "options.h"

#include <iostream>
#include <cassert>

/*-------------------------------------------------------------------*/
/*!

 */
ViewConfigDialog::ViewConfigDialog( QWidget * parent,
                                    const MainData & main_data )

    : QTabDialog( parent )
    , M_main_data( main_data )
{
    this->setCaption( tr( "View Preference" ) );

    //QFont font = this->font();
    //font.setPointSize( 8 );
    //this->setFont( font );

    createWidgets();

    this->setOkButton( QString::null ); // no ok button
}

/*-------------------------------------------------------------------*/
/*!

 */
ViewConfigDialog::~ViewConfigDialog()
{
    //std::cerr << "delete ViewConfigDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::createWidgets()
{
    // canvas
    {
        QVBox * box = new QVBox( this );

        createZoomControls( box );
        createObjectSizeControls( box );
        createCanvasSizeControls( box );
        createFieldStyleControls( box );
        createMiscControls( box );

        box->layout()->setResizeMode( QLayout::Fixed );
        this->addTab( box, tr( "Canvas" ) );
    }
    // show
    {
        QVBox * box = new QVBox( this );

        createShowControls( box );
        createPlayersDetailControls( box );
        createReverseControls( box );
        createGridStepControls( box );
        createCompGeomControls( box );

        box->layout()->setResizeMode( QLayout::Fixed );
        this->addTab( box, tr( "Show" ) );
    }
    // trace / future
    {
        QVBox * box = new QVBox( this );

        createFocusControls( box );
        createPlayerSelectionControls( box );
        createTraceControls( box );
        createInertiaMoveControls( box );

        box->layout()->setResizeMode( QLayout::Fixed );
        this->addTab( box, tr( "Trace" ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
ViewConfigDialog::createZoomControls( QWidget * parent )
{
    QVGroupBox * group_box = new QVGroupBox( tr( "Zoom" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    const QFontMetrics metrics = parent->fontMetrics();
    const int button_height = metrics.height() + 12;

    {
        QFrame * box = new QFrame( group_box );
        QHBoxLayout * layout = new QHBoxLayout( box );

        QPushButton * zoomin_button = new QPushButton( tr( "+" ), box );
        zoomin_button->setMaximumSize( 48, button_height );
        connect( zoomin_button, SIGNAL( clicked() ),
                 this, SLOT( zoomIn() ) );
        layout->addWidget( zoomin_button, 0, Qt::AlignLeft );
        //
        QPushButton * zoomout_button = new QPushButton( tr( "-" ), box );
        zoomout_button->setMaximumSize( 48, button_height );
        connect( zoomout_button, SIGNAL( clicked() ),
                 this, SLOT( zoomOut() ) );
        layout->addWidget( zoomout_button, 0, Qt::AlignLeft );
        //
        layout->addStretch( 1 );
        //
        QPushButton * fit_button = new QPushButton( tr( "Fit" ), box );
        fit_button->setMaximumSize( 60, button_height );
        connect( fit_button, SIGNAL( clicked() ),
                 this, SLOT( unzoom() ) );
        layout->addWidget( fit_button, 0, Qt::AlignRight );
    }
    {
        QHBox * box = new QHBox( group_box );

        new QLabel( tr( "Scale: " ), box );
        //
        M_scale_slider = new QSlider( Qt::Horizontal, box );
        M_scale_slider->setRange( static_cast< int >( Options::MIN_FIELD_SCALE * 10.0 ),
                                  static_cast< int >( Options::MAX_FIELD_SCALE * 10.0 ) );
        M_scale_slider->setValue( 80 );
        M_scale_slider->setSteps( 1, 1 );
        M_scale_slider->setMinimumSize( 300, 16 );
        connect( M_scale_slider, SIGNAL( sliderMoved( int ) ),
                 this, SLOT( slideFieldScale( int ) ) );
        //
        M_scale_text = new QLineEdit( tr( "0.0" ), box );
        M_scale_text->setValidator( new QDoubleValidator( 0.0, 400.0, 3, M_scale_text ) );
        M_scale_text->setMaximumSize( 48, 32 );
        M_scale_text->setMinimumSize( 40, 24 );
        connect( M_scale_text, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editFieldScale( const QString & ) ) );
    }

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
ViewConfigDialog::createObjectSizeControls( QWidget * parent )
{
    const Options & opt = Options::instance();

    QVGroupBox * group_box = new QVGroupBox( tr( "Object Size" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    M_enlarge_cb = new QCheckBox( tr( "Enlarge Objects" ), group_box );
    M_enlarge_cb->setChecked( opt.enlargeMode() );
    connect( M_enlarge_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickEnlarge( bool ) ) );

    QHBox * box = new QHBox( group_box );
    //
    new QLabel( tr( " Ball Size:" ), box );
    M_ball_size_text = new QLineEdit( tr( "0.35" ), box );
    M_ball_size_text->setValidator( new QDoubleValidator( 0.01, 100.0, 3, M_ball_size_text ) );
    M_ball_size_text->setMaximumSize( 48, 24 );
    connect( M_ball_size_text, SIGNAL( textChanged( const QString & ) ),
             this, SLOT( editBallSize( const QString & ) ) );
    //
    new QLabel( tr( " Player Size:" ), box );
    M_player_size_text = new QLineEdit( tr( "0.0" ), box );
    M_player_size_text->setValidator( new QDoubleValidator( 0.0, 100.0, 3, M_ball_size_text ) );
    M_player_size_text->setMaximumSize( 48, 24 );
    connect( M_player_size_text, SIGNAL( textChanged( const QString & ) ),
             this, SLOT( editPlayerSize( const QString & ) ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
ViewConfigDialog::createCanvasSizeControls( QWidget * parent )
{
    QHGroupBox * group_box = new QHGroupBox( tr( "Canvas Size" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    new QLabel( tr( " Width:" ), group_box );

    M_canvas_width_text = new QLineEdit( tr( "640" ), group_box );
    M_canvas_width_text->setValidator( new QIntValidator( 100, 3000, M_canvas_width_text ) );
    M_canvas_width_text->setMaximumSize( 48, 24 );

    new QLabel( tr( " Height:" ), group_box );

    M_canvas_height_text = new QLineEdit( tr( "480" ), group_box );
    M_canvas_height_text->setValidator( new QIntValidator( 100, 3000, M_canvas_height_text ) );
    M_canvas_height_text->setMaximumSize( 48, 24 );

    QPushButton * apply_canvas_size_btn = new QPushButton( tr( "apply" ), group_box );
    apply_canvas_size_btn->setMaximumSize( 60, parent->fontMetrics().height() + 12 );
    connect( apply_canvas_size_btn, SIGNAL( clicked() ),
             this, SLOT( applyCanvasSize() ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
ViewConfigDialog::createReverseControls( QWidget * parent )
{
    QHGroupBox * group_box = new QHGroupBox( tr( "Reverse" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    M_reverse_side_cb = new QCheckBox( tr( "Reverse Side" ), group_box );
    M_reverse_side_cb->setChecked( Options::instance().reverseSide() );
    connect( M_reverse_side_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickReverseSide( bool ) ) );

    M_player_reverse_draw_cb = new QCheckBox( tr( "Reverse Player Draw Order" ),
                                              group_box );
    M_player_reverse_draw_cb->setChecked( Options::instance().playerReverseDraw() );
    connect( M_player_reverse_draw_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickPlayerReverseDraw( bool ) ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
ViewConfigDialog::createPlayersDetailControls( QWidget * parent )
{
    const Options & opt = Options::instance();

    QVGroupBox * group_box = new QVGroupBox( tr( "Players\' Detail" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 0 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );
    {
        QHBox * box = new QHBox( group_box );
        box->setSpacing( 2 );

        //
        M_player_number_cb = new QCheckBox( tr( "Unum" ), box );
        M_player_number_cb->setChecked( opt.showPlayerNumber() );
        connect( M_player_number_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowPlayerNumber( bool ) ) );
        //
        M_player_type_cb = new QCheckBox( tr( "Type" ), box );
        M_player_type_cb->setChecked( opt.showPlayerType() );
        connect( M_player_type_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowPlayerType( bool ) ) );
        //
        M_stamina_cb = new QCheckBox( tr( "Stamina" ), box );
        M_stamina_cb->setChecked( opt.showStamina() );
        connect( M_stamina_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowStamina( bool ) ) );
        //
        M_stamina_capacity_cb = new QCheckBox( tr( "Stamina Capacity" ), box );
        M_stamina_capacity_cb->setChecked( opt.showStaminaCapacity() );
        connect( M_stamina_capacity_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowStaminaCapacity( bool ) ) );
    }
    {
        QHBox * box = new QHBox( group_box );
        box->setSpacing( 2 );

        //
        M_view_area_cb = new QCheckBox( tr( "View Area" ), box );
        M_view_area_cb->setChecked( opt.showViewArea() );
        connect( M_view_area_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowViewArea( bool ) ) );
        //
        M_body_shadow_cb = new QCheckBox( tr( "Body Shadow" ), box );
        M_body_shadow_cb->setChecked( opt.showBodyShadow() );
        connect( M_body_shadow_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowBodyShadow( bool ) ) );
        //
        M_catchable_area_cb = new QCheckBox( tr( "Catch Area" ), box );
        M_catchable_area_cb->setChecked( opt.showCatchableArea() );
        connect( M_catchable_area_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowCatchableArea( bool ) ) );
    }
    {
        QHBox * box = new QHBox( group_box );
        box->setSpacing( 2 );

        //
        M_tackle_area_cb = new QCheckBox( tr( "Tackle Area" ), box );
        M_tackle_area_cb->setChecked( opt.showTackleArea() );
        connect( M_tackle_area_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowTackleArea( bool ) ) );
        //
        M_kick_accel_area_cb = new QCheckBox( tr( "Kick Accel Area" ), box );
        M_kick_accel_area_cb->setChecked( opt.showKickAccelArea() );
        connect( M_kick_accel_area_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowKickAccelArea( bool ) ) );
    }
    {
        QHBox * box = new QHBox( group_box );
        box->setSpacing( 2 );

        //
        M_pointto_cb = new QCheckBox( tr( "Pointto" ), box );
        M_pointto_cb->setChecked( opt.showPointto() );
        connect( M_pointto_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowPointto( bool ) ) );
        //
        M_attentionto_cb = new QCheckBox( tr( "Attentionto" ), box );
        M_attentionto_cb->setChecked( opt.showAttentionto() );
        connect( M_attentionto_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowAttentionto( bool ) ) );
        //
        M_card_cb = new QCheckBox( tr( "Card" ), box );
        M_card_cb->setChecked( opt.showCard() );
        connect( M_card_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowCard( bool ) ) );
    }
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
ViewConfigDialog::createShowControls( QWidget * parent )
{
    const Options & opt = Options::instance();

    QVGroupBox * group_box = new QVGroupBox( tr( "Show" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    {
        QHBox * box = new QHBox( group_box );
        box->setSpacing( 2 );
        //
        M_anonymous_mode_cb = new QCheckBox( tr( "Anonymous Mode" ), box );
        M_anonymous_mode_cb->setChecked( opt.anonymousMode() );
        connect( M_anonymous_mode_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickAnonymousMode( bool ) ) );
    }
    {
        QHBox * top_box = new QHBox( group_box );
        top_box->setSpacing( 2 );

        //
        M_show_score_board_cb = new QCheckBox( tr( "Score Board" ), top_box );
        M_show_score_board_cb->setChecked( opt.showScoreBoard() );
        connect( M_show_score_board_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowScoreBoard( bool ) ) );
        //
        M_show_team_graphic_cb = new QCheckBox( tr( "Team Graphic" ), top_box );
        M_show_team_graphic_cb->setChecked( opt.showTeamGraphic() );
        connect( M_show_team_graphic_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowTeamGraphic( bool ) ) );
        //
        M_show_flags_cb = new QCheckBox( tr( "Flag" ), top_box );
        M_show_flags_cb->setChecked( opt.showFlags() );
        connect( M_show_flags_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowFlags( bool ) ) );
    }
    {
        QHBox * bottom_box = new QHBox( group_box );
        //
        M_show_ball_cb = new QCheckBox( tr( "Ball" ), bottom_box );
        M_show_ball_cb->setChecked( opt.showBall() );
        connect( M_show_ball_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowBall( bool ) ) );
        //
        M_show_players_cb = new QCheckBox( tr( "Player" ), bottom_box );
        M_show_players_cb->setChecked( opt.showPlayers() );
        connect( M_show_players_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowPlayers( bool ) ) );
        //
        M_show_offside_line_cb = new QCheckBox( tr( "Offside Line" ), bottom_box );
        M_show_offside_line_cb->setChecked( opt.showOffsideLine() );
        connect( M_show_offside_line_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowOffsideLine( bool ) ) );
    }

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
ViewConfigDialog::createCompGeomControls( QWidget * parent )
{
    const Options & opt = Options::instance();

    QVGroupBox * group_box = new QVGroupBox( tr( "Compututational Geometry" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    {
        QHBox * above_box = new QHBox( group_box );
        //
        M_show_voronoi_cb = new QCheckBox( tr( "VoronoiDiagram" ), above_box );
        M_show_voronoi_cb->setChecked( opt.showVoronoiDiagram() );
        connect( M_show_voronoi_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowVoronoiDiagram( bool ) ) );
        //
        M_show_delaunay_cb = new QCheckBox( tr( "DelaunayTrianglation" ),
                                            above_box );
        M_show_delaunay_cb->setChecked( opt.showDelaunayTriangulation() );
        connect( M_show_delaunay_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowDelaunayTriangulation( bool ) ) );
    }

    M_voronoi_choice = new QComboBox( group_box );
    M_voronoi_choice->insertItem( tr( "All" ) );
    M_voronoi_choice->insertItem( tr( "Left Team" ) );
    M_voronoi_choice->insertItem( tr( "Right Team" ) );
    M_voronoi_choice->setMaximumSize( this->fontMetrics().width( tr( "Right Team" ) ) + 32,
                                      this->fontMetrics().height() + 12 );
    M_voronoi_choice->setCurrentItem( 0 );
    connect( M_voronoi_choice, SIGNAL( activated( int ) ),
             this, SLOT( selectCompGeomSide( int ) ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
ViewConfigDialog::createFocusControls( QWidget * parent )
{
    QGroupBox * group_box = new QHButtonGroup( tr( "Focus" ),
                                               parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    //
    M_focus_ball_rb = new QRadioButton( tr( "Ball" ), group_box );
    connect( M_focus_ball_rb, SIGNAL( clicked() ),
             this, SLOT( clickFocusBall() ) );
    //
    M_focus_player_rb = new QRadioButton( tr( "Selected Player" ),
                                          group_box );
    connect( M_focus_player_rb, SIGNAL( clicked() ),
             this, SLOT( clickFocusPlayer() ) );
    //
    M_focus_fix_rb = new QRadioButton( tr( "Fix" ), group_box );
    connect( M_focus_fix_rb, SIGNAL( clicked() ),
             this, SLOT( setFocusFix() ) );
    //
    M_focus_fix_rb->setChecked( true );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
ViewConfigDialog::createPlayerSelectionControls( QWidget * parent )
{
    QVGroupBox * group_box = new QVGroupBox( tr( "Player Selection" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    // selection type
    {
        QGroupBox * box = new QHButtonGroup( group_box );
        box->setFlat( true );
        box->setInsideMargin( 0 );
        box->setInsideSpacing( 0 );
        //
        M_select_all_rb = new QRadioButton( tr( "Auto" ), box );
        connect( M_select_all_rb, SIGNAL( clicked() ),
                 this, SLOT( clickSelectAutoAll() ) );
        //
        M_select_left_rb = new QRadioButton( tr( "Auto left" ), box );
        connect( M_select_left_rb, SIGNAL( clicked() ),
                 this, SLOT( clickSelectAutoLeft() ) );
        //
        M_select_right_rb = new QRadioButton( tr( "Auto right" ), box );
        connect( M_select_right_rb, SIGNAL( clicked() ),
                 this, SLOT( clickSelectAutoRight() ) );
        //
        M_select_fix_rb = new QRadioButton( tr( "Fix" ), box );
        connect( M_select_fix_rb, SIGNAL( clicked() ),
                 this, SLOT( clickSelectFix() ) );
        //
        M_unselect_rb = new QRadioButton( tr( "Unselect" ), box );
        connect( M_unselect_rb, SIGNAL( clicked() ),
                 this, SLOT( setUnselect() ) );
        //
        M_unselect_rb->setChecked( true );
    }

    {
        QFrame * box = new QFrame( group_box );
        QHBoxLayout * layout = new QHBoxLayout( box );
        //layout->setMargin( 0 );
        //layout->setSpacing( 0 );

        //layout->addSpacing( 16 ); // need to add the space before next choice

        M_player_choice = new QComboBox( box );
        M_player_choice->insertItem( tr ( "No Selection" ) );
        for ( int i = 1; i <= 11; ++i )
        {
            M_player_choice->insertItem( QString( "Left number: %1" ).arg( i ) );
        }
        for ( int i = 1; i <= 11; ++i )
        {
            M_player_choice->insertItem( QString( "Right number: %1" ).arg( i ) );
        }
        M_player_choice->setCurrentItem( 0 );
        int id_width = this->fontMetrics().width( tr( "Right Number: 11" ) );
        M_player_choice->setMaximumWidth( id_width + 40 );
        connect( M_player_choice, SIGNAL( activated( int ) ),
                 this, SLOT( choicePlayer( int ) ) );
        layout->addWidget( M_player_choice, 0, Qt::AlignLeft );
    }

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
ViewConfigDialog::createFieldStyleControls( QWidget * parent )
{
    const Options & opt = Options::instance();

    QGroupBox * group_box = new QVGroupBox( tr( "Field Style" ),
                                            parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    {
        QHButtonGroup * hbox = new QHButtonGroup( group_box );
        hbox->setFlat( true );
        hbox->setInsideMargin( 0 );
        hbox->setInsideSpacing( 0 );
        //
        M_keepaway_mode_cb = new QCheckBox( tr( "Keepaway" ), hbox );
        M_keepaway_mode_cb->setChecked( opt.keepawayMode() );
        connect( M_keepaway_mode_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickKeepawayMode( bool ) ) );
        //
        M_grass_normal_rb = new QRadioButton( tr( "mono" ), hbox );
        connect( M_grass_normal_rb, SIGNAL( clicked() ),
                 this, SLOT( clickGrassNormal() ) );
        //
        M_grass_line_rb = new QRadioButton( tr( "lines" ), hbox );
        connect( M_grass_line_rb, SIGNAL( clicked() ),
                 this, SLOT( clickGrassLine() ) );
        //
        M_grass_checker_rb = new QRadioButton( tr( "checker" ), hbox );
        connect( M_grass_checker_rb, SIGNAL( clicked() ),
                 this, SLOT( clickGrassChecker() ) );
    }
    {
        M_draw_type_choice = new QComboBox( group_box );
        M_draw_type_choice->insertItem( tr( "default" ) );
        M_draw_type_choice->insertItem( tr( "rcssmonitor" ) );
        connect( M_draw_type_choice, SIGNAL( activated( int ) ),
                 this, SLOT( selectDrawType( int ) ) );
    }

    switch ( Options::instance().fieldGrassType() ) {
    case Options::GRASS_NORMAL:
        M_grass_normal_rb->setChecked( true );
        break;
    case Options::GRASS_LINES:
        M_grass_line_rb->setChecked( true );
        break;
    case Options::GRASS_CHECKER:
        M_grass_checker_rb->setChecked( true );
        break;
    default:
        M_grass_normal_rb->setChecked( true );
        break;
    }

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
ViewConfigDialog::createMiscControls( QWidget * parent )
{
    const Options & opt = Options::instance();

    QGroupBox * group_box = new QVGroupBox( tr( "Misc" ),
                                            parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );

    M_cursor_hide_cb = new QCheckBox( tr( "Audomatic Cursor Hide" ), group_box );
    M_cursor_hide_cb->setChecked( opt.cursorHide() );
    connect( M_cursor_hide_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickCursorHide( bool ) ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
ViewConfigDialog::createGridStepControls( QWidget * parent )
{
    const Options & opt = Options::instance();

    QGroupBox * group_box = new QVGroupBox( tr( "Grid Lines" ),
                                            parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    //     group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
    //                                            QSizePolicy::Fixed ) );


    M_show_grid_coord_cb = new QCheckBox( tr( "Grid Coordinate" ), group_box );
    M_show_grid_coord_cb->setChecked( opt.showGridCoord() );
    connect( M_show_grid_coord_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickShowGridCoord( bool ) ) );
    //
    //
    QHBox * hbox = new QHBox( group_box );
    //
    new QLabel( tr( "Step:" ), hbox );
    //
    M_grid_step_slider = new QSlider( Qt::Horizontal, hbox );
    M_grid_step_slider->setRange( 0, 1000 );
    M_grid_step_slider->setValue( 0 );
    M_grid_step_slider->setSteps( 1, 1 );
    //        M_grid_step_slider->setMinimumSize( 300, -1 );
    connect( M_grid_step_slider, SIGNAL( sliderMoved( int ) ),
             this, SLOT( slideGridStep( int ) ) );
    //
    M_grid_step_text = new QLineEdit( tr( "0.0" ), hbox );
    M_grid_step_text->setValidator( new QDoubleValidator( 0.0, 200.0, 2, M_grid_step_text ) );
    M_grid_step_text->setMaximumSize( 48, 24 );
    connect( M_grid_step_text, SIGNAL( textChanged( const QString & ) ),
             this, SLOT( editGridStep( const QString & ) ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
ViewConfigDialog::createTraceControls( QWidget * parent )
{
    QVGroupBox * group_box = new QVGroupBox( tr( "Trace" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    QVBox * top_box = new QVBox( group_box );
    {
        QHGroupBox * ball_box = new QHGroupBox( tr( "Ball" ),
                                                top_box );
        ball_box->setInsideMargin( 5 );
        ball_box->setInsideSpacing( 2 );
        ball_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                              QSizePolicy::Fixed ) );
        //
        M_ball_trace_start = new QLineEdit( tr( "0" ), ball_box );
        M_ball_trace_start->setValidator( new QIntValidator( 0, 20000, M_ball_trace_start ) );
        M_ball_trace_start->setMaximumSize( 36, 24 );
        connect( M_ball_trace_start, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editBallTraceStart( const QString & ) ) );
        //
        new QLabel( tr( "-" ), ball_box );
        //
        M_ball_trace_end = new QLineEdit( tr( "0" ), ball_box );
        M_ball_trace_end->setValidator( new QIntValidator( 0, 20000, M_ball_trace_end ) );
        M_ball_trace_end->setMaximumSize( 36, 24 );
        connect( M_ball_trace_end, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editBallTraceEnd( const QString & ) ) );
        //
        QPushButton * all_btn = new QPushButton( tr( "All" ), ball_box );
        all_btn->setMaximumSize( 48, this->fontMetrics().height() + 12 );
        all_btn->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
        connect( all_btn, SIGNAL( clicked() ),
                 this, SLOT( clickBallTraceAll() ) );
        //
        M_ball_trace_cb = new QCheckBox( tr( "Auto" ), ball_box );
        M_ball_trace_cb->setChecked( false );
        connect( M_ball_trace_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickAutoBallTrace( bool ) ) );
    }
    {
        QHGroupBox * player_box = new QHGroupBox( tr( "Player" ),
                                                  top_box );
        player_box->setInsideMargin( 5 );
        player_box->setInsideSpacing( 2 );
        player_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                                QSizePolicy::Fixed ) );
        //
        M_player_trace_start = new QLineEdit( tr( "0" ), player_box );
        M_player_trace_start->setValidator( new QIntValidator( 0, 20000, M_player_trace_start ) );
        M_player_trace_start->setMaximumSize( 36, 24 );
        connect( M_player_trace_start, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editPlayerTraceStart( const QString & ) ) );
        //
        new QLabel( tr( "-" ), player_box );
        //
        M_player_trace_end = new QLineEdit( tr( "0" ), player_box );
        M_player_trace_end->setValidator( new QIntValidator( 0, 20000, M_player_trace_end ) );
        M_player_trace_end->setMaximumSize( 36, 24 );
        connect( M_ball_trace_end, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editBallTraceEnd( const QString & ) ) );
        //
        QPushButton * all_btn = new QPushButton( tr( "All" ), player_box );
        all_btn->setMaximumSize( 48, this->fontMetrics().height() + 12 );
        all_btn->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
        connect( all_btn, SIGNAL( clicked() ),
                 this, SLOT( clickPlayerTraceAll() ) );
        //
        M_player_trace_cb =  new QCheckBox( tr( "Auto" ), player_box );
        M_player_trace_cb->setChecked( false );
        connect( M_player_trace_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickAutoPlayerTrace( bool ) ) );
    }

    QHBox * bottom_box = new QHBox( top_box );
    {
        QHGroupBox * trace_box = new QHGroupBox( tr( "Auto Trace Period" ),
                                                 bottom_box );
        trace_box->setInsideMargin( 5 );
        trace_box->setInsideSpacing( 2 );
        //
        new QLabel( tr( "Start:" ), trace_box );
        //
        M_auto_trace_start = new QLineEdit( tr( "-10" ), trace_box );
        M_auto_trace_start->setValidator( new QIntValidator( -20000, 20000, M_auto_trace_start ) );
        M_auto_trace_start->setMaximumSize( 36, 24 );
        connect( M_auto_trace_start, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editAutoTraceStart( const QString & ) ) );
        //
        new QLabel( tr( "Period:" ), trace_box );
        //
        M_auto_trace_period = new QSpinBox( trace_box );
        M_auto_trace_period->setValue( 10 );
        M_auto_trace_period->setRange( 1, 12000 );
        //M_auto_trace_period->setMaximumSize( 36, 24 );
        connect( M_auto_trace_period, SIGNAL( valueChanged( int ) ),
                 this, SLOT( changeAutoTracePeriod( int ) ) );
    }
    {
        QPushButton * line_point_btn = new QPushButton( tr( "Line/Point" ),
                                                        bottom_box );
        line_point_btn->setMaximumSize( 80, this->fontMetrics().height() + 12 );
        connect( line_point_btn, SIGNAL( clicked() ),
                 this, SLOT( clickLinePointButton() ) );
    }

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
QWidget *
ViewConfigDialog::createInertiaMoveControls( QWidget * parent )
{
    QHGroupBox * group_box = new QHGroupBox( tr( "Inertia Move" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    //
    new QLabel( tr( "Ball:" ), group_box );
    //
    M_ball_future = new QSpinBox( group_box );
    //M_ball_future->setMinimumSize( 60, 24 );
    M_ball_future->setValue( 0 );
    M_ball_future->setRange( 0, 50 );
    connect( M_ball_future, SIGNAL( valueChanged( int ) ),
             this, SLOT( changeBallFutureCycle( int ) ) );
    //
    new QLabel( tr( "Player:" ), group_box );
    //
    M_player_future = new QSpinBox( group_box );
    //M_player_future->setMaximumSize( 60, 24 );
    M_player_future->setValue( 0 );
    M_player_future->setRange( 0, 10 );
    connect( M_player_future, SIGNAL( valueChanged( int ) ),
             this, SLOT( changePlayerFutureCycle( int ) ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::showEvent( QShowEvent * )
{
    updateAll();
    //event->accept();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::updateAll()
{
    const Options & opt = Options::instance();

    M_enlarge_cb->setChecked( opt.enlargeMode() );
    M_ball_size_text->setText( QString::number( opt.ballSize() ) );
    M_player_size_text->setText( QString::number( opt.playerSize() ) );

    updateFieldScale();

    M_canvas_width_text->setText( QString::number( opt.canvasWidth() ) );
    M_canvas_height_text->setText( QString::number( opt.canvasHeight() ) );

    M_player_number_cb->setChecked( opt.showPlayerNumber() );
    M_player_type_cb->setChecked( opt.showPlayerType() );
    M_stamina_cb->setChecked( opt.showStamina() );
    M_stamina_capacity_cb->setChecked( opt.showStaminaCapacity() );
    M_view_area_cb->setChecked( opt.showViewArea() );
    M_body_shadow_cb->setChecked( opt.showBodyShadow() );
    M_catchable_area_cb->setChecked( opt.showCatchableArea() );
    M_tackle_area_cb->setChecked( opt.showTackleArea() );
    M_kick_accel_area_cb->setChecked( opt.showKickAccelArea() );
    M_pointto_cb->setChecked( opt.showPointto() );
    M_attentionto_cb->setChecked( opt.showAttentionto() );
    M_card_cb->setChecked( opt.showCard() );

    M_anonymous_mode_cb->setChecked( opt.anonymousMode() );
    M_show_score_board_cb->setChecked( opt.showScoreBoard() );
    M_show_ball_cb->setChecked( opt.showBall() );
    M_show_players_cb->setChecked( opt.showPlayers() );
    M_show_flags_cb->setChecked( opt.showFlags() );
    M_show_offside_line_cb->setChecked( opt.showOffsideLine() );

    switch ( opt.fieldGrassType() ) {
    case Options::GRASS_NORMAL:
        M_grass_normal_rb->setChecked( true );
        break;
    case Options::GRASS_LINES:
        M_grass_line_rb->setChecked( true );
        break;
    case Options::GRASS_CHECKER:
        M_grass_checker_rb->setChecked( true );
        break;
    default:
        M_grass_normal_rb->setChecked( true );
        break;
    }

    M_keepaway_mode_cb->setChecked( opt.keepawayMode() );

    //M_draw_type_choice->setCurrentItem( opt.drawType() );
    M_draw_type_choice->setCurrentItem( opt.paintStyle() );

    M_cursor_hide_cb->setChecked( opt.cursorHide() );

    M_show_voronoi_cb->setChecked( opt.showVoronoiDiagram() );
    M_show_delaunay_cb->setChecked( opt.showDelaunayTriangulation() );

    switch ( opt.voronoiTarget() ) {
    case rcsc::NEUTRAL:
        M_voronoi_choice->setCurrentItem( 0 );
        break;
    case rcsc::LEFT:
        M_voronoi_choice->setCurrentItem( 1 );
        break;
    case rcsc::RIGHT:
        M_voronoi_choice->setCurrentItem( 2 );
        break;
    default:
        M_voronoi_choice->setCurrentItem( 0 );
        break;
    }

    switch ( opt.focusType() ) {
    case Options::FOCUS_BALL:
        M_focus_ball_rb->setChecked( true );
        break;
    case Options::FOCUS_PLAYER:
        M_focus_player_rb->setChecked( true );
        break;
    case Options::FOCUS_POINT:
        M_focus_fix_rb->setChecked( true );
        break;
    default:
        M_focus_fix_rb->setChecked( true );
        break;
    }

    switch ( opt.playerSelectType() ) {
    case Options::SELECT_AUTO_ALL:
        M_select_all_rb->setChecked( true );
        break;
    case Options::SELECT_AUTO_LEFT:
        M_select_left_rb->setChecked( true );
        break;
    case Options::SELECT_AUTO_RIGHT:
        M_select_right_rb->setChecked( true );
        break;
    case Options::SELECT_UNSELECT:
        M_unselect_rb->setChecked( true );
        break;
    default:
        M_unselect_rb->setChecked( true );
        break;
    }

    M_ball_trace_start->setText( QString::number( opt.ballTraceStart() ) );
    M_ball_trace_end->setText( QString::number( opt.ballTraceEnd() ) );
    M_ball_trace_cb->setChecked( opt.ballAutoTrace() );

    M_player_trace_start->setText( QString::number( opt.playerTraceStart() ) );
    M_player_trace_end->setText( QString::number( opt.playerTraceEnd() ) );
    M_player_trace_cb->setChecked( opt.playerAutoTrace() );

    M_auto_trace_start->setText( QString::number( opt.autoTraceStart() ) );
    M_auto_trace_period->setValue( opt.autoTracePeriod() );

    M_ball_future->setValue( opt.ballFutureCycle() );
    M_player_future->setValue( opt.playerFutureCycle() );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::updateFieldScale()
{
    int iscale = static_cast< int >( rint( Options::instance().fieldScale() * 10.0 ) );
    if ( M_scale_slider->value() != iscale )
    {
        M_scale_slider->setValue( iscale );
    }

    char buf[16];
    snprintf( buf, 16, "%.2f", Options::instance().fieldScale() );
    M_scale_text->setText( QString::fromAscii( buf ) );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::zoomIn()
{
    Options::instance().zoomIn();

    updateFieldScale();

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::zoomOut()
{
    Options::instance().zoomOut();

    updateFieldScale();

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::fitToScreen()
{
    Options::instance().fitToScreen();
    updateFieldScale();
    M_focus_fix_rb->setChecked( true );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::slideFieldScale( int value )
{
    double scale = value * 0.1;

    if ( std::fabs( scale - Options::instance().fieldScale() ) >= 0.01 )
    {
        M_scale_text->setText( QString::number( scale ) );

        Options::instance().setFieldScale( scale );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::editFieldScale( const QString & text )
{
    bool ok = true;
    double value = text.toDouble( &ok );

    if ( ok
         && std::fabs( value - Options::instance().fieldScale() ) >= 0.01 )
    {
        int ivalue = static_cast< int >( rint( value * 10.0 ) );
        M_scale_slider->setValue( ivalue );

        Options::instance().setFieldScale( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickEnlarge( bool checked )
{
    if ( Options::instance().enlargeMode() != checked )
    {
        Options::instance().toggleEnlargeMode();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleEnlarge()
{
    Options::instance().toggleEnlargeMode();
    M_enlarge_cb->setChecked( Options::instance().enlargeMode() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::editBallSize( const QString & text )
{
    bool ok = true;
    double value = text.toDouble( &ok );

    if ( ok && value > 0.0 )
    {
        Options::instance().setBallSize( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::editPlayerSize( const QString & text )
{
    bool ok = true;
    double value = text.toDouble( &ok );

    if ( ok )
    {
        Options::instance().setPlayerSize( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::applyCanvasSize()
{
    bool ok_w = true;
    bool ok_h = true;
    int width = M_canvas_width_text->text().toInt( & ok_w );
    int height = M_canvas_height_text->text().toInt( & ok_h );

    if ( ok_w
         && ok_h
         && width > 0
         && height > 0 )
    {
        emit canvasResized( QSize( width, height ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickReverseSide( bool checked )
{
    if ( Options::instance().reverseSide() != checked )
    {
        Options::instance().toggleReverseSide();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleReverseSide()
{
    Options::instance().toggleReverseSide();
    M_reverse_side_cb->setChecked( Options::instance().reverseSide() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickPlayerReverseDraw( bool checked )
{
    if ( Options::instance().playerReverseDraw() != checked )
    {
        Options::instance().togglePlayerReverseDraw();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::togglePlayerReverseDraw()
{
    Options::instance().togglePlayerReverseDraw();
    M_player_reverse_draw_cb->setChecked( Options::instance().playerReverseDraw() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowPlayerNumber( bool checked )
{
    if ( Options::instance().showPlayerNumber() != checked )
    {
        Options::instance().toggleShowPlayerNumber();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowPlayerNumber()
{
    Options::instance().toggleShowPlayerNumber();
    M_player_number_cb->setChecked( Options::instance().showPlayerNumber() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowPlayerType( bool checked )
{
    if ( Options::instance().showPlayerType() != checked )
    {
        Options::instance().toggleShowPlayerType();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowPlayerType()
{
    Options::instance().toggleShowPlayerType();
    M_player_type_cb->setChecked( Options::instance().showPlayerType() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowStamina( bool checked )
{
    if ( Options::instance().showStamina() != checked )
    {
        Options::instance().toggleShowStamina();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowStamina()
{
    const Options & opt = Options::instance();

    if ( ! opt.showStamina()
         && ! opt.showStaminaCapacity() )
    {
        Options::instance().toggleShowStamina();
    }
    else if ( opt.showStamina()
              && ! opt.showStaminaCapacity() )
    {
        Options::instance().toggleShowStamina();
        Options::instance().toggleShowStaminaCapacity();
    }
    else if ( ! opt.showStamina()
              && opt.showStaminaCapacity() )
    {
        Options::instance().toggleShowStamina();
    }
    else
    {
        Options::instance().toggleShowStamina();
        Options::instance().toggleShowStaminaCapacity();
    }

    M_stamina_cb->setChecked( opt.showStamina() );
    M_stamina_capacity_cb->setChecked( opt.showStaminaCapacity() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowStaminaCapacity( bool checked )
{
    if ( Options::instance().showStaminaCapacity() != checked )
    {
        Options::instance().toggleShowStaminaCapacity();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowStaminaCapacity()
{
    Options::instance().toggleShowStaminaCapacity();
    M_stamina_capacity_cb->setChecked( Options::instance().showStaminaCapacity() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowViewArea( bool checked )
{
    if ( Options::instance().showViewArea() != checked )
    {
        Options::instance().toggleShowViewArea();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowViewArea()
{
    Options::instance().toggleShowViewArea();
    M_view_area_cb->setChecked( Options::instance().showViewArea() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowBodyShadow( bool checked )
{
    if ( Options::instance().showBodyShadow() != checked )
    {
        Options::instance().toggleShowBodyShadow();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowBodyShadow()
{
    Options::instance().toggleShowBodyShadow();
    M_body_shadow_cb->setChecked( Options::instance().showBodyShadow() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowCatchableArea( bool checked )
{
    if ( Options::instance().showCatchableArea() != checked )
    {
        Options::instance().toggleShowCatchableArea();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowCatchableArea()
{
    Options::instance().toggleShowCatchableArea();
    M_catchable_area_cb->setChecked( Options::instance().showCatchableArea() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowTackleArea( bool checked )
{
    if ( Options::instance().showTackleArea() != checked )
    {
        Options::instance().toggleShowTackleArea();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowTackleArea()
{
    Options::instance().toggleShowTackleArea();
    M_tackle_area_cb->setChecked( Options::instance().showTackleArea() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowKickAccelArea( bool checked )
{
    if ( Options::instance().showKickAccelArea() != checked )
    {
        Options::instance().toggleShowKickAccelArea();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowKickAccelArea()
{
    Options::instance().toggleShowKickAccelArea();
    M_kick_accel_area_cb->setChecked( Options::instance().showKickAccelArea() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowPointto( bool checked )
{
    if ( Options::instance().showPointto() != checked )
    {
        Options::instance().toggleShowPointto();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowPointto()
{
    Options::instance().toggleShowPointto();
    M_pointto_cb->setChecked( Options::instance().showPointto() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowAttentionto( bool checked )
{
    if ( Options::instance().showAttentionto() != checked )
    {
        Options::instance().toggleShowAttentionto();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowAttentionto()
{
    Options::instance().toggleShowAttentionto();
    M_attentionto_cb->setChecked( Options::instance().showAttentionto() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowCard( bool checked )
{
    if ( Options::instance().showCard() != checked )
    {
        Options::instance().toggleShowCard();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowCard()
{
    Options::instance().toggleShowCard();
    M_card_cb->setChecked( Options::instance().showCard() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickAnonymousMode( bool checked )
{
    if ( Options::instance().anonymousMode() != checked )
    {
        Options::instance().toggleAnonymousMode();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleAnonymousMode()
{
    Options::instance().toggleAnonymousMode();
    M_anonymous_mode_cb->setChecked( Options::instance().anonymousMode() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowScoreBoard( bool checked )
{
    if ( Options::instance().showScoreBoard() != checked )
    {
        Options::instance().toggleShowScoreBoard();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowScoreBoard()
{
    Options::instance().toggleShowScoreBoard();
    M_show_score_board_cb->setChecked( Options::instance().showScoreBoard() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowTeamGraphic( bool checked )
{
    if ( Options::instance().showTeamGraphic() != checked )
    {
        Options::instance().toggleShowTeamGraphic();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowTeamGraphic()
{
    Options::instance().toggleShowTeamGraphic();
    M_show_team_graphic_cb->setChecked( Options::instance().showTeamGraphic() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowBall( bool checked )
{
    if ( Options::instance().showBall() != checked )
    {
        Options::instance().toggleShowBall();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowBall()
{
    Options::instance().toggleShowBall();
    M_show_ball_cb->setChecked( Options::instance().showBall() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowPlayers( bool checked )
{
    if ( Options::instance().showPlayers() != checked )
    {
        Options::instance().toggleShowPlayers();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowPlayers()
{
    Options::instance().toggleShowPlayers();
    M_show_players_cb->setChecked( Options::instance().showPlayers() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowFlags( bool checked )
{
    if ( Options::instance().showFlags() != checked )
    {
        Options::instance().toggleShowFlags();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowFlags()
{
    Options::instance().toggleShowFlags();
    M_show_flags_cb->setChecked( Options::instance().showFlags() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowOffsideLine( bool checked )
{
    if ( Options::instance().showOffsideLine() != checked )
    {
        Options::instance().toggleShowOffsideLine();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleShowOffsideLine()
{
    Options::instance().toggleShowOffsideLine();
    M_show_offside_line_cb->setChecked( Options::instance().showOffsideLine() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickGrassNormal()
{
    if ( Options::instance().fieldGrassType() != Options::GRASS_NORMAL )
    {
        Options::instance().setFieldGrassType( Options::GRASS_NORMAL );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickGrassLine()
{
    if ( Options::instance().fieldGrassType() != Options::GRASS_LINES )
    {
        Options::instance().setFieldGrassType( Options::GRASS_LINES );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickGrassChecker()
{
    if ( Options::instance().fieldGrassType() != Options::GRASS_CHECKER )
    {
        Options::instance().setFieldGrassType( Options::GRASS_CHECKER );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickKeepawayMode( bool checked )
{
    if ( Options::instance().keepawayMode() != checked )
    {
        Options::instance().toggleKeepawayMode();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowGridCoord( bool checked )
{
    if ( Options::instance().showGridCoord() != checked )
    {
        Options::instance().toggleShowGridCoord();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::slideGridStep( int value )
{
    if ( value != Options::instance().gridStep() )
    {
        M_grid_step_text->setText( QString::number( value * 0.1 ) );
        Options::instance().setGridStep( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::editGridStep( const QString & text )
{
    bool ok = true;
    double value = text.toDouble( &ok );

    if ( ok )
    {
        int ivalue = static_cast< int >( rint( value * 10.0 ) );
        if ( ivalue != Options::instance().gridStep() )
        {
            M_grid_step_slider->setValue( ivalue );
            M_grid_step_text->setText( QString::number( ivalue * 0.1 ) );
            Options::instance().setGridStep( ivalue );

            emit configured();
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::selectDrawType( int index )
{
    Options::PaintStyle style = static_cast< Options::PaintStyle >( index );
    if ( Options::instance().paintStyle() != style )
    {
        Options::instance().setPaintStyle( style );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleKeepawayMode()
{
    Options::instance().toggleKeepawayMode();
    M_keepaway_mode_cb->setChecked( Options::instance().keepawayMode() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickCursorHide( bool checked )
{
    if ( Options::instance().cursorHide() != checked )
    {
        Options::instance().toggleCursorHide();
        //emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowVoronoiDiagram( bool checked )
{
    if ( Options::instance().showVoronoiDiagram() != checked )
    {
        Options::instance().toggleShowVoronoiDiagram();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickShowDelaunayTriangulation( bool checked )
{
    if ( Options::instance().showDelaunayTriangulation() != checked )
    {
        Options::instance().toggleShowDelaunayTriangulation();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::selectCompGeomSide( int index )
{
    switch ( index ) {
    case 0:
        if ( Options::instance().voronoiTarget() != rcsc::NEUTRAL )
        {
            Options::instance().setVoronoiTarget( rcsc::NEUTRAL );
            emit configured();
        }
        break;
    case 1:
        if ( Options::instance().voronoiTarget() != rcsc::LEFT )
        {
            Options::instance().setVoronoiTarget( rcsc::LEFT );
            emit configured();
        }
        break;
    case 2:
        if ( Options::instance().voronoiTarget() != rcsc::RIGHT )
        {
            Options::instance().setVoronoiTarget( rcsc::RIGHT );
            emit configured();
        }
        break;
    default:
        std::cerr << "ViewConfigDialog::selectCompGeomSide() unsupported index "
                  << index
                  << std::endl;
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickFocusBall()
{
    if ( Options::instance().focusType() != Options::FOCUS_BALL )
    {
        Options::instance().setFocusType( Options::FOCUS_BALL );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleFocusBall()
{
    if ( Options::instance().focusType() == Options::FOCUS_BALL )
    {
        setFocusFix();
    }
    else
    {
        Options::instance().setFocusType( Options::FOCUS_BALL );
        M_focus_ball_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickFocusPlayer()
{
    if ( Options::instance().focusType() != Options::FOCUS_PLAYER )
    {
        Options::instance().setFocusType( Options::FOCUS_PLAYER );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleFocusPlayer()
{
    if ( Options::instance().focusType() == Options::FOCUS_PLAYER )
    {
        setFocusFix();
    }
    else
    {
        Options::instance().setFocusType( Options::FOCUS_PLAYER );
        M_focus_player_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::setFocusFix()
{
    if ( Options::instance().focusType() != Options::FOCUS_POINT )
    {
        Options::instance().setFocusType( Options::FOCUS_POINT );
        M_focus_fix_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::setFocusPoint( const QPoint & point )
{
    Options::instance().setFocusPoint( point.x(), point.y() );
    M_focus_fix_rb->setChecked( true );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickSelectAutoAll()
{
    if ( Options::instance().playerSelectType() != Options::SELECT_AUTO_ALL )
    {
        Options::instance().setPlayerSelectType( Options::SELECT_AUTO_ALL );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleSelectAutoAll()
{
    if ( Options::instance().playerSelectType() == Options::SELECT_AUTO_ALL )
    {
        setUnselect();
    }
    else
    {
        Options::instance().setPlayerSelectType( Options::SELECT_AUTO_ALL );
        M_select_all_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickSelectAutoLeft()
{
    Options::PlayerSelectType type = ( Options::instance().reverseSide()
                                       ? Options::SELECT_AUTO_RIGHT
                                       : Options::SELECT_AUTO_LEFT );

    if ( Options::instance().playerSelectType() != type )
    {
        Options::instance().setPlayerSelectType( type );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleSelectAutoLeft()
{
    Options::PlayerSelectType type = ( Options::instance().reverseSide()
                                       ? Options::SELECT_AUTO_RIGHT
                                       : Options::SELECT_AUTO_LEFT );

    if ( Options::instance().playerSelectType() == type )
    {
        setUnselect();
    }
    else
    {
        Options::instance().setPlayerSelectType( type );
        M_select_left_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickSelectAutoRight()
{
    Options::PlayerSelectType type = ( Options::instance().reverseSide()
                                       ? Options::SELECT_AUTO_LEFT
                                       : Options::SELECT_AUTO_RIGHT );

    if ( Options::instance().playerSelectType() != type )
    {
        Options::instance().setPlayerSelectType( type );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleSelectAutoRight()
{
    Options::PlayerSelectType type = ( Options::instance().reverseSide()
                                       ? Options::SELECT_AUTO_LEFT
                                       : Options::SELECT_AUTO_RIGHT );

    if ( Options::instance().playerSelectType() == type )
    {
        setUnselect();
    }
    else
    {
        Options::instance().setPlayerSelectType( type );
        M_select_right_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickSelectFix()
{
    if ( Options::instance().playerSelectType() != Options::SELECT_FIX )
    {
        Options::instance().setPlayerSelectType( Options::SELECT_AUTO_RIGHT );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::toggleSelectFix()
{
    if ( Options::instance().playerSelectType() == Options::SELECT_FIX )
    {
        setUnselect();
    }
    else
    {
        Options::instance().setPlayerSelectType( Options::SELECT_FIX );
        M_select_fix_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::setUnselect()
{
    Options::instance().unselectPlayer();
    M_unselect_rb->setChecked( true );
    M_player_choice->setCurrentItem( 0 );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::selectPlayer( int number )
{
    // left side:  number = [1,11]
    // right side:  number = [12,22]
    if ( number <= 0
         || 22 < number )
    {
        setUnselect();
        return;
    }

    int orig_number = number;
    if ( Options::instance().reverseSide() )
    {
        if ( number <= 11 )
        {
            number += 11;
        }
        else
        {
            number -= 11;
        }
    }

    Options::instance().setPlayerSelectType( Options::SELECT_FIX );
    if ( number <= 11 )
    {
        Options::instance().setSelectedNumber( rcsc::LEFT, number );
    }
    else
    {
        Options::instance().setSelectedNumber( rcsc::RIGHT, number - 11 );
    }

    M_player_choice->setCurrentItem( orig_number );
    M_select_fix_rb->setChecked( true );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::choicePlayer( int number )
{
    int orig_number = number;
    if ( Options::instance().reverseSide() )
    {
        if ( number <= 11 )
        {
            number += 11;
        }
        else
        {
            number -= 11;
        }
    }

    if ( number <= 11
         && Options::instance().isSelectedPlayer( rcsc::LEFT, number ) )
    {
        return;
    }

    if ( number > 11
         && Options::instance().isSelectedPlayer( rcsc::RIGHT, number - 11 ) )
    {
        return;
    }

    selectPlayer( orig_number );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::editBallTraceStart( const QString & text )
{
    bool ok = true;
    int value = text.toInt( & ok );

    if ( ok )
    {
        Options::instance().setBallTraceStart( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::editBallTraceEnd( const QString & text )
{
    bool ok = true;
    int value = text.toInt( & ok );

    if ( ok )
    {
        Options::instance().setBallTraceEnd( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickBallTraceAll()
{
    const std::vector< MonitorViewPtr > & view
        = M_main_data.viewHolder().monitorViewCont();

    if ( ! view.empty() )
    {
        M_ball_trace_cb->setChecked( false );

        if ( Options::instance().ballTraceStart() == view.front()->cycle()
             && Options::instance().ballTraceEnd() == view.back()->cycle() )
        {
            M_ball_trace_start->setText( QString::number( 0 ) );
            M_ball_trace_end->setText( QString::number( 0 ) );

            Options::instance().setBallTraceStart( 0 );
            Options::instance().setBallTraceEnd( 0 );
        }
        else
        {
            M_ball_trace_start->setText( QString::number( view.front()->cycle() ) );
            M_ball_trace_end->setText( QString::number( view.back()->cycle() ) );

            Options::instance().setBallTraceStart( view.front()->cycle() );
            Options::instance().setBallTraceEnd( view.back()->cycle() );
        }

        if ( Options::instance().ballAutoTrace() )
        {
            Options::instance().toggleBallAutoTrace();
        }

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickAutoBallTrace( bool checked )
{
    if ( Options::instance().ballAutoTrace() != checked )
    {
        Options::instance().toggleBallAutoTrace();

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::editPlayerTraceStart( const QString & text )
{
    bool ok = true;
    int value = text.toInt( & ok );

    if ( ok )
    {
        Options::instance().setPlayerTraceStart( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::editPlayerTraceEnd( const QString & text )
{
    bool ok = true;
    int value = text.toInt( & ok );

    if ( ok )
    {
        Options::instance().setPlayerTraceEnd( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickPlayerTraceAll()
{
    const std::vector< MonitorViewPtr > & view
        = M_main_data.viewHolder().monitorViewCont();
    if ( ! view.empty() )
    {
        M_player_trace_cb->setChecked( false );

        if ( Options::instance().playerTraceStart() == view.front()->cycle()
             && Options::instance().playerTraceEnd() == view.back()->cycle() )
        {
            M_player_trace_start->setText( QString::number( 0 ) );
            M_player_trace_end->setText( QString::number( 0 ) );

            Options::instance().setPlayerTraceStart( 0 );
            Options::instance().setPlayerTraceEnd( 0 );
        }
        else
        {
            M_player_trace_start->setText( QString::number( view.front()->cycle() ) );
            M_player_trace_end->setText( QString::number( view.back()->cycle() ) );

            Options::instance().setPlayerTraceStart( view.front()->cycle() );
            Options::instance().setPlayerTraceEnd( view.back()->cycle() );
        }

        if ( Options::instance().playerAutoTrace() )
        {
            Options::instance().togglePlayerAutoTrace();
        }

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickAutoPlayerTrace( bool checked )
{
    if ( Options::instance().playerAutoTrace() != checked )
    {
        Options::instance().togglePlayerAutoTrace();

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::editAutoTraceStart( const QString & text )
{
    bool ok = true;
    int value = text.toInt( & ok );

    if ( ok )
    {
        Options::instance().setAutoTraceStart( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::changeAutoTracePeriod( int value )
{
    //std::cerr << "changedAutoTracePeriod " << value << std::endl;
    if ( value >= 1 )
    {
        Options::instance().setAutoTracePeriod( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::clickLinePointButton()
{
    Options::instance().toggleLineTrace();

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::changeBallFutureCycle( int value )
{
    Options::instance().setBallFutureCycle( value );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ViewConfigDialog::changePlayerFutureCycle( int value )
{
    Options::instance().setPlayerFutureCycle( value );

    emit configured();
}
