// -*-c++-*-

/*!
  \file main_window.cpp
  \brief main application window class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "main_window.h"

#include "color_setting_dialog.h"
#include "detail_dialog.h"
#include "font_setting_dialog.h"
#include "image_save_dialog.h"
#include "launcher_dialog.h"
#include "player_type_dialog.h"
#include "monitor_move_dialog.h"
#include "view_config_dialog.h"
#include "debug_message_window.h"
#include "field_canvas.h"
#include "debug_server.h"
#include "monitor_client.h"
#include "log_player.h"
#include "log_player_tool_bar.h"
//#include "dir_selector.h"
#include "options.h"

#include <rcsc/common/server_param.h>

#include <string>
#include <iostream>
#include <cstring>

#include <csignal> // ::kill()
#include <cstdlib> // system()
#include <sys/types.h> // SIGINT


#include "xpm/soccerwindow2.xpm"
#include "xpm/soccerwindow2-nostr.xpm"

#include "xpm/open_rcg.xpm"
#include "xpm/save.xpm"

#include "xpm/logplayer_live_mode.xpm"
#include "xpm/monitor_move_player.xpm"
#include "xpm/debug_server_switch.xpm"

#ifndef PACKAGE_NAME
#define PACKAGE_NAME "soccerwindow2"
#endif

/*-------------------------------------------------------------------*/
/*!

 */
MainWindow::MainWindow()
    : M_log_player( new LogPlayer( M_main_data, this ) )
    , M_detail_dialog( static_cast< DetailDialog * >( 0 ) )
    , M_player_type_dialog( static_cast< PlayerTypeDialog * >( 0 ) )
    , M_monitor_move_dialog( static_cast< MonitorMoveDialog * >( 0 ) )
    , M_view_config_dialog( static_cast< ViewConfigDialog * >( 0 ) )
    , M_launcher_dialog( static_cast< LauncherDialog * >( 0 ) )
    , M_debug_message_window( static_cast< DebugMessageWindow * >( 0 ) )
    , M_monitor_client( static_cast< MonitorClient * >( 0 ) )
    , M_debug_server( static_cast< DebugServer * >( 0 ) )
    , M_last_connected_host( "127.0.0.1" )
{
    readSettings();

    createActions();
    createMenus();
    createToolBars();
    createStatusBar();
    createFieldCanvas();
    createViewConfigDialog();

    connect( M_log_player, SIGNAL( updated() ),
             this, SIGNAL( viewUpdated() ) );

    this->setIcon( QPixmap( soccerwindow2_xpm ) );
    this->setCaption( tr( PACKAGE_NAME ) );
    /*
      this->setDockEnabled( Qt::DockBottom, true );
      this->setDockEnabled( Qt::DockLeft, false );
      this->setDockEnabled( Qt::DockRight, false );
    */
    this->setMinimumSize( 280, 220 );
    this->resize( Options::instance().frameWidth() > 0
                  ? Options::instance().frameWidth()
                  : 640,
                  Options::instance().frameHeight() > 0
                  ? Options::instance().frameHeight()
                  : 480 );
    this->move( Options::instance().framePosX() >= 0
                ? Options::instance().framePosX()
                : this->x(),
                Options::instance().framePosY() >= 0
                ? Options::instance().framePosY()
                : this->y() );

    // this->setWindowOpacity( 0.5 ); // window transparency

    this->setAcceptDrops( true );

    if ( Options::instance().hideToolBar() )
    {
        M_log_player_tool_bar->hide();
        //M_monitor_tool_bar->hide();
    }

    if ( Options::instance().hideStatusBar() )
    {
        this->statusBar()->hide();
    }

    if ( Options::instance().hideMenuBar() )
    {
        this->menuBar()->hide();
    }

}

/*-------------------------------------------------------------------*/
/*!

 */
MainWindow::~MainWindow()
{
    //std::cerr << "delete MainWindow" << std::endl;

    if ( Options::instance().killServer() )
    {
        killServer();
    }
    saveSettings();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::init()
{
    if ( ! Options::instance().gameLogFilePath().empty() )
    {
        //openRCG( QString::fromAscii( Options::instance().gameLogFilePath().c_str() ) );
        openRCG( Options::instance().gameLogFilePath() );
    }
    else if ( Options::instance().connect() )
    {
        std::string host = Options::instance().host();
        if ( host.empty() )
        {
            host = "127.0.0.1";
        }

        connectMonitorTo( host.c_str() );

        if ( Options::instance().debugServerMode() )
        {
            startDebugServer();
        }
    }

    if ( Options::instance().canvasWidth() > 0
         && Options::instance().canvasHeight() > 0 )
    {
        resizeCanvas( QSize( Options::instance().canvasWidth(),
                             Options::instance().canvasHeight() ) );
    }
    else if ( Options::instance().fullScreen() )
    {
        this->showFullScreen();
    }
    else if ( Options::instance().maximize() )
    {
        this->showMaximized();
    }


    if ( Options::instance().autoImageSave() )
    {
        QTimer::singleShot( 500,
                            this, SLOT( saveImageAndQuit() ) );
        this->setEnabled( false );
    }

    M_debug_message_window = new DebugMessageWindow( this,
                                                     M_main_data );
    connect( M_debug_message_window, SIGNAL( configured() ),
             this, SIGNAL( viewUpdated() ) );
    M_debug_message_window->hide();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::readSettings()
{
    QSettings settings( QSettings::Ini );
    settings.insertSearchPath( QSettings::Unix,
                               QDir::homeDirPath() );

    settings.beginGroup( "/.soccerwindow2-qt3./Options" );

    QString val;

    val = settings.readEntry( "gameLogDir" );
    if ( ! val.isNull()
         && Options::instance().gameLogDir().empty() )
    {
        Options::instance().setGameLogDir( val.ascii() );
    }

    val = settings.readEntry( "debugLogDir" );
    if ( ! val.isNull()
         && Options::instance().debugLogDir().empty() )
    {
        Options::instance().setDebugLogDir( val.ascii() );
    }

    settings.endGroup();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::saveSettings()
{
    QSettings settings( QSettings::Ini );
    settings.insertSearchPath( QSettings::Unix,
                               QDir::homeDirPath() );

    settings.beginGroup( "/.soccerwindow2-qt3./Options" );

    settings.writeEntry( "gameLogDir",
                         QString( Options::instance().gameLogDir() ) );
    settings.writeEntry( "debugLogDir",
                         QString( Options::instance().debugLogDir() ) );

    settings.endGroup();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createActions()
{
    createActionsFile();
    createActionsMonitor();
    createActionsView();
    createActionsDebug();
    createActionsHelp();

}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createActionsFile()
{
    M_open_rcg_act = new QAction( QIconSet( QPixmap( open_rcg_xpm ) ),
                                  tr( "&Open rcg file..." ),
                                  QKeySequence(),
                                  this );
    M_open_rcg_act->setAccel( Qt::CTRL + Qt::Key_O );
    M_open_rcg_act->setStatusTip( tr( "Open RoboCup Game Log file" ) );
    connect( M_open_rcg_act, SIGNAL( activated() ), this, SLOT( openRCG() ) );
    //
    M_save_rcg_act = new QAction( QIconSet( QPixmap( save_xpm ) ),
                                  tr( "Save rcg file as..." ),
                                  QKeySequence(),
                                  this );
    M_save_rcg_act->setStatusTip( tr( "Save RoboCup Game Log file" ) );
    connect( M_save_rcg_act, SIGNAL( activated() ), this, SLOT( saveRCG() ) );
    //
    M_open_debug_view_act = new QAction( QIconSet( QPixmap( open_rcg_xpm ) ),
                                         tr( "Open debug view" ),
                                         QKeySequence(),
                                         this );
    M_open_debug_view_act->setStatusTip( tr( "Open the directory where debug view logs exist" ) );
    connect( M_open_debug_view_act, SIGNAL( activated() ),
             this, SLOT( openDebugView() ) );
    //
    M_save_debug_view_act = new QAction( QIconSet( QPixmap( save_xpm ) ),
                                         tr( "Save debug view" ),
                                         QKeySequence(),
                                         this );
    M_save_debug_view_act->setStatusTip( tr( "Save debug view logs to the directory..." ) );
    connect( M_save_debug_view_act, SIGNAL( activated() ),
             this, SLOT( saveDebugView() ) );
    //
    M_show_image_save_dialog_act = new QAction( tr( "Save &Image" ),
                                                QKeySequence(),
                                                this );
    M_show_image_save_dialog_act->setStatusTip( tr( "Save game log data as image files" ) );
    connect( M_show_image_save_dialog_act, SIGNAL( activated() ),
             this, SLOT( showImageSaveDialog() ) );
    //
    M_exit_act = new QAction( tr( "&Quit" ),
                              QKeySequence(),
                              this );
    M_exit_act->setAccel( Qt::CTRL + Qt::Key_Q );
    M_exit_act->setStatusTip( tr( "Exit the application." ) );
    connect( M_exit_act, SIGNAL( activated() ), this, SLOT( close() ) );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createActionsMonitor()
{
    M_kick_off_act = new QAction( tr( "&KickOff" ),
                                  QKeySequence(),
                                  this );
    M_kick_off_act->setAccel( Qt::CTRL + Qt::Key_K );
    M_kick_off_act->setStatusTip( tr( "Start the game" ) );
    M_kick_off_act->setEnabled( false );
    connect( M_kick_off_act, SIGNAL( activated() ), this, SLOT( kickOff() ) );
    //
    M_set_live_mode_act = new QAction( QIconSet( QPixmap( logplayer_live_mode_xpm ) ),
                                       tr( "&Live Mode" ),
                                       QKeySequence(),
                                       this );
    M_set_live_mode_act->setAccel( Qt::CTRL + Qt::Key_L );
    M_set_live_mode_act->setStatusTip( tr( "set monitor to live mode" ) );
    M_set_live_mode_act->setEnabled( false );
    connect( M_set_live_mode_act, SIGNAL( activated() ),
             this, SLOT( setLiveMode() ) );
    //
    M_connect_monitor_act = new QAction( tr( "&Connect" ),
                                         QKeySequence(),
                                         this );
    M_connect_monitor_act->setAccel( Qt::CTRL + Qt::Key_C );
    M_connect_monitor_act->setStatusTip( "Connect to the rcssserver on localhost" );
    M_connect_monitor_act->setEnabled( true );
    connect( M_connect_monitor_act, SIGNAL( activated() ),
             this, SLOT( connectMonitor() ) );
    //
    M_connect_monitor_to_act = new QAction( tr( "Connect &to ..." ),
                                            QKeySequence(),
                                            this );
    M_connect_monitor_to_act->setStatusTip( tr( "Connect to the rcssserver on other host" ) );
    M_connect_monitor_to_act->setEnabled( true );
    connect( M_connect_monitor_to_act, SIGNAL( activated() ),
             this, SLOT( connectMonitorTo() ) );
    //
    M_disconnect_monitor_act = new QAction( tr( "&Disconnect" ),
                                            QKeySequence(),
                                            this );
    M_disconnect_monitor_act->setStatusTip( tr( "Disonnect from rcssserver" ) );
    M_disconnect_monitor_act->setEnabled( false );
    connect( M_disconnect_monitor_act, SIGNAL( activated() ),
             this, SLOT( disconnectMonitor() ) );
    //
    M_kill_server_act = new QAction( tr( "&Kill server" ),
                                     QKeySequence(),
                                     this );
    M_kill_server_act->setStatusTip( tr( "Kill the rcssserver process" ) );
    M_kill_server_act->setEnabled( false );
    connect( M_kill_server_act, SIGNAL( activated() ),
             this, SLOT( killServer() ) );
    //
    M_restart_server_act = new QAction( tr( "(Re)&start server" ),
                                        QKeySequence(),
                                        this );
    M_restart_server_act->setStatusTip( tr( "(Re)start rcssserver" ) );
    connect( M_restart_server_act, SIGNAL( activated() ),
             this, SLOT( restartServer() ) );
    //
    M_toggle_drag_move_mode_act = new QAction( QIconSet( QPixmap( monitor_move_player_xpm ) ),
                                               tr( "Drag Move Mode" ),
                                               QKeySequence(),
                                               this );
    M_toggle_drag_move_mode_act->setStatusTip( tr( "Toggle drag move mode." ) );
    M_toggle_drag_move_mode_act->setEnabled( false );
    M_toggle_drag_move_mode_act->setToggleAction( true );
    connect( M_toggle_drag_move_mode_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDragMoveMode( bool ) ) );
    //
    M_show_monitor_move_dialog_act = new QAction( tr( "Move Dialog" ),
                                                  QKeySequence(),
                                                  this );
    M_show_monitor_move_dialog_act->setStatusTip( tr( "Show player move dialog" ) );
    //M_show_monitor_move_dialog_act->setEnabled( false );
    connect( M_show_monitor_move_dialog_act, SIGNAL( activated() ),
             this, SLOT( showMonitorMoveDialog() ) );
    //
    M_show_launcher_dialog_act = new QAction( tr( "Launcher Dialog" ),
                                              QKeySequence(),
                                              this );
    M_show_launcher_dialog_act->setAccel( Qt::CTRL + Qt::Key_X );
    M_show_launcher_dialog_act->setStatusTip( tr( "Show launcher dialog" ) );
    connect( M_show_launcher_dialog_act, SIGNAL( activated() ),
             this, SLOT( showLauncherDialog() ) );
    //
    M_playmode_change_act_group = new QActionGroup( this );
    M_playmode_change_act_group->setText( tr( "Change Playmode" ) );
    M_playmode_change_act_group->setUsesDropDown( true );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createActionsView()
{
    M_toggle_menu_bar_act = new QAction( tr( "&Menu Bar" ),
                                         QKeySequence(),
                                         this );
    M_toggle_menu_bar_act->setStatusTip( tr( "Show/Hide Menu Bar" ) );
    M_toggle_menu_bar_act->setAccel( Qt::CTRL + Qt::Key_M );
    connect( M_toggle_menu_bar_act, SIGNAL( activated() ),
             this, SLOT( toggleMenuBar() ) );

    M_toggle_tool_bar_act = new QAction( tr( "&Tool Bar" ),
                                         QKeySequence(),
                                         this );
    M_toggle_tool_bar_act->setStatusTip( tr( "Show/Hide Tool Bar" ) );
    connect( M_toggle_tool_bar_act, SIGNAL( activated() ),
             this, SLOT( toggleToolBar() ) );

    M_toggle_status_bar_act = new QAction( tr( "&Status Bar" ),
                                           QKeySequence(),
                                           this );
    M_toggle_status_bar_act->setStatusTip( tr( "Show/Hide Status Bar" ) );
    connect( M_toggle_status_bar_act, SIGNAL( activated() ),
             this, SLOT( toggleStatusBar() ) );

    M_full_screen_act = new QAction( tr( "&Full Screen" ),
                                     QKeySequence(),
                                     this );
    M_full_screen_act->setAccel( Qt::Key_F11 ); //tr( "Alt+Enter" ) );
    //M_full_screen_act->setAccel( tr( "Alt+Enter" ) );
    //M_full_screen_act->setAccel( Qt::ALT + Qt::Key_Return );
    //M_full_screen_act->setAccel( Qt::ALT + Qt::Key_Enter );
    M_full_screen_act->setStatusTip( tr( "Toggle Full Screen" ) );
    connect( M_full_screen_act, SIGNAL( activated() ),
             this, SLOT( toggleFullScreen() ) );
    // qt4
    //(void) new QShortcut( Qt::ALT + Qt::Key_Enter, this, SLOT( toggleFullScreen() ) );
    //(void) new QShortcut( Qt::ALT + Qt::Key_Return,
    //                      this, SLOT( toggleFullScreen() ) );
    // qt3
    {
        QAccel * accel = new QAccel( this );
        accel->connectItem( accel->insertItem( Qt::ALT + Qt::Key_Return ),
                            this, SLOT( toggleFullScreen() ) );
        accel->connectItem( accel->insertItem( Qt::ALT + Qt::Key_Enter ),
                            this, SLOT( toggleFullScreen() ) );
        //accel->connectItem( accel->insertItem( Qt::Key_F11 ),
        //                    this, SLOT( toggleFullScreen() ) );
    }

    M_show_player_type_dialog_act = new QAction( tr( "&Player Type List" ),
                                                 QKeySequence(),
                                                 this );
    M_show_player_type_dialog_act->setAccel( Qt::CTRL + Qt::Key_H );
    M_show_player_type_dialog_act->setStatusTip( tr( "Show player type parameters dialog" ) );
    connect( M_show_player_type_dialog_act, SIGNAL( activated() ),
             this, SLOT( showPlayerTypeDialog() ) );

    M_show_detail_dialog_act = new QAction( tr( "&Object Detail" ),
                                            QKeySequence(),
                                            this );
    M_show_detail_dialog_act->setAccel( Qt::CTRL + Qt::Key_I );
    M_show_detail_dialog_act->setStatusTip( tr( "Show detail information dialog" ) );
    connect( M_show_detail_dialog_act, SIGNAL( activated() ),
             this, SLOT( showDetailDialog() ) );

    // qt style menu group
    QSignalMapper * style_mapper = new QSignalMapper( this );
    connect( style_mapper, SIGNAL( mapped( const QString & ) ),
             this, SLOT( changeStyle( const QString & ) ) );;
    M_style_act_group = new QActionGroup( this );
    QStringList style_list = QStyleFactory::keys();
    for ( QStringList::iterator it = style_list.begin();
          it != style_list.end();
          ++it )
    {
        QAction * subaction = new QAction( M_style_act_group );
        subaction->setText( tr( "%1 Style" ).arg( *it ) );
        subaction->setToggleAction( true );
        if ( (*it).lower() == QString( QApplication::style().name() ).lower() )
        {
            subaction->setOn( true );
        }
        connect( subaction, SIGNAL( activated() ),
                 style_mapper, SLOT( map() ) );
        style_mapper->setMapping( subaction, *it );
    }

    M_show_color_setting_dialog_act = new QAction( tr( "&Color Settings" ),
                                                   QKeySequence(),
                                                   this );
    M_show_color_setting_dialog_act->setStatusTip( tr( "Show color setting dialog" ) );
    connect( M_show_color_setting_dialog_act, SIGNAL( activated() ),
             this, SLOT( showColorSettingDialog() ) );

    M_show_font_setting_dialog_act = new QAction( tr( "&Font Settings" ),
                                                  QKeySequence(),
                                                  this );
    M_show_font_setting_dialog_act->setStatusTip( tr( "Show font setting dialog" ) );
    connect( M_show_font_setting_dialog_act, SIGNAL( activated() ),
             this, SLOT( showFontSettingDialog() ) );

    M_show_view_config_dialog_act = new QAction( tr( "&View Preference" ),
                                                 QKeySequence(),
                                                 this );
    M_show_view_config_dialog_act->setAccel( Qt::CTRL + Qt::Key_V );
    M_show_view_config_dialog_act->setStatusTip( tr( "Show view preference dialog" ) );
    connect( M_show_view_config_dialog_act, SIGNAL( activated() ),
             this, SLOT( showViewConfigDialog() ) );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createActionsDebug()
{
    M_show_debug_message_window_act = new QAction( tr( "Debug &Message" ),
                                                   QKeySequence(),
                                                   this );
    M_show_debug_message_window_act->setAccel( Qt::CTRL + Qt::Key_D );
    M_show_debug_message_window_act->setStatusTip( tr( "Show debug message window" ) );
    connect( M_show_debug_message_window_act, SIGNAL( activated() ),
             this, SLOT( showDebugMessageWindow() ) );
    //
    M_toggle_debug_server_mode_act
        = new QAction( QIconSet( QPixmap( debug_server_switch_xpm ) ),
                       tr( "Debug Server Mode" ),
                       QKeySequence(),
                       this );
    M_toggle_debug_server_mode_act->setAccel( Qt::CTRL + Qt::Key_S );
    M_toggle_debug_server_mode_act->setStatusTip( tr( "Start/Stop the debug server." ) );
    M_toggle_debug_server_mode_act->setEnabled( false );
    M_toggle_debug_server_mode_act->setToggleAction( true );
    connect( M_toggle_debug_server_mode_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugServer( bool ) ) );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createActionsHelp()
{
    M_about_act = new QAction( QIconSet( QPixmap( soccerwindow2_nostr_xpm ) ),
                               tr( "&About" ),
                               QKeySequence(),
                               this );
    M_about_act->setStatusTip( tr( "Show the about dialog." ) );
    connect( M_about_act, SIGNAL( activated() ), this, SLOT( about() ) );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createMenus()
{
    createMenuFile();
    createMenuMonitor();
    createMenuView();
    createMenuDebug();
    createMenuHelp();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createMenuFile()
{
    QPopupMenu * menu = new QPopupMenu( this );

    menu->insertTearOffHandle();
    M_open_rcg_act->addTo( menu );
    M_save_rcg_act->addTo( menu );

    menu->insertSeparator();
    M_open_debug_view_act->addTo( menu );
    M_save_debug_view_act->addTo( menu );

    menu->insertSeparator();
    M_show_image_save_dialog_act->addTo( menu );

    menu->insertSeparator();
    M_exit_act->addTo( menu );

    menuBar()->insertItem( tr( "&File" ), menu );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createMenuMonitor()
{
    QPopupMenu * menu = new QPopupMenu( this );

    menu->insertTearOffHandle();
    M_kick_off_act->addTo( menu );
    M_set_live_mode_act->addTo( menu );

    menu->insertSeparator();
    M_connect_monitor_act->addTo( menu );
    M_connect_monitor_to_act->addTo( menu );
    M_disconnect_monitor_act->addTo( menu );

    menu->insertSeparator();
    M_kill_server_act->addTo( menu );
    M_restart_server_act->addTo( menu );

    menu->insertSeparator();
    M_toggle_drag_move_mode_act->addTo( menu );
    M_show_monitor_move_dialog_act->addTo( menu );

    menu->insertSeparator();
    M_show_launcher_dialog_act->addTo( menu );

    menuBar()->insertItem( tr( "&Monitor" ), menu );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createMenuView()
{
    QPopupMenu * menu = new QPopupMenu( this );
    menu->insertTearOffHandle();

    M_toggle_menu_bar_act->addTo( menu );
    M_toggle_tool_bar_act->addTo( menu );
    M_toggle_status_bar_act->addTo( menu );

    menu->insertSeparator();
    M_full_screen_act->addTo( menu );

    menu->insertSeparator();
    M_show_player_type_dialog_act->addTo( menu );
    M_show_detail_dialog_act->addTo( menu );

    menu->insertSeparator();
    {
        QPopupMenu * submenu = new QPopupMenu( this );
        M_style_act_group->addTo( submenu );
        menu->insertItem( tr( "Qt &Style" ), submenu );
    }
    M_show_color_setting_dialog_act->addTo( menu );
    M_show_font_setting_dialog_act->addTo( menu );
    M_show_view_config_dialog_act->addTo( menu );

    menuBar()->insertItem( tr( "&View" ), menu );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createMenuDebug()
{
    QPopupMenu * menu = new QPopupMenu( this );
    menu->insertTearOffHandle();

    M_show_debug_message_window_act->addTo( menu );
    M_toggle_debug_server_mode_act->addTo( menu );

    menuBar()->insertItem( tr( "&Debug" ), menu );
}
/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createMenuHelp()
{
    QPopupMenu * menu = new QPopupMenu( this );
    M_about_act->addTo( menu );

//     QAction * act = new QAction( tr( "About Qt" ), QKeySequence(), this );
//     act->setStatusTip( tr( "Show about Qt." ) );
//     connect( act, SIGNAL( activated() ), qApp, SLOT( aboutQt() ) );
//     act->addTo( menu );
    menu->insertItem( tr( "About Qt" ) , qApp, SLOT( aboutQt() ) );

    menuBar()->insertItem( tr( "&Help" ), menu );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createToolBars()
{
//     M_monitor_tool_bar = new QToolBar( tr( "Monitor" ), this );

//     M_set_live_mode_act->addTo( M_monitor_tool_bar );
//     M_toggle_drag_move_mode_act->addTo( M_monitor_tool_bar );
//     M_toggle_debug_server_mode_act->addTo( M_monitor_tool_bar );

    //
    M_log_player_tool_bar = new LogPlayerToolBar( M_log_player,
                                                  M_main_data,
                                                  this );

    M_log_player_tool_bar->addSeparator();

    M_set_live_mode_act->addTo( M_log_player_tool_bar );
    M_toggle_drag_move_mode_act->addTo( M_log_player_tool_bar );
    M_toggle_debug_server_mode_act->addTo( M_log_player_tool_bar );

    connect( this, SIGNAL( viewUpdated() ),
             M_log_player_tool_bar, SLOT( updateSlider() ) );

    if ( Options::instance().toolBarArea() == "top" )
    {
        //this->addDockWindow( M_log_player_tool_bar, Qt::DockTop );
    }
    else if ( Options::instance().toolBarArea() == "bottom" )
    {
        this->addDockWindow( M_log_player_tool_bar, Qt::DockBottom );
    }
    else if ( Options::instance().toolBarArea() == "left" )
    {
        M_log_player_tool_bar->setOrientation( Qt::Vertical );
        this->addDockWindow( M_log_player_tool_bar, Qt::DockLeft );
    }
    else if ( Options::instance().toolBarArea() == "right" )
    {
        M_log_player_tool_bar->setOrientation( Qt::Vertical );
        this->addDockWindow( M_log_player_tool_bar, Qt::DockRight );
    }
    else
    {
        std::cerr << "***WARNING*** Unsupported tool bar position ["
                  << Options::instance().toolBarArea() << ']'
                  << std::endl;
    }

    //this->addToolBar( M_log_player_tool_bar ); // qt4
    //this->addDockWindow( M_log_player_tool_bar, Qt::DockTop ); // qt3
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createStatusBar()
{
    this->statusBar()->message( tr( "Ready" ) );

    M_position_label = new QLabel( tr( "(0.0, 0.0)" ), this->statusBar() );

    int min_width
        = M_position_label->fontMetrics().width(  tr( "(-60.0, -30.0)" ) )
        + 16;
    M_position_label->setMinimumWidth( min_width );
    M_position_label->setAlignment( Qt::AlignRight );

    //this->statusBar()->addPermanentWidget( M_position_label ); // qt4
    this->statusBar()->addWidget( M_position_label, 0, true ); // permanent

    //QSlider * slider = new QSlider( Qt::Horizontal );
    //this->statusBar()->addWidget( slider );

    //QLineEdit * edit = new QLineEdit( tr( "0" ) );
    //edit->setMaximumSize( 36, 20 );
    //this->statusBar()->addWidget( edit );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createFieldCanvas()
{
    M_field_canvas = new FieldCanvas( M_main_data, this );
    //M_field_canvas->setParent( this );
    this->setCentralWidget( M_field_canvas );
    M_field_canvas->setFocus();

    connect( this, SIGNAL( viewUpdated() ),
             M_field_canvas, SLOT( update() ) );
    //       M_field_canvas, SLOT( repaint() ) );

    connect( M_field_canvas, SIGNAL( mouseMoved( const QPoint & ) ),
             this, SLOT( updatePositionLabel( const QPoint & ) ) );

    connect( M_field_canvas, SIGNAL( dropBall( const QPoint & ) ),
             this, SLOT( dropBall( const QPoint & ) ) );
    connect( M_field_canvas, SIGNAL( freeKickLeft( const QPoint & ) ),
             this, SLOT( freeKickLeft( const QPoint & ) ) );
    connect( M_field_canvas, SIGNAL( freeKickRight( const QPoint & ) ),
             this, SLOT( freeKickRight( const QPoint & ) ) );

    connect( M_field_canvas, SIGNAL( playModeChanged( int, const QPoint & ) ),
             this, SLOT( changePlayMode( int, const QPoint & ) ) );

    connect( M_field_canvas, SIGNAL( playerMoved( const QPoint & ) ),
             this, SLOT( moveDraggedPlayer( const QPoint & ) ) );

    // create & set context menus
    {
        QPopupMenu * menu = new QPopupMenu( M_field_canvas );
        M_open_rcg_act->addTo( menu );
        M_connect_monitor_act->addTo( menu );
        M_restart_server_act->addTo( menu );

        M_field_canvas->setNormalMenu( menu );
    }
    {
        QPopupMenu * menu = new QPopupMenu( M_field_canvas );
        M_open_rcg_act->addTo( menu );
        M_connect_monitor_act->addTo( menu );
        menu->insertSeparator();
        M_kill_server_act->addTo( menu );
        M_restart_server_act->addTo( menu );

        M_field_canvas->setSystemMenu( menu );
    }
    {
        const char * playmode_strings[] = PLAYMODE_STRINGS;

        QSignalMapper * mapper = new QSignalMapper( this );
        connect( mapper, SIGNAL( mapped( int ) ),
                 M_field_canvas, SLOT( changePlayMode( int ) ) );
        for ( int mode = 0; mode < rcsc::PM_MAX; ++mode )
        {
            if ( mode == rcsc::PM_BeforeKickOff
                 || mode == rcsc::PM_PlayOn
                 || mode == rcsc::PM_KickIn_Left
                 || mode == rcsc::PM_KickIn_Right
                 || mode == rcsc::PM_CornerKick_Left
                 || mode == rcsc::PM_CornerKick_Right
                 || mode == rcsc::PM_GoalKick_Left
                 || mode == rcsc::PM_GoalKick_Right
                 || mode == rcsc::PM_Foul_Charge_Left
                 || mode == rcsc::PM_Foul_Charge_Right
                 || mode == rcsc::PM_Foul_Push_Left
                 || mode == rcsc::PM_Foul_Push_Right
                 || mode == rcsc::PM_Back_Pass_Left
                 || mode == rcsc::PM_Back_Pass_Right
                 || mode == rcsc::PM_GoalKick_Right
                 || mode == rcsc::PM_IndFreeKick_Left
                 || mode == rcsc::PM_IndFreeKick_Right )
            {
                QAction * act = new QAction( M_playmode_change_act_group );
                act->setText( QString::fromAscii( playmode_strings[mode] ) );
                connect( act, SIGNAL( activated() ), mapper, SLOT( map() ) );
                mapper->setMapping( act, mode );
            }
        }
    }
    {
        QPopupMenu * menu = new QPopupMenu( M_field_canvas );
        M_kick_off_act->addTo( menu );
        //
        menu->insertSeparator();
        //
        menu->insertItem( tr( "Drop Ball" ),
                          M_field_canvas, SLOT( dropBall() ) );
        {
            QAction * act = new QAction( tr( "Drop Ball There" ),
                                         QKeySequence(),
                                         this );
            act->setStatusTip( tr( "Drop ball at the current ball position." ) );
            connect( act, SIGNAL( activated() ),
                     this, SLOT( dropBallThere() ) );
            act->addTo( menu );
        }
        menu->insertItem( tr( "Free Kick Left" ),
                          M_field_canvas, SLOT( freeKickLeft() ) );
        menu->insertItem( tr( "Free Kick Right" ),
                          M_field_canvas, SLOT( freeKickRight() ) );
        //
        menu->insertSeparator();
        //
        M_playmode_change_act_group->addTo( menu );
        //
        menu->insertSeparator();
        {
            QAction * act = new QAction( tr( "Yellow Card" ),
                                         QKeySequence(),
                                         this );
            act->setStatusTip( tr( "Call yellow card to the selected player." ) );
            connect( act, SIGNAL( activated() ),
                     this, SLOT( yellowCard() ) );
            act->addTo( menu );
        }
        {
            QAction * act = new QAction( tr( "Red Card" ),
                                         QKeySequence(),
                                         this );
            act->setStatusTip( tr( "Call red card to the selected player." ) );
            connect( act, SIGNAL( activated() ),
                     this, SLOT( redCard() ) );
            act->addTo( menu );
        }
        //

        //
        M_field_canvas->setMonitorMenu( menu );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::createViewConfigDialog()
{
    if ( M_view_config_dialog )
    {
        return;
    }

    M_view_config_dialog = new ViewConfigDialog( this, M_main_data );
    M_view_config_dialog->hide();

    connect( M_view_config_dialog, SIGNAL( configured() ),
             this, SIGNAL( viewUpdated() ) );

    connect( M_view_config_dialog, SIGNAL( canvasResized( const QSize & ) ),
             this, SLOT( resizeCanvas( const QSize & ) ) );

    connect( M_field_canvas, SIGNAL( focusChanged( const QPoint & ) ),
             M_view_config_dialog, SLOT( setFocusPoint( const QPoint & ) ) );

    {
        // register short cut keys
        QAccel * accel = new QAccel( this );

        // zoom
        // z
        accel->connectItem( accel->insertItem( Qt::Key_Z ),
                            M_view_config_dialog, SLOT( zoomIn() ) );
        // +
        accel->connectItem( accel->insertItem( Qt::Key_Plus ),
                            M_view_config_dialog, SLOT( zoomIn() ) );
        // x
        accel->connectItem( accel->insertItem( Qt::Key_X ),
                            M_view_config_dialog, SLOT( zoomOut() ) );
        // Ctrl + +
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_Plus ),
                            M_view_config_dialog, SLOT( zoomOut() ) );
        // Ctrl + z
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_Z ),
                            M_view_config_dialog, SLOT( zoomOut() ) );
        // i
        accel->connectItem( accel->insertItem( Qt::Key_I ),
                            M_view_config_dialog, SLOT( unzoom() ) );
        // e
        accel->connectItem( accel->insertItem( Qt::Key_E ),
                            M_view_config_dialog, SLOT( toggleEnlarge() ) );

        // reverse mode
        // Ctrl + r
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_R ),
                            M_view_config_dialog, SLOT( toggleReverseSide() ) );
        // Ctrl + Shift + r
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::SHIFT + Qt::Key_R ),
                            M_view_config_dialog, SLOT( togglePlayerReverseDraw() ) );

        // player detail
        // n
        accel->connectItem( accel->insertItem( Qt::Key_N ),
                            M_view_config_dialog, SLOT( toggleShowPlayerNumber() ) );
        // h
        accel->connectItem( accel->insertItem( Qt::Key_H ),
                            M_view_config_dialog, SLOT( toggleShowPlayerType() ) );
        // s
        accel->connectItem( accel->insertItem( Qt::Key_S ),
                            M_view_config_dialog, SLOT( toggleShowStamina() ) );
        //accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_S ),
        //                    M_view_config_dialog, SLOT( toggleShowStaminaCapacity() ) );
        // v
        accel->connectItem( accel->insertItem( Qt::Key_V ),
                            M_view_config_dialog, SLOT( toggleShowViewArea() ) );
        // d
        accel->connectItem( accel->insertItem( Qt::Key_D ),
                            M_view_config_dialog, SLOT( toggleShowBodyShadow() ) );
        // c
        accel->connectItem( accel->insertItem( Qt::Key_C ),
                            M_view_config_dialog, SLOT( toggleShowCatchableArea() ) );
        // t
        accel->connectItem( accel->insertItem( Qt::Key_T ),
                            M_view_config_dialog, SLOT( toggleShowTackleArea() ) );
        // k
        accel->connectItem( accel->insertItem( Qt::Key_K ),
                            M_view_config_dialog, SLOT( toggleShowKickAccelArea() ) );
        // f
        accel->connectItem( accel->insertItem( Qt::Key_F ),
                            M_view_config_dialog, SLOT( toggleShowPointto() ) );
        // Ctrl + a
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_A ),
                            M_view_config_dialog, SLOT( toggleShowAttentionto() ) );

        // show/hide
        // Ctrl + t
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_T ),
                            M_view_config_dialog, SLOT( toggleShowScoreBoard() ) );
        // Shift + t
        accel->connectItem( accel->insertItem( Qt::SHIFT + Qt::Key_T ),
                            M_view_config_dialog, SLOT( toggleShowTeamLogo() ) );
        // Ctrl + b
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_B ),
                            M_view_config_dialog, SLOT( toggleShowBall() ) );
        // Ctrl + p
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_P ),
                            M_view_config_dialog, SLOT( toggleShowPlayers() ) );
        // Ctrl + f
        accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_F ),
                            M_view_config_dialog, SLOT( toggleShowFlags() ) );
        // o
        accel->connectItem( accel->insertItem( Qt::Key_O ),
                            M_view_config_dialog, SLOT( toggleShowOffsideLine() ) );

        // grass type
        // k
//         accel->connectItem( accel->insertItem( Qt::Key_K ),
//                             M_view_config_dialog, SLOT( toggleKeepawayMode() ) );

        // focus
        // b
        accel->connectItem( accel->insertItem( Qt::Key_B ),
                            M_view_config_dialog, SLOT( toggleFocusBall() ) );
        // p
        accel->connectItem( accel->insertItem( Qt::Key_P ),
                            M_view_config_dialog, SLOT( toggleFocusPlayer() ) );

        // player selection
        // a
        accel->connectItem( accel->insertItem( Qt::Key_A ),
                            M_view_config_dialog, SLOT( toggleSelectAutoAll() ) );
        // l
        accel->connectItem( accel->insertItem( Qt::Key_L ),
                            M_view_config_dialog, SLOT( toggleSelectAutoLeft() ) );
        // r
        accel->connectItem( accel->insertItem( Qt::Key_R ),
                            M_view_config_dialog, SLOT( toggleSelectAutoRight() ) );
        // u
        accel->connectItem( accel->insertItem( Qt::Key_U ),
                            M_view_config_dialog, SLOT( setUnselect() ) );
    }

    // number keys & minus
    {
        QAccel * select_accel = new QAccel( this );
        // number 1-9
        for ( int i = 1; i <= 9; ++i )
        {
            select_accel->insertItem( Qt::Key_0 + i, i );
            select_accel->insertItem( Qt::CTRL + Qt::Key_0 + i, 11 + i );
        }
        // number 10
        select_accel->insertItem( Qt::Key_0, 10 );
        select_accel->insertItem( Qt::CTRL + Qt::Key_0, 11 + 10 );
        // number 11
        select_accel->insertItem( Qt::Key_Minus, 11 );
        select_accel->insertItem( Qt::CTRL + Qt::Key_Minus, 11 + 11 );

        connect( select_accel, SIGNAL( activated( int ) ),
                 M_view_config_dialog, SLOT( selectPlayer( int ) ) );

    }
}

/*-------------------------------------------------------------------*/
/*!

 */
// void
// MainWindow::closeEvent( QCloseEvent * event )
// {
//     event->ignore();

//     qApp->quit();
// }

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::resizeEvent( QResizeEvent * )
{
    //event->accept(); qt4

    if ( M_view_config_dialog
         && M_view_config_dialog->isVisible() )
    {
        M_view_config_dialog->updateFieldScale();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::wheelEvent( QWheelEvent * event )
{
    if ( event->delta() < 0 )
    {
        M_log_player->stepForward();
    }
    else
    {
        M_log_player->stepBack();
    }

    event->accept();
}


/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::dragEnterEvent( QDragEnterEvent * event )
{
    event->accept( QUriDrag::canDecode( event ) );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::dropEvent( QDropEvent * event )
{
    QStringList files;

    if ( QUriDrag::decodeLocalFiles( event, files ) )
    {
        //         std::cerr << "files size = " << files.size() << std::endl;

        //         for ( QStringList::iterator it = files.begin();
        //               it != files.end();
        //               ++it )
        //         {
        //             std::cerr << "url: " << (*it).ascii()
        //                       << std::endl;
        //         }

        if ( files.empty() )
        {
            QMessageBox::information( this,
                                      tr( "Information" ),
                                      tr( "Dropped data has no file path." ),
                                      QMessageBox::Ok, QMessageBox::NoButton );
        }
        else if ( files.size() == 1 )
        {
            openRCG( files.front() );
        }
        else if ( files.size() > 1 )
        {
            QMessageBox::critical( this,
                                   tr( "Error" ),
                                   tr( "Too many files are dropped." ),
                                   QMessageBox::Ok, QMessageBox::NoButton );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::openRCG()
{
    //std::cerr << "MainWindow::openRCG()" << std::endl;

#ifdef HAVE_LIBRCSC_GZ
    QString filter( tr( "Game Log files (*.rcg *.rcg.gz);;"
                        "All files (*)" ) );
#else
    QString filter( tr( "Game Log files (*.rcg);;"
                        "All files (*)" ) );
#endif

    QString default_dir( Options::instance().gameLogDir().c_str() );

    QString file_path = QFileDialog::getOpenFileName( default_dir,
                                                      filter,
                                                      this,
                                                      tr( "open file dialog" ),
                                                      tr( "Choose a game log file to open" ) );

    if ( file_path.isEmpty() )
    {
        //std::cerr << "MainWindow::opneRCG() empty file path" << std::endl;
        return;
    }

    std::cerr << "selected file = [" << file_path.ascii() << std::endl;

    openRCG( file_path );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::openRCG( const QString & file_path )
{
    if ( ! QFile::exists( file_path ) )
    {
        std::cerr << "File [" << file_path.ascii()
                  << "] does not exist."
                  << std::endl;
        return;
    }

    disconnectMonitor();
    M_log_player->stop();

    if ( ! M_main_data.openRCG( std::string( file_path.ascii() ) ) )
    {
        QString err_msg = tr( "Failed to read [" );
        err_msg += file_path;
        err_msg += tr( "]" );
        QMessageBox::critical( this,
                               tr( "Error" ),
                               err_msg,
                               QMessageBox::Ok, QMessageBox::NoButton );
        this->setCaption( tr( PACKAGE_NAME ) );
        return;
    }

    if ( M_main_data.viewHolder().monitorViewCont().empty() )
    {
        QString err_msg = tr( "Empty log file [" );
        err_msg += file_path;
        err_msg += tr( "]" );
        QMessageBox::critical( this,
                               tr( "Error" ),
                               err_msg,
                               QMessageBox::Ok, QMessageBox::NoButton );
        this->setCaption( tr( PACKAGE_NAME ) );
        return;
    }

    QFileInfo file_info( file_path );

    Options::instance().setGameLogDir( file_info.absFilePath().ascii() );

    if ( M_player_type_dialog )
    {
        M_player_type_dialog->updateData();
    }

    if ( M_debug_message_window )
    {
        M_debug_message_window->clearAll();
    }

    if ( M_view_config_dialog )
    {
        M_view_config_dialog->fitToScreen();
    }

    if ( ! Options::instance().anonymousMode() )
    {
        //this->setCaption( file_info.baseName() + tr( " - "PACKAGE_NAME ) );
        QString name = file_info.fileName();
        if ( name.length() > 128 )
        {
            name.replace( 125, name.length() - 125, tr( "..." ) );
        }
        this->setCaption( name + tr( " - "PACKAGE_NAME ) );
    }

    emit viewUpdated();

}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::saveRCG()
{
    if ( M_main_data.viewHolder().monitorViewCont().empty() )
    {
        QMessageBox::warning( this,
                              tr( "Error" ),
                              tr( "No Monitor View Data!" ) );
        return;
    }

    QString default_file_name;
    {
        const MonitorViewConstPtr latest = M_main_data.viewHolder().latestViewData();

        if ( latest )
        {
            default_file_name
                = QDateTime::currentDateTime().toString( "yyyyMMddhhmm-" );

            QString left_team( latest->leftTeam().name() );
            QString left_score = QString::number( latest->leftTeam().score() );

            QString right_team( latest->rightTeam().name() );
            QString right_score = QString::number( latest->rightTeam().score() );

            default_file_name += left_team;
            default_file_name += tr( "_" );
            default_file_name += left_score;
            default_file_name += tr( "-" );
            default_file_name += right_team;
            default_file_name += tr( "_" );
            default_file_name += right_score;

            default_file_name += tr( ".rcg" );
        }
    }

#ifdef HAVE_LIBRCSC_GZ
    QString filter( tr( "Game Log files (*.rcg *.rcg.gz);;"
                        "All files (*)" ) );
#else
    QString filter( tr( "Game Log files (*.rcg);;"
                        "All files (*)" ) );
#endif

    QString default_dir( Options::instance().gameLogDir() );
    if ( ! default_file_name.isEmpty() )
    {
        default_dir += tr( "/" );
        default_dir += default_file_name;
    }

    QString file_path = QFileDialog::getSaveFileName( default_dir,
                                                      filter,
                                                      this,
                                                      tr( "open file dialog" ),
                                                      tr( "Save a game log file as" ) );

    if ( file_path.isEmpty() )
    {
        //std::cerr << "MainWindow::saveRCG() empty file path" << std::endl;
        return;
    }

    std::string file_path_std = file_path.ascii();

    std::cerr << "save game log data to the file = [" << file_path_std
              << std::endl;

    // update game log dir
    QFileInfo file_info( file_path );
    Options::instance().setGameLogDir( file_info.absFilePath().ascii() );

    // check gzip usability
    bool is_gzip = false;
    if ( file_path_std.length() > 3
         && file_path_std.compare( file_path_std.length() - 3, 3, ".gz" ) == 0 )
    {
#ifdef HAVE_LIBRCSC_GZ
        if ( file_path_std.length() <= 7
             || file_path_std.compare( file_path_std.length() - 4, 4, ".rcg.gz" ) != 0 )
        {
            file_path_std == ".rcg.gz";
        }
        is_gzip = true;
#else
        // erase '.gz'
        file_path_std.erase( file_path_std.length() - 3 );
#endif
    }

    // check the extention string
    if ( ! is_gzip )
    {
        if ( file_path_std.length() <= 4
             || file_path_std.compare( file_path_std.length() - 4, 4, ".rcg" ) != 0 )
        {
            file_path_std += ".rcg";
        }
    }

    M_main_data.saveRCG( file_path_std );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::openDebugView()
{
    //std::cerr << "MainWindow::openDebugView()" << std::endl;

    QString default_dir( Options::instance().debugLogDir() );
#if 1
    QString dir_path
        = QFileDialog::getExistingDirectory( default_dir,
                                             this,
                                             "get existing directory",
                                             tr( "Choose a debug view log directory" ),
                                             true ); // dir only

#else
    DirSelector selector( this,
                          tr( "Choose a debug view log directory" ),
                          default_dir );

    if ( ! selector.exec() )
    {
        //std::cerr << "MainWindow::openDebugView() canceled" << std::endl;
        return;
    }

    QString dir_path = selector.dirPath();
#endif
    if ( dir_path.isEmpty() )
    {
        //std::cerr << "MainWindow::openDebugView() empty dir path" << std::endl;
        return;
    }

    std::cerr << "open debug view. dir = [" << dir_path.ascii() << std::endl;

    M_main_data.getViewHolder().openDebugView( dir_path.ascii() );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::saveDebugView()
{
    //std::cerr << "MainWindow::saveDebugView()" << std::endl;
#if 1
    QString dir_path
        = QFileDialog::getExistingDirectory( QString::null, // means current path
                                             this,
                                             "get existing directory",
                                             tr( "Choose a directory to save a debug view logs" ),
                                             true ); // dir only

#else
    DirSelector selector( this,
                          tr( "Choose a debug view log directory" ),
                          QDir::current().absPath() );

    if ( ! selector.exec() )
    {
        //std::cerr << "MainWindow::openDebugView() canceled" << std::endl;
        return;
    }

    QString dir_path = selector.dirPath();
#endif
    if ( dir_path.isEmpty() )
    {
        //std::cerr << "MainWindow::openDebugView() empty dir path" << std::endl;
        return;
    }

    std::cerr << "save debug view. dir = [" << dir_path.ascii() << std::endl;


    M_main_data.viewHolder().saveDebugView( dir_path.ascii() );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::kickOff()
{
    //std::cerr << "MainWindow::kickOff()" << std::endl;
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        M_monitor_client->sendKickOff();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::setLiveMode()
{
    //std::cerr << "MainWindow::setLiveMode()" << std::endl;

    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        M_log_player->setLiveMode();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::connectMonitor()
{
    //std::cerr << "MainWindow::connectMonitor()" << std::endl;
    connectMonitorTo( "127.0.0.1" );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::connectMonitorTo()
{
    //std::cerr << "MainWindow::connectMonitorTo()" << std::endl;

    if ( M_last_connected_host.isEmpty() )
    {
        M_last_connected_host = "127.0.0.1";
    }

    bool ok = true;
    QString text = QInputDialog::getText( tr( "Input sserver host name" ),
                                          tr( "Host name: "),
                                          QLineEdit::Normal,
                                          M_last_connected_host,
                                          & ok,
                                          this );
    if ( ok
         && ! text.isEmpty() )
    {
        connectMonitorTo( text.ascii() );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::connectMonitorTo( const char * hostname )
{
    if ( std::strlen( hostname ) == 0 )
    {
        std::cerr << "Empty host name! Connection failed!" << std::endl;
        return;
    }

    std::cerr << "Connect to rcssserver on [" << hostname << "]" << std::endl;

    M_monitor_client = new MonitorClient( this,
                                          M_main_data.getViewHolder(),
                                          hostname,
                                          Options::instance().port(),
                                          Options::instance().clientVersion() );

    if ( ! M_monitor_client->isConnected() )
    {
        std::cerr << "Conenction failed." << std::endl;
        delete M_monitor_client;
        M_monitor_client = static_cast< MonitorClient * >( 0 );
        return;
    }

    M_last_connected_host = hostname;

    // reset all data
    M_main_data.clear();

    if ( M_player_type_dialog )
    {
        M_player_type_dialog->hide();
    }

    if ( M_debug_message_window )
    {
        M_debug_message_window->clearAll();
    }

    if ( M_view_config_dialog )
    {
        M_view_config_dialog->fitToScreen();
    }

    if ( M_toggle_debug_server_mode_act->isOn() )
    {
        stopDebugServer();
        startDebugServer();
    }

    Options::instance().setMonitorClientMode( true );

    M_save_rcg_act->setEnabled( false );

    M_kick_off_act->setEnabled( true );
    M_set_live_mode_act->setEnabled( true );
    M_connect_monitor_act->setEnabled( false );
    M_connect_monitor_to_act->setEnabled( false );
    M_disconnect_monitor_act->setEnabled( true );
    M_kill_server_act->setEnabled( true );
    M_toggle_drag_move_mode_act->setEnabled( true );
    //M_show_monitor_move_dialog_act->setEnabled( true );

    M_toggle_debug_server_mode_act->setEnabled( true );
    M_show_image_save_dialog_act->setEnabled( false );

    connect( M_monitor_client, SIGNAL( received() ),
             this, SLOT( receiveMonitorPacket() ) );
    connect( M_monitor_client, SIGNAL( timeout() ),
             this, SLOT( disconnectMonitor() ) );

    M_log_player->setLiveMode();

    // this method will be called automatically.
    //M_monitor_client->sendDispInit();

    if ( QApplication::overrideCursor() )
    {
        QApplication::restoreOverrideCursor();
    }

    this->setCaption( tr( PACKAGE_NAME ) );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::disconnectMonitor()
{
    //std::cerr << "MainWindow::disconnectMonitor()" << std::endl;

    if ( M_monitor_client )
    {
        M_monitor_client->disconnect();

        disconnect( M_monitor_client, SIGNAL( received() ),
                    this, SLOT( receiveMonitorPacket() ) );

        disconnect( M_monitor_client, SIGNAL( timeout() ),
                    this, SLOT( disconnectMonitor() ) );

        delete M_monitor_client;
        M_monitor_client = static_cast< MonitorClient * >( 0 );
    }

    if ( M_debug_server )
    {
        delete M_debug_server;
        M_debug_server = static_cast< DebugServer * >( 0 );
    }

    Options::instance().setMonitorClientMode( false );

    M_save_rcg_act->setEnabled( true );

    M_kick_off_act->setEnabled( false );
    M_set_live_mode_act->setEnabled( false );
    M_connect_monitor_act->setEnabled( true );
    M_connect_monitor_to_act->setEnabled( true );
    M_disconnect_monitor_act->setEnabled( false );
    M_kill_server_act->setEnabled( false );
    M_toggle_drag_move_mode_act->setEnabled( false );
    //M_show_monitor_move_dialog_act->setEnabled( false );

    M_toggle_debug_server_mode_act->setEnabled( false );
    M_show_image_save_dialog_act->setEnabled( true );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::killServer()
{
    //std::cerr << "MainWindow::killServer()" << std::endl;

    disconnectMonitor();

    Options::instance().setKillServer( false );

    if ( Options::instance().serverPID() != 0 )
    {
        Options::instance().setServerPID( 0 );
        ::kill( pid_t( Options::instance().serverPID() ), SIGINT );
    }
    else
    {
#if 1
        int r = std::system( "killall -INT rcssserver" );
        if ( r == -1 )
        {
            std::cerr << "could not kill the rcssserver." << std::endl;
        }
#else
        // qt4
        //QString command( "killall -s INT rcssserver" );
        //QProcess::execute( command );
#endif
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::startServer()
{
    Options::instance().setServerPID( 0 );
    Options::instance().setKillServer( true );

    std::string server_command;
    if ( M_server_command.isEmpty() )
    {
        server_command = Options::instance().serverPath();
    }
    else
    {
        server_command = M_server_command.ascii();
        M_server_command.setLength( 0 );
    }

    if ( server_command.empty() )
    {
        return;
    }

#if 1
    //command += " > /dev/null 2>&1 &";
    server_command += " &";
    int r = std::system( server_command.c_str() );
    if ( r == -1 )
    {
        std::cerr << "could not start the rcssserver." << std::endl;
        return;
    }
#else
    QProcess * process = new QProcess( server_command, this );
    process->start();
#endif
    if ( ! QApplication::overrideCursor() )
    {
        QApplication::setOverrideCursor( QCursor( Qt::WaitCursor ) );
    }

    QTimer::singleShot( 1000,
                        this, SLOT( connectMonitor() ) );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::restartServer()
{
    restartServer( Options::instance().serverPath() );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::restartServer( const QString & command )
{
    static bool s_last_auto_start = false;

    M_server_command = command;

    bool auto_start = false;
    if ( command.find( "server::team_l_start" ) != -1 )
    {
        auto_start = true;
    }

    if ( M_monitor_client )
    {
        killServer();

        if ( ! QApplication::overrideCursor() )
        {
            QApplication::setOverrideCursor( QCursor( Qt::WaitCursor ) );
        }

        int timer_msec = 1000;
        if ( s_last_auto_start )
        {
            timer_msec = 3000;
        }

        QTimer::singleShot( timer_msec,
                            this, SLOT( startServer() ) );
    }
    else
    {
        startServer();
    }

    s_last_auto_start = auto_start;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::toggleDragMoveMode( bool on )
{
    if ( M_main_data.trainerData().dragMode() != on )
    {
        M_main_data.getTrainerData().toggleDragMode();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::showLauncherDialog()
{
    if ( M_launcher_dialog )
    {
        if ( M_launcher_dialog->isVisible() )
        {
            M_launcher_dialog->hide();
        }
        else
        {
            M_launcher_dialog->show();
        }
    }
    else
    {
        M_launcher_dialog = new LauncherDialog( this );
        M_launcher_dialog->show();

        //QSize size = M_launcher_dialog->size();
        //M_launcher_dialog->setMinimumSize( size );
        //M_launcher_dialog->setMaximumSize( size );
        QSize size = M_launcher_dialog->size();
        M_launcher_dialog->setMinimumSize( size );
        M_launcher_dialog->setMaximumSize( 1024, size.height() );

        connect( M_launcher_dialog, SIGNAL( launchServer( const QString & ) ),
                 this, SLOT( restartServer( const QString & ) ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::toggleMenuBar()
{
    if ( this->menuBar()->isVisible() )
    {
        this->menuBar()->hide();
    }
    else
    {
        this->menuBar()->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::toggleToolBar()
{
    if ( M_log_player_tool_bar->isVisible() )
    {
        M_log_player_tool_bar->hide();
    }
    else
    {
        M_log_player_tool_bar->show();
    }

//     if ( M_monitor_tool_bar->isVisible() )
//     {
//         M_monitor_tool_bar->hide();
//     }
//     else
//     {
//         M_monitor_tool_bar->show();
//     }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::toggleStatusBar()
{
    if ( this->statusBar()->isVisible() )
    {
        this->statusBar()->hide();
    }
    else
    {
        this->statusBar()->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::toggleFullScreen()
{
    if ( this->isFullScreen() )
    {
        this->showNormal();
    }
    else
    {
        this->showFullScreen();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::showPlayerTypeDialog()
{
    if ( M_player_type_dialog )
    {
        if ( M_player_type_dialog->isVisible() )
        {
            M_player_type_dialog->hide();
        }
        else
        {
            M_player_type_dialog->show();
        }
    }
    else
    {
        M_player_type_dialog = new PlayerTypeDialog( this, M_main_data );
        M_player_type_dialog->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::showMonitorMoveDialog()
{
    if ( M_monitor_move_dialog )
    {
        if ( M_monitor_move_dialog->isVisible() )
        {
            M_monitor_move_dialog->hide();
        }
        else
        {
            M_monitor_move_dialog->show();
        }
    }
    else
    {
        M_monitor_move_dialog
            = new MonitorMoveDialog( this,
                                     M_main_data,
                                     M_main_data.getTrainerData() );
        connect( M_monitor_move_dialog, SIGNAL( executed() ),
                 this, SLOT( moveObjects() ) );

        M_monitor_move_dialog->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::showDetailDialog()
{
    //std::cerr << "MainWindow::showDetailDialog()" << std::endl;

    if ( M_detail_dialog )
    {
        if ( M_detail_dialog->isVisible() )
        {
            M_detail_dialog->hide();
        }
        else
        {
            M_detail_dialog->show();
        }
    }
    else
    {
        M_detail_dialog = new DetailDialog( this, M_main_data );
        connect( this,  SIGNAL( viewUpdated() ),
                 M_detail_dialog, SLOT( updateLabels() ) );

        M_detail_dialog->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::changeStyle( const QString & style )
{
    QApplication::setStyle( style );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::showColorSettingDialog()
{
    ColorSettingDialog dlg( this );

    connect( &dlg, SIGNAL( colorChanged() ),
             M_field_canvas, SLOT( update() ) );

    dlg.exec();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::showFontSettingDialog()
{
    FontSettingDialog dlg( this );

    connect( &dlg, SIGNAL( fontChanged() ),
             M_field_canvas, SLOT( update() ) );

    dlg.exec();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::showViewConfigDialog()
{
    if ( M_view_config_dialog->isVisible() )
    {
        M_view_config_dialog->hide();
    }
    else
    {
        M_view_config_dialog->show();

        QSize size = M_view_config_dialog->size();
        M_view_config_dialog->setMinimumSize( size );
        M_view_config_dialog->setMaximumSize( size );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::showDebugMessageWindow()
{
    //std::cerr << "MainWindow::showDebugMessageWindow()" << std::endl;

    if ( M_debug_message_window )
    {
        if ( M_debug_message_window->isVisible() )
        {
            M_debug_message_window->hide();
        }
        else
        {
            M_debug_message_window->show();
        }
    }
//     else
//     {
//         M_debug_message_window = new DebugMessageWindow( this,
//                                                          M_main_data );
//         connect( M_debug_message_window, SIGNAL( configured() ),
//                  this, SIGNAL( viewUpdated() ) );

//         M_debug_message_window->show();
//     }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::toggleDebugServer( bool on )
{
    if ( on )
    {
        startDebugServer();
    }
    else
    {
        stopDebugServer();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::startDebugServer()
{
    //std::cerr << "MainWindow::startDebugServer()" << std::endl;

    if ( M_debug_server )
    {
        // the server instance has already existed.
        return;
    }

    std::cerr << "Start Debug Server" << std::endl;

    int port = static_cast< int >( Options::instance().debugServerPort() );

    M_debug_server = new DebugServer( this,
                                      M_main_data.getViewHolder(),
                                      port );

    if ( ! M_debug_server->isConnected() )
    {
        std::cerr << "failed to create Debug Server" << std::endl;
        delete M_debug_server;
        M_debug_server = static_cast< DebugServer * >( 0 );
        return;
    }

    M_toggle_debug_server_mode_act->setOn( true );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::stopDebugServer()
{
    //std::cerr << "MainWindow::stopDebugServer()" << std::endl;

    if ( M_debug_server )
    {
        //std::cerr << "Stop Debug Server" << std::endl;
        delete M_debug_server;
        M_debug_server = static_cast< DebugServer * >( 0 );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::showImageSaveDialog()
{
    //std::cerr << "MainWindow::showImageSaveDialog()" << std::endl;

    M_log_player->stop();

    ImageSaveDialog dlg( this,
                         M_field_canvas,
                         M_main_data );

    dlg.exec();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::about()
{
    //QString msg( "soccerwindow2" ); //QString msg( tr( PACKAGE ) );
    //msg += tr( "-" );
    //msg += tr( "2.0.0" ); //msg += tr( VERSION );
    QString msg( tr( PACKAGE "-" VERSION ) );
    msg += tr( "\n\n" );
    msg += tr( "soccerwindow2 is a viewer applicaton for\n"
               "the RoboCup Soccer Simulator.\n"
               "  http://sserver.sourceforge.net/\n"
               "\n"
               "soccerwindow2 Development Site:\n"
               "  http://rctools.sourceforge.jp/\n"
               "Author:\n"
               "  Hidehisa AKIYAMA <akky@users.sourceforge.jp>" );

    QMessageBox::about( this,
                        tr( "About soccerwindow2" ),
                        msg );

    // from Qt 4.1 documents
    /*
      about() looks for a suitable icon in four locations:

      1. It prefers parent->icon() if that exists.
      2. If not, it tries the top-level widget containing parent.
      3. If that fails, it tries the active window.
      4. As a last resort it uses the Information icon.

      The about box has a single button labelled "OK".
    */
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::resizeCanvas( const QSize & size )
{
    if ( size.width() < this->minimumWidth()
         || size.height() < this->minimumHeight() )
    {
        std::cerr << "Too small canvas size ("
                  << size.width() << " "
                  << size.height() << ")"
                  << std::endl;
        return;
    }

    if ( centralWidget() )
    {
        if ( this->isMaximized()
             || this->isFullScreen() )
        {
            this->showNormal();
        }

        QRect rect = this->geometry();

        int width_diff = rect.width() - centralWidget()->width();
        int height_diff = rect.height() - centralWidget()->height();

        rect.setWidth( size.width() + width_diff );
        rect.setHeight( size.height() + height_diff );

        this->setGeometry( rect );

        //std::cerr << "centralWidget width = " << centralWidget()->width()
        //          << " height = " << centralWidget()->height()
        //          << std::endl;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::saveImageAndQuit()
{
    if ( ! this->isVisible() )
    {
        QTimer::singleShot( 500,
                            this, SLOT( saveImageAndQuit() ) );
        return;
    }

    ImageSaveDialog dlg( this,
                         M_field_canvas,
                         M_main_data );
    dlg.show();
    dlg.selectAllCycle();
    dlg.executeSave();

    qApp->quit(); // QApplication::exit( 0 );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::receiveMonitorPacket()
{
    //std::cerr << "receive monitor packet" << std::endl;

    if ( M_log_player->isLiveMode() )
    {
        M_log_player->showLive();
    }
    else
    {
        //std::cerr << "receive monitor packet  no live" << std::endl;
        M_log_player_tool_bar->updateSlider();
    }


    if ( Options::instance().autoQuitMode() )
    {
        if ( M_main_data.viewHolder().lastPlayMode() == rcsc::PM_TimeOver )
        {
            static QDateTime s_game_end_time;

            if ( ! s_game_end_time.isValid() )
            {
                s_game_end_time = QDateTime::currentDateTime();
            }
            else
            {
                if ( s_game_end_time.secsTo( QDateTime::currentDateTime() )
                     >= Options::instance().waitSeconds() )
                {
                    std::cerr << "Elapsed " << Options::instance().waitSeconds()
                              << " seconds after game end\n"
                              << "Exit..."
                              << std::endl;
                    qApp->quit();
                }
            }
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::updatePositionLabel( const QPoint & point )
{
    if ( M_position_label
         && statusBar()
         && statusBar()->isVisible()
         )
    {
        double x = Options::instance().fieldX( point.x() );
        double y = Options::instance().fieldY( point.y() );

        char buf[32];
        snprintf( buf, 32, "(%.2f, %.2f)", x, y );

        M_position_label->setText( QString::fromAscii( buf ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::dropBallThere()
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        MonitorViewConstPtr view = M_main_data.viewHolder().latestViewData();
        if ( view )
        {
            //std::cerr << "drop ball at current position "
            //          << std::endl;
            M_monitor_client->sendDropBall( view->ball().x(),
                                            view->ball().y() );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::dropBall( const QPoint & point )
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        double x = Options::instance().fieldX( point.x() );
        double y = Options::instance().fieldY( point.y() );

        if ( Options::instance().reverseSide() )
        {
            x = -x;
            y = -y;
        }
        std::cerr << "drop ball to ("
                  << x << ", " << y << ")"
                  << std::endl;
        M_monitor_client->sendDropBall( x, y );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::freeKickLeft( const QPoint & point )
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        double x = Options::instance().fieldX( point.x() );
        double y = Options::instance().fieldY( point.y() );

        if ( Options::instance().reverseSide() )
        {
            x = -x;
            y = -y;
            M_monitor_client->sendFreeKickRight( x, y );
        }
        else
        {
            //std::cerr << "free kick left at ("
            //          << x << ", " << y << ")"
            //          << std::endl;
            M_monitor_client->sendFreeKickLeft( x, y );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::freeKickRight( const QPoint & point )
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        double x = Options::instance().fieldX( point.x() );
        double y = Options::instance().fieldY( point.y() );

        if ( Options::instance().reverseSide() )
        {
            x = -x;
            y = -y;
            M_monitor_client->sendFreeKickLeft( x, y );
        }
        else
        {
            //std::cerr << "free kick right at ("
            //          << x << ", " << y << ")"
            //          << std::endl;
            M_monitor_client->sendFreeKickRight( x, y );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::moveDraggedPlayer( const QPoint & point )
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        rcsc::SideID side = M_main_data.trainerData().draggedPlayerSide();
        int unum = M_main_data.trainerData().draggedPlayerNumber();

        if ( side != rcsc::NEUTRAL
             && unum != 0 )
        {
            double x = Options::instance().fieldX( point.x() );
            double y = Options::instance().fieldY( point.y() );
            if ( Options::instance().reverseSide() )
            {
                x = -x;
                y = -y;
            }

            //std::cerr << "move player side=" << side
            //          << " unum = " << unum
            //          << " pos=" << x << ' ' << y << std::endl;
            M_main_data.getTrainerData().unsetDrag();
            M_monitor_client->sendMovePlayer( side, unum, x, y, 0.0 );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::moveObjects()
{
    if ( ! M_monitor_client
         || ! M_monitor_client->isConnected() )
    {
        return;
    }

    TrainerData & data = M_main_data.getTrainerData();

    // ball
    if ( data.ballPos().isValid() )
    {
        if ( rcsc::ServerParam::i().coachMode()
             || rcsc::ServerParam::i().coachWithRefereeMode() )
        {
            if ( data.ballVel().isValid() )
            {
                M_monitor_client->sendTrainerMoveBall( data.ballPos().x,
                                                       data.ballPos().y,
                                                       data.ballVel().x,
                                                       data.ballVel().y );
            }
            else
            {
                M_monitor_client->sendTrainerMoveBall( data.ballPos().x,
                                                       data.ballPos().y );
            }
        }
        else
        {
            if ( data.playMode() == rcsc::PM_FreeKick_Left )
            {
                M_monitor_client->sendFreeKickLeft( data.ballPos().x,
                                                    data.ballPos().y );
            }
            else if ( data.playMode() == rcsc::PM_FreeKick_Right )
            {
                M_monitor_client->sendFreeKickRight( data.ballPos().x,
                                                     data.ballPos().y );
            }
            else
            {
                M_monitor_client->sendDropBall( data.ballPos().x,
                                                data.ballPos().y );
            }
        }
    }

    // left
    for ( int unum = 1; unum <= 11; ++unum )
    {
        rcsc::Vector2D pos = data.playerMovePos( rcsc::LEFT, unum );
        rcsc::AngleDeg body = data.playerBody( rcsc::LEFT, unum );
        if ( pos.isValid() )
        {
            M_monitor_client->sendMovePlayer( rcsc::LEFT, unum,
                                              pos.x, pos.y,
                                              body.degree() );
        }
    }

    // right
    for ( int unum = 1; unum <= 11; ++unum )
    {
        rcsc::Vector2D pos = data.playerMovePos( rcsc::RIGHT, unum );
        rcsc::AngleDeg body = data.playerBody( rcsc::RIGHT, unum );
        if ( pos.isValid() )
        {
            M_monitor_client->sendMovePlayer( rcsc::RIGHT, unum,
                                              pos.x, pos.y,
                                              body.degree() );
        }
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::yellowCard()
{
    if ( ! M_monitor_client
         || ! M_monitor_client->isConnected() )
    {
        return;
    }

    int unum = Options::instance().selectedNumber();
    if ( unum == 0 )
    {
        return;
    }

    rcsc::SideID side = rcsc::LEFT;
    if ( unum < 0 )
    {
        unum = -unum;
        side = rcsc::RIGHT;
    }

    M_monitor_client->sendCard( side, unum, rcsc::YELLOW );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::redCard()
{
    if ( ! M_monitor_client
         || ! M_monitor_client->isConnected() )
    {
        return;
    }

    int unum = Options::instance().selectedNumber();
    if ( unum == 0 )
    {
        return;
    }

    rcsc::SideID side = rcsc::LEFT;
    if ( unum < 0 )
    {
        unum = -unum;
        side = rcsc::RIGHT;
    }

    M_monitor_client->sendCard( side, unum, rcsc::RED );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
MainWindow::changePlayMode( int mode,
                            const QPoint & point )
{
    if ( ! M_monitor_client
         || ! M_monitor_client->isConnected() )
    {
        return;
    }

    if ( ! rcsc::ServerParam::i().coachMode()
         && ! rcsc::ServerParam::i().coachWithRefereeMode() )
    {
        QMessageBox::warning( this,
                              tr( "PlayMode Change" ),
                              tr( "Server is not running with coach mode." ) );
        return;
    }

    rcsc::PlayMode pmode = static_cast< rcsc::PlayMode >( mode );

    double x = Options::instance().fieldX( point.x() );
    double y = Options::instance().fieldY( point.y() );

    x = std::min( x, + rcsc::ServerParam::i().pitchHalfLength() );
    x = std::max( x, - rcsc::ServerParam::i().pitchHalfLength() );
    y = std::min( y, + rcsc::ServerParam::i().pitchHalfWidth() );
    y = std::max( y, - rcsc::ServerParam::i().pitchHalfWidth() );

    switch ( pmode ) {
    case rcsc::PM_BeforeKickOff:
        M_monitor_client->sendTrainerMoveBall( 0.0, 0.0, 0.0, 0.0 );
        M_monitor_client->sendChangeMode( pmode );
        break;
    case rcsc::PM_PlayOn:
        M_monitor_client->sendTrainerMoveBall( x, y, 0.0, 0.0 );
        M_monitor_client->sendChangeMode( pmode );
        break;
    case rcsc::PM_KickIn_Left:
    case rcsc::PM_KickIn_Right:
        y = ( y > 0.0
              ? + rcsc::ServerParam::i().pitchHalfWidth()
              : - rcsc::ServerParam::i().pitchHalfWidth() );

        M_monitor_client->sendTrainerMoveBall( x, y, 0.0, 0.0 );
        M_monitor_client->sendChangeMode( pmode );
        break;
    case rcsc::PM_CornerKick_Left:
        x = + rcsc::ServerParam::i().pitchHalfLength()
            - rcsc::ServerParam::i().cornerKickMargin();
        y = ( y > 0.0
              ? ( + rcsc::ServerParam::i().pitchHalfWidth()
                  - rcsc::ServerParam::i().cornerKickMargin() )
              : ( - rcsc::ServerParam::i().pitchHalfWidth()
                  + rcsc::ServerParam::i().cornerKickMargin() ) );

        M_monitor_client->sendTrainerMoveBall( x, y, 0.0, 0.0 );
        M_monitor_client->sendChangeMode( pmode );
        break;
    case rcsc::PM_CornerKick_Right:
        x = - rcsc::ServerParam::i().pitchHalfLength()
            + rcsc::ServerParam::i().cornerKickMargin();
        y = ( y > 0.0
              ? ( + rcsc::ServerParam::i().pitchHalfWidth()
                  - rcsc::ServerParam::i().cornerKickMargin() )
              : ( - rcsc::ServerParam::i().pitchHalfWidth()
                  + rcsc::ServerParam::i().cornerKickMargin() ) );

        M_monitor_client->sendTrainerMoveBall( x, y, 0.0, 0.0 );
        M_monitor_client->sendChangeMode( pmode );
        break;
    case rcsc::PM_GoalKick_Left:
        x = - rcsc::ServerParam::i().pitchHalfLength()
            + rcsc::ServerParam::i().goalAreaLength();
        y = ( y > 0.0
              ? + rcsc::ServerParam::i().goalAreaHalfWidth()
              : - rcsc::ServerParam::i().goalAreaHalfWidth() );

        M_monitor_client->sendTrainerMoveBall( x, y, 0.0, 0.0 );
        M_monitor_client->sendChangeMode( pmode );
        break;
    case rcsc::PM_GoalKick_Right:
        x = + rcsc::ServerParam::i().pitchHalfLength()
            - rcsc::ServerParam::i().goalAreaLength();
        y = ( y > 0.0
              ? + rcsc::ServerParam::i().goalAreaHalfWidth()
              : - rcsc::ServerParam::i().goalAreaHalfWidth() );

        M_monitor_client->sendTrainerMoveBall( x, y, 0.0, 0.0 );
        M_monitor_client->sendChangeMode( pmode );
        break;
    case rcsc::PM_Foul_Charge_Left:
        M_monitor_client->sendTrainerMoveBall( x, y, 0.0, 0.0 );
        M_monitor_client->sendChangeMode( pmode );
        break;
    case rcsc::PM_Foul_Charge_Right:
        M_monitor_client->sendTrainerMoveBall( x, y, 0.0, 0.0 );
        M_monitor_client->sendChangeMode( pmode );
        break;
    case rcsc::PM_Foul_Push_Left:
        M_monitor_client->sendTrainerMoveBall( x, y, 0.0, 0.0 );
        M_monitor_client->sendChangeMode( pmode );
        break;
    case rcsc::PM_Foul_Push_Right:
        M_monitor_client->sendTrainerMoveBall( x, y, 0.0, 0.0 );
        M_monitor_client->sendChangeMode( pmode );
        break;
    case rcsc::PM_Back_Pass_Right:
        if ( x < ( + rcsc::ServerParam::i().pitchHalfLength()
                   - rcsc::ServerParam::i().penaltyAreaLength() ) )
        {
            x = + rcsc::ServerParam::i().pitchHalfLength()
                - rcsc::ServerParam::i().penaltyAreaLength();
        }

        if ( std::fabs( y ) > rcsc::ServerParam::i().penaltyAreaHalfWidth() )
        {
            y = ( y > 0.0
                  ? + rcsc::ServerParam::i().penaltyAreaHalfWidth()
                  : - rcsc::ServerParam::i().penaltyAreaHalfWidth() );
        }
    case rcsc::PM_IndFreeKick_Left:
        if ( x >= ( + rcsc::ServerParam::i().pitchHalfLength()
                    - rcsc::ServerParam::i().goalAreaLength() )
             && std::fabs( y ) <= rcsc::ServerParam::i().goalAreaHalfWidth() )
        {
            x = + rcsc::ServerParam::i().pitchHalfLength()
                - rcsc::ServerParam::i().goalAreaLength();
        }

        M_monitor_client->sendTrainerMoveBall( x, y, 0.0, 0.0 );
        M_monitor_client->sendChangeMode( pmode );
        break;
    case rcsc::PM_Back_Pass_Left:
        if ( x > ( - rcsc::ServerParam::i().pitchHalfLength()
                   + rcsc::ServerParam::i().penaltyAreaLength() ) )
        {
            x = - rcsc::ServerParam::i().pitchHalfLength()
                + rcsc::ServerParam::i().penaltyAreaLength();
        }

        if ( std::fabs( y ) > rcsc::ServerParam::i().penaltyAreaHalfWidth() )
        {
            y = ( y > 0.0
                  ? + rcsc::ServerParam::i().penaltyAreaHalfWidth()
                  : - rcsc::ServerParam::i().penaltyAreaHalfWidth() );
        }
    case rcsc::PM_IndFreeKick_Right:
        if ( x <= ( - rcsc::ServerParam::i().pitchHalfLength()
                    + rcsc::ServerParam::i().goalAreaLength() )
             && std::fabs( y ) <= rcsc::ServerParam::i().goalAreaHalfWidth() )
        {
            x = - rcsc::ServerParam::i().pitchHalfLength()
                + rcsc::ServerParam::i().goalAreaLength();
        }

        M_monitor_client->sendTrainerMoveBall( x, y, 0.0, 0.0 );
        M_monitor_client->sendChangeMode( pmode );
        break;
    default:
        QMessageBox::warning( this,
                              tr( "PlayMode Change" ),
                              tr( "Unsupported playmode " ) );
        break;
    }
}
