// -*-c++-*-

/*!
  \file feditor_data.cpp
  \brief formation editor data class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "fedit_data.h"

#include "fedit_config.h"

//#include <rcsc/formation/formation_factory.h>
#include <rcsc/formation/formation.h>
#include <rcsc/formation/formation_dt.h>
#include <rcsc/formation/formation_knn.h>
#include <rcsc/common/server_param.h>
#include <rcsc/math_util.h>
#include <rcsc/timer.h>

#include <algorithm>
#include <sstream>
#include <fstream>
#include <iostream>

#include <cassert>
#include <ctime>
#include <cstdio>
#include <cstdarg>

using namespace rcsc;
using namespace rcsc::formation;


const double FEditData::MAX_X = ServerParam::DEFAULT_PITCH_LENGTH * 0.5 + 2.0;
const double FEditData::MAX_Y = ServerParam::DEFAULT_PITCH_WIDTH * 0.5 + 2.0;

namespace {

inline
Vector2D
round_coordinates( const double & x,
                   const double & y )
{
    return Vector2D( rint( bound( - FEditData::MAX_X, x, FEditData::MAX_X )
                           / SampleData::PRECISION )
                     * SampleData::PRECISION,
                     rint( bound( - FEditData::MAX_Y, y, FEditData::MAX_Y )
                           / SampleData::PRECISION )
                     * SampleData::PRECISION );
}

}

/*-------------------------------------------------------------------*/
/*!

 */
FEditData::FEditData()
    : M_training_type_name( "" )
    , M_filepath( "" )
    , M_modified( false )
    , M_our_dragged( 11, 0 )
    , M_our_players( 11 )
    , M_opp_players( 11 )
    , M_symmetry_mode( true )
    , M_select_type( NO_SELECT )
    , M_select_index( 0 )
    , M_ball_draggable( true )
    , M_player_auto_move( true )
    , M_data_auto_select( true )
{
    init();
}

/*-------------------------------------------------------------------*/
/*!

 */
FEditData::~FEditData()
{
    //saveChanged();
    //std::cerr << "delete FEditData" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::init()
{
    M_modified = false;
    M_samples.reset();

    M_ball.assign( FEditConfig::instance().initialBallX(),
                   FEditConfig::instance().initialBallY() );

    M_our_dragged.assign( 11, 0 );
    for ( int i = 1; i <= 11; ++i )
    {
        M_our_players[i-1].setPolar( -3.0 * i, -37.0 );
        M_opp_players[i-1].assign( 3.0 * i, -37.0 );
    }

    M_select_type = NO_SELECT;
    M_select_index = 0;
    M_ball_draggable = true;
    M_player_auto_move = true;

    updateTriangulation();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::backup( const std::string & filepath )
{
    if ( filepath.empty() )
    {
        return;
    }

    std::ostringstream time_str;
    std::time_t current = std::time( NULL );
    std::tm * local_time = std::localtime( &current );
    if ( local_time )
    {
        char buf[64];
        std::strftime( buf, 64, ".%Y%m%d-%H%M.", local_time );
        time_str << buf;
    }
    else
    {
        time_str << "." << current << ".";
    }

    std::string backup_filepath = ".";
    backup_filepath += time_str.str();
    backup_filepath += filepath;

    std::ifstream fin( filepath.c_str() );
    std::ofstream fout( backup_filepath.c_str() );

    if ( fin && fout )
    {
        std::istreambuf_iterator< char > in_itr( fin );
        std::ostreambuf_iterator< char > out_itr( fout );
        std ::copy( in_itr, std::istreambuf_iterator< char >(),
                    out_itr );

        fin.close();
        fout.close();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::createDefaultData()
{
    M_formation = Formation::create( M_training_type_name );

    if ( ! M_formation )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " ***ERROR*** Failed to create formation."
                  << std::endl;
        return;
    }

    M_samples = SampleDataSet::Ptr( new SampleDataSet() );
    M_formation->setSamples( M_samples );
    M_formation->createDefaultData();
    train();
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
FEditData::openConf( const std::string & filepath )
{
    std::ifstream fin( filepath.c_str() );
    if ( ! fin.is_open() )
    {
        std::cerr << "Failed to open formation file [" << filepath << "]"
                  << std::endl;
        return false;
    }

    M_samples.reset();
    M_formation = Formation::create( fin );

    if ( ! M_formation )
    {
        std::cerr << "Invalid formation format in the file ["
                  << filepath << "]\n"
                  << "You must write the training method type name"
                  << " at the top of the formation file"
                  << std::endl;
        return false;
    }

    fin.seekg( 0 );
    if ( ! M_formation->read( fin ) )
    {
        fin.close();
        M_formation.reset();
        return false;
    }

    fin.close();
    M_filepath = filepath;

    M_samples = M_formation->samples();
    updateTriangulation();

    moveBallTo( FEditConfig::instance().initialBallX(),
                FEditConfig::instance().initialBallY() );
    updatePlayerPosition();
    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
FEditData::openData( const std::string & filepath )
{
    if ( ! M_formation )
    {
        std::cerr << "No formation! create a new one or open a exsiting one." << std::endl;
        return false;
    }

    M_triangulation.clear();

    M_samples = SampleDataSet::Ptr( new SampleDataSet() );
    if ( ! M_samples->open( filepath ) )
    {
        M_samples.reset();
        return false;
    }

    //M_modified = false;

    M_formation->setSamples( M_samples );
    updateTriangulation();

    moveBallTo( FEditConfig::instance().initialBallX(),
                FEditConfig::instance().initialBallY() );
    updatePlayerPosition();

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
FEditData::openBackgroundConf( const std::string & filepath )
{
    std::ifstream fin( filepath.c_str() );
    if ( ! fin.is_open() )
    {
        std::cerr << "Failed to open a background formation file [" << filepath << "]"
                  << std::endl;
        return false;
    }

    M_background_formation = Formation::create( fin );

    if ( ! M_background_formation )
    {
        std::cerr << "Invalid formation format in the file ["
                  << filepath << "]\n"
                  << "You must write the training method type name"
                  << " at the top of the formation file"
                  << std::endl;
        return false;
    }

    fin.seekg( 0 );
    if ( ! M_background_formation->read( fin ) )
    {
        fin.close();
        M_background_formation.reset();
        return false;
    }

    fin.close();

    const Rect2D pitch( Vector2D( - FEditConfig::PITCH_LENGTH * 0.5,
                                  - FEditConfig::PITCH_WIDTH * 0.5 ),
                        Size2D( FEditConfig::PITCH_LENGTH,
                                FEditConfig::PITCH_WIDTH ) );
    M_background_triangulation.init( pitch );

    const SampleDataSet::DataCont::const_iterator end = M_background_formation->samples()->dataCont().end();
    for ( SampleDataSet::DataCont::const_iterator it = M_background_formation->samples()->dataCont().begin();
          it != end;
          ++it )
    {
        M_background_triangulation.addVertex( it->ball_ );
    }

    M_background_triangulation.compute();

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
FEditData::saveConf()
{
    if ( M_filepath.empty() )
    {
        return false;
    }

    return saveConfAs( M_filepath );
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
FEditData::saveConfAs( const std::string & filepath )
{
    // create backup file
    if ( FEditConfig::instance().autoBackup() )
    {
        backup( M_filepath );
    }

    std::ofstream fout( filepath.c_str() );
    if ( ! fout.is_open() )
    {
        fout.close();
        return false;
    }

    if ( ! saveConf( fout ) )
    {
        return false;
    }

    fout.flush();

    M_filepath = filepath;
    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
FEditData::saveConf( std::ostream & os )
{
    if ( ! M_formation )
    {
        return false;
    }

    if ( ! M_formation->print( os ) )
    {
        return false;
    }

    M_modified = false;
    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
FEditData::saveDataAs( const std::string & filepath )
{
    if ( ! M_samples )
    {
        return false;
    }

    return M_samples->save( filepath );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::updatePlayerPosition()
{
    if ( ! M_player_auto_move )
    {
        return;
    }

    if ( ! M_formation )
    {
        //std::cerr << "No formation param set" << std::endl;
        return;
    }

    // update all players' position using formation param
    M_formation->getPositions( M_ball, M_our_players );

    for ( std::vector< Vector2D >::iterator it = M_our_players.begin();
          it != M_our_players.end();
          ++it )
    {
        *it = round_coordinates( it->x, it->y );
    }

    M_our_dragged.assign( 11, 0 );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::updateTriangulation()
{
    if ( ! M_samples )
    {
        return;
    }

    const Rect2D pitch( Vector2D( - FEditConfig::PITCH_LENGTH * 0.5,
                                  - FEditConfig::PITCH_WIDTH * 0.5 ),
                        Size2D( FEditConfig::PITCH_LENGTH,
                                FEditConfig::PITCH_WIDTH ) );
    M_triangulation.init( pitch );

    const SampleDataSet::DataCont::const_iterator end = M_samples->dataCont().end();
    for ( SampleDataSet::DataCont::const_iterator it = M_samples->dataCont().begin();
          it != end;
          ++it )
    {
        M_triangulation.addVertex( it->ball_ );
    }

    M_triangulation.compute();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::reverseY()
{
    if ( ! M_formation )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " ***ERROR*** No formation!!"
                  << std::endl;
        return;
    }

    if ( ! M_samples )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " ***ERROR*** No formation data!!"
                  << std::endl;
        return;
    }

    M_ball.y *= -1.0;

    reverseY( M_our_players );

    // TODO: right side players

    //
    // update index
    //
    if ( M_data_auto_select )
    {
        int index = 0;
        const SampleDataSet::DataCont::const_iterator end = M_samples->dataCont().end();
        for ( SampleDataSet::DataCont::const_iterator it = M_samples->dataCont().begin();
              it != end;
              ++it, ++index )
        {
            if ( it->ball_.dist2( M_ball ) < 1.0*1.0 )
            {
                M_ball = it->ball_;
                M_current_index = index;
                break;
            }
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::reverseY( std::vector< Vector2D > & our_players )
{
    if ( ! M_formation )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " ***ERROR*** No formation!!"
                  << std::endl;
        return;
    }

    std::vector< Vector2D > old_positions = our_players;

    int unum = 1;
    for ( std::vector< Vector2D >::iterator it = our_players.begin();
          it != our_players.end();
          ++it )
    {
        if ( M_formation->isCenterType( unum ) )
        {
            it->y *= -1.0;
        }
        else if ( M_formation->isSymmetryType( unum ) )
        {
            int symmetry_unum = M_formation->getSymmetryNumber( unum );
            if ( symmetry_unum == 0 ) continue;
            it->x = old_positions[symmetry_unum - 1].x;
            it->y = old_positions[symmetry_unum - 1].y * -1.0;
        }
        else if ( M_formation->isSideType( unum ) )
        {
            it->y *= -1.0;
            for ( int iunum = 1; iunum <= 11; ++iunum )
            {
                if ( M_formation->getSymmetryNumber( iunum ) == unum )
                {
                    it->x = old_positions[iunum - 1].x;
                    it->y = old_positions[iunum - 1].y * -1.0;
                }
            }
        }
        ++unum;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::moveBallTo( const double & x,
                       const double & y )
{
    Vector2D pos = round_coordinates( x, y );

    M_ball = pos;
    if ( pos.absY() < 1.0 )
    {
        M_ball.y = 0.0;
    }

    if ( M_data_auto_select
         && M_samples )
    {
        SampleDataSet::IndexData d = M_samples->nearestData( pos, 1.0 );
        if ( d.second )
        {
            M_current_index = d.first;
            M_ball = d.second->ball_;
        }
    }

    updatePlayerPosition();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::selectObject( const double & x,
                         const double & y )
{
    const Vector2D pos( x, y );
    const double dist2_thr = 1.5 * 1.5;

    SelectType old_selection = M_select_type;

    double mindist2 = 200.0 * 200.0;

    if ( M_ball_draggable )
    {
        double d2 = M_ball.dist2( pos );
        if ( d2 < dist2_thr )
        {
            //std::cerr << "selection update ball" << std::endl;
            M_select_type = SELECT_BALL;
            mindist2 = d2;
        }
    }
    //mindist2 = M_ball.dist2( pos );

#if 0
    // check oppnent
    for ( std::vector< Vector2D >::iterator it = M_opp_players.begin();
          it != M_opp_players.end();
          ++it )
    {
        double d2 = it->dist2( pos );
        if ( d2 < dist2_thr
             && d2 < mindist2 )
        {
            M_select_type = SELECT_OPP;
            M_select_index = std::distance( M_opp_players.begin(), it );
            mindist2 = d2;
        }
    }
#endif

    // check our players
    int unum = 1;
    for ( std::vector< Vector2D >::iterator it = M_our_players.begin();
          it != M_our_players.end();
          ++it, ++unum )
    {
        double d2 = it->dist2( pos );
        if ( d2 < dist2_thr
             && d2 < mindist2 )
        {
            M_select_type = SELECT_OUR;
            M_select_index = std::distance( M_our_players.begin(), it );
            mindist2 = d2;
        }
    }

    if ( old_selection != M_select_type )
    {
        if ( M_select_type == SELECT_BALL )
        {
            M_ball = pos;
        }
        else if ( M_select_type == SELECT_OUR )
        {
            M_our_players[M_select_index] = pos;
        }
        else if ( M_select_type == SELECT_OPP )
        {
            M_opp_players[M_select_index] = pos;
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::releaseObject( const double &,
                          const double & )
{
    M_select_type = NO_SELECT;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::setObjectDragPoint( double x,
                               double y )
{
    if ( M_select_type == SELECT_BALL )
    {
        moveBallTo( x, y );
        return;
    }

    const Vector2D pos = round_coordinates( x, y );

    switch ( M_select_type ) {
    case SELECT_OUR:
        M_our_dragged[M_select_index] = 1;
        M_our_players[M_select_index] = pos;

        if ( M_symmetry_mode
             && M_ball.absY() < 0.1
             && M_formation )
        {
            if ( M_formation->isSymmetryType( M_select_index + 1 ) )
            {
                int symmetry_unum = M_formation->getSymmetryNumber( M_select_index + 1 );
                if ( symmetry_unum != 0 )
                {
                    M_our_players[symmetry_unum-1].x = pos.x;
                    M_our_players[symmetry_unum-1].y = -pos.y;
                }
            }
            else if ( M_formation->isSideType( M_select_index + 1 ) )
            {
                for ( int unum = 1; unum <= 11; ++unum )
                {
                    if ( M_formation->getSymmetryNumber( unum )
                         == static_cast< int >( M_select_index + 1 ) )
                    {
                        M_our_players[unum-1].x = pos.x;
                        M_our_players[unum-1].y = -pos.y;
                    }
                }
            }
        }

        break;
    case SELECT_OPP:
        M_opp_players[M_select_index] = pos;
        break;
    default:
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::setTrainingTypeName( const std::string & type_name )
{
    M_training_type_name = type_name;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::updateRoleData( const int unum,
                           const int symmetry_unum,
                           const std::string & role_name )
{
    if ( ! M_formation )
    {
        return;
    }

    if ( M_formation->updateRole( unum, symmetry_unum, role_name ) )
    {
        M_modified = true;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::setBallPosition( const double & x,
                            const double & y )
{
    M_ball = round_coordinates( x, y );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::setPlayerPosition( const int unum,
                              const double & x,
                              const double & y )
{
    M_our_players[unum - 1]  = round_coordinates( x, y );
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
FEditData::focusData( const int idx )
{
    if ( idx < 0 )
    {
        return false;
    }

    if ( ! M_samples )
    {
        return false;
    }

    const SampleData * d = M_samples->data( size_t( idx ) );
    if ( d )
    {
        M_ball = d->ball_;
        for ( std::size_t i = 0; i < 11; ++i )
        {
            M_our_players[i] = d->players_[i];
        }

        M_current_index = idx;
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
std::string
FEditData::recordData()
{
    if ( ! M_formation )
    {
        return std::string( "No formation." );
    }

    if ( ! M_samples )
    {
        return std::string( "No formation data." );
    }

    size_t old_size = M_samples->dataCont().size();
    SampleDataSet::ErrorType err
        = M_samples->addData( *M_formation,
                              SampleData( M_ball, M_our_players ),
                              M_symmetry_mode );
    if ( err != SampleDataSet::NO_ERROR
         && old_size == M_samples->dataCont().size() )
    {
        std::cerr << "recordData() error occured. error_code=" << err
                  << std::endl;
        return std::string( "error record data." );
    }


    M_ball = M_samples->dataCont().back().ball_;
    M_our_players = M_samples->dataCont().back().players_;

    updateTriangulation();

    if ( M_formation->methodName() == FormationDT::name()
         || M_formation->methodName() == FormationKNN::name() )
    {
        train();
    }

    return std::string();
}

/*-------------------------------------------------------------------*/
/*!

 */
std::string
FEditData::insertData( const int idx )
{
    if ( ! M_formation )
    {
        return std::string( "No formation." );
    }

    if ( ! M_samples )
    {
        return std::string( "No formation data." );
    }

    if ( idx < 0 )
    {
        return std::string( "Illegal index value." );
    }

    SampleData data( M_ball, M_our_players );

    size_t old_size = M_samples->dataCont().size();
    SampleDataSet::ErrorType err
        = M_samples->insertData( *M_formation,
                                 static_cast< size_t >( idx ),
                                 data,
                                 M_symmetry_mode );
    if ( err != SampleDataSet::NO_ERROR
         && old_size == M_samples->dataCont().size() )
    {
        std::cerr << "insertData() error occured. error_code=" << err
                  << std::endl;
        return std::string( "error insert data." );
    }

    M_ball = data.ball_;
    M_our_players = data.players_;

    updateTriangulation();

    if ( M_formation->methodName() == FormationDT::name()
         || M_formation->methodName() == FormationKNN::name() )
    {
        train();
    }

    return std::string();
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
FEditData::replaceData( const int idx )
{
    if ( ! M_formation )
    {
        std::cerr << "replaceData() no formation." << std::endl;
        return false;
    }

    if ( ! M_samples )
    {
        std::cerr << "replaceData() no formation data." << std::endl;
        return false;
    }

    if ( idx < 0
         || static_cast< int >( M_samples->dataCont().size() ) <= idx )
    {
        std::cerr << "replaceData() illegal index range. "
                  << idx << std::endl;
        return false;
    }

    SampleData data( M_ball, M_our_players );

    SampleDataSet::ErrorType err
        = M_samples->replaceData( *M_formation,
                                  static_cast< size_t >( idx ),
                                  data,
                                  M_symmetry_mode );


    if ( err != SampleDataSet::NO_ERROR )
    {
        std::cerr << "replaceData() error occured. error_code=" << err
                  << std::endl;
        return false;
    }


    M_ball = data.ball_;
    M_our_players = data.players_;

    updateTriangulation();

    if ( M_formation->methodName() == FormationDT::name()
         || M_formation->methodName() == FormationKNN::name() )
    {
        train();
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
FEditData::deleteData( const int idx )
{

    if ( ! M_samples )
    {
        std::cerr << "deleteData() no formation data." << std::endl;
        return false;
    }

    if ( idx < 0
         || static_cast< int >( M_samples->dataCont().size() ) <= idx )
    {
        std::cerr << "deleteData() illegal index range. "
                  << idx << std::endl;
        return false;
    }

    SampleDataSet::ErrorType err = M_samples->removeData( static_cast< size_t >( idx ) );

    if ( err != SampleDataSet::NO_ERROR )
    {
        std::cerr << "deleteData() error occured. error_code=" << err
                  << std::endl;
        return false;
    }

    M_current_index = -1;
    updateTriangulation();

    if ( M_formation
         && ( M_formation->methodName() == FormationDT::name()
              || M_formation->methodName() == FormationKNN::name() )
         )
    {
        train();
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FEditData::train()
{
    if ( ! M_formation )
    {
        return;
    }

    M_formation->train();

    M_modified = true;

    updatePlayerPosition();
}
