// -*-c++-*-

/*!
  \file pass_generator.cpp
  \brief pass course generator Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "pass_generator.h"

#include "field_analyzer.h"

#include <rcsc/action/kick_table.h>
#include <rcsc/player/world_model.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/player_type.h>
#include <rcsc/timer.h>

#include <cmath>

#define DEBUG_PROFILE
// #define DEBUG_UPDATE_PASSER
// #define DEBUG_CREATE_RECEIVERS
// #define DEBUG_CREATE_PASS_COURSES
// #define DEBUG_PREDICT_RECEIVER_REACH_STEP
// #define DEBUG_PREDICT_OPPONENT_REACH_STEP

// #define DEBUG_PRINT_SUCCESS_PASS
// #define DEBUG_PRINT_FAILED_PASS

using namespace rcsc;

/*-------------------------------------------------------------------*/
/*!

 */
PassGenerator::PassGenerator()
{
    M_receiver_candidates.reserve( 11 );
    M_courses.reserve( 1024 );

    clear();
}

/*-------------------------------------------------------------------*/
/*!

 */
PassGenerator &
PassGenerator::instance()
{
    static PassGenerator s_instance;
    return s_instance;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PassGenerator::clear()
{
    M_total_count = 0;
    M_passer = static_cast< AbstractPlayerObject * >( 0 );
    M_start_time.assign( -1, 0 );
    M_first_point.invalidate();
    M_receiver_candidates.clear();
    M_courses.clear();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PassGenerator::update( const WorldModel & wm )
{
    static GameTime s_update_time( -1, 0 );
    if ( s_update_time == wm.time() )
    {
        return;
    }
    s_update_time = wm.time();

    clear();

    if ( wm.time().stopped() > 0
         || wm.gameMode().isPenaltyKickMode() )
    {
        return;
    }

#ifdef DEBUG_PROFILE
    Timer timer;
#endif

    updatePasser( wm );

    if ( ! M_passer
         || ! M_first_point.isValid() )
    {
        dlog.addText( Logger::PASS,
                      __FILE__" (update) passer not found." );
        return;
    }

    createReceivers( wm );

    if ( M_receiver_candidates.empty() )
    {
        dlog.addText( Logger::PASS,
                      __FILE__" (update) no receiver." );
        return;
    }

    createCourses( wm );

#ifdef DEBUG_PROFILE
    dlog.addText( Logger::PASS,
                  __FILE__" (update) course_size=%d/%d elapsed %f [ms]",
                  (int)M_courses.size(),
                  M_total_count,
                  timer.elapsedReal() );
    //     std::cerr << "PassGenerator: total_count=" << M_total_count
    //               << " success_count=" << M_courses.size()
    //               << std::endl;
#endif
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PassGenerator::updatePasser( const WorldModel & wm )
{
    if ( wm.self().isKickable() )
    {
        M_passer = &wm.self();
        M_start_time = wm.time();
        M_first_point = wm.ball().pos();
#ifdef DEBUG_UPDATE_PASSER
        dlog.addText( Logger::PASS,
                      __FILE__" (updatePasser) self kickable." );
#endif
        return;
    }

#if 0
    int s_min = wm.interceptTable()->selfReachCycle();
    int t_min = wm.interceptTable()->teammateReachCycle();
    int o_min = wm.interceptTable()->opponentReachCycle();

    int our_min = std::min( s_min, t_min );
    if ( o_min < std::min( our_min - 4, (int)rint( our_min * 0.9 ) ) )
    {
        dlog.addText( Logger::PASS,
                      __FILE__" (updatePasser) opponent ball." );
        return;
    }

    if ( s_min <= t_min )
    {
        M_passer = &wm.self();
    }
    else
    {
        M_passer = wm.interceptTable()->fastestTeammate();
    }

    if ( ! M_passer )
    {
        dlog.addText( Logger::PASS,
                      __FILE__" (updatePasser) NULL passer teammate." );
        return;
    }

    M_start_time = wm.time();
    if ( ! wm.gameMode().isServerCycleStoppedMode() )
    {
        M_start_time.addCycle( t_min );
    }

    M_first_point = wm.ball().inertiaPoint( t_min );

#ifdef DEBUG_UPDATE_PASSER
    dlog.addText( Logger::PASS,
                  __FILE__" (updatePasser) passer=%d(%.1f %.1f) reachStep=%d startPos=(%.1f %.1f)",
                  M_passer->unum(),
                  M_passer->pos().x, M_passer->pos().y,
                  t_min,
                  M_first_point.x, M_first_point.y );
#endif
#endif
}

/*-------------------------------------------------------------------*/
/*!

 */
struct ReceiverAngleCompare {

    bool operator()( const PassGenerator::Receiver & lhs,
                     const PassGenerator::Receiver & rhs ) const
      {
          return lhs.angle_from_ball_.degree() < rhs.angle_from_ball_.degree();
      }
};

/*-------------------------------------------------------------------*/
/*!

 */
void
PassGenerator::createReceivers( const WorldModel & wm )
{
    const double max_dist2 = std::pow( 40.0, 2 ); // Magic Number

    const AbstractPlayerCont::const_iterator end = wm.allTeammates().end();
    for ( AbstractPlayerCont::const_iterator p = wm.allTeammates().begin();
          p != end;
          ++p )
    {
        if ( *p == M_passer ) continue;

        if ( M_passer->unum() == wm.self().unum() )
        {
            if ( (*p)->isGhost() ) continue;
            if ( (*p)->unumCount() > 10 ) continue;
            if ( (*p)->unum() == Unum_Unknown ) continue;
            if ( (*p)->posCount() > 10 ) continue;
            if ( (*p)->pos().x > wm.offsideLineX() ) continue;
            //if ( (*p)->isTackling() ) continue;
            if ( (*p)->goalie()
                 && (*p)->pos().x < ServerParam::i().ourPenaltyAreaLineX() + 15.0 )
            {
                continue;
            }
        }

        if ( (*p)->pos().dist2( M_first_point ) > max_dist2 ) continue;

        M_receiver_candidates.push_back( Receiver( *p, M_first_point ) );
    }

    std::sort( M_receiver_candidates.begin(),
               M_receiver_candidates.end(),
               ReceiverAngleCompare() );

#ifdef DEBUG_CREATE_RECEIVERS
    for ( ReceiverCont::const_iterator p = M_receiver_candidates.begin();
          p != M_receiver_candidates.end();
          ++p )
    {
        dlog.addText( Logger::PASS,
                      __FILE__" (createReceivers) receiver=%d angle=%.1f",
                      p->player_->unum(),
                      p->angle_from_ball_.degree() );
    }
#endif
}

/*-------------------------------------------------------------------*/
/*!

 */
AngleDeg
PassGenerator::createReceiversFor( const rcsc::AngleDeg & angle,
                                   AbstractPlayerCont * candidates )
{
    candidates->clear();

    //
    // create receiver candidates for 'angle'
    // and update target angle if direct pass candidate exists
    //

    double min_angle_diff = 180.0;
    AngleDeg min_angle = angle;
    for ( ReceiverCont::iterator p = M_receiver_candidates.begin();
          p != M_receiver_candidates.end();
          ++p )
    {
        double diff = ( angle - p->angle_from_ball_ ).abs();
        if ( diff < 31.0 ) // Magic Number
        {
            candidates->push_back( p->player_ );

            if ( diff < min_angle_diff )
            {
                min_angle_diff = diff;
                min_angle = p->angle_from_ball_;
            }
        }
    }

    if ( min_angle_diff < 1.0 )
    {
        return min_angle;
    }

    return angle;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PassGenerator::createCourses( const WorldModel & wm )
{
    AngleDeg angle = -180.0;
    for ( int i = 0; i < 60; ++i, angle += 6.0 ) // Magic Number
    {
        createCourses( wm, angle );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PassGenerator::createCourses( const rcsc::WorldModel & wm,
                              const rcsc::AngleDeg & angle )
{
    //
    // create receiver candidates for 'angle'
    // update pass angle if a direct pass candidate exists.
    //

    static AbstractPlayerCont receivers;

    const AngleDeg pass_angle = createReceiversFor( angle, &receivers );;

    if ( receivers.empty() )
    {
#ifdef DEBUG_CREATE_PASS_COURSES
        dlog.addText( Logger::PASS,
                      __FILE__" (createCourses) angle=%.1f no receiver candidate",
                      pass_angle.degree() );
#endif
        return;
    }

#ifdef DEBUG_CREATE_PASS_COURSES
    dlog.addText( Logger::PASS,
                  __FILE__" (createCourses) angle=%.1f receiverCandidates=%d",
                  pass_angle.degree(),
                  (int)receivers.size() );
#endif

    //
    // create patterns of ball first speed
    //

    const ServerParam & SP = ServerParam::i();
    const double max_first_speed = ( wm.gameMode().type() == GameMode::PlayOn
                                     ? SP.ballSpeedMax()
                                     : wm.self().isKickable()
                                     ? wm.self().kickRate() * SP.maxPower()
                                     : SP.kickPowerRate() * SP.maxPower() );
    const double min_first_speed = ServerParam::i().defaultPlayerSpeedMax();
    const double max_last_speed = SP.ballSpeedMax() * std::pow( SP.ballDecay(), 5 );
    const Vector2D our_goal( -SP.pitchHalfLength(), 0.0 );

    for ( double first_speed = max_first_speed;
          first_speed > min_first_speed;
          first_speed -= 0.3 )
    {
        ++M_total_count;

        const Vector2D first_vel = Vector2D::from_polar( first_speed, pass_angle );

        if ( pass_angle.abs() > 90.0 )
        {
            Vector2D final_pos = inertia_final_point( M_first_point, first_vel, SP.ballDecay() );
            if ( final_pos.x < -SP.pitchHalfLength() + 10.0
                 || final_pos.dist2( our_goal ) < 20.0*20.0 ) // Magic Number
            {
                continue;
            }
        }

        int our_reach_step = 1000;
        const AbstractPlayerObject * receiver = static_cast< const AbstractPlayerObject * >( 0 );

        for ( AbstractPlayerCont::const_iterator p = receivers.begin();
              p != receivers.end();
              ++p )
        {
            int step = predictReceiverReachStep( *p, M_first_point, first_vel );

            if ( step >= 0
                 && step < our_reach_step )
            {
                our_reach_step = step;
                receiver = *p;
            }
        }

        if ( ! receiver )
        {
            continue;
        }

        //
        // TODO: change the max_last_speed depending on the receiver's state or pass type
        //
        if ( first_speed * std::pow( SP.ballDecay(), our_reach_step ) > max_last_speed )
        {
            // over the max speed
#ifdef DEBUG_PRINT_FAILED_PASS
            dlog.addText( Logger::PASS,
                          "XXX (createCourses) %d bpos=(%.1f %.1f) pass_angle=%.1f Fspeed=%.2f Lspeed=%.2f receiver=%d step=%d over_speed",
                          M_total_count,
                          M_first_point.x, M_first_point.y,
                          pass_angle.degree(),
                          first_speed,
                          first_speed * std::pow( SP.ballDecay(), our_reach_step ),
                          receiver->unum(),
                          our_reach_step );
#endif
            continue;
        }

        int opp_reach_step = predictOpponentReachStep( wm, M_first_point, first_vel, our_reach_step );

        //
        // TODO: change the condtion for through pass
        //

        Vector2D receive_point = inertia_n_step_point( M_first_point,
                                                       first_vel,
                                                       our_reach_step,
                                                       SP.ballDecay() );
        if ( opp_reach_step < 0
             || opp_reach_step > our_reach_step )
        {
            Pass::Ptr pass( new Pass( M_start_time,
                                      M_passer->unum(),
                                      receiver->unum(),
                                      M_first_point,
                                      receive_point,
                                      first_speed ) );
            M_courses.push_back( pass );
#ifdef DEBUG_PRINT_SUCCESS_PASS
            dlog.addText( Logger::PASS,
                          "OOO (createCourses) %d bpos=(%.1f %.1f) pass_angle=%.1f Fspeed=%.2f Lspeed=%.2f receiver=%d step=%d opp=%d",
                          M_total_count,
                          M_first_point.x, M_first_point.y,
                          pass_angle.degree(),
                          first_speed,
                          first_speed * std::pow( SP.ballDecay(), our_reach_step ),
                          receiver->unum(),
                          our_reach_step,
                          opp_reach_step );
            dlog.addRect( Logger::PASS,
                          receive_point.x - 0.1, receive_point.y - 0.1,
                          0.2, 0.2,
                          "#ff0000" );
            char num[8];
            snprintf( num, 8, "%d", M_total_count );
            dlog.addMessage( Logger::PASS,
                             receive_point, num );
#endif
        }
#ifdef DEBUG_PRINT_FAILED_PASS
        else
        {
            // opponent can intercept the pass for this angle.
            dlog.addText( Logger::PASS,
                          "XXX (createCourses) %d bpos=(%.1f %.1f) pass_angle=%.1f Fspeed=%.2f Lspeed=%.2f receiver=%d step=%d opp=%d",
                          M_total_count,
                          M_first_point.x, M_first_point.y,
                          pass_angle.degree(),
                          first_speed,
                          first_speed * std::pow( SP.ballDecay(), our_reach_step ),
                          receiver->unum(),
                          our_reach_step,
                          opp_reach_step );
            dlog.addRect( Logger::PASS,
                          receive_point.x - 0.1, receive_point.y - 0.1,
                          0.2, 0.2,
                          "#00ff00" );
            char num[8];
            snprintf( num, 8, "%d", M_total_count );
            dlog.addMessage( Logger::PASS,
                             receive_point, num );
        }
#endif
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
int
PassGenerator::predictReceiverReachStep( const AbstractPlayerObject * receiver,
                                         const Vector2D & first_ball_pos,
                                         const Vector2D & first_ball_vel )
{
    const ServerParam & SP = ServerParam::i();
    const PlayerType * ptype = receiver->playerTypePtr();

    const double max_x = SP.pitchHalfLength() - 1.0;
    const double max_y = SP.pitchHalfWidth() - 1.0;

    const Vector2D & player_pos = ( receiver->seenPosCount() <= receiver->posCount()
                                    ? receiver->seenPos()
                                    : receiver->pos() );
    const Vector2D & player_vel = ( receiver->seenVelCount() <= receiver->velCount()
                                    ? receiver->seenVel()
                                    : receiver->vel() );
    const double player_speed = player_vel.r();

    const int min_cycle = FieldAnalyzer::estimate_min_reach_cycle( player_pos,
                                                                   ptype->realSpeedMax(),
                                                                   first_ball_pos,
                                                                   first_ball_vel.th() );

    if ( min_cycle < 0 )
    {
        return -1;
    }

    const double penalty_distance = FieldAnalyzer::estimate_virtual_dash_distance( receiver );

    for ( int cycle = min_cycle; cycle < 30; ++cycle )
    {
        Vector2D ball_pos = inertia_n_step_point( first_ball_pos,
                                                  first_ball_vel,
                                                  cycle,
                                                  SP.ballDecay() );
        if ( ball_pos.absX() > max_x
             || ball_pos.absY() > max_y )
        {
            break;
        }

        Vector2D inertia_pos = inertia_n_step_point( player_pos,
                                                     player_vel,
                                                     cycle,
                                                     ptype->playerDecay() );
        double target_dist = inertia_pos.dist( ball_pos );

        double dash_dist = target_dist - ptype->kickableArea() * 0.5;
        dash_dist += penalty_distance; // add penalty distance

        if ( dash_dist < 0.001 )
        {
#ifdef DEBUG_PREDICT_RECEIVER_REACH_STEP
            dlog.addText( Logger::PASS,
                          "____ receiver=%d(%.2f %.2f) step=%d already there. dist=%.1f penalty=%.1f",
                          receiver->unum(), receiver->pos().x, receiver->pos().y,
                          cycle,
                          target_dist, penalty_distance );
#endif
            return cycle;
        }

        if ( dash_dist > ptype->realSpeedMax() * cycle )
        {
            continue;
        }

        //
        // dash
        //

        // 1 step penalty because players can start to dash after observing the ball moving.
        int n_dash = ptype->cyclesToReachDistance( dash_dist ) + 1;
        if ( receiver->isTackling() )
        {
            n_dash += 5;
        }

        if ( n_dash > cycle )
        {
            continue;
        }

        //
        // turn
        // TODO: check intentional moving (role type or arm directin??)
        //
        int n_turn = ( receiver->bodyCount() > 1
                       ? 2
                       : FieldAnalyzer::predict_player_turn_cycle( ptype,
                                                                   receiver->body(),
                                                                   player_speed,
                                                                   target_dist,
                                                                   ( ball_pos - inertia_pos ).th(),
                                                                   ptype->kickableArea() * 0.9 ) );

        if ( n_turn + n_dash <= cycle )
        {

#ifdef DEBUG_PREDICT_RECEIVER_REACH_STEP
            dlog.addText( Logger::PASS,
                          "____ receiver=%d(%.2f %.2f) step=%d(t:%d,d:%d) penalty=%.1f",
                          receiver->unum(), receiver->pos().x, receiver->pos().y,
                          cycle, n_turn, n_dash,
                          penalty_distance );
#endif
            return cycle;
        }
    }

    return -1;
}

/*-------------------------------------------------------------------*/
/*!

 */
int
PassGenerator::predictOpponentReachStep( const rcsc::WorldModel & wm,
                                         const rcsc::Vector2D & first_ball_pos,
                                         const rcsc::Vector2D & first_ball_vel,
                                         const int max_cycle )
{
    const AngleDeg ball_move_angle = first_ball_vel.th();

    int min_step = 1000;
    const PlayerObject * fastest_opponent = static_cast< PlayerObject * >( 0 );

    const PlayerPtrCont::const_iterator o_end = wm.opponentsFromBall().end();
    for ( PlayerPtrCont::const_iterator o = wm.opponentsFromBall().begin();
          o != o_end;
          ++o )
    {
        int step = predictOpponentReachStep( **o,
                                             first_ball_pos,
                                             first_ball_vel,
                                             ball_move_angle,
                                             max_cycle );
        if ( step >= 0
             && step < min_step )
        {
            min_step = step;
            fastest_opponent = *o;
        }
    }

    if ( min_step == 1000 )
    {
        return -1;
    }

#if 0
    //#ifdef DEBUG_PREDICT_OPPONENT_REACH_STEP
    dlog.addText( Logger::PASS,
                  "______ opponent=%d(%.1f %.1f) step=%d",
                  fastest_opponent->unum(),
                  fastest_opponent->pos().x, fastest_opponent->pos().y,
                  min_step );
#endif

    return min_step;
}

/*-------------------------------------------------------------------*/
/*!

 */
int
PassGenerator::predictOpponentReachStep( const PlayerObject & opponent,
                                         const Vector2D & first_ball_pos,
                                         const Vector2D & first_ball_vel,
                                         const AngleDeg & ball_move_angle,
                                         const int max_cycle )
{
    static const Rect2D penalty_area( Vector2D( ServerParam::i().theirPenaltyAreaLineX(),
                                                -ServerParam::i().penaltyAreaHalfWidth() ),
                                      Size2D( ServerParam::i().penaltyAreaLength(),
                                              ServerParam::i().penaltyAreaWidth() ) );

    const ServerParam & SP = ServerParam::i();

    const Vector2D next_ball_pos = first_ball_pos + first_ball_vel;

    const Vector2D & player_pos = ( opponent.seenPosCount() <= opponent.posCount()
                                    ? opponent.seenPos()
                                    : opponent.pos() );
    const Vector2D & player_vel = ( opponent.seenVelCount() <= opponent.velCount()
                                    ? opponent.seenVel()
                                    : opponent.vel() );
    const double player_speed = player_vel.r();

    const PlayerType * ptype = opponent.playerTypePtr();

    const int min_cycle = FieldAnalyzer::estimate_min_reach_cycle( player_pos,
                                                                   ptype->realSpeedMax(),
                                                                   first_ball_pos,
                                                                   ball_move_angle );
    if ( min_cycle < 0 )
    {
#ifdef DEBUG_PREDICT_OPPONENT_REACH_STEP
        dlog.addText( Logger::PASS,
                      "____ opponent=%d(%.1f %.1f) never reach(1)",
                      opponent.unum(),
                      opponent.pos().x, opponent.pos().y );
#endif
        return -1;
    }

    double control_area = ( opponent.goalie() && penalty_area.contains( next_ball_pos )
                            ? SP.catchableArea()
                            : ptype->kickableArea() );

    const double bonus_distance = FieldAnalyzer::estimate_virtual_dash_distance( &opponent );

    for ( int cycle = min_cycle; cycle <= max_cycle + 5; ++cycle )
    {
        Vector2D ball_pos = inertia_n_step_point( first_ball_pos,
                                                  first_ball_vel,
                                                  cycle,
                                                  SP.ballDecay() );
        Vector2D inertia_pos = ptype->inertiaPoint( player_pos, player_vel, cycle );
        double target_dist = inertia_pos.dist( ball_pos );

        control_area = ( opponent.goalie() && penalty_area.contains( ball_pos )
                         ? SP.catchableArea()
                         : ptype->kickableArea() + 0.1 );

        double dash_dist =  target_dist - control_area;
        dash_dist -= bonus_distance; // subtract bonus distance

        if ( dash_dist < 0.001 )
        {
#ifdef DEBUG_PREDICT_OPPONENT_REACH_STEP
            dlog.addText( Logger::PASS,
                          "____ opponent=%d(%.1f %.1f) step=%d already there. dist=%.1f bonus=%.1f",
                          opponent.unum(),
                          opponent.pos().x, opponent.pos().y,
                          cycle,
                          target_dist, bonus_distance );
#endif
            return cycle;
        }

        if ( dash_dist > ptype->realSpeedMax() * cycle )
        {
            //             dlog.addText( Logger::PASS,
            //                           "______ opponent=%d(%.1f %.1f) cycle=%d dash_dist=%.1f reachable=%.1f",
            //                           opponent.unum(),
            //                           opponent.pos().x, opponent.pos().y,
            //                           cycle, dash_dist, ptype->realSpeedMax()*cycle );
            continue;
        }

        //
        // dash
        //

        // 1 step penalty because players can start to dash after observing the ball moving.
        int n_dash = ptype->cyclesToReachDistance( dash_dist ) + 1;
        if ( opponent.isTackling() )
        {
            n_dash += 2;
        }

        if ( n_dash > cycle )
        {
            //             dlog.addText( Logger::PASS,
            //                           "______ opponent=%d(%.1f %.1f) cycle=%d dash_dist=%.1f n_dash=%d",
            //                           opponent.unum(),
            //                           opponent.pos().x, opponent.pos().y,
            //                           cycle, dash_dist, n_dash );
            continue;
        }

        //
        // turn
        //
        int n_turn = ( opponent.bodyCount() > 1
                       ? 0
                       : FieldAnalyzer::predict_player_turn_cycle( ptype,
                                                                   opponent.body(),
                                                                   player_speed,
                                                                   target_dist,
                                                                   ( ball_pos - inertia_pos ).th(),
                                                                   control_area ) );

        if ( n_turn + n_dash <= cycle )
        {
#ifdef DEBUG_PREDICT_OPPONENT_REACH_STEP
            dlog.addText( Logger::PASS,
                          "____ opponent=%d(%.1f %.1f) step=%d(t:%d,d:%d) bonus=%.1f",
                          opponent.unum(),
                          opponent.pos().x, opponent.pos().y,
                          cycle, n_turn, n_dash,
                          bonus_distance );
#endif
            return cycle;
        }
    }

#ifdef DEBUG_PREDICT_OPPONENT_REACH_STEP
    dlog.addText( Logger::PASS,
                  "____ opponent=%d(%.1f %.1f) never reach(2)",
                  opponent.unum(),
                  opponent.pos().x, opponent.pos().y );
#endif

    return -1;
}

/*-------------------------------------------------------------------*/
/*!

 */
double
PassGenerator::evaluatePassCourse( const WorldModel &,
                                   const Pass & )
{
    return 0.0;
}
