// -*-c++-*-

/*!
  \file field_analyzer.cpp
  \brief miscellaneous field analysis Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA, Hiroki SHIMORA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "field_analyzer.h"

#include <rcsc/player/world_model.h>
#include <rcsc/player/player_predicate.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/player_type.h>
#include <rcsc/common/logger.h>
#include <rcsc/timer.h>

//#define DEBUG_PRINT

using namespace rcsc;

/*-------------------------------------------------------------------*/
/*!

 */
FieldAnalyzer::FieldAnalyzer()
{

}

/*-------------------------------------------------------------------*/
/*!

 */
FieldAnalyzer &
FieldAnalyzer::instance()
{
    static FieldAnalyzer s_instance;
    return s_instance;
}

/*-------------------------------------------------------------------*/
/*!

 */
int
FieldAnalyzer::predict_self_reach_cycle( const WorldModel & wm,
                                         const Vector2D & target_point,
                                         const double & dist_thr,
                                         const int wait_cycle,
                                         const bool save_recovery,
                                         StaminaModel * stamina )
{
    if ( wm.self().inertiaPoint( wait_cycle ).dist2( target_point ) < std::pow( dist_thr, 2 ) )
    {
        return 0;
    }

    const ServerParam & SP = ServerParam::i();
    const PlayerType & ptype = wm.self().playerType();
    const double recover_dec_thr = SP.recoverDecThrValue();

    const double first_speed = wm.self().vel().r() * std::pow( ptype.playerDecay(), wait_cycle );

    StaminaModel first_stamina_model = wm.self().staminaModel();
    if ( wait_cycle > 0 )
    {
        first_stamina_model.simulateWaits( ptype, wait_cycle );
    }

    for ( int cycle = std::max( 0, wait_cycle ); cycle < 30; ++cycle )
    {
        const Vector2D inertia_pos = wm.self().inertiaPoint( cycle );
        const double target_dist = inertia_pos.dist( target_point );

        if ( target_dist < dist_thr )
        {
            return cycle;
        }

        double dash_dist = target_dist - dist_thr * 0.5;

        if ( dash_dist > ptype.realSpeedMax() * ( cycle - wait_cycle ) )
        {
            continue;
        }

        AngleDeg target_angle = ( target_point - inertia_pos ).th();

        //
        // turn
        //

        int n_turn = predict_player_turn_cycle( &ptype,
                                                wm.self().body(),
                                                first_speed,
                                                target_dist,
                                                target_angle,
                                                dist_thr );
        if ( wait_cycle + n_turn >= cycle )
        {
            continue;
        }

        StaminaModel stamina_model = first_stamina_model;
        if ( n_turn > 0 )
        {
            stamina_model.simulateWaits( ptype, n_turn );
        }

        //
        // dash
        //

        int n_dash = ptype.cyclesToReachDistance( dash_dist );
        if ( wait_cycle + n_turn + n_dash > cycle )
        {
            continue;
        }

        double speed = first_speed * std::pow( ptype.playerDecay(), n_turn );
        double reach_dist = 0.0;

        n_dash = 0;
        while ( wait_cycle + n_turn + n_dash < cycle )
        {
            double dash_power = std::min( SP.maxDashPower(), stamina_model.stamina() );
            if ( save_recovery
                 && stamina_model.stamina() - dash_power < recover_dec_thr )
            {
                dash_power = std::max( 0.0, stamina_model.stamina() - recover_dec_thr );
                if ( dash_power < 1.0 )
                {
                    break;
                }
            }

            double accel = dash_power * ptype.dashPowerRate() * stamina_model.effort();
            speed += accel;
            if ( speed > ptype.playerSpeedMax() )
            {
                speed = ptype.playerSpeedMax();
            }

            reach_dist += speed;
            speed *= ptype.playerDecay();

            stamina_model.simulateDash( ptype, dash_power );

            ++n_dash;

            if ( reach_dist >= dash_dist )
            {
                break;
            }
        }

        if ( reach_dist >= dash_dist )
        {
            if ( stamina )
            {
                *stamina = stamina_model;
            }
            return wait_cycle + n_turn + n_dash;
        }
    }

    return 1000;
}

/*-------------------------------------------------------------------*/
/*!

 */
int
FieldAnalyzer::predict_player_reach_cycle( const AbstractPlayerObject * player,
                                           const Vector2D & target_point,
                                           const double & dist_thr,
                                           const double & penalty_distance,
                                           const int body_count_thr,
                                           const int default_n_turn,
                                           const int wait_cycle )
{
    const PlayerType * ptype = player->playerTypePtr();

    const Vector2D & first_player_pos = ( player->seenPosCount() <= player->posCount()
                                          ? player->seenPos()
                                          : player->pos() );
    const Vector2D & first_player_vel = ( player->seenVelCount() <= player->velCount()
                                          ? player->seenVel()
                                          : player->vel() );
    const double first_player_speed = first_player_vel.r() * std::pow( ptype->playerDecay(), wait_cycle );

    int final_reach_cycle = -1;
    {
        Vector2D inertia_pos = ptype->inertiaFinalPoint( first_player_pos, first_player_vel );
        double target_dist = inertia_pos.dist( target_point );

        int n_turn = ( player->bodyCount() > body_count_thr
                       ? default_n_turn
                       : predict_player_turn_cycle( ptype,
                                                    player->body(),
                                                    first_player_speed,
                                                    target_dist,
                                                    ( target_point - inertia_pos ).th(),
                                                    dist_thr ) );
        int n_dash = ptype->cyclesToReachDistance( target_dist + penalty_distance );

        final_reach_cycle = wait_cycle + n_turn + n_dash;
    }

    const int max_cycle = 6; // Magic Number

    if ( final_reach_cycle > max_cycle )
    {
        return final_reach_cycle;
    }

    for ( int cycle = std::max( 0, wait_cycle ); cycle <= max_cycle; ++cycle )
    {
        Vector2D inertia_pos = ptype->inertiaPoint( first_player_pos, first_player_vel, cycle );
        double target_dist = inertia_pos.dist( target_point ) + penalty_distance;

        if ( target_dist < dist_thr )
        {
            return cycle;
        }

        double dash_dist = target_dist - dist_thr * 0.5;

        if ( dash_dist < 0.001 )
        {
            return cycle;
        }

        if ( dash_dist > ptype->realSpeedMax() * ( cycle - wait_cycle ) )
        {
            continue;
        }

        int n_dash = ptype->cyclesToReachDistance( dash_dist );

        if ( wait_cycle + n_dash > cycle )
        {
            continue;
        }

        int n_turn = ( player->bodyCount() > body_count_thr
                       ? default_n_turn
                       : predict_player_turn_cycle( ptype,
                                                    player->body(),
                                                    first_player_speed,
                                                    target_dist,
                                                    ( target_point - inertia_pos ).th(),
                                                    dist_thr ) );

        if ( wait_cycle + n_turn + n_dash <= cycle )
        {
            return cycle;
        }

    }

    return final_reach_cycle;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
FieldAnalyzer::can_shoot_from( const WorldModel & wm,
                               const Vector2D & pos,
                               const int valid_opponent_threshold )
{
    const double SHOOT_DIST_THR2 = 16.0*16.0;
    const double SHOOT_ANGLE_THRESHOLD = 20.0;

    if ( ServerParam::i().theirTeamGoalPos().dist2( pos )
         > SHOOT_DIST_THR2 )
    {
        return false;
    }

    return get_shoot_free_angle_from_pos( wm, pos, valid_opponent_threshold )
        >= SHOOT_ANGLE_THRESHOLD;
}

/*-------------------------------------------------------------------*/
/*!

 */
double
FieldAnalyzer::get_shoot_free_angle_from_pos( const WorldModel & wm,
                                              const Vector2D & pos,
                                              const long valid_opponent_threshold )
{
    const double goal_half_width = ServerParam::i().goalHalfWidth();
    const double field_half_length = ServerParam::i().pitchHalfLength();

    const Vector2D goal_center = ServerParam::i().theirTeamGoalPos();
    const Vector2D goal_left( goal_center.x, +goal_half_width );
    const Vector2D goal_right( goal_center.x, -goal_half_width );

    const Vector2D goal_center_left( field_half_length,
                                     (+ goal_half_width - 1.5) / 2.0 );

    const Vector2D goal_center_right( field_half_length,
                                      (- goal_half_width + 1.5) / 2.0 );


    double center_angle = get_shoot_minimum_free_angle( wm,
                                                        goal_center,
                                                        pos,
                                                        valid_opponent_threshold );

    double left_angle   = get_shoot_minimum_free_angle( wm,
                                                        goal_left,
                                                        pos,
                                                        valid_opponent_threshold );

    double right_angle  = get_shoot_minimum_free_angle( wm,
                                                        goal_right,
                                                        pos,
                                                        valid_opponent_threshold );

    double center_left_angle  = get_shoot_minimum_free_angle( wm,
                                                              goal_center_left,
                                                              pos,
                                                              valid_opponent_threshold );

    double center_right_angle = get_shoot_minimum_free_angle( wm,
                                                              goal_center_right,
                                                              pos,
                                                              valid_opponent_threshold );
#if 0
    dlog.addText( Logger::RECURSIVE,
                  "get_shoot_free_angle_from_pos pos=(%.3f %.3f)",
                  pos.x, pos.y );
    dlog.addText( Logger::RECURSIVE,
                  "__ center_angle=%f", center_angle );
    dlog.addText( Logger::RECURSIVE,
                  "__ left_angle=%f", left_angle );
    dlog.addText( Logger::RECURSIVE,
                  "__ right_angle=%f", right_angle );
    dlog.addText( Logger::RECURSIVE,
                  "__ center_left_angle=%f", center_left_angle );
    dlog.addText( Logger::RECURSIVE,
                  "__ center_right_angle=%f", center_right_angle );
#endif
    return std::max( center_angle,
                     std::max( left_angle,
                               std::max( right_angle,
                                         std::max( center_left_angle,
                                                   center_right_angle ) ) ) );
}

/*-------------------------------------------------------------------*/
/*!

 */
double
FieldAnalyzer::get_shoot_minimum_free_angle( const WorldModel & wm,
                                             const Vector2D & goal,
                                             const Vector2D & pos,
                                             const int valid_opponent_threshold )
{
    AngleDeg test_dir = ( goal - pos ).th();

    double shoot_course_cone = +360.0;

    const PlayerPtrCont::const_iterator o_end = wm.opponentsFromSelf().end();
    for ( PlayerPtrCont::const_iterator o = wm.opponentsFromSelf().begin();
          o != o_end;
          ++o )
    {
        if ( (*o)->posCount() > valid_opponent_threshold )
        {
            continue;
        }

        double controllable_dist = ( (*o)->goalie()
                                     ? ServerParam::i().catchAreaLength()
                                     : (*o)->playerTypePtr()->kickableArea() );

        Vector2D relative_player = (*o)->pos() - pos;

        double hide_angle_radian = ( std::asin( std::min( controllable_dist / relative_player.r(),
                                                          1.0 ) ) );
        double angle_diff = std::max( (relative_player.th() - test_dir).abs()
                                      - hide_angle_radian / M_PI * 180,
                                      0.0 );

        if ( shoot_course_cone > angle_diff )
        {
            shoot_course_cone = angle_diff;
        }
    }

    return shoot_course_cone;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FieldAnalyzer::update( const WorldModel & wm )
{
    static GameTime s_update_time( 0, 0 );

    if ( s_update_time == wm.time() )
    {
        return;
    }
    s_update_time = wm.time();

#ifdef DEBUG_PRINT
    Timer timer;
#endif

    updateVoronoiDiagram( wm );
    updatePlayerGraph( wm );

#ifdef DEBUG_PRINT
    dlog.addText( Logger::TEAM,
                  "FieldAnalyzer::update() elapsed %f [ms]",
                  timer.elapsedReal() );
    writeDebugLog();
#endif

}

/*-------------------------------------------------------------------*/
/*!

 */
void
FieldAnalyzer::updateVoronoiDiagram( const WorldModel & wm )
{
    const Rect2D rect = Rect2D::from_center( 0.0, 0.0,
                                             ServerParam::i().pitchLength() - 10.0,
                                             ServerParam::i().pitchWidth() - 10.0 );

    M_all_players_voronoi_diagram.clear();
    M_teammates_voronoi_diagram.clear();
    M_pass_voronoi_diagram.clear();

    const SideID our = wm.ourSide();

    const AbstractPlayerCont::const_iterator end = wm.allPlayers().end();
    for ( AbstractPlayerCont::const_iterator p = wm.allPlayers().begin();
          p != end;
          ++p )
    {
        M_all_players_voronoi_diagram.addPoint( (*p)->pos() );

        if ( (*p)->side() == our )
        {
            M_teammates_voronoi_diagram.addPoint( (*p)->pos() );
        }
        else
        {
            M_pass_voronoi_diagram.addPoint( (*p)->pos() );
        }
    }

    // our goal
    M_pass_voronoi_diagram.addPoint( Vector2D( - ServerParam::i().pitchHalfLength() + 5.5, 0.0 ) );
    //     M_pass_voronoi_diagram.addPoint( Vector2D( - ServerParam::i().pitchHalfLength() + 5.5,
    //                                                - ServerParam::i().goalHalfWidth() ) );
    //     M_pass_voronoi_diagram.addPoint( Vector2D( - ServerParam::i().pitchHalfLength() + 5.5,
    //                                                + ServerParam::i().goalHalfWidth() ) );

    // opponent side corners
    M_pass_voronoi_diagram.addPoint( Vector2D( + ServerParam::i().pitchHalfLength() + 10.0,
                                               - ServerParam::i().pitchHalfWidth() - 10.0 ) );
    M_pass_voronoi_diagram.addPoint( Vector2D( + ServerParam::i().pitchHalfLength() + 10.0,
                                               + ServerParam::i().pitchHalfWidth() + 10.0 ) );

    M_pass_voronoi_diagram.setBoundingRect( rect );

    M_all_players_voronoi_diagram.compute();
    M_teammates_voronoi_diagram.compute();
    M_pass_voronoi_diagram.compute();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FieldAnalyzer::updatePlayerGraph( const WorldModel & wm )
{
    //
    // update field player graph
    //
    M_field_player_graph.setPredicate( new AndPlayerPredicate
                                       ( new FieldPlayerPredicate(),
                                         new XCoordinateForwardPlayerPredicate( wm.ball().pos().x - 50.0 ),
                                         new XCoordinateBackwardPlayerPredicate( wm.ball().pos().x + 50.0 ) ) );
    M_field_player_graph.update( wm );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FieldAnalyzer::writeDebugLog()
{

    if ( dlog.isEnabled( Logger::PASS ) )
    {
        const VoronoiDiagram::Segment2DCont::const_iterator s_end = M_pass_voronoi_diagram.resultSegments().end();
        for ( VoronoiDiagram::Segment2DCont::const_iterator s = M_pass_voronoi_diagram.resultSegments().begin();
              s != s_end;
              ++s )
        {
            dlog.addLine( Logger::PASS,
                          s->origin(), s->terminal(),
                          "#0000ff" );
        }

        const VoronoiDiagram::Ray2DCont::const_iterator r_end = M_pass_voronoi_diagram.resultRays().end();
        for ( VoronoiDiagram::Ray2DCont::const_iterator r = M_pass_voronoi_diagram.resultRays().begin();
              r != r_end;
              ++r )
        {
            dlog.addLine( Logger::PASS,
                          r->origin(), r->origin() + Vector2D::polar2vector( 20.0, r->dir() ),
                          "#0000ff" );
        }
    }

    if ( dlog.isEnabled( Logger::MARK ) )
    {
        //
        // field player graph
        //
        const std::vector< PlayerGraph::Connection >::const_iterator e_end = M_field_player_graph.connections().end();
        for ( std::vector< PlayerGraph::Connection >::const_iterator e = M_field_player_graph.connections().begin();
              e != e_end;
              ++e )
        {
            dlog.addLine( Logger::MARK,
                          e->first_->pos(), e->second_->pos(),
                          "#ff00ff" );
        }
    }
}
