// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_set_play_indirect_free_kick.h"

#include "strategy.h"

#include "bhv_set_play.h"
#include "bhv_prepare_set_play_kick.h"
#include "bhv_go_to_static_ball.h"

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/body_kick_one_step.h>
#include <rcsc/action/body_kick_collide_with_ball.h>
#include <rcsc/action/body_pass.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/say_message_builder.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/circle_2d.h>
#include <rcsc/math_util.h>

using namespace rcsc;

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_SetPlayIndirectFreeKick::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::TEAM,
                  __FILE__": Bhv_SetPlayIndirectFreeKick" );
    //---------------------------------------------------
    const WorldModel & wm = agent->world();

    const bool our_kick = ( ( wm.gameMode().type() == GameMode::BackPass_
                              && wm.gameMode().side() == wm.theirSide() )
                            || ( wm.gameMode().type() == GameMode::IndFreeKick_
                                 && wm.gameMode().side() == wm.ourSide() )
                            || ( wm.gameMode().type() == GameMode::FoulCharge_
                                 && wm.gameMode().side() == wm.theirSide() )
                            || ( wm.gameMode().type() == GameMode::FoulPush_
                                 && wm.gameMode().side() == wm.theirSide() )
                            );

    if ( our_kick )
    {
        if ( wm.teammatesFromBall().empty()
             || wm.teammatesFromBall().front()->distFromBall() > wm.self().distFromBall() )
        {
            doKick( agent );
        }
        else
        {
            doOffenseMove( agent );
        }
    }
    else
    {
        doDefenseMove( agent );
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
Bhv_SetPlayIndirectFreeKick::doKick( PlayerAgent * agent )
{
    // go to ball
    if ( Bhv_GoToStaticBall( 0.0 ).execute( agent ) )
    {
        return;
    }

    //
    // wait
    //

    if ( doKickWait( agent ) )
    {
        return;
    }

    //
    // kick to the teammate exist at the front of their goal
    //

    if ( doKickToShooter( agent ) )
    {
        return;
    }

    const WorldModel & wm = agent->world();

    const double max_kick_speed = wm.self().kickRate() * ServerParam::i().maxPower();

    //
    // pass
    //
    {
        const double max_kick_speed = wm.self().kickRate() * rcsc::ServerParam::i().maxPower();

        rcsc::Vector2D target_point;
        double ball_speed = 0.0;

        if  ( Body_Pass::get_best_pass( wm,
                                        &target_point,
                                        &ball_speed,
                                        NULL )
              && target_point.x > 35.0
              && ( target_point.x > wm.self().pos().x - 1.0
                   || target_point.x > 48.0 ) )
        {
            ball_speed = std::min( ball_speed, max_kick_speed );
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                __FILE__":  pass to (%.1f %.1f) speed=%.2f",
                                target_point.x, target_point.y,
                                ball_speed );
            Body_KickOneStep( target_point, ball_speed ).execute( agent );
            agent->setNeckAction( new Neck_ScanField() );
            return;
        }
    }

    //
    // wait(2)
    //
    if ( wm.setplayCount() <= 3 )
    {
        Body_TurnToPoint( Vector2D( 50.0, 0.0 ) ).execute( agent );
        agent->setNeckAction( new Neck_ScanField() );
        return;
    }

    //
    // no teammate
    //
    if ( wm.teammatesFromBall().empty()
         || wm.teammatesFromBall().front()->distFromSelf() > 35.0
         || wm.teammatesFromBall().front()->pos().x < -30.0 )
    {
        const int real_set_play_count
            = static_cast< int >( wm.time().cycle() - wm.lastSetPlayStartTime().cycle() );

        if ( real_set_play_count <= ServerParam::i().dropBallTime() - 3 )
        {
            dlog.addText( Logger::TEAM,
                          __FILE__": (doKick) real set play count = %d <= drop_time-3, wait...",
                          real_set_play_count );
            Body_TurnToPoint( Vector2D( 50.0, 0.0 ) ).execute( agent );
            agent->setNeckAction( new Neck_ScanField() );
            return;
        }

        Vector2D target_point( ServerParam::i().pitchHalfLength(),
                               static_cast< double >( -1 + 2 * wm.time().cycle() % 2 )
                               * ( ServerParam::i().goalHalfWidth() - 0.8 ) );
        double ball_speed = max_kick_speed;

        agent->debugClient().addMessage( "IndKick:ForceShoot" );
        agent->debugClient().setTarget( target_point );
        dlog.addText( Logger::TEAM,
                      __FILE__":  kick to goal (%.1f %.1f) speed=%.2f",
                      target_point.x, target_point.y,
                      ball_speed );

        Body_KickOneStep( target_point, ball_speed ).execute( agent );
        agent->setNeckAction( new Neck_ScanField() );
        return;
    }

    //
    // kick to the teammate nearest to opponent goal
    //

    const Vector2D goal( ServerParam::i().pitchHalfLength(),
                         wm.self().pos().y * 0.8 );

    double min_dist = 100000.0;
    const PlayerObject * receiver = static_cast< const PlayerObject * >( 0 );

    const PlayerPtrCont::const_iterator t_end = wm.teammatesFromBall().end();
    for ( PlayerPtrCont::const_iterator t = wm.teammatesFromBall().begin();
          t != t_end;
          ++t )
    {
        if ( (*t)->posCount() > 5 ) continue;
        if ( (*t)->distFromBall() < 1.5 ) continue;
        if ( (*t)->distFromBall() > 20.0 ) continue;
        if ( (*t)->pos().x > wm.offsideLineX() ) continue;

        double dist = (*t)->pos().dist( goal ) + (*t)->distFromBall();
        if ( dist < min_dist )
        {
            min_dist = dist;
            receiver = (*t);
        }
    }

    Vector2D target_point = goal;
    double target_dist = 10.0;
    if ( ! receiver )
    {
        target_dist = wm.teammatesFromSelf().front()->distFromSelf();
        target_point = wm.teammatesFromSelf().front()->pos();
    }
    else
    {
        target_dist = receiver->distFromSelf();
        target_point = receiver->pos();
        target_point.x += 0.6;
    }

    double ball_speed = calc_first_term_geom_series_last( 1.8, // end speed
                                                          target_dist,
                                                          ServerParam::i().ballDecay() );
    ball_speed = std::min( ball_speed, max_kick_speed );

    agent->debugClient().addMessage( "IndKick:ForcePass%.3f", ball_speed );
    agent->debugClient().setTarget( target_point );
    dlog.addText( Logger::TEAM,
                  __FILE__":  pass to nearest teammate (%.1f %.1f) speed=%.2f",
                  target_point.x, target_point.y,
                  ball_speed );


    Body_KickOneStep( target_point, ball_speed ).execute( agent );
    agent->setNeckAction( new Neck_ScanField() );
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_SetPlayIndirectFreeKick::doKickWait( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    const Vector2D face_point( 50.0, 0.0 );
    const AngleDeg face_angle = ( face_point - wm.self().pos() ).th();

    //     if ( wm.setplayCount() <= 3 )
    //     {
    //         Body_TurnToPoint( face_point ).execute( agent );
    //         agent->setNeckAction( new Neck_ScanField() );
    //         return;
    //     }

    if ( wm.time().stopped() > 0 )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__":  stoppage time" );

        Body_TurnToPoint( face_point ).execute( agent );
        agent->setNeckAction( new Neck_ScanField() );
        return true;
    }

    if ( ( face_angle - wm.self().body() ).abs() > 5.0 )
    {
        agent->debugClient().addMessage( "IndKick:TurnTo" );
        agent->debugClient().setTarget( face_point );

        Body_TurnToPoint( face_point ).execute( agent );
        agent->setNeckAction( new Neck_ScanField() );
        return true;
    }

    if ( wm.setplayCount() <= 10
         && wm.teammatesFromSelf().empty() )
    {
        agent->debugClient().addMessage( "IndKick:NoTeammate" );
        agent->debugClient().setTarget( face_point );

        Body_TurnToPoint( face_point ).execute( agent );
        agent->setNeckAction( new Neck_ScanField() );
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_SetPlayIndirectFreeKick::doKickToShooter( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();


    const Vector2D goal( ServerParam::i().pitchHalfLength(),
                         wm.self().pos().y * 0.8 );

    double min_dist = 100000.0;
    const PlayerObject * receiver = static_cast< const PlayerObject * >( 0 );

    const PlayerPtrCont::const_iterator t_end = wm.teammatesFromBall().end();
    for ( PlayerPtrCont::const_iterator t = wm.teammatesFromBall().begin();
          t != t_end;
          ++t )
    {
        if ( (*t)->posCount() > 5 ) continue;
        if ( (*t)->distFromBall() < 1.5 ) continue;
        if ( (*t)->distFromBall() > 20.0 ) continue;
        if ( (*t)->pos().x > wm.offsideLineX() ) continue;
        if ( (*t)->pos().x < wm.ball().pos().x ) continue;
        if ( (*t)->pos().absY() > ServerParam::i().goalHalfWidth() * 0.5 ) continue;

        double dist = (*t)->pos().dist( goal ) + (*t)->distFromBall();
        if ( dist < min_dist )
        {
            min_dist = dist;
            receiver = (*t);
        }
    }

    if ( ! receiver )
    {
        return false;
    }

    const double max_ball_speed = wm.self().kickRate() * ServerParam::i().maxPower();

    Vector2D target_point = receiver->pos() + receiver->vel();
    target_point.x += 0.6;

    double target_dist = wm.ball().pos().dist( target_point );

    int ball_reach_step
        = static_cast< int >( std::ceil( calc_length_geom_series( max_ball_speed,
                                                                  target_dist,
                                                                  ServerParam::i().ballDecay() ) ) );
    double ball_speed = calc_first_term_geom_series( target_dist,
                                                     ServerParam::i().ballDecay(),
                                                     ball_reach_step );

    ball_speed = std::min( ball_speed, max_ball_speed );

    agent->debugClient().addMessage( "IndKick:KickToShooter%.3f", ball_speed );
    agent->debugClient().setTarget( target_point );
    dlog.addText( Logger::TEAM,
                  __FILE__":  pass to nearest teammate (%.1f %.1f) ball_speed=%.2f reach_step=%d",
                  target_point.x, target_point.y,
                  ball_speed, ball_reach_step );

    Body_KickOneStep( target_point, ball_speed ).execute( agent );
    agent->setNeckAction( new Neck_ScanField() );

    return true;

}

namespace {

Vector2D
get_avoid_circle_point( const WorldModel & world,
                        const Vector2D & point,
                        int depth )
{
    if ( depth > 5 )
    {
        return point;
    }

    const double circle_r = world.gameMode().type() == GameMode::BackPass_
        ? ServerParam::i().goalAreaLength() + 1.5
        : ServerParam::i().centerCircleR() + 1.5;

    if ( ( world.self().pos().x > -ServerParam::i().pitchHalfLength()
           || world.self().pos().absY() > ServerParam::i().goalHalfWidth() - 0.5
           || world.ball().distFromSelf() < world.self().pos().dist( point ) )
         && ( Line2D( world.self().pos(), point ).dist2( world.ball().pos() )
              < circle_r * circle_r )
         && ( ( world.ball().pos() - point ).th() - ( world.self().pos() - point ).th() ).abs()
         < 90.0
         && ( world.ball().angleFromSelf() - ( point - world.self().pos() ).th() ).abs()
         < 90.0
         )
    {
        Vector2D new_point = world.ball().pos();
        AngleDeg self2target = ( point - world.self().pos() ).th();
        if ( world.ball().angleFromSelf().isLeftOf( self2target ) )
        {
            new_point += Vector2D::polar2vector( circle_r + 0.5,
                                                 self2target + 90.0 );
        }
        else
        {
            new_point += Vector2D::polar2vector( circle_r + 0.5,
                                                 self2target - 90.0 );
        }

        // recursive
        return get_avoid_circle_point( world, new_point, depth + 1 );
    }

    return point;
}


Vector2D
get_avoid_circle_point( const WorldModel & world,
                        Vector2D point )
{
    const double circle_r = world.gameMode().type() == GameMode::BackPass_
        ? ServerParam::i().goalAreaLength() + 0.5
        : ServerParam::i().centerCircleR() + 0.5;
    const double circle_r2 = std::pow( circle_r, 2 );

    dlog.addText( Logger::TEAM,
                  __FILE__": (get_avoid_circle_point) point=(%.1f %.1f)",
                  point.x, point.y );

//     if ( point.x > - ServerParam::i().pitchHalfLength() + 0.5
//          && world.ball().pos().dist2( point ) < std::pow( circle_r, 2 ) )
//     {
//         point = world.ball().pos() + ( point - world.ball().pos() ).setLengthVector( circle_r );
//         dlog.addText( Logger::TEAM,
//                       __FILE__": adjust position to (%.1f %.1f)",
//                       point.x, point.y );
//     }

    while ( point.x < world.ball().pos().x
            && point.x > - ServerParam::i().pitchHalfLength()
            && world.ball().pos().dist2( point ) < circle_r2 )
    {
        point.x -= 0.2;
        dlog.addText( Logger::TEAM,
                      __FILE__": adjust x (%.1f %.1f)",
                      point.x, point.y );
    }

    if ( world.ball().pos().x - circle_r < point.x
         && point.x < world.ball().pos().x
         && world.ball().pos().dist2( point ) < circle_r2 )
    {
        if ( world.self().pos().x < - ServerParam::i().pitchHalfLength() + 0.1
             && world.self().pos().absY() < ServerParam::i().goalHalfWidth() - 0.5 )
        {
            dlog.addText( Logger::TEAM,
                          __FILE__": can go to the point directly." );
            return point;
        }

        if ( world.self().pos().x < world.ball().pos().x - circle_r )
        {
            dlog.addText( Logger::TEAM,
                          __FILE__": can go to the Y. my_pos.x=%.1f < ball_x-r=%f",
                          world.self().pos().x,
                          world.ball().pos().x - circle_r );
            return Vector2D( world.self().pos().x, point.y );
        }

        Vector2D tmp_point( world.ball().pos().x - circle_r, point.y );

        dlog.addText( Logger::TEAM,
                      __FILE__": modified recursive. tmp_point=(%.1f %.1f)",
                      tmp_point.x, tmp_point.y );
        return get_avoid_circle_point( world, tmp_point, 0 );
    }

    dlog.addText( Logger::TEAM,
                  __FILE__": normal recursive" );

    return get_avoid_circle_point( world, point, 0 );
}

} // end noname namespace

/*-------------------------------------------------------------------*/
/*!

 */
void
Bhv_SetPlayIndirectFreeKick::doOffenseMove( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    Vector2D target_point = Strategy::i().getPosition( wm.self().unum() );
    target_point.x = std::min( wm.offsideLineX() - 1.0, target_point.x );

    double nearest_dist = 1000.0;
    const PlayerObject * teammate = wm.getTeammateNearestTo( target_point, 10, &nearest_dist );
    if ( nearest_dist < 2.5 )
    {
        target_point += ( target_point - teammate->pos() ).setLengthVector( 2.5 );
        target_point.x = std::min( wm.offsideLineX() - 1.0, target_point.x );
    }

    double dash_power = wm.self().getSafetyDashPower( ServerParam::i().maxDashPower() );

    double dist_thr = wm.ball().distFromSelf() * 0.07;
    if ( dist_thr < 0.5 ) dist_thr = 0.5;

    agent->debugClient().addMessage( "IndFK:OffenseMove" );
    agent->debugClient().setTarget( target_point );
    agent->debugClient().addCircle( target_point, dist_thr );

    if ( ! Body_GoToPoint( target_point,
                           dist_thr,
                           dash_power
                           ).execute( agent ) )
    {
        // already there
        Vector2D turn_point( ServerParam::i().pitchHalfLength(), 0.0 );
        AngleDeg angle = ( turn_point - wm.self().pos() ).th();
        if ( angle.abs() > 100.0 )
        {
            turn_point = Vector2D::polar2vector( 10.0, angle + 100.0 );
            if ( turn_point.x < 0.0 )
            {
                turn_point.rotate( 180.0 );
            }
            turn_point += wm.self().pos();
        }

        Body_TurnToPoint( turn_point ).execute( agent );
        dlog.addText( Logger::TEAM,
                      __FILE__":  our kick. turn to (%.1f %.1f)",
                      turn_point.x, turn_point.y );
    }

    if ( target_point.x > 36.0
         && ( wm.self().pos().dist( target_point )
              > std::max( wm.ball().pos().dist( target_point ) * 0.2, dist_thr ) + 6.0
              || wm.self().stamina() < ServerParam::i().staminaMax() * 0.7 )
         )
    {
        if ( ! wm.self().staminaModel().capacityIsEmpty() )
        {
            agent->debugClient().addMessage( "Sayw" );
            agent->addSayMessage( new WaitRequestMessage() );
        }
    }

    agent->setNeckAction( new Neck_TurnToBallOrScan() );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
Bhv_SetPlayIndirectFreeKick::doDefenseMove( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    Vector2D target_point = Strategy::i().getPosition( wm.self().unum() );
    Vector2D new_target = get_avoid_circle_point( wm, target_point );
    dlog.addText( Logger::TEAM,
                  __FILE__":  their kick adjust target to (%.1f %.1f)->(%.1f %.1f) ",
                  target_point.x, target_point.y,
                  new_target.x, new_target.y );
    target_point = new_target;

    double dash_power = wm.self().getSafetyDashPower( ServerParam::i().maxDashPower() );

    double dist_thr = wm.ball().distFromSelf() * 0.07;
    if ( dist_thr < 0.5 ) dist_thr = 0.5;

    agent->debugClient().addMessage( "IndFKMove" );
    agent->debugClient().setTarget( target_point );
    agent->debugClient().addCircle( target_point, dist_thr );

    if ( ! Body_GoToPoint( target_point,
                           dist_thr,
                           dash_power
                           ).execute( agent ) )
    {
        // already there
        Body_TurnToBall().execute( agent );
        dlog.addText( Logger::TEAM,
                      __FILE__":  their kick. turn to ball" );
    }

    agent->setNeckAction( new Neck_TurnToBall() );
}
