// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_set_play.h"

#include "strategy.h"

#include "bhv_goalie_free_kick.h"
#include "bhv_set_play_free_kick.h"
#include "bhv_set_play_goal_kick.h"
#include "bhv_set_play_kick_off.h"
#include "bhv_set_play_kick_in.h"
#include "bhv_set_play_indirect_free_kick.h"
#include "bhv_their_goal_kick_move.h"

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/bhv_before_kick_off.h>
#include <rcsc/action/bhv_scan_field.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/line_2d.h>

#include <limits>

using namespace rcsc;

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_SetPlay::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::TEAM,
                  __FILE__": Bhv_SetPlay" );

    const WorldModel & wm = agent->world();

#ifdef DEBUG
    rcsc::Vector2D home_pos = Strategy::i().getPosition( world().self().unum() );
    if ( ! home_pos.isValid() )
    {
        std::cerr << config().teamName() << ": "
                  << world().self().unum()
                  << " ***ERROR*** illegal home position."
                  << std::endl;
        home_pos.assign( 0.0, 0.0 );
    }

    debugClient().addLine( rcsc::Vector2D( home_pos.x, home_pos.y - 0.25 ),
                           rcsc::Vector2D( home_pos.x, home_pos.y + 0.25 ) );
    debugClient().addLine( rcsc::Vector2D( home_pos.x - 0.25, home_pos.y ),
                           rcsc::Vector2D( home_pos.x + 0.25, home_pos.y ) );
    debugClient().addCircle( home_pos, 0.5 );
#endif


    if ( wm.self().goalie() )
    {
        if ( wm.gameMode().type() != rcsc::GameMode::BackPass_
             && wm.gameMode().type() != rcsc::GameMode::IndFreeKick_ )
        {
            Bhv_GoalieFreeKick().execute( agent );
        }
        else
        {
            Bhv_SetPlayIndirectFreeKick().execute( agent );
        }

        return true;
    }

    switch ( wm.gameMode().type() ) {
    case GameMode::KickOff_:
        if ( wm.gameMode().side() == wm.ourSide() )
        {
            return Bhv_SetPlayKickOff().execute( agent );
        }
        else
        {
            doBasicTheirSetPlayMove( agent );
        }
        break;
    case GameMode::KickIn_:
    case GameMode::CornerKick_:
        if ( wm.gameMode().side() == wm.ourSide() )
        {
            return Bhv_SetPlayKickIn().execute( agent );
        }
        else
        {
            doBasicTheirSetPlayMove( agent );
        }
        break;
    case GameMode::GoalKick_:
        if ( wm.gameMode().side() == wm.ourSide() )
        {
            return Bhv_SetPlayGoalKick().execute( agent );
        }
        else
        {
            return Bhv_TheirGoalKickMove().execute( agent );
        }
        break;
    case GameMode::BackPass_:
    case GameMode::IndFreeKick_:
        return Bhv_SetPlayIndirectFreeKick().execute( agent );
        break;
    case GameMode::FoulCharge_:
    case GameMode::FoulPush_:
        if ( wm.ball().pos().x < ServerParam::i().ourPenaltyAreaLineX() + 1.0
             && wm.ball().pos().absY() < ServerParam::i().penaltyAreaHalfWidth() + 1.0 )
        {
            return Bhv_SetPlayIndirectFreeKick().execute( agent );
        }
        else if ( wm.ball().pos().x > ServerParam::i().theirPenaltyAreaLineX() - 1.0
                  && wm.ball().pos().absY() < ServerParam::i().penaltyAreaHalfWidth() + 1.0 )
        {
            return Bhv_SetPlayIndirectFreeKick().execute( agent );
        }
        break;
#if 0
    case GameMode::FreeKick_:
    case GameMode::CornerKick_:
    case GameMode::GoalieCatch_: // after catch
    case GameMode::Offside_:
    case GameMode::FreeKickFault_:
    case GameMode::CatchFault_:
#endif
    default:
        break;
    }

    if ( wm.gameMode().isOurSetPlay( wm.ourSide() ) )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": our set play" );
        return Bhv_SetPlayFreeKick().execute( agent );
    }
    else
    {
        doBasicTheirSetPlayMove( agent );
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
double
Bhv_SetPlay::get_set_play_dash_power( const PlayerAgent * agent )
{
#if 1
    const WorldModel & wm = agent->world();

    if ( ! wm.gameMode().isOurSetPlay( wm.ourSide() ) )
    {
        Vector2D target_point = Strategy::i().getPosition( wm.self().unum() );
        if ( target_point.x > wm.self().pos().x )
        {
            double rate = 0.0;
            if ( wm.self().stamina() > ServerParam::i().staminaMax() * 0.8 )
            {
                rate = 1.5 * wm.self().stamina() / ServerParam::i().staminaMax();
            }
            else
            {
                rate = 0.9
                    * ( wm.self().stamina() - ServerParam::i().recoverDecThrValue() )
                    / ServerParam::i().staminaMax();
                rate = std::max( 0.0, rate );
            }

            return ( wm.self().playerType().staminaIncMax()
                     * wm.self().recovery()
                     * rate );
        }
    }

    return agent->world().self().getSafetyDashPower( ServerParam::i().maxDashPower() );
#else
    if ( agent->world().gameMode().type() == GameMode::PenaltySetup_ )
    {
        return agent->world().self().getSafetyDashPower( ServerParam::i().maxDashPower() );
    }

    double rate;
    if ( agent->world().self().stamina() > ServerParam::i().staminaMax() * 0.8 )
    {
        rate = 1.5
            * agent->world().self().stamina()
            / ServerParam::i().staminaMax();
    }
    else
    {
        rate = 0.9
            * ( agent->world().self().stamina()
                - ServerParam::i().recoverDecThrValue() )
            / ServerParam::i().staminaMax();
        rate = std::max( 0.0, rate );
    }

    return ( agent->world().self().playerType().staminaIncMax()
             * agent->world().self().recovery()
             * rate );
#endif
}

/*-----------------------------------------------------------------------------*
 * recursive function
 *
 *-----------------------------------------------------------------------------*/
namespace {

Vector2D
get_avoid_circle_point( const WorldModel & wm,
                        const Vector2D & point,
                        int depth )
{
    if ( depth > 5 )
    {
        return point;
    }

    if ( wm.ball().distFromSelf() < wm.self().pos().dist( point )
         && ( ( wm.ball().pos() - point ).th()
              - ( wm.self().pos() - point ).th() ).abs() < 90.0
         && ( wm.ball().angleFromSelf()
              - ( point - wm.self().pos() ).th() ).abs() < 90.0
         && ( Line2D( wm.self().pos(), point).dist2( wm.ball().pos() )
              < 10.0 * 10.0 )
         )
    {
        Vector2D new_point = wm.ball().pos();
        AngleDeg self2target = ( point - wm.self().pos() ).th();
        if ( wm.ball().angleFromSelf().isLeftOf( self2target ) )
        {
            new_point += Vector2D::polar2vector( 11.5,
                                                 self2target + 90.0 );
        }
        else
        {
            new_point += Vector2D::polar2vector( 11.5,
                                                 self2target - 90.0 );
        }
        // recursive
        return get_avoid_circle_point( wm, new_point, depth + 1 );
    }

    return point;
}

} // end noname namespace


/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_SetPlay::is_kicker( const PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    //if ( setplayCount() < 1 )
    //     if ( wm.lastSetPlayStartTime().cycle() > wm.time().cycle() - 2 )
    //     {
    //         return false;
    //     }

    if ( wm.gameMode().type() == GameMode::GoalieCatch_
         && wm.gameMode().side() == wm.ourSide()
         && ! wm.self().goalie() )
    {
        return false;
    }

    int kicker_unum = 0;
    double min_dist2 = std::numeric_limits< double >::max();
    int second_kicker_unum = 0;
    double second_min_dist2 = std::numeric_limits< double >::max();
    for ( int unum = 1; unum <= 11; ++unum )
    {
        if ( unum == wm.teammateGoalieUnum() ) continue;

        Vector2D home_pos = Strategy::i().getPosition( unum );
        if ( ! home_pos.isValid() ) continue;

        double d2 = home_pos.dist2( wm.ball().pos() );
        if ( d2 < second_min_dist2 )
        {
            second_kicker_unum = unum;
            second_min_dist2 = d2;

            if ( second_min_dist2 < min_dist2 )
            {
                std::swap( second_kicker_unum, kicker_unum );
                std::swap( second_min_dist2, min_dist2 );
            }
        }
    }

    const AbstractPlayerObject * kicker = static_cast< AbstractPlayerObject* >( 0 );
    const AbstractPlayerObject * second_kicker = static_cast< AbstractPlayerObject* >( 0 );

    if ( kicker_unum != 0 )
    {
        kicker = wm.teammate( kicker_unum );
    }

    if ( second_kicker_unum != 0 )
    {
        second_kicker = wm.teammate( second_kicker_unum );
    }

    if ( ! kicker )
    {
        if ( ! wm.teammatesFromBall().empty()
             && wm.teammatesFromBall().front()->distFromBall() < wm.ball().distFromSelf() * 0.9 )
        {
            return false;
        }

        return true;
    }

    if ( kicker
         && second_kicker
         && ( kicker->unum() == wm.self().unum()
              || second_kicker->unum() == wm.self().unum() ) )
    {
        if ( std::sqrt( min_dist2 ) < std::sqrt( second_min_dist2 ) * 0.95 )
        {
            return ( kicker->unum() == wm.self().unum() );
        }
        else if ( kicker->distFromBall() < second_kicker->distFromBall() * 0.95 )
        {
            return ( kicker->unum() == wm.self().unum() );
        }
        else if ( second_kicker->distFromBall() < kicker->distFromBall() * 0.95 )
        {
            return ( second_kicker->unum() == wm.self().unum() );
        }
        else  if ( ! wm.teammatesFromBall().empty()
                   && wm.teammatesFromBall().front()->distFromBall() < wm.self().distFromBall() * 0.95 )
        {
            return false;
        }
        else
        {
            return true;
        }
    }

    return ( kicker->unum() == wm.self().unum() );
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_SetPlay::is_delaying_tactics_situation( const PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    long cycle_thr = std::max( 0,
                               ServerParam::i().nrNormalHalfs()
                               * ( ServerParam::i().halfTime() * 10 )
                               - 500 );

    if ( wm.time().cycle() < cycle_thr )
    {
        return false;
    }

    int our_score = ( wm.ourSide() == LEFT
                      ? wm.gameMode().scoreLeft()
                      : wm.gameMode().scoreRight() );
    int opp_score = ( wm.ourSide() == LEFT
                      ? wm.gameMode().scoreRight()
                      : wm.gameMode().scoreLeft() );

    if ( our_score > opp_score
         && our_score - opp_score <= 1 )
    {
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!
  execute action
*/
void
Bhv_SetPlay::doBasicTheirSetPlayMove( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    Vector2D target_point = Strategy::i().getPosition( wm.self().unum() );
    if ( wm.gameMode().type() == GameMode::KickOff_
         && ServerParam::i().kickoffOffside() )
    {
        target_point.x = std::min( -0.5, target_point.x );
    }

    dlog.addText( Logger::TEAM,
                  __FILE__": their set play. position=(%.1f, %.1f)",
                  target_point.x, target_point.y );

    double dash_power = Bhv_SetPlay::get_set_play_dash_power( agent );

    Vector2D ball_to_target = target_point - wm.ball().pos();

    if ( ball_to_target.r() < 11.0 )
    {
        if ( wm.ball().pos().x < ServerParam::i().ourPenaltyAreaLineX() + 1.0 )
        {
            // ball is in the behind of penalty area line
            double ydiff = std::sqrt( std::pow( 11.0, 2 ) - std::pow( ball_to_target.x, 2 ) );
            if ( wm.ball().pos().y < 0.0 )
            {
                target_point.y = agent->world().ball().pos().y + ydiff;
            }
            else
            {
                target_point.y = agent->world().ball().pos().y - ydiff;
            }
        }
        else
        {
            target_point.x
                = wm.ball().pos().x
                - std::sqrt( std::pow( 11.0, 2 ) - std::pow( ball_to_target.y, 2 ) );
            if ( target_point.x < -45.0 )
            {
                target_point = wm.ball().pos();
                target_point += ball_to_target.setLengthVector( 11.0 );
            }
        }
    }

    if ( wm.self().pos().absY() > ServerParam::i().pitchHalfWidth()
         && wm.self().pos().x < wm.ball().pos().x + 11.0
         && wm.ball().pos().x < wm.self().pos().x )
    {
        // subtarget may be out of area.
        // at first, player should back to safety area
        target_point = wm.ball().pos();
        target_point.x += 12.0;
        target_point.y *= 0.9;
    }
    else
    {
        // recursive search
        target_point = get_avoid_circle_point( wm, target_point, 0 );
    }

    if ( wm.gameMode().type() == GameMode::KickOff_
         && ServerParam::i().kickoffOffside() )
    {
        target_point.x = std::min( -0.1, target_point.x );
    }

    double dist_thr = wm.ball().distFromSelf() * 0.1;
    if ( dist_thr < 0.7 ) dist_thr = 0.7;

    agent->debugClient().setTarget( target_point );
    agent->debugClient().addCircle( target_point, dist_thr );

    if ( ! Body_GoToPoint( target_point,
                           dist_thr,
                           dash_power
                           ).execute( agent ) )
    {
        // already there
        AngleDeg body_angle = wm.ball().angleFromSelf();
        if ( body_angle.degree() < 0.0 ) body_angle -= 90.0;
        else body_angle += 90.0;

        Body_TurnToAngle( body_angle ).execute( agent );
    }

    agent->setNeckAction( new Neck_TurnToBall() );
}
