// -*-c++-*-

/*!
  \file bhv_penalty_shootouts_goalie.cpp
  \brief aggressive goalie behavior
*/

/*
 *Copyright:

 Copyright (C) Hiroki SHIMORA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_penalty_shootouts_goalie.h"

#include "strategy.h"

#include "bhv_basic_tackle.h"
#include "body_goalie_go_to_point.h"

#include <rcsc/action/bhv_scan_field.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/body_intercept.h>
#include <rcsc/action/body_turn_to_angle.h>
#include <rcsc/action/body_turn_to_point.h>
#include <rcsc/action/body_stop_dash.h>
#include <rcsc/action/body_clear_ball.h>
#include <rcsc/action/body_smart_kick.h>
#include <rcsc/action/neck_turn_to_ball.h>
#include <rcsc/action/neck_turn_to_point.h>
#include <rcsc/action/view_synch.h>
#include <rcsc/action/view_wide.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/player_predicate.h>
#include <rcsc/player/penalty_kick_state.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/angle_deg.h>
#include <rcsc/geom/rect_2d.h>
#include <rcsc/math_util.h>

#include <algorithm>
#include <cmath>
#include <cstdio>

#if 1
# define VISUAL_DEBUG
#endif

//
// all revert switch
//

// uncomment this, to revert all unstable behaviors
#if 0
# define OLD_STABLE
#endif



//
// each revert switch
//

//#define PENALTY_SHOOTOUT_BLOCK_IN_PENALTY_AREA


// don't chase ball side of penalty area
#define INHIBIT_SIDE_AREA_CHASE


// !!!!!!
// !!!!!!
// uncomment this not to do forward positioning
// new behavior is not stable, may be lose point by long shoot.
#if 1
# define OLD_AGGRESSIVE_POSITIONING
#endif
// !!!!!!
// !!!!!!


// uncomment this to disable shifting to shoot point
// this shifting is effective to opponent's fast passing near goal
// but has side effect, may weak far corner shooting at side 1 on 1.
#if 0
# define DISABLE_GOAL_LINE_POSITIONING_SHIFT_TO_SHOOT_POINT
#endif

// uncomment this to use old condition for decision which positioning
// mode, normal positioning or goal line positioning
#if 0
# define OLD_GOAL_LINE_POSITINING_CONDITION
#endif

// uncomment this to use old aggressive defensing at mid field
// new emergent advance is semi stable
#if 1
# define OLD_EMERGENT_ADVANCE
#endif

#ifdef OLD_STABLE
# define OLD_AGGRESSIVE_POSITIONING
# define OLD_EMERGENT_ADVANCE
# define DISABLE_GOAL_LINE_POSITIONING_SHIFT_TO_SHOOT_POINT
# define OLD_GOAL_LINE_POSITINING_CONDITION
#endif


#if 1
# define DO_EMERGENT_ADVANCE
#endif
#if 0
# define DO_GOAL_LINE_POSITIONONG
#endif
#if 0
# define FORCE_GOAL_LINE_POSITIONONG
#endif

using namespace rcsc;

static const double EMERGENT_ADVANCE_DIR_DIFF_THRESHOLD = 3.0; //degree

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_PenaltyShootoutsGoalie::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::TEAM,
                  __FILE__": Bhv_PenaltyShootoutsGoalie::execute()" );

    const WorldModel & wm = agent->world();

    switch( wm.gameMode().type() ) {
    case GameMode::PenaltyTaken_:
        {
            const PenaltyKickState * pen = wm.penaltyKickState();

            if ( pen->currentTakerSide() == wm.ourSide() )
            {
                Body_TurnToPoint( Vector2D( 0.0, 0.0 ) ).execute( agent );
                agent->setNeckAction( new Neck_TurnToBall() );
                agent->setViewAction( new View_Wide() );
                return true;
            }
            else
            {
                if ( doMove( agent,
                             true,
                             (pen->onfieldSide() != wm.ourSide()) ) )
                {
                    agent->setNeckAction( new Neck_TurnToBall() );
                    agent->setViewAction( new View_Synch() );
                    return true;
                }
            }

            break;
        }

    default:
        break;
    }

    return false;
}


/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_PenaltyShootoutsGoalie::doMove( PlayerAgent * agent,
                                    bool penalty_kick_mode,
                                    bool reverse_x )
{
    const WorldModel & wm = agent->world();
    const ServerParam & param = ServerParam::i();

    //
    // set some parameters for thinking
    //
    const int teammate_reach_cycle = wm.interceptTable()->teammateReachCycle();
    const int opponent_reach_cycle = wm.interceptTable()->opponentReachCycle();
    const int self_reach_cycle = wm.interceptTable()->selfReachCycle();
    int predict_step = std::min( opponent_reach_cycle,
                                 std::min( teammate_reach_cycle,
                                           self_reach_cycle ) );
    const Vector2D intercept_point = wm.ball().inertiaPoint( self_reach_cycle );

    dlog.addText( Logger::TEAM,
                  __FILE__": self pos = [%f, %f]",
                  wm.self().pos().x, wm.self().pos().y );
    dlog.addText( Logger::TEAM,
                  __FILE__": self reach cycle = %d",
                  self_reach_cycle );
    dlog.addText( Logger::TEAM,
                  __FILE__": teammate reach cycle = %d",
                  teammate_reach_cycle );
    dlog.addText( Logger::TEAM,
                  __FILE__": opponent reach cycle = %d",
                  opponent_reach_cycle );
    dlog.addText( Logger::TEAM,
                  __FILE__": ball pos = [%f, %f]",
                  wm.ball().pos().x, wm.ball().pos().y );
    dlog.addText( Logger::TEAM,
                  __FILE__": ball dist = %f",
                  wm.ball().distFromSelf() );
    dlog.addText( Logger::TEAM,
                  __FILE__": self intercept cycle = %d, point = (%f, %f)",
                  self_reach_cycle,
                  intercept_point.x, intercept_point.y );
    dlog.addText( Logger::TEAM,
                  __FILE__": tackle probability = %f",
                  wm.self().tackleProbability() );


    //
    // set default neck action
    //
    agent->setNeckAction( new Neck_TurnToBall() );


    //
    // if catchable, do catch
    //
    const double max_self_error = 0.5;
    const double max_ball_error = 0.5;

    const bool ball_cachable_if_reach
        = ( wm.self().goalie()
            && ( wm.gameMode().type() == GameMode::PlayOn
                 || wm.gameMode().type() == GameMode::PenaltyTaken_ )
            && wm.time().cycle()
            >= agent->effector().getCatchTime().cycle()
            + ServerParam::i().catchBanCycle()
            && wm.ball().pos().x < ( param.ourPenaltyAreaLineX()
                                     + param.ballSize() * 2
                                     - max_ball_error )
            && wm.ball().pos().absY()  < ( param.penaltyAreaHalfWidth()
                                           + param.ballSize() * 2
                                           - max_ball_error )
            && wm.self().pos().x < param.ourPenaltyAreaLineX() - max_self_error
            && wm.self().pos().absY() < param.penaltyAreaHalfWidth() - max_self_error
            && ( ! wm.existKickableTeammate() || wm.existKickableOpponent() ) );


    // if catchable situation
    if ( ball_cachable_if_reach
         && wm.time().cycle() > wm.self().catchTime().cycle() + param.catchBanCycle()
         && wm.ball().distFromSelf() < wm.self().catchableArea() - 0.01 )
    {
        // if near opponent is not exist, catch the ball,
        // otherwise don't catch it.
        //
        // (this check is for avoiding back pass foul)
        if ( penalty_kick_mode
             || ( wm.ball().pos().x <= 40.0
                  || ! wm.getPlayerCont( new AndPlayerPredicate
                                         ( new OpponentOrUnknownPlayerPredicate( wm ),
                                           new PointNearPlayerPredicate( wm.self().pos(),
                                                                         5.0 ) ) ).empty() )
             )
        {
            return agent->doCatch();
        }
    }


    //
    // if kickable, do kick
    //
    if ( wm.self().isKickable() )
    {
        if ( ! wm.existKickableTeammate()
             || wm.teammatesFromBall().front()->distFromBall() > wm.ball().distFromSelf() )
        {
            return this->doKick( agent );
        }
    }


    agent->debugClient().addMessage( "ball dist=%f", wm.ball().distFromSelf() );


    //
    // set parameters
    //
    const double goal_half_width = ServerParam::i().goalHalfWidth();

    const Vector2D goal_center = translateXVec( reverse_x,
                                                ServerParam::i().ourTeamGoalPos() );
    const Vector2D wide_goal_left_post( goal_center.x,
                                        translateX( reverse_x,
                                                    +( goal_half_width + 1.0 ) ) );
    const Vector2D wide_goal_right_post( goal_center.x,
                                         translateX( reverse_x,
                                                     -( goal_half_width + 1.0 ) ) );


    bool is_shoot_ball;

    is_shoot_ball = ( ( (wide_goal_left_post - wm.ball().pos() ).th()
                        - wm.ball().vel().th() ).degree() < 0
                      && ( ( wide_goal_right_post - wm.ball().pos() ).th()
                           - wm.ball().vel().th() ).degree() > 0 );

    dlog.addText( Logger::TEAM,
                  __FILE__": is_shoot_ball = %s",
                  ( is_shoot_ball ? "true" : "false" ) );

#ifdef VISUAL_DEBUG
    if ( is_shoot_ball )
    {
        agent->debugClient().addLine( wm.self().pos()
                                      + Vector2D( -2.0, -2.0 ),
                                      wm.self().pos()
                                      + Vector2D( -2.0, +2.0 ) );
    }
#endif


    //
    // tackle
    //
    double tackle_prob_threshold = 0.8;
    bool despair_mode = false;

    if ( is_shoot_ball
         && translateXIsLessEqualsTo
         ( reverse_x,
           translateX( reverse_x, intercept_point.x ),
           translateX( reverse_x, param.ourTeamGoalLineX() ) ) )
    {
        despair_mode = true;

#ifdef VISUAL_DEBUG
        agent->debugClient().addLine( Vector2D
                                      ( param.ourTeamGoalLineX() - 2.0,
                                        - param.goalHalfWidth() ),
                                      Vector2D
                                      ( param.ourTeamGoalLineX() - 2.0,
                                        + param.goalHalfWidth() ) );
#endif

        tackle_prob_threshold = EPS;

        Vector2D ball_next = wm.ball().inertiaPoint( 1 );
        bool next_step_goal = ( ball_next.x
                                > ServerParam::i().ourTeamGoalLineX()
                                + param.ballRand() );

        if ( next_step_goal )
        {
            dlog.addText( Logger::TEAM,
                          __FILE__": next step, ball is in goal [%f, %f]",
                          ball_next.x, ball_next.y );

            double next_tackle_prob_forward
                = getSelfNextTackleProbabilityWithDash( wm, + param.maxDashPower() );
            double next_tackle_prob_backword
                = getSelfNextTackleProbabilityWithDash( wm, param.minDashPower() );

            dlog.addText( Logger::TEAM,
                          __FILE__": next_tackle_prob_forward = %f",
                          next_tackle_prob_forward );
            dlog.addText( Logger::TEAM,
                          __FILE__": next_tackle_prob_backword = %f",
                          next_tackle_prob_backword );

            if ( next_tackle_prob_forward > wm.self().tackleProbability() )
            {
                dlog.addText( Logger::TEAM,
                              __FILE__": dash forward expect next tackle" );

                agent->doDash( param.maxDashPower() );

                return true;
            }
            else if ( next_tackle_prob_backword > wm.self().tackleProbability() )
            {
                dlog.addText( Logger::TEAM,
                              __FILE__": dash backward expect next tackle" );
                agent->doDash( param.minDashPower() );

                return true;
            }
        }

        if ( ! next_step_goal )
        {
            double next_tackle_prob_with_turn = getSelfNextTackleProbabilityWithTurn( wm );
            dlog.addText( Logger::TEAM,
                          __FILE__": next_tackle_prob_with_turn = %f",
                          next_tackle_prob_with_turn );
            if ( next_tackle_prob_with_turn > wm.self().tackleProbability() )
            {
                dlog.addText( Logger::TEAM,
                              __FILE__": turn to next ball position for tackling" );
                Body_TurnToPoint( ball_next, 1 ).execute( agent );
                return true;
            }
        }
    }

    if ( Bhv_BasicTackle( tackle_prob_threshold, 160.0 ).execute( agent ) )
    {
        agent->debugClient().addMessage( "Savior:Tackle(%f)",
                                         wm.self().tackleProbability() );
        return true;
    }


    const Rect2D shrinked_penalty_area
        ( Vector2D( param.ourTeamGoalLineX(),
                    - (param.penaltyAreaHalfWidth() - 1.0) ),
          Size2D( param.penaltyAreaLength() - 1.0,
                  (param.penaltyAreaHalfWidth() - 1.0) * 2.0 ) );


    //
    // chase ball
    //
    if ( despair_mode )
    {
#ifdef VISUAL_DEBUG
        agent->debugClient().addLine( wm.self().pos()
                                      + Vector2D( +2.0, -2.0 ),
                                      wm.self().pos()
                                      + Vector2D( +2.0, +2.0 ) );
#endif

        agent->debugClient().addMessage( "Savior:DespairChase" );
        if ( this->doChaseBall( agent ) )
        {
            return true;
        }
    }


    if ( wm.gameMode().type() == GameMode::PlayOn
         && ! wm.existKickableTeammate() )
    {
        if ( self_reach_cycle + 1 < opponent_reach_cycle
#ifdef INHIBIT_SIDE_AREA_CHASE
             && ( ( self_reach_cycle + 5 <= teammate_reach_cycle
                    || intercept_point.absY() < param.penaltyAreaHalfWidth() - 1.0 )
                  && shrinked_penalty_area.contains( intercept_point ) )
#endif
             && ( self_reach_cycle <= teammate_reach_cycle
                  || ( shrinked_penalty_area.contains( intercept_point )
                       && self_reach_cycle <= teammate_reach_cycle + 3 ) )
             )
        {
            agent->debugClient().addMessage( "Savior:Chase" );

            if ( this->doChaseBall( agent ) )
            {
                return true;
            }
        }

#if 1
        // 2009-12-13 akiyama
        if ( self_reach_cycle + 3 < opponent_reach_cycle
             && self_reach_cycle + 2 <= teammate_reach_cycle
             && intercept_point.x > param.ourPenaltyAreaLineX()
             && intercept_point.absY() < param.penaltyAreaHalfWidth() - 1.0 )
        {
            agent->debugClient().addMessage( "Savior:ChaseAggressive" );

            if ( this->doChaseBall( agent ) )
            {
                return true;
            }
        }
#endif
    }


    if ( penalty_kick_mode
         && ( self_reach_cycle + 1 < opponent_reach_cycle
#ifdef PENALTY_SHOOTOUT_BLOCK_IN_PENALTY_AREA
              && ( shrinked_penalty_area.contains( intercept_point )
                   || translateXIsLessEqualsTo
                   ( reverse_x,
                     intercept_point.x,
                     translateX( reverse_x, param.ourTeamGoalLineX() ) ) )

#endif
              ) )
    {
        agent->debugClient().addMessage( "Savior:ChasePenaltyKickMode" );

        if ( Body_Intercept().execute( agent ) )
        {
            agent->setNeckAction( new Neck_TurnToBall() );

            return true;
        }
    }


    //
    // check ball
    //
    long ball_premise_accuracy = 2;
    bool brief_ball_check = false;

    if ( wm.self().pos().x >= -30.0 )
    {
        ball_premise_accuracy = 6;
        brief_ball_check = true;
    }
    else if ( opponent_reach_cycle >= 3 )
    {
        ball_premise_accuracy = 3;
        brief_ball_check = true;
    }

    if ( wm.ball().seenPosCount() > ball_premise_accuracy
         || ( brief_ball_check
              && wm.ball().posCount() > ball_premise_accuracy ) )
    {
        agent->debugClient().addMessage( "Savior:FindBall" );
        dlog.addText( Logger::TEAM,
                      __FILE__ ": find ball" );
        return Neck_TurnToBall().execute( agent );
    }


    //
    // set predicted ball pos
    //
    Vector2D ball_pos = getBoundPredictBallPos( wm, predict_step, 0.5 );
    Vector2D current_ball_pos = wm.ball().pos();


    //
    // positioning
    //
    bool in_region;
    bool dangerous;
    bool aggressive;
    bool emergent_advance;
    bool goal_line_positioning;

    Vector2D best_position = this->getBasicPosition
        ( agent,
          penalty_kick_mode,
          despair_mode,
          reverse_x,
          &in_region,
          &dangerous,
          &aggressive,
          &emergent_advance,
          &goal_line_positioning );


#ifdef VISUAL_DEBUG
    if ( emergent_advance )
    {
        Vector2D hat_base_vec = ( best_position - wm.self().pos() ).normalizedVector();

        // draw arrow
        agent->debugClient().addLine( wm.self().pos() + hat_base_vec * 3.0,
                                      wm.self().pos() + hat_base_vec * 3.0
                                      + Vector2D::polar2vector
                                      ( 1.5, hat_base_vec.th() + 150.0 ) );
        agent->debugClient().addLine( wm.self().pos() + hat_base_vec * 3.0,
                                      wm.self().pos() + hat_base_vec * 3.0
                                      + Vector2D::polar2vector
                                      ( 1.5, hat_base_vec.th() - 150.0 ) );
        agent->debugClient().addLine( wm.self().pos() + hat_base_vec * 4.0,
                                      wm.self().pos() + hat_base_vec * 4.0
                                      + Vector2D::polar2vector
                                      ( 1.5, hat_base_vec.th() + 150.0 ) );
        agent->debugClient().addLine( wm.self().pos() + hat_base_vec * 4.0,
                                      wm.self().pos() + hat_base_vec * 4.0
                                      + Vector2D::polar2vector
                                      ( 1.5, hat_base_vec.th() - 150.0 ) );
    }

    if ( goal_line_positioning )
    {
        agent->debugClient().addLine( Vector2D
                                      ( param.ourTeamGoalLineX() - 1.0,
                                        - param.goalHalfWidth() ),
                                      Vector2D
                                      ( param.ourTeamGoalLineX() - 1.0,
                                        + param.goalHalfWidth() ) );
    }
#endif


#if 0
    //
    // attack to near ball
    //
    if ( ! dangerous
         && ball_cachable_if_reach
         && wm.ball().distFromSelf() < 5.0 )
    {
        agent->debugClient().addMessage( "Savior:Attack" );
        return this->doChaseBall( agent );
    }
#endif


#ifdef VISUAL_DEBUG
    agent->debugClient().addLine( wm.self().pos(), best_position );
#endif


    double max_position_error = 0.70;
    double goal_line_positioning_y_max_position_error = 0.3;
    double dash_power = param.maxDashPower();

    if ( aggressive )
    {
        if ( wm.ball().distFromSelf() >= 30.0 )
        {
            max_position_error
                = bound( 2.0, 2.0 + (wm.ball().distFromSelf() - 30.0) / 20.0, 3.0 );
        }
    }


    double diff = (best_position - wm.self().pos()).r();

    if ( wm.ball().distFromSelf() >= 30.0
         && wm.ball().pos().x >= -20.0
         && diff < 5.0
         && ! penalty_kick_mode )
    {
        dash_power = param.maxDashPower() * 0.7;

#ifdef VISUAL_DEBUG
        agent->debugClient().addLine
            ( wm.self().pos() + Vector2D( -1.0, +3.0 ),
              wm.self().pos() + Vector2D( +1.0, +3.0 ) );

        agent->debugClient().addLine
            ( wm.self().pos() + Vector2D( -1.0, -3.0 ),
              wm.self().pos() + Vector2D( +1.0, -3.0 ) );
#endif
    }

    if ( emergent_advance )
    {
        if ( diff >= 10.0 )
        {
            // for speedy movement
            max_position_error = 1.9;
        }
        else if ( diff >= 5.0 )
        {
            // for speedy movement
            max_position_error = 1.0;
        }

#ifdef VISUAL_DEBUG
        agent->debugClient().addCircle( wm.self().pos(), 3.0 );
#endif
    }
    else if ( goal_line_positioning )
    {
        if ( std::fabs( wm.self().pos().x - best_position.x ) > 5.0 )
        {
            max_position_error = 2.0;
            goal_line_positioning_y_max_position_error = 1.5;
        }

        if ( ball_pos.x > -10.0
             && current_ball_pos.x > +10.0
             && wm.defenseLineX() > -20.0
             && std::fabs( wm.self().pos().x - best_position.x ) < 5.0 )
        {
            // safety, save stamina
            dash_power = param.maxDashPower() * 0.5;
            max_position_error = 2.0;
            goal_line_positioning_y_max_position_error = 1.5;
        }
    }
    else if ( dangerous )
    {
        if ( diff >= 10.0 )
        {
            // for speedy movement
            max_position_error = 1.9;
        }
        else if ( diff >= 5.0 )
        {
            // for speedy movement
            max_position_error = 1.0;
        }
    }
    else
    {
        if ( diff >= 10.0 )
        {
            // for speedy movement
            max_position_error = 1.5;
        }
        else if ( diff >= 5.0 )
        {
            // for speedy movement
            max_position_error = 1.0;
        }
    }

    if ( despair_mode )
    {
        if ( max_position_error < 1.5 )
        {
            max_position_error = 1.5;

            dlog.addText( Logger::TEAM,
                          __FILE__": Savior: despair_mode, "
                          "position error changed to %f",
                          max_position_error );
        }
    }


    dlog.addText( Logger::TEAM,
                  __FILE__": Savior: positioning target = [%f, %f], "
                  "max_position_error = %f, dash_power = %f",
                  best_position.x, best_position.y,
                  max_position_error, dash_power );

    bool force_back_dash = false;

    if ( goal_line_positioning
         && ! despair_mode )
    {
        if ( doGoalLinePositioning( agent ,
                                    best_position,
                                    2.0,
                                    max_position_error,
                                    goal_line_positioning_y_max_position_error,
                                    dash_power,
                                    reverse_x ) )
        {
            return true;
        }
    }
    else
    {
        if ( Body_GoalieGoToPoint( best_position,
                                   max_position_error,
                                   dash_power,
                                   true,
                                   force_back_dash,
                                   despair_mode ).execute( agent ) )
        {
            agent->debugClient().addMessage( "PenGoalie:Positioning" );
#ifdef VISUAL_DEBUG
            agent->debugClient().setTarget( best_position );
            agent->debugClient().addCircle( best_position, max_position_error );
#endif
            dlog.addText( Logger::TEAM,
                          __FILE__ ": go to (%.2f %.2f) error=%.3f  dash_power=%.1f  force_back=%d",
                          best_position.x, best_position.y,
                          max_position_error,
                          dash_power,
                          static_cast< int >( force_back_dash ) );
            return true;
        }
    }


    //
    // emergency stop
    //
    if ( opponent_reach_cycle <= 1
         && (current_ball_pos - wm.self().pos()).r() < 10.0
         && wm.self().vel().r() >= 0.05 )
    {
        if ( Body_StopDash( true ).execute( agent ) )
        {
            agent->debugClient().addMessage( "Savior:EmergemcyStop" );
            dlog.addText( Logger::TEAM,
                          __FILE__ ": emergency stop" );
            return true;
        }
    }


    //
    // go to position with minimum error
    //
    const PlayerObject * firstAccessOpponent;
    firstAccessOpponent = wm.interceptTable()->fastestOpponent();

    if ( (firstAccessOpponent
          && ( firstAccessOpponent->pos() - wm.self().pos() ).r() >= 5.0)
         && opponent_reach_cycle >= 3 )
    {
        if ( goal_line_positioning
             && ! despair_mode )
        {
            if ( doGoalLinePositioning( agent,
                                        best_position,
                                        2.0,
                                        0.3,
                                        0.3,
                                        dash_power,
                                        reverse_x ) )
            {
                return true;
            }
        }
        else
        {
            if ( Body_GoalieGoToPoint( best_position,
                                       0.3,
                                       dash_power,
                                       true,
                                       false,
                                       despair_mode ).execute( agent ) )
            {
                agent->debugClient().addMessage( "Savior:TunePositioning" );
#ifdef VISUAL_DEBUG
                agent->debugClient().setTarget( best_position );
                agent->debugClient().addCircle( best_position, 0.3 );
#endif
                dlog.addText( Logger::TEAM,
                              __FILE__ ": go to position with minimum error. target=(%.2f %.2f) dash_power=%.1f",
                              best_position.x, best_position.y,
                              dash_power );
                return true;
            }
        }
    }


    //
    // stop
    //
    if ( wm.self().vel().rotatedVector( - wm.self().body() ).absX() >= 0.01 )
    {
        if ( Body_StopDash( true ).execute( agent ) )
        {
            agent->debugClient().addMessage( "Savior:Stop" );
            dlog.addText( Logger::TEAM,
                          __FILE__ ": stop. line=%d", __LINE__ );
            return true;
        }
    }


    //
    // turn body angle against ball
    //
    const Vector2D final_body_pos = wm.self().inertiaFinalPoint();

    AngleDeg target_body_angle = 0.0;

    if ( ((ball_pos - final_body_pos).r() <= 5.0
          && (wm.existKickableOpponent() || wm.existKickableTeammate()))
         || goal_line_positioning )
    {
        if ( goal_line_positioning )
        {
            target_body_angle = sign( final_body_pos.y ) * 90.0;
        }
        else
        {
            AngleDeg diff_body_angle = 0.0;

            if ( final_body_pos.y > 0.0 )
            {
                diff_body_angle = + 90.0;
            }
            else
            {
                diff_body_angle = - 90.0;
            }

            if ( translateXIsLessEqualsTo( reverse_x, final_body_pos.x, -45.0 ) )
            {
                diff_body_angle *= -1.0;
            }

            target_body_angle = (ball_pos - final_body_pos).th()
                + diff_body_angle;
        }
    }

    target_body_angle = this->translateTheta( reverse_x, target_body_angle );

    if ( ( ! penalty_kick_mode
           || ( target_body_angle - wm.self().body() ).abs() > 10.0 )
         && Body_TurnToAngle( target_body_angle ).execute( agent ) )
    {
        agent->debugClient().addMessage( "Savior:TurnTo%.0f",
                                         target_body_angle.degree() );
        dlog.addText( Logger::TEAM,
                      __FILE__ ": turn to angle %.1f",
                      target_body_angle.degree() );
        return true;
    }


#if ! defined( PENALTY_SHOOTOUT_BLOCK_IN_PENALTY_AREA )
    if ( penalty_kick_mode )
    {
        double dash_dir;

        if ( wm.self().body().degree() > 0.0 )
        {
            dash_dir = this->translateTheta( reverse_x, -90.0 ).degree();
        }
        else
        {
            dash_dir = this->translateTheta( reverse_x, +90.0 ).degree();
        }

        agent->doDash( param.maxPower(), dash_dir );
        agent->debugClient().addMessage( "Savior:SideChase" );
        dlog.addText( Logger::TEAM,
                      __FILE__ ": side chase" );
        return true;
    }
#endif


    if ( Body_TurnToAngle( target_body_angle ).execute( agent ) )
    {
        agent->debugClient().addMessage( "Savior:AdjustTurnTo%.0f",
                                         target_body_angle.degree() );
        dlog.addText( Logger::TEAM,
                      __FILE__ ": adjust turn to angle %.1f",
                      target_body_angle.degree() );
        return true;
    }


    agent->debugClient().addMessage( "Savior:NoAction" );
    dlog.addText( Logger::TEAM,
                  __FILE__ ": no action" );
    agent->doTurn( 0.0 );
    return true;
}

bool
Bhv_PenaltyShootoutsGoalie::doGoalLinePositioning( PlayerAgent * agent,
                                                   const Vector2D & target_position,
                                                   double low_priority_x_position_error,
                                                   double max_x_position_error,
                                                   double max_y_position_error,
                                                   double dash_power,
                                                   bool reverse_x )
{
    const WorldModel & wm = agent->world();
    const double dist = target_position.dist( wm.self().pos() );
    const double x_diff = std::fabs( target_position.x - wm.self().pos().x );
    const double y_diff = std::fabs( target_position.y - wm.self().pos().y );

    if ( x_diff < max_x_position_error
         && y_diff < max_y_position_error )
    {
        return false;
    }

    Vector2D p = target_position;

#ifdef USE_GO_TO_POINT_2008
    const bool use_back_dash = ( translateXIsLessEqualsTo
                                 ( reverse_x,
                                   agent->world().ball().pos().x,
                                   -20.0 ) );
#endif

    const bool x_near = ( x_diff < max_x_position_error );
    const bool y_near = ( y_diff < max_y_position_error );


#ifdef VISUAL_DEBUG
    agent->debugClient().addRectangle( Rect2D::from_center
                                       ( target_position,
                                         max_x_position_error * 2.0,
                                         max_y_position_error * 2.0 ) );
#endif

    dlog.addText( Logger::TEAM,
                  __FILE__": (doGoalLinePositioning): "
                  "max_position_error = %f, "
                  "enlarged_max_position_error = %f, "
                  "dist = %f, "
                  "x_diff = %f, y_diff = %f, "
                  "x_near = %s, y_near = %s",
                  max_x_position_error,
                  max_y_position_error,
                  dist,
                  x_diff, y_diff,
                  ( x_near ? "true" : "false" ),
                  ( y_near ? "true" : "false" ) );

    if ( x_diff > low_priority_x_position_error )
    {
#ifdef USE_GO_TO_POINT_2008
        if ( Body_GoToPoint2008( p,
                                 std::min( max_x_position_error,
                                           max_y_position_error ),
                                 dash_power,
                                 use_back_dash,
                                 false ).execute( agent ) )
#else
            if ( Body_GoToPoint( p,
                                 std::min( max_x_position_error,
                                           max_y_position_error ),
                                 dash_power ).execute( agent ) )
#endif
            {
                agent->debugClient().addMessage( "Savior:GoalLinePositioning:normal" );
#ifdef VISUAL_DEBUG
                agent->debugClient().setTarget( p );
#endif
                dlog.addText( Logger::TEAM,
                              __FILE__ ": go to (%.2f %.2f) error=%.3f dash_power=%.1f",
                              p.x, p.y,
                              std::min( max_x_position_error,
                                        max_y_position_error ),
                              dash_power );

                return true;
            }
    }
    else if ( ! y_near )
    {
        p.assign( wm.self().pos().x, p.y );

#ifdef USE_GO_TO_POINT_2008
        if ( Body_GoToPoint2008( p,
                                 std::min( max_x_position_error,
                                           max_y_position_error ),
                                 dash_power,
                                 use_back_dash,
                                 false ).execute( agent ) )
#else
            if ( Body_GoToPoint( p,
                                 std::min( max_x_position_error,
                                           max_y_position_error ),
                                 dash_power ).execute( agent ) )
#endif
            {
                agent->debugClient().addMessage( "Savior:GoalLinePositioning:AdjustY" );
#ifdef VISUAL_DEBUG
                agent->debugClient().setTarget( p );
#endif
                dlog.addText( Logger::TEAM,
                              __FILE__ ": go to (%.2f %.2f) error=%.3f dash_power=%.1f",
                              p.x, p.y, low_priority_x_position_error, dash_power );
                return true;
            }
    }
    else
    {
        double dir_target;

        if ( wm.self().body().degree() > 0.0 )
        {
            dir_target = this->translateTheta( reverse_x, +90.0 ).degree();
        }
        else
        {
            dir_target = this->translateTheta( reverse_x, -90.0 ).degree();
        }

        if ( std::fabs( wm.self().body().degree() - dir_target ) > 2.0 )
        {
            if ( Body_TurnToAngle( dir_target ).execute( agent ) )
            {
                return true;
            }
        }


        double side_dash_dir;

        if ( wm.self().body().degree() > 0.0 )
        {
            side_dash_dir = this->translateTheta( reverse_x, +90.0 ).degree();
        }
        else
        {
            side_dash_dir = this->translateTheta( reverse_x, -90.0 ).degree();
        }

        if ( wm.self().pos().x < target_position.x )
        {
            side_dash_dir *= -1.0;
        }

        const double side_dash_rate = wm.self().dashRate() * ServerParam::i().dashDirRate( side_dash_dir );

        double side_dash_power = x_diff / std::max( side_dash_rate, EPS );
        side_dash_power = std::min( side_dash_power, dash_power );

        agent->doDash( side_dash_power, side_dash_dir );
        agent->debugClient().addMessage( "Savior:GoalLinePositioning:SideDash" );
        dlog.addText( Logger::TEAM,
                      __FILE__ ": goal line posisitioning side dash" );

        return true;
    }

    return false;
}


bool
Bhv_PenaltyShootoutsGoalie::doKick( PlayerAgent * agent )
{
    //
    // if exist near opponent, clear the ball
    //
    Body_ClearBall().execute( agent );
    agent->setNeckAction( new Neck_TurnToBall() );
    agent->debugClient().addMessage( "clear ball" );
    dlog.addText( Logger::TEAM,
                  __FILE__": clear ball" );
    return true;
}

Vector2D
Bhv_PenaltyShootoutsGoalie::getBasicPosition( PlayerAgent * agent,
                                              bool penalty_kick_mode,
                                              bool despair_mode,
                                              bool reverse_x,
                                              bool * in_region,
                                              bool * dangerous,
                                              bool * aggressive,
                                              bool * emergent_advance,
                                              bool * goal_line_positioning ) const
{
    if ( in_region )
    {
        *in_region = false;
    }

    if ( dangerous )
    {
        *dangerous = false;
    }

    if ( aggressive )
    {
        *aggressive = false;
    }

    if ( emergent_advance )
    {
        *emergent_advance = false;
    }

    if ( goal_line_positioning )
    {
        *goal_line_positioning = false;
    }

    const WorldModel & wm = agent->world();
    const ServerParam & param = ServerParam::i();

    const int teammate_reach_cycle = wm.interceptTable()->teammateReachCycle();
    const int opponent_reach_cycle = wm.interceptTable()->opponentReachCycle();
    const int self_reach_cycle = wm.interceptTable()->selfReachCycle();
    int predict_step = std::min( opponent_reach_cycle,
                                 std::min( teammate_reach_cycle,
                                           self_reach_cycle ) );

    const Vector2D self_pos = wm.self().pos();
    const Vector2D ball_pos = getBoundPredictBallPos( wm, predict_step, 0.5 );

    dlog.addText( Logger::TEAM,
                  __FILE__": (getBasicPosition) ball predict pos = (%f,%f)",
                  ball_pos.x, ball_pos.y );

    Vector2D positioning_base_pos = ball_pos;

#if 0
    const AbstractPlayerObject * holder = wm.interceptTable()->fastestOpponent();

    if ( holder )
    {
        double ball_rate = 3.0 / 4.0;

        positioning_base_pos = ball_pos * ball_rate
            + holder->pos() * (1.0 - ball_rate);
    }
#endif


    Rect2D penalty_area;
    Rect2D conservative_area;
    Vector2D goal_pos;

    if ( reverse_x )
    {
        penalty_area.assign( Vector2D( param.theirPenaltyAreaLineX(),
                                       - param.penaltyAreaHalfWidth() ),
                             Size2D( param.penaltyAreaLength(),
                                     param.penaltyAreaWidth() ) );

#if defined( OLD_GOAL_LINE_POSITINING_CONDITION )
        conservative_area.assign( Vector2D( param.theirPenaltyAreaLineX() - 15.0,
                                            - param.penaltyAreaHalfWidth() ),
                                  Size2D( param.penaltyAreaLength() + 15.0,
                                          param.penaltyAreaWidth() ) );
#else
        conservative_area.assign( Vector2D( param.theirPenaltyAreaLineX() - 15.0,
                                            - param.pitchHalfWidth() ),
                                  Size2D( param.penaltyAreaLength() + 15.0,
                                          param.pitchWidth() ) );
#endif

        goal_pos = ServerParam::i().theirTeamGoalPos();
    }
    else
    {
        penalty_area.assign( Vector2D( param.ourTeamGoalLineX(),
                                       - param.penaltyAreaHalfWidth() ),
                             Size2D( param.penaltyAreaLength(),
                                     param.penaltyAreaWidth() ) );

#if defined( OLD_GOAL_LINE_POSITINING_CONDITION )
        conservative_area.assign( Vector2D( param.ourTeamGoalLineX(),
                                            - param.penaltyAreaHalfWidth() ),
                                  Size2D( param.penaltyAreaLength() + 15.0,
                                          param.penaltyAreaWidth() ) );
#else
        conservative_area.assign( Vector2D( param.ourTeamGoalLineX(),
                                            - param.pitchHalfWidth() ),
                                  Size2D( param.penaltyAreaLength() + 15.0,
                                          param.pitchWidth() ) );
#endif


        goal_pos = ServerParam::i().ourTeamGoalPos();
    }

    const size_t num_teammates_in_conservative_area
        = wm.getPlayerCont( new AndPlayerPredicate
                            ( new TeammatePlayerPredicate( wm ),
                              new ContainsPlayerPredicate< Rect2D >
                              ( conservative_area ) ) ).size();

    const bool ball_is_in_conservative_area
        = conservative_area.contains( wm.ball().pos() );

    dlog.addText( Logger::TEAM,
                  __FILE__": (getBasicPosition) number of teammates in conservative area is %d",
                  (int)num_teammates_in_conservative_area );

    dlog.addText( Logger::TEAM,
                  __FILE__": (getBasicPosition) ball in area = %d",
                  (int)(conservative_area.contains( wm.ball().pos() )) );

    double base_dist = 14.0;

    dlog.addText( Logger::TEAM,
                  __FILE__": (getBasicPosition) basic base dist = %f",
                  base_dist );

    dlog.addText( Logger::TEAM,
                  __FILE__": (getBasicPosition) defenseLineX = %f",
                  wm.defenseLineX() );

    bool goal_line_positioning_flag = false;

#if defined( OLD_GOAL_LINE_POSITINING_CONDITION )
    dlog.addText( Logger::TEAM,
                  __FILE__": (getBasicPosition) ball dir from goal = %f",
                  getDirFromGoal( ball_pos ).degree() );

#if 1
    bool is_side_angle = ( getDirFromGoal( ball_pos ).abs() > 30.0 );
# elif 0
    bool is_side_angle = ( getDirFromGoal( ball_pos ).abs() > 35.0 );
# elif 1
    bool is_side_angle = ( getDirFromGoal( ball_pos ).abs() > 40.0 );
#else
    bool is_side_angle = ( getDirFromGoal( ball_pos ).abs() > 45.0 );
#endif

#ifdef FORCE_GOAL_LINE_POSITIONONG
    if ( ((! is_side_angle && wm.defenseLineX() < -15.0)
          || (wm.defenseLineX() < -15.0 && wm.defenseLineX() > -40.0))
         && ! penalty_kick_mode )
    {
        goal_line_positioning_flag = true;

        if ( goal_line_positioning )
        {
            *goal_line_positioning = goal_line_positioning_flag;
        }

        dlog.addText( Logger::TEAM,
                      __FILE__": (getBasicPosition) "
                      "force goal line positioning" );
    }
    else
#endif
        // many teammates are in penalty area, do back positioning
        if ( ! is_side_angle
             && penalty_area.contains( wm.ball().pos() )
             && wm.getPlayerCont
             ( new AndPlayerPredicate
               ( new TeammatePlayerPredicate( wm ),
                 new ContainsPlayerPredicate< Rect2D >
                 ( penalty_area ) ) ).size() >= 3 )
        {
            base_dist = std::min( base_dist,
                                  std::max( wm.defenseLineX()
                                            - param.ourPenaltyAreaLineX() - 2.0,
                                            2.0 ) );
            dlog.addText( Logger::TEAM,
                          __FILE__": (getBasicPosition) back positioning, "
                          "base dist = %f", base_dist );

            if ( std::fabs( ball_pos.y ) < param.goalHalfWidth() + 5.0 )
            {
                goal_line_positioning_flag = true;

                if ( goal_line_positioning )
                {
                    *goal_line_positioning = goal_line_positioning_flag;
                }
            }
        }
#else
    if ( isGoalLinePositioningSituation( wm, ball_pos, penalty_kick_mode ) )
    {
        goal_line_positioning_flag = true;

        if ( goal_line_positioning )
        {
            *goal_line_positioning = goal_line_positioning_flag;
        }
    }
#endif

    // if ball is not in penalty area and teammates are in penalty area
    // (e.g. our clear ball succeded), do back positioning
    if ( ! ball_is_in_conservative_area
         && num_teammates_in_conservative_area >= 2 )
    {
#ifdef DO_GOAL_LINE_POSITIONONG
        dlog.addText( Logger::TEAM,
                      __FILE__": goal line positioning" );

        goal_line_positioning_flag = true;

        if ( goal_line_positioning )
        {
            *goal_line_positioning = goal_line_positioning_flag;
        }
#else
        if ( ( goal_pos - ball_pos ).r() > 20.0 )
        {
            base_dist = std::min( base_dist,
                                  5.0 + ( ( goal_pos - ball_pos ).r() - 20.0 ) * 0.1 );
        }
        else
        {
            base_dist = 5.0;
        }
#endif
    }

    // aggressive positioning
#if defined( OLD_AGGRESSIVE_POSITIONING )
# if 0
    const double additional_forward_positioning_max = 18.0;
# else
    const double additional_forward_positioning_max = 16.0;
# endif
#else
    const double additional_forward_positioning_max = 25.0;
#endif

    const double emergent_advance_dist = (goal_pos - ball_pos).r() - 3.0;

    bool emergent_advance_mode = false;
#if defined( DO_EMERGENT_ADVANCE )
    const AbstractPlayerObject * fastestOpponent = wm.interceptTable()->fastestOpponent();
    // if opponent player will be have the ball at backward of our
    // defense line, do 1-on-1 aggressive defense
#if defined( OLD_EMERGENT_ADVANCE )
    if ( ball_pos.x <= -20.0
         && ball_pos.x >= param.ourPenaltyAreaLineX() + 5.0
         && opponent_reach_cycle < teammate_reach_cycle
         && ball_pos.x <= wm.defenseLineX() + 5.0
         && (getDirFromGoal( ball_pos )
             - getDirFromGoal( wm.self().inertiaPoint(1) )).abs()
         < EMERGENT_ADVANCE_DIR_DIFF_THRESHOLD
         && canShootFrom( wm, ball_pos, 20 )
         && ! wm.existKickableOpponent()
         && ! penalty_kick_mode
         && fastestOpponent
         && wm.getPlayerCont
         ( new AndPlayerPredicate
           ( new TeammatePlayerPredicate( wm ),
             new XCoordinateBackwardPlayerPredicate
             ( fastestOpponent->pos().x ),
             new YCoordinatePlusPlayerPredicate
             ( fastestOpponent->pos().y - 1.0 ),
             new YCoordinateMinusPlayerPredicate
             ( fastestOpponent->pos().y + 1.0 ) ) ).empty()
         && emergent_advance_dist < 30.0 )
#else
        if ( ball_pos.x >= param.ourPenaltyAreaLineX() + 5.0
             && opponent_reach_cycle < teammate_reach_cycle
             && ball_pos.x <= wm.defenseLineX() + 5.0
             && (getDirFromGoal( ball_pos )
                 - getDirFromGoal( wm.self().inertiaPoint(1) )).abs()
             < EMERGENT_ADVANCE_DIR_DIFF_THRESHOLD
             && canShootFrom( wm, ball_pos, 20 )
             && ! penalty_kick_mode
             && fastestOpponent
             && wm.getPlayerCont
             ( new AndPlayerPredicate
               ( new TeammatePlayerPredicate( wm ),
                 new XCoordinateBackwardPlayerPredicate
                 ( fastestOpponent->pos().x ),
                 new YCoordinatePlusPlayerPredicate
                 ( fastestOpponent->pos().y - 1.0 ),
                 new YCoordinateMinusPlayerPredicate
                 ( fastestOpponent->pos().y + 1.0 ) ) ).empty()
             && getDirFromGoal( ball_pos ).abs() < 20.0 )
#endif
        {
            dlog.addText( Logger::TEAM,
                          __FILE__": (getBasicPosition) emergent advance mode" );
            emergent_advance_mode = true;

            if ( emergent_advance )
            {
                *emergent_advance = true;
            }

            agent->debugClient().addMessage( "EmergentAdvance" );
        }
#endif


    if ( penalty_kick_mode )
    {
#if defined( PENALTY_SHOOTOUT_BLOCK_IN_PENALTY_AREA )
# if 0
        base_dist = std::min( (goal_pos - ball_pos).r() - 3.0,
                              base_dist );
# endif
#else
# if 0
        base_dist = (goal_pos - ball_pos).r() - 3.0;
# else
        base_dist = (goal_pos - ball_pos).r() - 4.0;
# endif

        if ( (wm.self().pos() - goal_pos).r() > base_dist )
        {
            base_dist = (wm.self().pos() - goal_pos).r();
        }
#endif

        dlog.addText( Logger::TEAM,
                      __FILE__": (getBasicPosition) penalty kick mode, "
                      "new base dist = %f", base_dist );
    }
    else if ( emergent_advance_mode )
    {
        base_dist = emergent_advance_dist;
    }
#if defined( OLD_AGGRESSIVE_POSITIONING )
    else if ( wm.defenseLineX() >= - additional_forward_positioning_max )
#else
    else if ( wm.defenseLineX() >= -20.0 )
#endif
    {
#if 0
        static const double AGGRESSIVE_POSITIONING_STAMINA_RATE_THRESHOLD = 0.85;
#else
        static const double AGGRESSIVE_POSITIONING_STAMINA_RATE_THRESHOLD = 0.6;
#endif
        if ( wm.self().stamina()
             >= ServerParam::i().staminaMax()
             * AGGRESSIVE_POSITIONING_STAMINA_RATE_THRESHOLD )
        {
            base_dist += std::min( wm.defenseLineX(), 0.0 )
                + additional_forward_positioning_max;

            dlog.addText( Logger::TEAM,
                          __FILE__ ": (getBasicPosition) aggressive positioning,"
                          " base_dist = %f",
                          base_dist );
            agent->debugClient().addMessage( "AggressivePositioniong" );

            if ( aggressive )
            {
                *aggressive = true;
            }
        }
    }
    else
    {
        if ( aggressive )
        {
            *aggressive = false;
        }
    }

    dlog.addText( Logger::TEAM,
                  __FILE__ ": (getBasicPosition) base dist = %f", base_dist );

    if ( in_region )
    {
        *in_region = true;
    }

    if ( dangerous )
    {
        *dangerous = false;
    }

    if ( emergent_advance )
    {
        *emergent_advance = emergent_advance_mode;
    }

    if ( ! reverse_x )
    {
        return getBasicPositionFromBall( agent,
                                         positioning_base_pos,
                                         base_dist,
                                         wm.self().pos(),
                                         penalty_kick_mode,
                                         despair_mode,
                                         in_region,
                                         dangerous,
                                         emergent_advance_mode,
                                         goal_line_positioning );
    }
    else
    {
        Vector2D reversed_ball_pos( - positioning_base_pos.x,
                                    positioning_base_pos.y );

        Vector2D self_pos( - wm.self().pos().x, wm.self().pos().y );

        Vector2D result_pos = getBasicPositionFromBall
            ( agent,
              reversed_ball_pos,
              base_dist,
              self_pos,
              penalty_kick_mode,
              despair_mode,
              in_region,
              dangerous,
              emergent_advance_mode,
              goal_line_positioning );

        return Vector2D( - result_pos.x, result_pos.y );
    }
}

Vector2D
Bhv_PenaltyShootoutsGoalie::getBasicPositionFromBall( PlayerAgent * agent,
                                                      const Vector2D & ball_pos,
                                                      double base_dist,
                                                      const Vector2D & self_pos,
                                                      bool no_danger_angle,
                                                      bool despair_mode,
                                                      bool * in_region,
                                                      bool * dangerous,
                                                      bool emergent_advance,
                                                      bool * goal_line_positioning ) const
{
    static const double GOAL_LINE_POSITIONINIG_GOAL_X_DIST = 1.5;
    static const double MINIMUM_GOAL_X_DIST = 1.0;

    const WorldModel & wm = agent->world();

    const double goal_line_x = ServerParam::i().ourTeamGoalLineX();
    const double goal_half_width = ServerParam::i().goalHalfWidth();

    const double alpha = std::atan2( goal_half_width, base_dist );

    const double min_x = (goal_line_x + MINIMUM_GOAL_X_DIST);

    const Vector2D goal_center = ServerParam::i().ourTeamGoalPos();
    const Vector2D goal_left_post( goal_center.x, +goal_half_width );
    const Vector2D goal_right_post( goal_center.x, -goal_half_width );

    Line2D line_1( ball_pos, goal_left_post );
    Line2D line_2( ball_pos, goal_right_post );

    AngleDeg line_dir = AngleDeg::normalize_angle
        ( AngleDeg::normalize_angle
          ( (goal_left_post - ball_pos).th().degree()
            + (goal_right_post - ball_pos).th().degree() )
          / 2.0 );

    Line2D line_m( ball_pos, line_dir );

    Line2D goal_line( goal_left_post, goal_right_post );

    if ( ! emergent_advance
         && ( goal_line_positioning != static_cast<bool *>(0)
              && *goal_line_positioning ) )
    {
        return this->getGoalLinePositioningTarget
            ( agent, wm, GOAL_LINE_POSITIONINIG_GOAL_X_DIST,
              ball_pos, despair_mode );
    }


    Vector2D p = goal_line.intersection( line_m );

    if ( ! p.isValid() )
    {
        if ( ball_pos.x > 0.0 )
        {
            return Vector2D( min_x, goal_left_post.y );
        }
        else if ( ball_pos.x < 0.0 )
        {
            return Vector2D( min_x, goal_right_post.y );
        }
        else
        {
            return Vector2D( min_x, goal_center.y );
        }
    }

    if ( in_region != static_cast<bool *>(0) )
    {
        *in_region = false;
    }

    // angle -> dist
    double dist_from_goal = (line_1.dist(p) + line_2.dist(p)) / 2.0
        / std::sin(alpha);

    if ( dist_from_goal <= goal_half_width )
    {
        dist_from_goal = goal_half_width;
    }

    if ( (ball_pos - p).r() + 1.5 < dist_from_goal )
    {
        dist_from_goal = (ball_pos - p).r() + 1.5;
    }

    AngleDeg current_position_line_dir = (self_pos - p).th();

    AngleDeg position_error = line_dir - current_position_line_dir;

    const double danger_angle = 21.0;

    dlog.addText( Logger::TEAM,
                  __FILE__": position angle error = %f",
                  position_error.abs() );

    agent->debugClient().addMessage( "angleError = %f", position_error.abs() );

    if ( ! no_danger_angle
         && position_error.abs() > danger_angle
         && (ball_pos - goal_center).r() < 50.0
         && (p - self_pos).r() > 5.0 )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": position angle error too big,"
                      " dangerous" );

        dist_from_goal *= (1.0 - (position_error.abs() - danger_angle)
                           / (180.0 - danger_angle) ) / 3.0;

        if ( dist_from_goal < goal_half_width - 1.0 )
        {
            dist_from_goal = goal_half_width - 1.0;
        }

        if ( dangerous != static_cast<bool *>(0) )
        {
            *dangerous = true;
        }

#if defined( OLD_GOAL_LINE_POSITINING_CONDITION )
        if ( self_pos.x < -45.0
             && ( ball_pos - self_pos ).r() < 20.0 )
        {
            dlog.addText( Logger::TEAM,
                          "change to goal line positioning" );

            if ( goal_line_positioning != static_cast<bool *>(0) )
            {
                *goal_line_positioning = true;
            }

            return this->getGoalLinePositioningTarget
                ( agent, wm, GOAL_LINE_POSITIONINIG_GOAL_X_DIST,
                  ball_pos, despair_mode );
        }
#endif
    }
#if 0
    else if ( ! no_danger_angle
              && position_error.abs() > 5.0
              && (p - self_pos).r() > 5.0 )
#else
    else if ( position_error.abs() > 5.0
              && (p - self_pos).r() > 5.0 )
#endif
    {
        double current_position_dist = (p - self_pos).r();

        if ( dist_from_goal < current_position_dist )
        {
            dist_from_goal = current_position_dist;
        }

        dist_from_goal = (p - self_pos).r();

        dlog.addText( Logger::TEAM,
                      __FILE__": position angle error big,"
                      " positioning has changed to projection point, "
                      "new dist_from_goal = %f", dist_from_goal );

        if ( dangerous != static_cast<bool *>(0) )
        {
            *dangerous = true;
        }

#if defined( OLD_GOAL_LINE_POSITINING_CONDITION )
        if ( self_pos.x < -45.0
             && ( ball_pos - self_pos ).r() < 20.0 )
        {
            dlog.addText( Logger::TEAM,
                          "change to goal line positioning" );

            if ( goal_line_positioning != static_cast<bool *>(0) )
            {
                *goal_line_positioning = true;
            }

            return this->getGoalLinePositioningTarget
                ( agent, wm, GOAL_LINE_POSITIONINIG_GOAL_X_DIST,
                  ball_pos, despair_mode );
        }
#endif

        {
            const Vector2D r = p + ( ball_pos - p ).setLengthVector( dist_from_goal );

#ifdef VISUAL_DEBUG
            agent->debugClient().addLine( p, r );
            agent->debugClient().addLine( self_pos, r );
#endif
        }
    }
    else if ( ! no_danger_angle
              && position_error.abs() > 10.0 )
    {
        dist_from_goal = std::min( dist_from_goal, 14.0 );

        dlog.addText( Logger::TEAM,
                      __FILE__": position angle error is big,"
                      "new dist_from_goal = %f", dist_from_goal );
    }


    Vector2D result = p + ( ball_pos - p ).setLengthVector( dist_from_goal );

    if ( result.x < min_x )
    {
        result.assign( min_x, result.y );
    }

    dlog.addText( Logger::TEAM,
                  __FILE__": positioning target = [%f,%f]",
                  result.x, result.y );

    return result;
}




Vector2D
Bhv_PenaltyShootoutsGoalie::getGoalLinePositioningTarget( PlayerAgent * agent,
                                                          const WorldModel & wm,
                                                          const double goal_x_dist,
                                                          const Vector2D & ball_pos,
                                                          bool despair_mode ) const
{
    const double goal_line_x = ServerParam::i().ourTeamGoalLineX();

    const double goal_half_width = ServerParam::i().goalHalfWidth();
    const Vector2D goal_center = ServerParam::i().ourTeamGoalPos();
    const Vector2D goal_left_post( goal_center.x, +goal_half_width );
    const Vector2D goal_right_post( goal_center.x, -goal_half_width );

    AngleDeg line_dir = getDirFromGoal( ball_pos );

    Line2D line_m( ball_pos, line_dir );

    Line2D goal_line( goal_left_post, goal_right_post );

    Line2D target_line( goal_left_post
                        + Vector2D( goal_x_dist, 0.0 ),
                        goal_right_post
                        + Vector2D( goal_x_dist, 0.0 ) );

    if ( despair_mode )
    {
        double target_x = std::max( wm.self().inertiaPoint( 1 ).x,
                                    goal_line_x );

        Line2D positioniong_line( Vector2D( target_x, -1.0 ),
                                  Vector2D( target_x, +1.0 ) );

        if ( wm.ball().vel().r() < EPS )
        {
            return wm.ball().pos();
        }

        Line2D ball_line( ball_pos, wm.ball().vel().th() );

        Vector2D c = positioniong_line.intersection( ball_line );

        if ( c.isValid() )
        {
            return c;
        }
    }


    Vector2D p = target_line.intersection( line_m );

    if ( ! p.isValid() )
    {
        double target_y;

        if ( ball_pos.absY() > ServerParam::i().goalHalfWidth() )
        {
            target_y = sign( ball_pos.y )
                * std::min( ball_pos.absY(),
                            ServerParam::i().goalHalfWidth()
                            + 2.0 );
        }
        else
        {
            target_y = ball_pos.y * 0.8;
        }

        p = Vector2D( goal_line_x + goal_x_dist, target_y );
    }


#if 1
    const int teammate_reach_cycle = wm.interceptTable()->teammateReachCycle();
    const int opponent_reach_cycle = wm.interceptTable()->opponentReachCycle();
    int predict_step = std::min( opponent_reach_cycle, teammate_reach_cycle );

    Vector2D predict_ball_pos = wm.ball().inertiaPoint( predict_step );

    dlog.addText( Logger::TEAM,
                  __FILE__": predict_step = %d, "
                  "predict_ball_pos = [%f, %f]",
                  predict_step, predict_ball_pos.x, predict_ball_pos.y );

    bool can_shoot_from_predict_ball_pos = canShootFrom( wm, predict_ball_pos );

    dlog.addText( Logger::TEAM,
                  __FILE__": can_shoot_from_predict_ball_pos = %s",
                  ( can_shoot_from_predict_ball_pos? "true": "false" ) );

# if ! defined( DISABLE_GOAL_LINE_POSITIONING_SHIFT_TO_SHOOT_POINT )
    if ( can_shoot_from_predict_ball_pos )
    {
        Vector2D old_p = p;

        //
        // shift position to shoot point
        //
#if 1
        p += Vector2D( 0.0, predict_ball_pos.y - p.y ) * 0.2;
#else
        p += Vector2D( 0.0, predict_ball_pos.y - p.y ) * 0.3;
#endif

#ifdef VISUAL_DEBUG
        agent->debugClient().addCircle( old_p, 0.5 );

        agent->debugClient().addLine( p + Vector2D( -1.5, -1.5 ),
                                      p + Vector2D( +1.5, +1.5 ) );
        agent->debugClient().addLine( p + Vector2D( -1.5, +1.5 ),
                                      p + Vector2D( +1.5, -1.5 ) );

        agent->debugClient().addLine( old_p, p );
#endif

        dlog.addText( Logger::TEAM,
                      __FILE__": new shift position = [%f, %f]",
                      p.x, p.y );
    }
# else
    static_cast< void >( agent );
# endif

#endif


    if ( p.absY() > ServerParam::i().goalHalfWidth() + 2.0 )
    {
        p.assign( p.x,
                  sign( p.y )
                  * ( ServerParam::i().goalHalfWidth() + 2.0 ) );
    }

    return p;
}


/*-------------------------------------------------------------------*/
AngleDeg
Bhv_PenaltyShootoutsGoalie::getDirFromGoal( const Vector2D & pos )
{
    const double goal_half_width = ServerParam::i().goalHalfWidth();

    const Vector2D goal_center = ServerParam::i().ourTeamGoalPos();
    const Vector2D goal_left_post( goal_center.x, +goal_half_width );
    const Vector2D goal_right_post( goal_center.x, -goal_half_width );

    return AngleDeg( ( (pos - goal_left_post).th().degree()
                       + (pos - goal_right_post).th().degree() ) / 2.0 );
}


/*-------------------------------------------------------------------*/
Vector2D
Bhv_PenaltyShootoutsGoalie::getBallFieldLineCrossPoint( const Vector2D & ball_from,
                                                        const Vector2D & ball_to,
                                                        const Vector2D & p1,
                                                        const Vector2D & p2,
                                                        double field_back_offset )
{
    const Segment2D line( p1, p2 );
    const Segment2D ball_segment( ball_from, ball_to );

    Vector2D cross_point = ball_segment.intersection( line, true );

    if ( cross_point.isValid() )
    {
        if ( Vector2D( ball_to - ball_from ).r() <= EPS )
        {
            return cross_point;
        }

        return cross_point
            + Vector2D::polar2vector
            ( field_back_offset,
              Vector2D( ball_from - ball_to ).th() );
    }

    return ball_to;
}

/*-------------------------------------------------------------------*/
Vector2D
Bhv_PenaltyShootoutsGoalie::getBoundPredictBallPos( const WorldModel & wm,
                                                    int predict_step,
                                                    double field_back_offset )
{
    Vector2D current_pos = wm.ball().pos();
    Vector2D predict_pos = wm.ball().inertiaPoint( predict_step );

    const double wid = ServerParam::i().pitchHalfWidth();
    const double len = ServerParam::i().pitchHalfLength();

    const Vector2D corner_1( +len, +wid );
    const Vector2D corner_2( +len, -wid );
    const Vector2D corner_3( -len, -wid );
    const Vector2D corner_4( -len, +wid );

    const Rect2D pitch_rect = Rect2D::from_center( Vector2D( 0.0, 0.0 ),
                                                   len * 2, wid * 2 );

    if ( ! pitch_rect.contains( current_pos )
         && ! pitch_rect.contains( predict_pos ) )
    {
        const Vector2D result( bound( -len, current_pos.x, +len ),
                               bound( -wid, current_pos.y, +wid ) );

        dlog.addText( Logger::TEAM,
                      __FILE__": getBoundPredictBallPos "
                      "out of field, "
                      "current_pos = [%f, %f], predict_pos = [%f, %f], "
                      "result = [%f, %f]",
                      current_pos.x, current_pos.y,
                      predict_pos.x, predict_pos.y,
                      result.x, result.y );

        return result;
    }


    Vector2D p0 = predict_pos;
    Vector2D p1;
    Vector2D p2;
    Vector2D p3;
    Vector2D p4;

    p1 = getBallFieldLineCrossPoint( current_pos, p0, corner_1, corner_2, field_back_offset );
    p2 = getBallFieldLineCrossPoint( current_pos, p1, corner_2, corner_3, field_back_offset );
    p3 = getBallFieldLineCrossPoint( current_pos, p2, corner_3, corner_4, field_back_offset );
    p4 = getBallFieldLineCrossPoint( current_pos, p3, corner_4, corner_1, field_back_offset );

    dlog.addText( Logger::TEAM,
                  __FILE__": getBoundPredictBallPos "
                  "current_pos = [%f, %f], predict_pos = [%f, %f], "
                  "result = [%f, %f]",
                  current_pos.x, current_pos.y,
                  predict_pos.x, predict_pos.y,
                  p4.x, p4.y );

    return p4;
}


/*-------------------------------------------------------------------*/
double
Bhv_PenaltyShootoutsGoalie::getTackleProbability( const Vector2D & body_relative_ball )
{
    const ServerParam & param = ServerParam::i();

    double tackle_length;

    if ( body_relative_ball.x > 0.0 )
    {
        if ( ServerParam::i().tackleDist() > EPS )
        {
            tackle_length = param.tackleDist();
        }
        else
        {
            return 0.0;
        }
    }
    else
    {
        if ( param.tackleBackDist() > EPS )
        {
            tackle_length = param.tackleBackDist();
        }
        else
        {
            return 0.0;
        }
    }

    if ( param.tackleWidth() < EPS )
    {
        return 0.0;
    }


    double prob = 1.0;

    // vertical dist penalty
    prob -= std::pow( body_relative_ball.absX() / tackle_length,
                      param.tackleExponent() );

    // horizontal dist penalty
    prob -= std::pow( body_relative_ball.absY() / param.tackleWidth(),
                      param.tackleExponent() );

    // don't allow negative value by calculation error.
    return std::max( 0.0, prob );
}

/*-------------------------------------------------------------------*/
Vector2D
Bhv_PenaltyShootoutsGoalie::getSelfNextPosWithDash( const WorldModel & wm,
                                                    double dash_power )
{
    return wm.self().inertiaPoint( 1 )
        + Vector2D::polar2vector
        ( dash_power * wm.self().playerType().dashPowerRate(),
          wm.self().body() );
}

double
Bhv_PenaltyShootoutsGoalie::getSelfNextTackleProbabilityWithDash( const WorldModel & wm,
                                                                  double dash_power )
{
    return getTackleProbability( ( wm.ball().inertiaPoint( 1 )
                                   - getSelfNextPosWithDash( wm, dash_power ) )
                                 .rotatedVector( - wm.self().body() ) );
}

double
Bhv_PenaltyShootoutsGoalie::getSelfNextTackleProbabilityWithTurn( const WorldModel & wm )
{
    Vector2D self_next = wm.self().pos() + wm.self().vel();
    Vector2D ball_next = wm.ball().pos() + wm.ball().vel();
    Vector2D ball_rel = ball_next - self_next;
    double ball_dist = ball_rel.r();

    if ( ball_dist > ServerParam::i().tackleDist()
         && ball_dist > ServerParam::i().tackleWidth() )
    {
        return 0.0;
    }

    double max_turn = wm.self().playerType().effectiveTurn( ServerParam::i().maxMoment(),
                                                            wm.self().vel().r() );
    AngleDeg ball_angle = ball_rel.th() - wm.self().body();
    ball_angle = std::max( 0.0, ball_angle.abs() - max_turn );

    return getTackleProbability( Vector2D::polar2vector( ball_dist, ball_angle ) );
}


/*-------------------------------------------------------------------*/
double
Bhv_PenaltyShootoutsGoalie::translateX( bool reverse, double x )
{
    if ( reverse )
    {
        return -x;
    }
    else
    {
        return x;
    }
}

Vector2D
Bhv_PenaltyShootoutsGoalie::translateXVec( bool reverse, const Vector2D & vec )
{
    if ( reverse )
    {
        return Vector2D( - vec.x, vec.y );
    }
    else
    {
        return vec;
    }
}

AngleDeg
Bhv_PenaltyShootoutsGoalie::translateTheta( bool reverse, const AngleDeg & theta )
{
    if ( reverse )
    {
        return theta + 180.0;
    }
    else
    {
        return theta;
    }
}

bool
Bhv_PenaltyShootoutsGoalie::translateXIsLessEqualsTo( bool reverse,
                                                      double x, double threshold )
{
    if ( reverse )
    {
        return (-x >= threshold);
    }
    else
    {
        return (x <= threshold);
    }
}


namespace rcsc_ext {
inline
double
ServerParam_getDistFromOurNearGoalPost( const rcsc::Vector2D & point )
{
    return std::min( point.dist( rcsc::Vector2D
                                 ( - rcsc::ServerParam::i().pitchHalfLength(),
                                   - rcsc::ServerParam::i().goalHalfWidth() ) ),
                     point.dist( rcsc::Vector2D
                                 ( - rcsc::ServerParam::i().pitchHalfLength(),
                                   + rcsc::ServerParam::i().goalHalfWidth() ) ) );
}
}

bool
Bhv_PenaltyShootoutsGoalie::canShootFrom( const WorldModel & wm,
                                          const Vector2D & pos,
                                          long valid_teammate_threshold )
{
    static const double SHOOTABLE_ANGLE_THRESHOLD = 12.0;
    static const double SHOOTABLE_LENGTH_THRESHOLD = 25.0;

    return (getFreeAngleFromPos( wm, valid_teammate_threshold, pos )
            >= SHOOTABLE_ANGLE_THRESHOLD
            && rcsc_ext::ServerParam_getDistFromOurNearGoalPost( pos )
            < SHOOTABLE_LENGTH_THRESHOLD);
}

double
Bhv_PenaltyShootoutsGoalie::getFreeAngleFromPos( const WorldModel & wm,
                                                 long valid_teammate_threshold,
                                                 const Vector2D & pos )
{
    const double goal_half_width = ServerParam::i().goalHalfWidth();
    const double field_half_length = ServerParam::i().pitchHalfLength();

    const Vector2D goal_center = ServerParam::i().ourTeamGoalPos();
    const Vector2D goal_left( goal_center.x, +goal_half_width );
    const Vector2D goal_right( goal_center.x, -goal_half_width );

    const Vector2D goal_center_left( field_half_length,
                                     (+ goal_half_width - 1.5) / 2.0 );

    const Vector2D goal_center_right( field_half_length,
                                      (- goal_half_width + 1.5) / 2.0 );


    double center_angle = getMinimumFreeAngle( wm,
                                               goal_center,
                                               valid_teammate_threshold,
                                               pos );

    double left_angle   = getMinimumFreeAngle( wm,
                                               goal_left,
                                               valid_teammate_threshold,
                                               pos );

    double right_angle  = getMinimumFreeAngle( wm,
                                               goal_right,
                                               valid_teammate_threshold,
                                               pos );

    double center_left_angle  = getMinimumFreeAngle
        ( wm,
          goal_center_left,
          valid_teammate_threshold,
          pos );

    double center_right_angle = getMinimumFreeAngle
        ( wm,
          goal_center_right,
          valid_teammate_threshold,
          pos );

    return std::max( center_angle,
                     std::max( left_angle,
                               std::max( right_angle,
                                         std::max( center_left_angle,
                                                   center_right_angle ) ) ) );
}


double
Bhv_PenaltyShootoutsGoalie::getMinimumFreeAngle( const WorldModel & wm,
                                                 const Vector2D & goal,
                                                 long valid_teammate_threshold,
                                                 const Vector2D & pos )
{
    AngleDeg test_dir = (goal - pos).th();

    double shoot_course_cone = +360.0;

    AbstractPlayerCont team_set;
    team_set = wm.getPlayerCont( new AndPlayerPredicate
                                 ( new TeammatePlayerPredicate
                                   ( wm ),
                                   new CoordinateAccuratePlayerPredicate
                                   ( valid_teammate_threshold ) ) );

    AbstractPlayerCont::const_iterator t_end = team_set.end();
    for ( AbstractPlayerCont::const_iterator t = team_set.begin();
          t != t_end;
          ++t )
    {
        double controllable_dist;

        if ( (*t)->goalie() )
        {
            controllable_dist = ServerParam::i().catchAreaLength();
        }
        else
        {
            controllable_dist = (*t)->playerTypePtr()->kickableArea();
        }

        Vector2D relative_player = (*t)->pos() - pos;

        double hide_angle_radian = ( std::asin
                                     ( std::min( controllable_dist
                                                 / relative_player.r(),
                                                 1.0 ) ) );
        double angle_diff = std::max( (relative_player.th() - test_dir).abs()
                                      - hide_angle_radian / M_PI * 180,
                                      0.0 );

        if ( shoot_course_cone > angle_diff )
        {
            shoot_course_cone = angle_diff;
        }
    }

    return shoot_course_cone;
}


/*-------------------------------------------------------------------*/
bool
Bhv_PenaltyShootoutsGoalie::doChaseBall( PlayerAgent * agent )
{
    const int self_reach_cycle = agent->world().interceptTable()->selfReachCycle();
    Vector2D intercept_point = agent->world().ball().inertiaPoint( self_reach_cycle );

    dlog.addText( Logger::TEAM,
                  __FILE__": (doChaseBall): "
                  "intercept point = [%f, %f]",
                  intercept_point.x, intercept_point.y );

    if ( intercept_point.x > ServerParam::i().ourTeamGoalLineX() - 1.0 )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": (doChaseBall): normal intercept" );

        if ( Body_Intercept().execute( agent ) )
        {
            agent->setNeckAction( new Neck_TurnToBall() );

            return true;
        }
    }


    //
    // go to goal line
    //
    const Vector2D predict_ball_pos = getBoundPredictBallPos( agent->world(),
                                                              self_reach_cycle,
                                                              1.0 );

    const double target_error = std::max( agent->world().self().catchableArea(),
                                          ServerParam::i().tackleDist() );

    dlog.addText( Logger::TEAM,
                  __FILE__": (doChaseBall): "
                  "goal line intercept point = [%f, %f]",
                  predict_ball_pos.x, predict_ball_pos.y );

    if ( Body_GoalieGoToPoint( predict_ball_pos,
                               target_error,
                               ServerParam::i().maxPower(),
                               true,
                               true,
                               true ).execute( agent ) )
    {
        agent->setNeckAction( new Neck_TurnToBall() );

        return true;
    }

    return false;
}


bool
Bhv_PenaltyShootoutsGoalie::isGoalLinePositioningSituation( const WorldModel & wm,
                                                            const Vector2D & ball_pos,
                                                            bool penalty_kick_mode )
{
    if ( penalty_kick_mode )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": (isGoalLinePositioningSituation): "
                      "penalty_kick_mode, return false" );
        return false;
    }

    if ( wm.defenseLineX() >= -15.0 )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": (isGoalLinePositioningSituation): "
                      "defense line(%f) too forward, "
                      "no need to goal line positioning",
                      wm.defenseLineX() );
        return false;
    }

    AngleDeg ball_dir = getDirFromGoal( ball_pos );

    dlog.addText( Logger::TEAM,
                  __FILE__": (isGoalLinePositioningSituation) "
                  "ball direction from goal = %f",
                  ball_dir.degree() );

#if 0
# if 1
    bool is_side_angle = ( ball_dir.abs() > 30.0 );
# elif 0
    bool is_side_angle = ( ball_dir.abs() > 35.0 );
# elif 1
    bool is_side_angle = ( ball_dir.abs() > 40.0 );
# else
    bool is_side_angle = ( ball_dir.abs() > 45.0 );
# endif
#else
    bool is_side_angle = ( ball_dir.abs() > 50.0 );
#endif

    if ( is_side_angle )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": (isGoalLinePositioningSituation) "
                      "side angle, not goal line positioning" );

        return false;
    }

    if ( wm.defenseLineX() <= -15.0 )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": (isGoalLinePositioningSituation): "
                      "defense line(%f) too back, "
                      "return true",
                      wm.defenseLineX() );
        return true;
    }


    dlog.addText( Logger::TEAM,
                  __FILE__": (isGoalLinePositioningSituation) "
                  "return false" );

    return false;
}
