// -*-c++-*-

/*!
  \file stamina_model.cpp
  \brief player's stamina model Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 3 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "stamina_model.h"

#include "body_sensor.h"
#include "fullstate_sensor.h"

#include <rcsc/common/server_param.h>
#include <rcsc/common/player_type.h>
#include <rcsc/game_time.h>

#include <algorithm>

namespace rcsc {

/*-------------------------------------------------------------------*/
/*!

*/
StaminaModel:: StaminaModel()
    : M_stamina( ServerParam::DEFAULT_STAMINA_MAX )
    , M_effort( ServerParam::DEFAULT_EFFORT_INIT )
    , M_recovery( ServerParam::DEFAULT_RECOVER_INIT )
{

}

/*-------------------------------------------------------------------*/
/*!

*/
void
StaminaModel::init( const PlayerType & player_type )
{
    M_stamina = ServerParam::i().staminaMax();
    M_effort = player_type.effortMax();
    M_recovery = ServerParam::i().recoverInit();
    M_stamina_capacity = ServerParam::i().staminaCapacity();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
StaminaModel::update( const PlayerType & player_type,
                      const double & dash_power )
{
    // substitute dash power from stamina value
    if ( dash_power >= 0.0 )
    {
        M_stamina -= dash_power;
    }
    else
    {
        M_stamina -= dash_power * -2.0;
    }

    // update recovery value
    if ( M_stamina <= ServerParam::i().recoverDecThrValue() )
    {
        if ( M_recovery > ServerParam::i().recoverMin() )
        {
            M_recovery = std::max( ServerParam::i().recoverMin(),
                                   M_recovery - ServerParam::i().recoverDec() );
            std::cerr << "recover dec " << M_recovery << std::endl;
        }
    }

    // update effort value
    // !!! using HETERO PLAYER PARAMS !!!
    if ( M_stamina <= ServerParam::i().effortDecThrValue() )
    {
        if ( M_effort > player_type.effortMin() )
        {
            M_effort = std::max( player_type.effortMin(),
                                 M_effort - ServerParam::i().effortDec() );
        }
    }
    else if ( M_stamina >= ServerParam::i().effortIncThrValue() )
    {
        if ( M_effort < player_type.effortMax() )
        {
            M_effort = std::min( player_type.effortMax(),
                                 M_effort + ServerParam::i().effortInc() );
        }
    }

    // recover stamina value & update stamina capacity

    double max_recover = std::min( std::max( M_stamina_capacity,
                                             player_type.extraStamina() ),
                                   M_recovery * player_type.staminaIncMax() );
    double new_stamina = std::min( ServerParam::i().staminaMax(),
                                   M_stamina + max_recover );
    M_stamina_capacity = std::max( 0.0,
                                   M_stamina_capacity - ( new_stamina - M_stamina ) );
    M_stamina = new_stamina;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
StaminaModel::updateAfterSense( const BodySensor & body_sensor,
                                const GameTime & current )
{
    // set directly
    M_stamina = body_sensor.stamina();
    M_effort = body_sensor.effort();
    M_stamina_capacity = body_sensor.staminaCapacity();

    const ServerParam & param = ServerParam::i();

    // reset recover value, when new harf start

    if ( param.halfTime() < 0 // server setting is normal game mode
         && param.nrNormalHalfs() >= 0
         && current.cycle() <= param.halfTime() * param.nrNormalHalfs() + 1
         && current.cycle() % param.halfTime() == 1 ) // just after kickoff
    {
        M_recovery = ServerParam::i().recoverInit();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
StaminaModel::updateAfterFullstate( const FullstateSensor::PlayerT & state )
{
    M_stamina = state.stamina_;
    M_effort = state.effort_;
    M_recovery = state.recovery_;
    M_stamina_capacity = state.stamina_capacity_;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
StaminaModel::simulateWait( const PlayerType & player_type,
                            const int n_wait )
{
    const ServerParam & SP = ServerParam::i();

    for ( int i = 0; i < n_wait; ++i )
    {
        // update effort
        if ( M_stamina <= SP.effortDecThrValue() )
        {
            if ( M_effort > player_type.effortMin() )
            {
                M_effort -= SP.effortDec();
                if ( M_effort < player_type.effortMin() )
                {
                    M_effort = player_type.effortMin();
                }
            }
        }
        else if ( M_stamina > SP.effortIncThrValue() )
        {
            if ( M_effort < player_type.effortMax() )
            {
                M_effort += SP.effortInc();
                if ( M_effort > player_type.effortMax() )
                {
                    M_effort = player_type.effortMax();
                }
            }
        }

        // update stamina
        double max_recover = std::min( std::max( M_stamina_capacity,
                                                 player_type.extraStamina() ),
                                       M_recovery * player_type.staminaIncMax() );
        double new_stamina = std::min( SP.staminaMax(),
                                       M_stamina + max_recover );

        // update stamina capacity
        M_stamina_capacity = std::max( 0.0,
                                       M_stamina_capacity - ( new_stamina - M_stamina ) );
        M_stamina = new_stamina;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
StaminaModel::simulateDash( const PlayerType & player_type,
                            const int n_dash,
                            const double & dash_power )
{
    const ServerParam & SP = ServerParam::i();
    const double consumption = ( dash_power > 0.0
                                 ? dash_power
                                 : dash_power * -2.0 );

    for ( int i = 0; i < n_dash; ++i )
    {
        // decrease stamina
        M_stamina -= consumption;
        M_stamina = std::max( 0.0, M_stamina );

        // update recovery
        if ( M_stamina <= SP.recoverDecThrValue() )
        {
            if ( M_recovery > SP.recoverMin() )
            {
                M_recovery -= SP.recoverDec();
                M_recovery = std::max( M_recovery, SP.recoverMin() );
            }
        }

        simulateWait( player_type, 1 );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
StaminaModel::simulate( const PlayerType & player_type,
                        const int n_wait,
                        const int n_dash,
                        const double & dash_power )
{
    simulateWait( player_type, n_wait );
    simulateDash( player_type, n_dash, dash_power );
}

}
