// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "role_442_side_back.h"

#include "strategy.h"

#include "bhv_cross.h"
#include "bhv_basic_defensive_kick.h"
#include "bhv_basic_offensive_kick.h"
#include "bhv_keep_shoot_chance.h"

#include "bhv_basic_tackle.h"

#include "bhv_basic_move.h"
#include "bhv_defender_basic_block_move.h"
#include "bhv_danger_area_tackle.h"
#include "bhv_block_dribble.h"
#include "bhv_block_side_attack.h"

#include "body_advance_ball_test.h"
#include <rcsc/action/body_dribble2008.h>
#include <rcsc/action/body_intercept2008.h>
#include <rcsc/action/body_hold_ball2008.h>
#include "bhv_pass_test.h"

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/bhv_go_to_point_look_ball.h>
#include <rcsc/action/bhv_scan_field.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>
#include <rcsc/action/neck_turn_to_player_or_scan.h>
#include <rcsc/action/neck_turn_to_low_conf_teammate.h>

#include <rcsc/formation/formation.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/player/debug_client.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/line_2d.h>
#include <rcsc/geom/ray_2d.h>
#include <rcsc/soccer_math.h>
#include <rcsc/math_util.h>

//#include "bhv_side_back_cross_block_move.h"
#include "bhv_side_back_aggressive_cross_block.h"
#include "bhv_side_back_danger_move.h"
#include "bhv_side_back_defensive_move.h"
#include "bhv_side_back_stopper_move.h"
#include "bhv_mark_pass_line.h"

#include "neck_default_intercept_neck.h"
#include "neck_offensive_intercept_neck.h"

using namespace rcsc;

const std::string Role442SideBack::NAME( "442SideBack" );

/*-------------------------------------------------------------------*/
/*!

 */
bool
Role442SideBack::execute( rcsc::PlayerAgent * agent )
{
    //////////////////////////////////////////////////////////////
    // play_on play
    bool kickable = agent->world().self().isKickable();
    if ( agent->world().existKickableTeammate()
         && agent->world().teammatesFromBall().front()->distFromBall()
         < agent->world().ball().distFromSelf() )
    {
        kickable = false;
    }

    if ( kickable )
    {
        // kick
        doKick( agent );
    }
    else
    {
        // positioning
        doMove( agent );
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
Role442SideBack::doKick( rcsc::PlayerAgent* agent )
{
    const double nearest_opp_dist
        = agent->world().getDistOpponentNearestToSelf( 5 );

    switch ( Strategy::get_ball_area(agent->world().ball().pos()) ) {
    case Strategy::BA_CrossBlock:
        if ( nearest_opp_dist < 3.0 )
        {
            rcsc::dlog.addText( rcsc::Logger::ROLE,
                                __FILE__": doKick() danger" );
            if ( ! rcsc::Bhv_PassTest().execute( agent ) )
            {
                Body_AdvanceBallTest().execute( agent );

                if ( agent->effector().queuedNextBallKickable() )
                {
                    agent->setNeckAction( new rcsc::Neck_ScanField() );
                }
                else
                {
                    agent->setNeckAction( new rcsc::Neck_TurnToBall() );
                }
            }
        }
        else
        {
            Bhv_BasicDefensiveKick().execute( agent );
        }
        break;
    case Strategy::BA_Stopper:
    case Strategy::BA_Danger:
        if ( nearest_opp_dist < 3.0 )
        {
            rcsc::dlog.addText( rcsc::Logger::ROLE,
                                __FILE__": doKick(). very danger" );
            if ( ! rcsc::Bhv_PassTest().execute( agent ) )
            {
                Body_AdvanceBallTest().execute( agent );

                if ( agent->effector().queuedNextBallKickable() )
                {
                    agent->setNeckAction( new rcsc::Neck_ScanField() );
                }
                else
                {
                    agent->setNeckAction( new rcsc::Neck_TurnToBall() );
                }
            }
        }
        else
        {
            Bhv_BasicDefensiveKick().execute( agent );
        }
        break;
    case Strategy::BA_DribbleBlock:
    case Strategy::BA_DefMidField:
        Bhv_BasicDefensiveKick().execute( agent );
        break;
    case Strategy::BA_DribbleAttack:
    case Strategy::BA_OffMidField:
        Bhv_BasicOffensiveKick().execute( agent );
        break;
    case Strategy::BA_Cross:
        Bhv_Cross().execute( agent );
        break;
    case Strategy::BA_ShootChance:
        Bhv_KeepShootChance().execute( agent );
        break;
    default:
        rcsc::dlog.addText( rcsc::Logger::ROLE,
                            __FILE__": unknown ball area" );
        rcsc::Body_HoldBall2008().execute( agent );
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
Role442SideBack::doMove( rcsc::PlayerAgent * agent )
{
    rcsc::Vector2D home_pos = Strategy::i().getPosition( agent->world().self().unum() );
    if ( ! home_pos.valid() ) home_pos.assign( 0.0, 0.0 );

    switch ( Strategy::get_ball_area( agent->world() ) ) {
    case Strategy::BA_CrossBlock:
        //Bhv_SideBackCrossBlockMove().execute( agent );
        Bhv_SideBackAggressiveCrossBlock().execute( agent );
        break;
    case Strategy::BA_Stopper:
        Bhv_SideBackStopperMove().execute( agent );
        break;
    case Strategy::BA_Danger:
        Bhv_SideBackDangerMove().execute( agent );
        break;
    case Strategy::BA_DribbleBlock:
        Bhv_SideBackDefensiveMove().execute( agent );
        break;
    case Strategy::BA_DefMidField:
        Bhv_SideBackDefensiveMove().execute( agent );
        break;
    case Strategy::BA_DribbleAttack:
    case Strategy::BA_OffMidField:
    case Strategy::BA_Cross:
    case Strategy::BA_ShootChance:
        doBasicMove( agent, home_pos );
        break;
    default:
        rcsc::dlog.addText( rcsc::Logger::ROLE,
                            __FILE__": unknown ball area" );
        Bhv_BasicMove( home_pos ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
Role442SideBack::doBasicMove( rcsc::PlayerAgent * agent,
                              const rcsc::Vector2D & home_pos )
{
    rcsc::dlog.addText( rcsc::Logger::ROLE,
                        __FILE__": doBasicMove" );

    const rcsc::WorldModel & wm = agent->world();

    //-----------------------------------------------
    // tackle
    if ( Bhv_BasicTackle( 0.85, 60.0 ).execute( agent ) )
    {
        return;
    }

    //--------------------------------------------------
    // intercept

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( self_min <= 3
         || ( self_min < 20
              && ( self_min < mate_min
                   || ( self_min < mate_min + 3 && mate_min > 3 ) )
              && self_min <= opp_min + 1 )
         || ( self_min < mate_min
              && opp_min >= 2
              && self_min <= opp_min + 1 )
         )
    {
        rcsc::dlog.addText( rcsc::Logger::ROLE,
                            __FILE__": doBasicMove() intercept" );
        rcsc::Body_Intercept2008().execute( agent );

        if ( opp_min >= self_min + 3 )
        {
            agent->setNeckAction( new Neck_OffensiveInterceptNeck() );
        }
        else
        {
            agent->setNeckAction( new Neck_DefaultInterceptNeck
                                  ( new Neck_TurnToBallOrScan() ) );
        }

        return;
    }

    /*--------------------------------------------------------*/

    if ( Bhv_MarkPassLine().execute( agent ) )
    {
        return;
    }

    // decide move target point
    rcsc::Vector2D target_point = getBasicMoveTarget( agent, home_pos );

    // decide dash power
    double dash_power = Strategy::get_defender_dash_power( wm, target_point );
    // decide threshold
    double dist_thr = std::fabs( wm.ball().pos().x - wm.self().pos().x ) * 0.1; //wm.ball().distFromSelf() * 0.1;
    if ( dist_thr < 1.0 ) dist_thr = 1.0;

    agent->debugClient().addMessage( "%.0f", dash_power );
    agent->debugClient().setTarget( target_point );
    agent->debugClient().addCircle( target_point, dist_thr );

    if ( ! rcsc::Body_GoToPoint( target_point, dist_thr, dash_power ).execute( agent ) )
    {
        rcsc::AngleDeg body_angle = ( wm.ball().pos().y < wm.self().pos().y
                                      ? -90.0
                                      : 90.0 );
        rcsc::Body_TurnToAngle( body_angle ).execute( agent );
    }

    if ( wm.ball().distFromSelf() < 20.0
         && ( wm.existKickableOpponent()
              || opp_min <= 3 )
         )
    {
        agent->setNeckAction( new rcsc::Neck_TurnToBall() );
    }
    else
    {
        agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
rcsc::Vector2D
Role442SideBack::getBasicMoveTarget( rcsc::PlayerAgent * agent,
                                     const rcsc::Vector2D & home_pos )
{
    const rcsc::WorldModel & wm = agent->world();

    rcsc::Vector2D target_point = home_pos;

    const int opp_min = wm.interceptTable()->opponentReachCycle();

    const rcsc::Vector2D opp_trap_pos = ( opp_min < 100
                                          ? wm.ball().inertiaPoint( opp_min )
                                          : wm.ball().inertiaPoint( 20 ) );

    // decide wait position
    // adjust to the defence line

    if ( wm.ball().pos().x < 10.0
         && -36.0 < home_pos.x
         && home_pos.x < 10.0
         && wm.self().pos().x > home_pos.x
         && wm.ball().pos().x > wm.self().pos().x + 3.0 )
    {
        // make static line
        double tmpx = home_pos.x;
        for ( double x = -12.0; x > -27.0; x -= 8.0 )
        {
            if ( opp_trap_pos.x > x )
            {
                tmpx = x - 3.3;
                break;
            }
        }

        if ( std::fabs( wm.self().pos().y - opp_trap_pos.y ) > 5.0 )
        {
            tmpx -= 3.0;
        }
        target_point.x = tmpx;

        agent->debugClient().addMessage( "LineDef%.1f", tmpx );
    }
    else
    {
        agent->debugClient().addMessage( "SB:Normal" );
    }

    return target_point;
}

/*-------------------------------------------------------------------*/
/*!

 */
namespace {
rcss::RegHolder role = SoccerRole::creators().autoReg( &Role442SideBack::create,
                                                       Role442SideBack::NAME );
}
