// -*-c++-*-

/*!
  \file predict_player_object.h
  \brief predict player object class Header File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA, Hiroki SHIMORA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_PREDICT_PLAYER_OBJECT_H
#define RCSC_PREDICT_PLAYER_OBJECT_H

#include <rcsc/player/abstract_player_object.h>

#include <rcsc/common/player_type.h>
#include <rcsc/geom/vector_2d.h>
#include <rcsc/geom/angle_deg.h>
#include <rcsc/types.h>

namespace rcsc {

/*!
  \class PredictPlayerObject
  \brief predict player object class
*/
class PredictPlayerObject
    : public AbstractPlayerObject {
protected:

    bool M_is_valid;
    bool M_is_self;
    bool M_is_ghost;
    int M_ghost_count;

    /*!
      \brief initialize member variables using abstract player object
      \param pl base player information
    */
    void copyPlayerInfo( const AbstractPlayerObject & pl )
      {
          M_is_valid = true;
          M_is_self = pl.isSelf();
          M_is_ghost = pl.isGhost();
          M_ghost_count = pl.ghostCount();

          M_side = pl.side();
          M_unum = pl.unum();
          M_unum_count = pl.unumCount();
          M_goalie = pl.goalie();

          M_type = pl.type();
          M_player_type = pl.playerTypePtr();

          M_pos = pl.pos();
          M_pos_count = pl.posCount();

          M_seen_pos = pl.seenPos();
          M_seen_pos_count = pl.seenPosCount();

          M_heard_pos = pl.heardPos();
          M_heard_pos_count = pl.heardPosCount();

          M_vel = pl.vel();
          M_vel_count = pl.velCount();

          M_seen_vel = pl.seenVel();
          M_seen_vel_count = pl.seenVelCount();

          M_body = pl.body();
          M_body_count = pl.bodyCount();

          M_face = pl.face();
          M_face_count = pl.faceCount();

          M_dist_from_ball = pl.distFromBall();
      }

public:

    /*!
      \brief initialize member variables using abstract player object
      \param pl base player information
    */
    PredictPlayerObject()
        : AbstractPlayerObject()
        , M_is_valid( false )
        , M_is_self( false )
        , M_is_ghost( false )
        , M_ghost_count( 0 )
      {
      }

    /*!
      \brief initialize member variables using abstract player object
      \param pl base player information
    */
    PredictPlayerObject( const AbstractPlayerObject & player )
      {
          copyPlayerInfo( player );
      }

    /*!
      \brief initialize member variables using abstract player object
      \param player base player information
      \param override_pos position to overriding
    */
    PredictPlayerObject( const AbstractPlayerObject & player,
                         const Vector2D & override_pos )
      {
          copyPlayerInfo( player );

          M_pos = override_pos;
      }

    /*!
      \brief destructor. nothing to do
    */
    virtual
    ~PredictPlayerObject()
      { }


    // ------------------------------------------
    /*!
      \brief check if this player is valid or not
      \return true if this player is valid
     */
    virtual
    bool valid() const
      {
          return M_is_valid;
      }

    // ------------------------------------------
    /*!
      \brief check if this player is self or not
      \return true if this player is self
     */
    virtual
    bool isSelf() const
      {
          return M_is_self;
      }

    /*!
      \brief check if this player is ghost object or not
      \return true if this player may be ghost object
     */
    virtual
    bool isGhost() const
      {
          return M_is_ghost;
      }

    /*!
      \brief get the counter value as a ghost recognition
      \return count as a ghost recognition
     */
    virtual
    int ghostCount() const
      {
          return M_ghost_count;
      }

    // ------------------------------------------
    /*!
      \brief set positon
      \param p new position
     */
    virtual
    void setPos( const rcsc::Vector2D & p )
      {
          M_pos = p;
      }
};

}

#endif
