// -*-c++-*-

/*!
  \file bhv_cross_area_kick.cpp
  \brief cross area kick behavior.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_cross_area_kick.h"

#include "strategy.h"
#include "bhv_cross.h"
#include "bhv_pass_test.h"
#include "body_kick_to_corner.h"

#include <rcsc/action/body_dribble2008.h>
#include <rcsc/action/body_hold_ball2008.h>

#include <rcsc/action/neck_turn_to_point.h>
#include <rcsc/action/neck_turn_to_low_conf_teammate.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/common/player_type.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/logger.h>
#include <rcsc/geom/sector_2d.h>

using namespace rcsc;

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_CrossAreaKick::execute( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    dlog.addText( Logger::TEAM,
                  __FILE__": Bhv_CrossAreaKick" );

    if ( ! wm.self().isKickable() )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " not ball kickable!"
                  << std::endl;
        dlog.addText( Logger::TEAM,
                      __FILE__":  not kickable" );
        return false;
    }

    //-------------------------------------------------------
    // !!!check stamina!!!
    static bool S_recover_mode = false;
    if ( wm.self().stamina()
         < ServerParam::i().recoverDecThrValue() + 200.0 )
    {
        S_recover_mode = true;
    }
    else if ( wm.self().stamina()
              > ServerParam::i().staminaMax() * 0.6 )
    {
        S_recover_mode = false;
    }


    const Vector2D next_self_pos = wm.self().pos() + wm.self().vel();

    //-------------------------------------------------------
    // check opponent fielder & goalie distance
    const PlayerObject * nearest_opp = wm.getOpponentNearestToSelf( 10 );
    const double opp_dist
        = ( nearest_opp
            ? nearest_opp->distFromSelf()
            : 1000.0 );
    const Vector2D opp_pos
        = ( nearest_opp
            ? nearest_opp->pos() + nearest_opp->vel()
            : Vector2D( -1000.0, 0.0 ) );

    const PlayerObject * opp_goalie = wm.getOpponentGoalie();
    bool goalie_near = false;
    bool goalie_verynear = false;
    if ( opp_goalie )
    {
        if ( opp_goalie->pos().x > ServerParam::i().theirPenaltyAreaLineX()
             && opp_goalie->pos().absY() < ServerParam::i().penaltyAreaHalfWidth() )
        {
            if ( opp_goalie->distFromBall()
                 < ( ServerParam::i().catchAreaLength()
                     + ServerParam::i().defaultPlayerSpeedMax() * 4.0 ) )
            {
                goalie_near = true;
            }
            if ( opp_goalie->distFromBall()
                 < ( ServerParam::i().catchAreaLength()
                     + ServerParam::i().defaultPlayerSpeedMax() * 2.0 ) )
            {
                goalie_verynear = true;
            }
        }
    }

    //-------------------------------------------------------
    if ( goalie_verynear )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": goalie very near. keep away from goalie " );
        // kick to the far side corner
        agent->debugClient().addMessage( "XAreaGKNear" );
        Body_KickToCorner( ( wm.self().pos().y > 0.0 )
                           ).execute( agent );
        agent->setNeckAction( new Neck_TurnToLowConfTeammate() );
        return true;
    }

    //-------------------------------------------------------

    const Bhv_PassTest::PassRoute * pass = Bhv_PassTest::get_best_pass( wm );

    const bool receiver_free
        = ( pass
            && ! wm.existOpponentIn( Circle2D( pass->receiver_->pos(), 5.0 ),
                                     10,
                                     true ) );

    if ( pass
         && pass->receive_point_.x > 37.0
         && receiver_free
         && pass->receive_point_.absY() < 12.0
         && ! wm.existOpponentIn( Circle2D( pass->receive_point_, 6.0 ),
                                  10,
                                  true )
         && ( ! opp_goalie
              || opp_goalie->pos().dist( pass->receive_point_ ) > 6.0 )
         )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": CrossAreaKick. very chance pass." );
        agent->debugClient().addMessage( "XAreaPassChance1" );
        Bhv_PassTest().execute( agent );
        return true;
    }

    if ( pass )
    {
        Rect2D shoot_rect = Rect2D::from_center( 52.5 - 7.0, 0.0,
                                                 14.0, 32.0 );
        if ( shoot_rect.contains( pass->receive_point_ )
             && receiver_free
             && wm.countOpponentsIn( shoot_rect, 10, false ) <= 2
             && ! wm.existOpponentIn( Circle2D( pass->receive_point_, 5.0 ),
                                      10
                                      , true ) )
        {
            dlog.addText( Logger::ROLE,
                          __FILE__": CrossAreaKick. very chance pass(2)." );
            agent->debugClient().addMessage( "XAreaPassChance2" );
            Bhv_PassTest().execute( agent );
            return true;
        }
    }

    if ( pass
         && pass->receive_point_.x > 39.0
         && pass->receive_point_.absY() < 7.0 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": can cross pass." );
        agent->debugClient().addMessage( "XAreaPass1" );
        Bhv_PassTest().execute( agent );
        return true;
    }

    // very side attack
    if ( pass
         && wm.self().pos().absY() > 27.0
         && opp_dist < 4.0
         && pass->receive_point_.x > 26.0 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": very side attack.  cross" );
        agent->debugClient().addMessage( "XAreaPass2" );
        Bhv_PassTest().execute( agent );
        return true;
    }

    //-------------------------------------------------------
    // cross to center
    // exist teammate at cross area
    if ( opp_dist < 3.0
         && Bhv_Cross::get_best_point( agent, NULL ) )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": teammate in cross area. cross to center" );
        agent->debugClient().addMessage( "XAreaCross1" );
        Bhv_Cross().execute( agent );
        return true;
    }

    //-------------------------------------------------------
    // dribble cource is blocked
    if ( pass
         && receiver_free
         && wm.self().pos().x > 40.0
         && wm.self().pos().absY() > 20.0 )
    {
        Vector2D rect_center( 47.0,
                              ( wm.self().pos().absY() + 9.0 ) * 0.5 );
        if ( wm.self().pos().y < 0.0 ) rect_center.y *= -1.0;

        Rect2D side_rect
            = Rect2D::from_center( rect_center,
                                   10.0,
                                   wm.self().pos().absY() - 9.0 );
        agent->debugClient().addRectangle( side_rect );
        if ( wm.countOpponentsIn( side_rect, 10, false ) >= 3 ) // 2 )
        {
            dlog.addText( Logger::ROLE,
                          __FILE__": CrossAreaKick. dribble cource blocked." );
            agent->debugClient().addMessage( "XAreaPassBlocked" );
            Bhv_PassTest().execute( agent );
            return true;
        }
    }

    //-------------------------------------------------------
    if ( S_recover_mode
         && ! goalie_near
         && opp_dist > 4.0 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": recovering" );
        agent->debugClient().addMessage( "XAreaHold" );
        Body_HoldBall2008().execute( agent );
        agent->setNeckAction( new Neck_TurnToLowConfTeammate() );
        return true;
    }

    //-------------------------------------------------------
    // keep dribble
    if ( ! goalie_near
         // && opp_dist > ServerParam::i().tackleDist() + 0.5
         // 2008-07-19
         && opp_dist > ServerParam::i().tackleDist() + 1.0
         //          && ( ( opp_dist > ServerParam::i().tackleDist() + 0.5
         //                 && opp_pos.x < next_self_pos.x - 0.5 )
         //               || opp_dist > ServerParam::i().tackleDist() + 1.5 )
         )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": goalie not near and not tackle opp. wing dribble. opp_dist=%.2f",
                      opp_dist );
        agent->debugClient().addMessage( "XAreaDrib1" );
        doDribble( agent );
        return true;
    }

    //-------------------------------------------------------
    // forcely pass
    if ( pass )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": enforce pass" );
        agent->debugClient().addMessage( "XAreaPass3" );
        Bhv_PassTest().execute( agent );
        return true;
    }

    //-------------------------------------------------------
    // not exist opp on my body dir
    Sector2D body_sector( wm.self().pos(),
                          0.5, 8.0,
                          wm.self().body() - 25.0,
                          wm.self().body() + 25.0 );
    // oponent check with goalie
    if ( ! wm.existOpponentIn( body_sector, 10, true ) )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": exsit opp in my body dir. wing dribble" );
        agent->debugClient().addMessage( "XAreaDrib2" );
        doDribble( agent );
        return true;
    }

    //-------------------------------------------------------
    // forcely cross
    if ( S_recover_mode && opp_dist < 2.5 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": enforce cross" );
        agent->debugClient().addMessage( "XAreaCross2" );
        Bhv_Cross().execute( agent );
        return true;
    }

    doDribble( agent );

    return true;
}


/*-------------------------------------------------------------------*/
/*!

 */
void
Bhv_CrossAreaKick::doDribble( PlayerAgent * agent )
{
    dlog.addText( Logger::ROLE,
                  __FILE__": (doDribble)" );

    const Vector2D drib_target = getDribbleTarget( agent );
    AngleDeg target_angle = ( drib_target - agent->world().self().pos() ).th();

    // decide dash count & looking point
    int dash_count = getDribbleAttackDashStep( agent, target_angle );

    bool dodge = true;
    if ( agent->world().self().pos().x > 43.0
         && agent->world().self().pos().absY() < 20.0 )
    {
        dodge = false;
    }

    agent->debugClient().setTarget( drib_target );
    agent->debugClient().addMessage( "DribAtt%d", dash_count );

    Body_Dribble2008( drib_target,
                      1.0,
                      ServerParam::i().maxDashPower(),
                      dash_count,
                      dodge
                      ).execute( agent );

    if ( agent->world().dirCount( target_angle ) > 1 )
    {
        target_angle
            = ( drib_target - agent->effector().queuedNextMyPos() ).th();
        AngleDeg next_body = agent->effector().queuedNextMyBody();
        if ( ( target_angle - next_body ).abs() < 90.0 )
        {
            dlog.addText( Logger::ROLE,
                          __FILE__": SB Dribble LookDribTarget" );
            agent->debugClient().addMessage( "LookDribTarget" );
            agent->setNeckAction( new Neck_TurnToPoint( drib_target ) );
            return;
        }
    }

    agent->setNeckAction( new Neck_TurnToLowConfTeammate() );
}

/*-------------------------------------------------------------------*/
/*!

 */
Vector2D
Bhv_CrossAreaKick::getDribbleTarget( PlayerAgent * agent )
{
    const double base_x = 47.5; //50.0; // old 45.0
    const double second_x = 40.0;

    const WorldModel & wm = agent->world();

    Vector2D drib_target( Vector2D::INVALIDATED );

    bool goalie_near = false;
    //--------------------------------------------------
    // goalie check
    const PlayerObject * opp_goalie = agent->world().getOpponentGoalie();
    if ( opp_goalie
         && opp_goalie->pos().x > ServerParam::i().theirPenaltyAreaLineX()
         && opp_goalie->pos().absY() < ServerParam::i().penaltyAreaHalfWidth()
         && ( opp_goalie->distFromSelf()
              < ServerParam::i().catchAreaLength()
              + ServerParam::i().defaultPlayerSpeedMax() * 3.5 )
         )
    {
        goalie_near = true;
        dlog.addText( Logger::ROLE,
                      __FILE__": dribble. goalie close" );
    }

    //--------------------------------------------------
    // goalie is close
    if ( goalie_near
         && std::fabs( opp_goalie->pos().y - wm.self().pos().y ) < 3.0 )
    {
        drib_target.assign( base_x, 25.0 );
        if ( wm.self().pos().y < 0.0 ) drib_target.y *= -1.0;

        dlog.addText( Logger::ROLE,
                      __FILE__": getDribbleTarget. goalie colse. normal target" );

        return drib_target;
    }

    // goalie is safety
    const PlayerObject * nearest_opp = wm.getOpponentNearestToSelf( 6 );

    if ( nearest_opp
         && nearest_opp->distFromSelf() < 2.0
         && wm.self().body().abs() < 40.0 )
    {
        Vector2D tmp_target
            = wm.self().pos()
            + Vector2D::polar2vector( 5.0, wm.self().body() );
        if ( tmp_target.x < 50.0 )
        {
            Sector2D body_sector( wm.self().pos(),
                                  0.5, 10.0,
                                  wm.self().body() - 30.0,
                                  wm.self().body() + 30.0 );
            if ( ! wm.existOpponentIn( body_sector, 10, true ) )
            {
                dlog.addText( Logger::ROLE,
                              __FILE__": getDribbleTarget. opp near. dribble to my body dir" );
                return tmp_target;
            }
        }
    }


    // cross area
    if ( wm.self().pos().x > second_x )
    {
        Vector2D rect_center( ( wm.self().pos().x + base_x ) * 0.5,
                              wm.self().pos().y < 0.0
                              ? wm.self().pos().y + 4.0
                              : wm.self().pos().y - 4.0 );
        Rect2D side_rect
            = Rect2D::from_center( rect_center, 6.0, 8.0 );
        agent->debugClient().addRectangle( side_rect );

        if ( wm.countOpponentsIn( side_rect, 10, false ) <= 1 )
        {
            drib_target.assign( base_x, 12.0 );
            dlog.addText( Logger::ROLE,
                          __FILE__": getDribbleTarget. cut in (%.1f, %.1f)",
                          drib_target.x, drib_target.y );
        }
        else
        {
            drib_target.assign( second_x,
                                min_max( 15.0,
                                         wm.self().pos().absY() - 1.0,
                                         25.0 ) );
            dlog.addText( Logger::ROLE,
                          __FILE__": getDribbleTarget. keep away (%.1f, %.1f)",
                          drib_target.x, drib_target.y );
        }

        if ( Strategy::i().getPositionType( wm.self().unum() ) == Position_Left )
        {
            drib_target.y *= -1.0;
        }


        if ( opp_goalie )
        {
            Vector2D drib_rel = drib_target - wm.self().pos();
            AngleDeg drib_angle = drib_rel.th();
            Line2D drib_line( wm.self().pos(), drib_target );
            if ( ( opp_goalie->angleFromSelf() - drib_angle ).abs() < 90.0
                 && drib_line.dist( opp_goalie->pos() ) < 4.0
                 )
            {
                double drib_dist = std::max( 0.0, opp_goalie->distFromSelf() - 5.0 );
                drib_target
                    = wm.self().pos()
                    + drib_rel.setLengthVector( drib_dist );

                dlog.addText( Logger::ROLE,
                              __FILE__": getDribbleTarget. attend goalie. dist=%.2f (%.1f, %.1f)",
                              drib_dist,
                              drib_target.x, drib_target.y );
            }
        }

        return drib_target;
    }

    if ( wm.self().pos().x > 36.0 )
    {
        if ( wm.self().pos().absY() < 5.0 )
        {
            drib_target.assign( base_x, 22.0 );
            dlog.addText( Logger::ROLE,
                          __FILE__": getDribbleTarget. keep away to (%.1f, %.1f)",
                          drib_target.x, drib_target.y );
        }
        else if ( wm.self().pos().absY() > 19.0 )
        {
            drib_target.assign( base_x, 12.0 );
            dlog.addText( Logger::ROLE,
                          __FILE__": getDribbleTarget. go their penalty area (%.1f, %.1f)",
                          drib_target.x, drib_target.y );
        }
        else
        {
            drib_target.assign( base_x, wm.self().pos().absY() );
            if ( drib_target.y > 30.0 ) drib_target.y = 30.0;
            dlog.addText( Logger::ROLE,
                          __FILE__": getDribbleTarget. go my Y (%.1f, %.1f)",
                          drib_target.x, drib_target.y );
        }

        if ( Strategy::i().getPositionType( wm.self().unum() ) == Position_Left )
        {
            drib_target.y *= -1.0;
        }

        return drib_target;
    }

    drib_target.assign( base_x, wm.self().pos().absY() );
    if ( drib_target.y > 30.0 ) drib_target.y = 30.0;

    if ( Strategy::i().getPositionType( wm.self().unum() ) == Position_Left )
    {
        drib_target.y *= -1.0;
    }

    dlog.addText( Logger::ROLE,
                  __FILE__": getDribbleTarget. normal target (%.1f, %.1f)",
                  drib_target.x, drib_target.y );

    return drib_target;
}

/*-------------------------------------------------------------------*/
/*!

 */
int
Bhv_CrossAreaKick::getDribbleAttackDashStep( const PlayerAgent * agent,
                                             const AngleDeg & target_angle )
{
    const WorldModel & wm = agent->world();

    bool goalie_near = false;
    //--------------------------------------------------
    // goalie check
    const PlayerObject * opp_goalie = wm.getOpponentGoalie();
    if ( opp_goalie
         && opp_goalie->pos().x > ServerParam::i().theirPenaltyAreaLineX()
         && opp_goalie->pos().absY() < ServerParam::i().penaltyAreaHalfWidth()
         && ( opp_goalie->distFromSelf()
              < ServerParam::i().catchAreaLength()
              + ServerParam::i().defaultPlayerSpeedMax() * 3.5 )
         )
    {
        goalie_near = true;
        dlog.addText( Logger::ROLE,
                      __FILE__": dribble. goalie close" );
    }

    const Sector2D target_sector( wm.self().pos(),
                                  0.5, 10.0,
                                  target_angle - 30.0,
                                  target_angle + 30.0 );
    const int default_step = 10;
    double dash_dist = 4.0;
    try
    {
        dash_dist = wm.self().playerType().dashDistanceTable().at( default_step );
        dash_dist += wm.self().playerType().inertiaTravel( wm.self().vel(), default_step ).r();
    }
    catch ( ... )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " Exception caught" << std::endl;
    }

    Vector2D next_point
        = wm.self().pos()
        + Vector2D::polar2vector( dash_dist, target_angle );
    dlog.addText( Logger::ROLE,
                  __FILE__": getDribbleAttackDashStep. my_dash_dsit= %.2f  point(%.1f %.1f)",
                  dash_dist, next_point.x, next_point.y  );

    bool exist_opp = false;

    const PlayerPtrCont::const_iterator o_end = wm.opponentsFromSelf().end();
    for ( PlayerPtrCont::const_iterator it = wm.opponentsFromSelf().begin();
          it != o_end;
          ++it )
    {
        if ( (*it)->posCount() >= 10 )
        {
            continue;
        }

        if ( ( (*it)->angleFromSelf() - target_angle ).abs() > 120.0 )
        {
            continue;
        }

        if ( (*it)->pos().dist( next_point ) <
             PlayerTypeSet::i().dummyType().realSpeedMax()
             * ( default_step + std::min( 4, (*it)->posCount() ) )
             + 1.0
             )
        {
            exist_opp = true;
            dlog.addText( Logger::ROLE,
                          __FILE__": getDribbleAttackDashStep. opp%d(%.1f %.1f) can reach my target point faster",
                          (*it)->unum(),
                          (*it)->pos().x, (*it)->pos().y );
            break;
        }

        if ( (*it)->distFromSelf() > 10.0 )
        {
            break;
        }
        if ( target_sector.contains( (*it)->pos() ) )
        {
            exist_opp = true;
            break;
        }
    }

    if ( exist_opp )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": getDribbleAttackDashStep. exist opp" );
        return 6;
    }

    if ( wm.self().pos().x > 36.0 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": getDribbleAttackDashStep. x > 36" );
        return 8;
        //return 3;
        //return 2;
    }

    dlog.addText( Logger::ROLE,
                  __FILE__": getDribbleAttackDashStep. default" );
    return 10;
    //return 3;
    //return 2;
}
