// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hiroki SHIMORA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "actgen_voronoi_pass.h"
#include <rcsc/geom/voronoi_diagram.h>

#include "simple_pass_checker.h"
#include "world_model_ext.h"
#include <rcsc/common/server_param.h>
#include <rcsc/common/logger.h>
#include <boost/shared_ptr.hpp>

//#define DEBUG_PRINT

using rcsc::dlog;

ActGen_VoronoiPass::ActGen_VoronoiPass()
{
}

ActGen_VoronoiPass::~ActGen_VoronoiPass()
{
}

// XXX: should unify with bhv_basic_kick.cpp
static
double
s_get_ball_speed_for_pass( const double & distance );

void
ActGen_VoronoiPass::addCandidates( std::vector< rcsc::ActionStatePair > * result,
                                   const rcsc::PredictState & state,
                                   const rcsc::WorldModel &,
                                   const std::vector< rcsc::ActionStatePair > & ) const
{
    static const int VALID_PLAYER_THRESHOLD = 8;

    const rcsc::AbstractPlayerObject * holder = state.ballHolder();

    static boost::shared_ptr< rcsc::VoronoiDiagram > voronoi;
    static rcsc::GameTime voronoi_time;

    if ( voronoi_time != state.time() )
    {
        voronoi_time = state.time();

        voronoi = boost::shared_ptr<rcsc::VoronoiDiagram>( new rcsc::VoronoiDiagram );

        const rcsc::PlayerCont::const_iterator end = state.opponents().end();
        for ( rcsc::PlayerCont::const_iterator it = state.opponents().begin();
              it != end;
              ++it )
        {
            if ( (*it).posCount() > VALID_PLAYER_THRESHOLD
                 || (*it).isGhost() )
            {
                continue;
            }

            voronoi->addPoint( (*it).pos() );
        }

        voronoi->compute();
    }

    dlog.addText( rcsc::Logger::PASS,
                  "voronoi_pass: ball = (%f, %f)",
                  state.ballPos().x, state.ballPos().y );

    std::vector< rcsc::Vector2D > cand;

    //
    // add pass candidates (point)
    //
    const rcsc::VoronoiDiagram::Vector2DCont & points = voronoi->resultPoints();
    const rcsc::VoronoiDiagram::Vector2DCont::const_iterator p_end = points.end();
    for ( rcsc::VoronoiDiagram::Vector2DCont::const_iterator it = points.begin();
          it != p_end;
          ++it )
    {
        cand.push_back( *it );
    }


    //
    // add pass candidates (on segment)
    //
    const rcsc::VoronoiDiagram::Segment2DCont & segments = voronoi->resultSegments();

    const rcsc::VoronoiDiagram::Segment2DCont::const_iterator s_end = segments.end();
    for ( rcsc::VoronoiDiagram::Segment2DCont::const_iterator it = segments.begin();
          it != s_end;
          ++it )
    {
        static int N = 8;
        for ( int d = 1; d < N; ++d )
        {
            cand.push_back( (*it).origin()
                            * static_cast<double>(d) / N
                            + (*it).terminal()
                            * static_cast<double>(N - d) / N );
        }
    }

    const rcsc::SimplePassChecker pass_check;

    const rcsc::PredictState::PlayerCont::const_iterator pl_end = state.allTeammates().end();
    for ( rcsc::PredictState::PlayerCont::const_iterator pl = state.allTeammates().begin();
          pl != pl_end;
          ++pl )
    {
        if ( ! (*pl).valid()
             || (*pl).posCount() > VALID_PLAYER_THRESHOLD
             || (*pl).isGhost()
             || (*pl).unum() == state.ballHolderUnum()
             || (*pl).unum() == -1 )
        {
            continue;
        }

#ifdef DEBUG_PRINT
        dlog.addText( rcsc::Logger::PASS,
                      "voronoi_pass: checking teammate %d, pos = (%f, %f)",
                      (*pl).unum(), (*pl).pos().x, (*pl).pos().y );
#endif

        const std::vector< rcsc::Vector2D >::const_iterator end = cand.end();
        for ( std::vector< rcsc::Vector2D >::const_iterator it = cand.begin();
              it != end;
              ++it )
        {
            const rcsc::Vector2D & receive_point = *it;

            if ( receive_point.absX() >= rcsc::ServerParam::i().pitchHalfLength()
                 || receive_point.absY() >= rcsc::ServerParam::i().pitchHalfWidth() )
            {
                continue;
            }

#ifdef DEBUG_PRINT
            dlog.addText( rcsc::Logger::PASS,
                          "voronoi_pass: receive_point = (%f, %f)",
                          receive_point.x, receive_point.y );
#endif

            rcsc::Vector2D teammate_pos = (*pl).pos();

            if ( (*pl).velCount() == 0 )
            {
                teammate_pos += (*pl).vel() * 2.0;
            }


            const double ball_dist = ( state.ballPos() - receive_point ).r();
            const double ball_first_speed = s_get_ball_speed_for_pass( ball_dist );
            const double receiver_dist = ( teammate_pos - receive_point ).r();

            long ball_step = rcsc::calc_length_geom_series
                             ( ball_first_speed,
                               ball_dist,
                               rcsc::ServerParam::i().ballDecay() );

            double receiver_step = (*pl).playerTypePtr()->cyclesToReachDistance( receiver_dist );

            if ( receiver_step + 3 > ball_step )
            {
                continue;
            }

            if ( ! pass_check( state, holder, &(*pl), receive_point ) )
            {
                continue;
            }

            result->push_back( rcsc::ActionStatePair
                               ( rcsc::CooperativeAction
                                 ( rcsc::CooperativeAction::ActionType::Pass,
                                   receive_point,
                                   (*pl).unum(),
                                   ball_first_speed,
                                   "voronoi" ),
                                 rcsc::PredictState
                                 ( state, ball_step,
                                   (*pl).unum(), receive_point ) ) );
        }
    }
}


static
double
s_get_ball_speed_for_pass( const double & distance )
{
    if ( distance >= 20.0 )
    {
        return 3.0;
    }
    else if ( distance >= 8.0 )
    {
        return 2.5;
    }
    else if ( distance >= 5.0 )
    {
        return 1.8;
    }
    else
    {
        return 1.5;
    }
}
