// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hiroki SHIMORA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "actgen_dribble_forward.h"

#include "world_model_ext.h"
#include <rcsc/common/logger.h>

static
bool
s_dribble_check( const rcsc::WorldModel & wm,
                 const rcsc::AbstractPlayerObject * from );

static
unsigned long
dribble_spend_time( double dribble_length );


void
ActGen_DribbleForward::addCandidates( std::vector< rcsc::ActionStatePair > * result,
                                      const rcsc::PredictState & state,
                                      const rcsc::WorldModel & current_wm,
                                      const std::vector< rcsc::ActionStatePair > & ) const
{
    static const double DRIBBLE_OFFSET = 10.0;

    const rcsc::AbstractPlayerObject * holder = state.ballHolder();

    //
    // add dribble candidate
    //
    if ( s_dribble_check( current_wm, holder ) )
    {
        rcsc::dlog.addText( rcsc::Logger::ACTION,
                            "holder = %d, dribble ok", holder->unum() );

        rcsc::Vector2D dribble_target;

        if ( holder->pos().x > + 35.0 )
        {
              dribble_target = rcsc_ext::ServerParam_opponentTeamFarGoalPostPos
                               ( holder->pos() );
        }
        else
        {
            dribble_target = holder->pos() + rcsc::Vector2D( 10.0, 0.0 );
        }

        rcsc::Vector2D dribble_predict_pos;
        if ( ( dribble_target - holder->pos() ).r() > DRIBBLE_OFFSET )
        {
            dribble_predict_pos = dribble_target;
        }
        else
        {
            dribble_predict_pos = holder->pos()
                                  + ( dribble_target - holder->pos() )
                                    .setLengthVector( DRIBBLE_OFFSET );
        }

        rcsc::PredictState new_state( current_wm,
                                      dribble_spend_time
                                      ( ( dribble_target - holder->pos() ).r() ),
                                      holder->unum(), dribble_predict_pos );

        result->push_back( rcsc::ActionStatePair
                           ( rcsc::CooperativeAction
                             ( rcsc::CooperativeAction::ActionType::Dribble,
                               dribble_target,
                               holder->unum() ),
                             new_state ) );
    }
    else
    {
        rcsc::dlog.addText( rcsc::Logger::ACTION,
                            "holder = %d, cannot dribble", holder->unum() );
    }
}


static
bool
s_dribble_check( const rcsc::WorldModel & wm,
                 const rcsc::AbstractPlayerObject * from )
{
    if ( (wm.gameMode().type() == rcsc::GameMode::KickOff_
          && wm.gameMode().side() == wm.ourSide())
         || (wm.gameMode().type() == rcsc::GameMode::FreeKick_
             && wm.gameMode().side() == wm.ourSide())
         || (wm.gameMode().type() == rcsc::GameMode::KickIn_
             && wm.gameMode().side() == wm.ourSide()))
    {
        return false;
    }

    const long VALID_PLAYER_ACCURACY = 20;
    const long VALID_OPPONENT_ACCURACY = VALID_PLAYER_ACCURACY;
    const double FREE_WIDTH = 1.75;

    if ( ! from
         || from->posCount() > VALID_PLAYER_ACCURACY
         || from->isGhost() )
    {
        return false;
    }

    if ( from->pos().x > wm.offsideLineX() )
    {
#if DEBUG_PRINT
        dlog.addText( rcsc::Logger::PASS,
                      "checking push %d: offside position",
                      from->unum() );
#endif
        return false;
    }


#if 1
    static const double WIDE_FREE_WIDTH = 10.0;
#else
    static const double WIDE_FREE_WIDTH = 5.0;
#endif

    if ( from->isSelf()
#if 0
         || from->pos().x >= wm.offsideLineX() - 20.0
#endif
       )
    {
        rcsc::AbstractPlayerCont opp_set;
        opp_set = wm.getPlayerCont( new rcsc::AndPlayerPredicate
                                    ( new rcsc::OpponentOrUnknownPlayerPredicate( wm ),
                                      new rcsc::CoordinateAccuratePlayerPredicate
                                      ( VALID_PLAYER_ACCURACY ),
                                      new rcsc::XCoordinateForwardPlayerPredicate
                                      ( from->pos().x ),
                                      new rcsc::XCoordinateBackwardPlayerPredicate
                                      ( wm.offsideLineX() + 1.0 ),
                                      new rcsc::YCoordinatePlusPlayerPredicate
                                      ( from->pos().y - WIDE_FREE_WIDTH ),
                                      new rcsc::YCoordinateMinusPlayerPredicate
                                      ( from->pos().y + WIDE_FREE_WIDTH ) ) );

        double free_length = rcsc_ext::AbstractPlayerCont_getMinimumEvaluation
                             ( opp_set,
                               new XPosPlayerEvaluator )
                             - from->pos().x;

#if 0
        if ( free_length >= 12.5 )
#else
        if ( free_length >= 5.0 )
#endif
        {
            rcsc::dlog.addText( rcsc::Logger::PASS,
                                "checking push %d: first check ok",
                                from->unum() );
            return true;
        }
    }


#if 0
    if ( from->pos().x <= wm.defenseLineX() + 10.0 )
    {
# if DEBUG_PRINT
        rcsc::dlog.addText( rcsc::Logger::PASS,
                            "checking push %d: near our defense line",
                            from->unum() );
# endif
        return false;
    }
#endif

#if 0
    if ( from->pos().x < rcsc_ext::WorldModel_forwardLineX( wm ) - 5.0 )
    {
# if DEBUG_PRINT
        rcsc::dlog.addText( rcsc::Logger::PASS,
                            "checking push %d: too far from our forward line: x = %.2f, forward line = %.2f",
                            from->unum(),
                            from->pos().x, rcsc_ext::WorldModel_forwardLineX( wm ) );
# endif
        return false;
    }
#endif

    rcsc::PlayerCont::const_iterator o_end = wm.opponents().end();
    for ( rcsc::PlayerCont::const_iterator opp = wm.opponents().begin();
          opp != o_end;
          ++opp )
    {
        if ( (*opp).posCount() > VALID_OPPONENT_ACCURACY )
        {
            continue;
        }

        if ( (from->pos().x < (*opp).pos().x
#if 1
              && (*opp).pos().x < from->pos().x + 5.0)
#else
              && (*opp).pos().x < from->pos().x + 2.0)
#endif
             && std::fabs( (*opp).pos().y - from->pos().y ) < FREE_WIDTH )
        {
# if DEBUG_PRINT
            rcsc::dlog.addText( rcsc::Logger::PASS,
                                "checking push %d: too narrow free width",
                                from->unum() );
# endif
            return false;
        }
    }

    return true;
}

static
unsigned long
dribble_spend_time( double dribble_length )
{
    return static_cast< unsigned long >( dribble_length * 2.0 );
}
